/********************
 * Callback functions
 ********************/
/*
 * Include declarations
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <glob.h>

#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>
#include <glib/gprintf.h>
#include <glib/gfileutils.h>
#include <pango/pango.h>

#include <libraw/libraw.h>

#include "interface.h"
#include "callbacks.h"
#include "../config.h"

/*
 * Load image from filename
 */
void load_image(app_data_t *myapp, gchar *imgfname) {
	GtkWidget *infobanner;
	GtkWidget *errdialog;
	GdkPixbufLoader *imgloader;
	libraw_data_t *rawdata;
	gboolean imgerr = FALSE;
	const gchar *paramformat = "<span size=\"xx-small\" weight=\"bold\" face=\"arial\">%s\n%s %s\n%-f s\nf/%-.1f\nISO %-.0f\n%-.0f mm</span>";

	g_free(myapp->currfname);
	myapp->currfname = g_strndup(g_path_get_basename(imgfname),MAXFNLEN);
	infobanner = hildon_banner_show_informationf(GTK_WIDGET(myapp->image),NULL,"Loading image '%s' ...",myapp->currfname);
	hildon_banner_set_timeout(HILDON_BANNER(infobanner),1000);
	rawdata = libraw_init(0);
	/* DEBUG */
	/* g_print("Supported RAW formats: %s\n",g_strjoinv("\n",(gchar**) libraw_cameraList())); */
	if (libraw_open_file(rawdata,imgfname) == LIBRAW_SUCCESS) {
		if (libraw_unpack_thumb(rawdata) == LIBRAW_SUCCESS) {
			imgloader = gdk_pixbuf_loader_new();
			if (gdk_pixbuf_loader_write(imgloader,rawdata->thumbnail.thumb,rawdata->thumbnail.tlength,NULL)) {
				switch(rawdata->sizes.flip) {
					case 3:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_UPSIDEDOWN);
						break;
					case 5:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
						break;
					case 6:
						myapp->imgbuffer = gdk_pixbuf_rotate_simple(gdk_pixbuf_loader_get_pixbuf(imgloader),GDK_PIXBUF_ROTATE_CLOCKWISE);
						break;
					case 0:
					default:
						myapp->imgbuffer = gdk_pixbuf_loader_get_pixbuf(imgloader);
						break;
				}
			} else imgerr = TRUE;
			gdk_pixbuf_loader_close(imgloader,NULL);
		} else imgerr = TRUE;
	} else imgerr = TRUE;
	g_free(myapp->imgparam);
	if (!imgerr) {
		myapp->imgparam = g_strdup_printf(paramformat,myapp->currfname,rawdata->idata.make,rawdata->idata.model,rawdata->other.shutter,rawdata->other.aperture,rawdata->other.iso_speed,rawdata->other.focal_len);
		scale_image(GTK_WINDOW (myapp->mainwin)->screen,myapp);
	} else {
		g_object_unref(myapp->imgbuffer);
		gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image),NULL);
		errdialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_ERROR,
											GTK_BUTTONS_CLOSE,
											"Error loading image '%s'",
											g_path_get_basename(imgfname));
		 gtk_dialog_run(GTK_DIALOG (errdialog));
		 gtk_widget_destroy(errdialog);
	}
	libraw_close(rawdata);
	/*
	if (!(myapp->imgbuffer = gdk_pixbuf_new_from_file(imgfname,NULL)))
    	g_print("Failed loading image %s\n", imgfname);
    else scale_image(NULL,myapp);
    */
}

gboolean scale_image(GdkScreen *screen,app_data_t *myapp) {
	GdkPixbuf *tmpbuffer;
	PangoLayout *textbuff;
	cairo_t *cr;
	GdkPixmap *tmppm;
	GdkBitmap *tmpbm;
	gint dispwidth, dispheight, imgwidth, imgheight, scaledwidth, scaledheight, x0, y0, r, txtwidth, txtheight;
	gdouble dispratio, imgratio;

	if (myapp->imgbuffer != NULL) {
		gtk_widget_queue_draw(GTK_WIDGET (myapp->mainwin));
		/* Get actual image widget dimensions */
		/* dispwidth = gdk_screen_get_width(screen);
		dispheight = gdk_screen_get_height(screen);*/
		gtk_window_get_size(GTK_WINDOW (myapp->mainwin),&dispwidth,&dispheight);
		dispratio = (double) dispwidth / (double) dispheight;
		imgwidth = gdk_pixbuf_get_width(myapp->imgbuffer);
		imgheight = gdk_pixbuf_get_height(myapp->imgbuffer);
		imgratio = (double) imgwidth / (double) imgheight;
		/* Eventually scale image */
		if ((imgwidth > dispwidth) || (imgheight > dispheight)) {
			if (imgratio >= dispratio) {
				scaledwidth = dispwidth;
				scaledheight = (gint) round((gdouble) scaledwidth * (1 / imgratio));
			} else {
				scaledheight = dispheight;
				scaledwidth = (gint) round((gdouble) scaledheight * imgratio);
			}
			tmpbuffer = gdk_pixbuf_scale_simple(myapp->imgbuffer,scaledwidth,scaledheight,GDK_INTERP_BILINEAR);
		} else tmpbuffer = NULL;
		/* Eventually add image details */
		if (myapp->showimgparam) {
			gdk_pixbuf_render_pixmap_and_mask((tmpbuffer ? tmpbuffer : myapp->imgbuffer),&tmppm,&tmpbm,0);

			cr = gdk_cairo_create(GDK_DRAWABLE (tmppm));
			textbuff = pango_cairo_create_layout(cr);
			pango_layout_set_markup(textbuff,myapp->imgparam,-1);
			pango_layout_get_pixel_size(textbuff,&txtwidth,&txtheight);
			/* Draw a rounded rectangle */
			x0 = y0 = 15;
			r = 5;
			cairo_set_source_rgba(cr,0,0,0,0.4);
			cairo_move_to(cr,x0,y0 - r);
			cairo_line_to(cr,x0 + txtwidth,y0 -r);
			cairo_curve_to(cr,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0 -r,x0 + txtwidth + r,y0);
			cairo_line_to(cr,x0 + txtwidth + r,y0 + txtheight);
			cairo_curve_to(cr,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth + r,y0 + txtheight + r,x0 + txtwidth,y0 + txtheight + r);
			cairo_line_to(cr,x0,y0 + txtheight + r);
			cairo_curve_to(cr,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight + r,x0 -r,y0 + txtheight);
			cairo_line_to(cr,x0 - r,y0);
			cairo_curve_to(cr,x0-r,y0 -r,x0 -r,y0 -r,x0,y0 - r);
			cairo_close_path(cr);
			cairo_fill(cr);
			/* Layout the text on it */
			cairo_new_path(cr);
			cairo_move_to(cr,x0,y0);
			cairo_set_line_width(cr,1);
			cairo_set_source_rgb(cr,0.8,0.8,0.8);
			pango_cairo_update_layout(cr,textbuff);
			pango_cairo_layout_path(cr,textbuff);
			cairo_fill_preserve(cr);
			/*cairo_set_line_width(cr,0.5);
			cairo_set_source_rgb(cr,0,0,0);
			cairo_stroke(cr);*/
			cairo_destroy(cr);
			g_object_unref(textbuff);

			gtk_image_set_from_pixmap(GTK_IMAGE (myapp->image),tmppm,tmpbm);
			g_object_unref(tmppm);
			g_object_unref(tmpbm);
		} else gtk_image_set_from_pixbuf(GTK_IMAGE (myapp->image), (tmpbuffer ? tmpbuffer : myapp->imgbuffer));
		if (tmpbuffer) g_object_unref(tmpbuffer);
	}
	/* DEBUG */
	/* g_print("\n\nOriginal dim.: %dx%d (%f) - Window dim.: %dx%d (%f) - Scaled dim.: %dx%d\n\n",imgwidth,imgheight,imgratio,dispwidth,dispheight,dispratio,scaledwidth,scaledheight); */
	return TRUE;
}

/*
 * Image file selector
 */
void open_image_file(GtkWidget *callerobj,app_data_t *myapp) {
	GtkWidget *dialog;
	GtkFileFilter *filter;
	gint retval;
	gchar *filename;
	gchar *filterext;
	guint i;
	GList *li = NULL;

	dialog = hildon_file_chooser_dialog_new(GTK_WINDOW (myapp->mainwin),GTK_FILE_CHOOSER_ACTION_OPEN);
	if (myapp->currdir != NULL) gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (dialog),myapp->currdir);
	/* Show only supported RAW files */
	filter = gtk_file_filter_new();
	for(i = 0;myapp->rawext[i] != NULL;i++) {
		filterext = g_strconcat("*.",myapp->rawext[i],NULL);
		gtk_file_filter_add_pattern(filter,filterext);
		g_free(filterext);
	}
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER (dialog),filter);
	retval = gtk_dialog_run(GTK_DIALOG (dialog));
	if (retval == GTK_RESPONSE_ACCEPT || retval == GTK_RESPONSE_OK)
		filename = g_strndup(gtk_file_chooser_get_filename(GTK_FILE_CHOOSER (dialog)),(MAXDIRLEN + MAXFNLEN + 1));
	gtk_widget_destroy(dialog);
	if (filename != NULL) {
		g_free(myapp->currdir);
		myapp->currdir = g_strndup(g_path_get_dirname(filename),MAXDIRLEN);
		store_filenames_in_dir(myapp);
		li = g_list_find_custom(myapp->allimgindir,(gconstpointer) g_path_get_basename(filename),(GCompareFunc) g_strcmp0);
		myapp->currfindex = g_list_position(myapp->allimgindir,li);
		/* DEBUG */
		/* g_print("Current directory: %s - Current file: %s - Current index in list: %d\n",myapp->currdir,g_path_get_basename(filename),myapp->currfindex); */
		load_image(myapp,filename);
	}
	g_free(filename);
}

/*
 * Store image file names in current directory into application's data structure
 */
void store_filenames_in_dir(app_data_t *myapp) {
	gchar *tmpstr;
	glob_t allimages;
	guint i;
	GList *li = NULL;

	/* First free the allocated memory in the list */
	if (myapp->allimgindir != NULL) {
		for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
			g_free(li->data);
		g_list_free(li);
		g_list_free(myapp->allimgindir);
		myapp->allimgindir = NULL;
	}
	/* Then search for other image files */
	tmpstr = g_strconcat(myapp->currdir,"/*.{",g_strjoinv(",",(gchar**) myapp->rawext),"}",NULL);
	if (glob(tmpstr,GLOB_ERR | GLOB_BRACE,NULL,&allimages) == 0) {
		for(i = 0;i < allimages.gl_pathc;i++) {
			myapp->allimgindir = g_list_prepend(myapp->allimgindir,g_strndup(g_path_get_basename((gchar*) allimages.gl_pathv[i]),MAXFNLEN));
		}
		if (myapp->allimgindir != NULL) myapp->allimgindir = g_list_reverse(myapp->allimgindir);
	}
	if (allimages.gl_pathc <= 1) toggle_prev_next_menu_button("disable",myapp);
	else toggle_prev_next_menu_button("enable",myapp);
	globfree(&allimages);
	g_free(tmpstr);
	/* DEBUG */
	/* for(li = myapp->allimgindir;li != NULL;li = g_list_next(li))
		g_print("File in list: %s\n",li->data);
	g_list_free(li);*/
}

/*
 * Display some picture details over picture image
 */
void show_image_details(GtkWidget *callerobj,app_data_t *myapp) {
	myapp->showimgparam = TRUE;
	scale_image(GTK_WINDOW (myapp->mainwin)->screen,myapp);
}

/*
 * Hide picture details
 */
void hide_image_details(GtkWidget *callerobj,app_data_t *myapp) {
	myapp->showimgparam = FALSE;
	scale_image(GTK_WINDOW (myapp->mainwin)->screen,myapp);
}

/*
 * Open next image in the same directory (if any)
 */
void load_next_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *nextfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,++myapp->currfindex);
			if (nextfn == NULL) {
				myapp->currfindex = 0;
				nextfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			nextfn = g_strconcat(myapp->currdir,"/",nextfn,NULL);
			if (!g_file_test(nextfn,G_FILE_TEST_EXISTS)) {
				g_free(nextfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex > 0) ? (myapp->currfindex - 1) : (g_list_length(myapp->allimgindir) - 1));
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,nextfn);
				g_free(nextfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Open previous image in the same directory (if any)
 */
void load_prev_image(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *prevfn;
	GtkWidget *dialog;

	if (myapp->allimgindir != NULL) {
		if (g_list_length(myapp->allimgindir) > 1) {
			prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,--myapp->currfindex);
			if (prevfn == NULL) {
				myapp->currfindex = g_list_length(myapp->allimgindir) - 1;
				prevfn = (gchar*) g_list_nth_data(myapp->allimgindir,myapp->currfindex);
			}
			prevfn = g_strconcat(myapp->currdir,"/",prevfn,NULL);
			if (!g_file_test(prevfn,G_FILE_TEST_EXISTS)) {
				g_free(prevfn);
				g_free(g_list_nth_data(myapp->allimgindir,myapp->currfindex));
				myapp->allimgindir = g_list_delete_link(myapp->allimgindir,g_list_nth(myapp->allimgindir,myapp->currfindex));
				myapp->currfindex = ((myapp->currfindex != (g_list_length(myapp->allimgindir) - 1)) ? (myapp->currfindex + 1) : 0);
				load_next_image(callerobj,myapp);
			} else {
				load_image(myapp,prevfn);
				g_free(prevfn);
			}
		} else {
			toggle_prev_next_menu_button("disable",myapp);
			dialog = gtk_message_dialog_new(GTK_WINDOW (myapp->mainwin),
											GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_MESSAGE_WARNING,
											GTK_BUTTONS_CLOSE,
											"No more images in directory '%s'",
											myapp->currdir);
			 gtk_dialog_run(GTK_DIALOG (dialog));
			 gtk_widget_destroy(dialog);
		}
	}
}

/*
 * Enable / disable "previous" and "next image" menu buttons
 */
void toggle_prev_next_menu_button(const gchar *mode,app_data_t *myapp) {
	GList *menuitems = NULL;

	menuitems = hildon_app_menu_get_items(myapp->appmenu);
	if (menuitems != NULL) {
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,0)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		gtk_widget_set_sensitive(GTK_WIDGET (g_list_nth_data(menuitems,1)),((g_strcmp0(mode,"enable") == 0) ? TRUE : FALSE));
		g_list_free(menuitems);
	}
}

/*
 * Manage button presses on image to load previous / next image in the same directory
 */
gboolean detect_button_press_on_image (GtkWidget *eventbox,GdkEventButton *event,app_data_t *myapp) {
	gint dispwidth;
	const guint mintime = 3000;

	if (g_list_length(myapp->allimgindir) > 1) {
		/* DEBUG */
		/* g_print ("Event box clicked at coordinates (%f,%f) at time %d\n",event->x,event->y,event->time); */
		if (event->type == GDK_BUTTON_PRESS) myapp->pressbuttontime = event->time;
		else if ((event->type == GDK_BUTTON_RELEASE) & ((event->time - myapp->pressbuttontime) > mintime)) {
			myapp->pressbuttontime = 0;
			dispwidth = gdk_screen_get_width(GTK_WINDOW (myapp->mainwin)->screen);
			if (event->x >= ((gdouble) dispwidth / (gdouble) 2)) load_next_image(eventbox,myapp);
			else load_prev_image(eventbox,myapp);
		}
	}
	/* Returning TRUE means we handled the event, so the signal
	 * emission should be stopped (don't call any further
	 * callbacks that may be connected). Return FALSE
	 * to continue invoking callbacks.
	 */
	return TRUE;
}

/*
 * Show the About dialog
 */
void create_about(GtkWidget *callerobj,app_data_t *myapp) {
	gchar *authors[] = {"Luca Donaggio <donaggio@gmail.com>",NULL};
	GdkPixbuf *appicon = NULL;

	/* DEBUG */
	/* appicon = gdk_pixbuf_new_from_file(g_strconcat("./data/",PACKAGE,".png",NULL),NULL); */
	appicon = gdk_pixbuf_new_from_file(g_strconcat(HILDONPIXMAPDIR,"/",PACKAGE,".png",NULL),NULL);
	gtk_show_about_dialog(GTK_WINDOW (myapp->mainwin),
			"logo",((appicon != NULL) ? appicon : NULL),
			"program-name",PACKAGE,
			"version",VERSION,
			"authors",authors,
			"copyright","(C) 2009 Luca Donaggio",
			"license","This program is free software; you can redistribute it and/or\n"
					"modify it under the terms of the GNU Lesser General Public\n"
					"License as published by the Free Software Foundation; either\n"
					"version 2.1 of the License, or (at your option) any later version.\n\n"
					"This program is distributed in the hope that it will be useful,\n"
					"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
					"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU\n"
					"Lesser General Public License for more details.\n\n"
					"You should have received a copy of the GNU Lesser General Public\n"
					"License along with this program; if not, write to the Free Software\n"
					"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA\n",
			"wrap-license",FALSE,
			"comments",g_strconcat(PACKAGE," is an image viewer for RAW image file formats as those produced by modern DSLRs.\n"
					"All the DSLRs from the most important camera manufacturers are supported.\n",
					PACKAGE," is based on LibRaw-Lite which in turn is (C) 2008-2009 LibRaw LLC <info@libraw.org>",NULL),
			NULL);
	if (appicon != NULL) g_object_unref(appicon);
}
