/*******************************************************************************
 *
 * This code example is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the package
 *
 */

/*
 ============================================================================
 Name        : main.c
 Author      : Dart
 Version     : 1.3
 Description : Hildon GUI Application in C
 ============================================================================
 */

/* Sun Nov 01 16:33:34 2009 lcuk : added focus event handlers to ensure no drained battery when in the background */
/* Sun Nov 01 16:33:34 2009 lcuk : discussed in #maemo irc with RST38H, qwerty12 and a few others prior to implementing */

/* Includes */
#include <hildon/hildon-program.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <libosso.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <pulse/simple.h>
#include <pulse/error.h>
#include <pulse/gccmacro.h>
#include <gst/gst.h>
#include <mce/dbus-names.h>

#include "localisation.h"
/* Defines to add the application to dbus and keep it running
 * Please do not modify "APP_NAME" (or other defines) to different name
 */
#define APP_NAME "moobox"
#define APP_VER "1.3"
#define APP_SERVICE "com.nokia.moobox"
#define APP_METHOD "/com/nokia/moobox"
#define MCE_MATCH_RULE "type='signal',interface='" MCE_SIGNAL_IF "',member='" MCE_DEVICE_ORIENTATION_SIG "'"
/*#define MOODIR "/home/jerome/MaemoProj/MooBox/media"
*/
#define MOODIR "/opt/moobox/media"
#define USERMOODIR "/home/user/MyDocs/MooBox"
/* end defines */

static int down=0;
static unsigned int orientation_enabled = 0; /* Sun Nov 01 16:33:34 2009 lcuk : storage for the timer (needed to know we can remove it) */
static gchar *soundfilepath=NULL;
static gboolean landscape;

static void eos_message_received (GstBus* bus, GstMessage* message, GstElement* pipeline)
{
  /* stop playback and free pipeline */
  gst_element_set_state (pipeline, GST_STATE_NULL);
  gst_object_unref (GST_OBJECT(pipeline));
  pipeline = NULL;
}

static gint play_sound ()
{
	GstElement *pipeline;
	GstElement *filesrc;
	GstBus *bus;
	GError *error = NULL;

	/* setup pipeline and configure elements */
	pipeline = gst_parse_launch ("filesrc name=my_filesrc ! decodebin ! audioconvert ! audioresample !pulsesink",&error);
	if (!pipeline) {
		g_printerr("Parse error: %s\n", error->message);
	  goto error;
	}

	filesrc = gst_bin_get_by_name (GST_BIN (pipeline), "my_filesrc");
	if (!filesrc) {
	  g_printerr("Parse error: no filesrc\n");
	  goto error;
	}

	g_object_set (G_OBJECT (filesrc), "location", soundfilepath, NULL);

	/* setup message handling */
	bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
	gst_bus_add_signal_watch_full (bus, G_PRIORITY_HIGH);
	g_signal_connect (bus, "message::eos", (GCallback) eos_message_received,pipeline);
	gst_object_unref (GST_OBJECT(bus));

	/* start playback */
	gst_element_set_state (pipeline, GST_STATE_PLAYING);
	return 0;

	error:
	  gst_object_unref (GST_OBJECT(pipeline));
	  pipeline = NULL;
	  return -1;
}

void enable_accelerometer(osso_context_t *osso_cont)
{
	if (!osso_rpc_run_system(osso_cont, MCE_SERVICE, MCE_REQUEST_PATH,
			MCE_REQUEST_IF,	MCE_ACCELEROMETER_ENABLE_REQ , NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_printerr("WARN: Cannot enable accelerometer\n");
	}
}

void disable_accelerometer(osso_context_t *osso_cont)
{
	if (!osso_rpc_run_system(osso_cont, MCE_SERVICE, MCE_REQUEST_PATH,
			MCE_REQUEST_IF,	MCE_ACCELEROMETER_DISABLE_REQ , NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_printerr("WARN: Cannot enable accelerometer\n");
	}
}

void check_orientation(const gchar *mode){

	printf("mode: %s", mode);
	if (landscape) {
		if ((down==0) && (!strcmp(mode, "landscape (inverted)"))){
			down=1;
		}
		else if ((down==1) && (!strcmp(mode, "landscape"))) {
			down=0;
			play_sound();
		}
	} else {
		if ((down==0) && (!strcmp(mode, "portrait (inverted)"))){
			down=1;
		}
		else if ((down==1) && (!strcmp(mode, "portrait"))) {
			down=0;
			play_sound();
		}
	}
}

static DBusHandlerResult dbus_handle_mce_message(DBusConnection *con, DBusMessage *msg, gpointer data)
{
    DBusMessageIter iter;
    const gchar *mode = NULL;

    if (dbus_message_is_signal(msg, MCE_SIGNAL_IF, MCE_DEVICE_ORIENTATION_SIG)) {
        if (dbus_message_iter_init(msg, &iter)) {
            dbus_message_iter_get_basic(&iter, &mode);
            check_orientation(mode);
        }
    }
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


static gint dbus_callback (const gchar *interface, const gchar *method,
               GArray *arguments, gpointer data,
               osso_rpc_t *retval)
{
  printf ("dbus: %s, %s\n", interface, method);

  if (!strcmp (method, "top_application"))
      gtk_window_present (GTK_WINDOW (data));

  return DBUS_TYPE_INVALID;
}


/* Sun Nov 01 16:35:54 2009 lcuk : added event handlers for focus */

static gint GTKFocus_In_Handler(GtkWidget *SrcWidget,GdkEventFocus *Event,gpointer Arg)
{
    /* got focus, enable accelerometer */
	printf("listening\n");
    if(!orientation_enabled)
    {
        enable_accelerometer(Arg);
        orientation_enabled=1;
    }
    return 0;
}
static gint GTKFocus_Out_Handler(GtkWidget *SrcWidget,GdkEventFocus *Event,gpointer Arg)
{
    printf("stopped listenning...\n");
	/* focusing out, lets disable the accelerometer */
    if(orientation_enabled)
    {
    	disable_accelerometer(Arg);
        orientation_enabled=0;
    }
    return 0;
}

static void show_box_window (GtkIconView *Iconview,
							 GtkTreePath *selected,
							 osso_context_t *osso_cont)
{
    GtkWidget *win;
    GtkWidget *vbox;
    GtkWidget *image;
    gchar *name, *path;
    GtkTreeModel *model;
    GtkTreeIter iter;
	DBusConnection *dbus_con;

    model = gtk_icon_view_get_model (Iconview);

   	gtk_tree_model_get_iter (model, &iter, selected);
    gtk_tree_model_get (model, &iter, 0, &name, 2, &path, -1);

    gchar *imagename = g_strconcat(name, ".jpg", NULL);
    gchar *imagenamepath = g_build_filename (path, imagename, NULL);
    gchar *soundfile = g_strconcat(name, ".mp3", NULL);

    /* If we already played a file, free the value */
    if (soundfilepath!=NULL) g_free(soundfilepath);
    soundfilepath = g_build_filename (path, soundfile, NULL);

    /* Create the box window */
	win = hildon_stackable_window_new();

    /* Setting a picture in the window */
	/* Here check if the picture is in landscape or portrait mode */
	/* Then load the picture in the correct orientation so that it's displayed nicely */
	/*  and fills the screen as much as possible */
	int width, height;
	GdkPixbufFormat *PictFormat=gdk_pixbuf_get_file_info(imagenamepath,&width,&height);
	GdkPixbuf *Rpixbuf;
	if (width>height){
		landscape=TRUE;
		Rpixbuf = gdk_pixbuf_new_from_file_at_scale(imagenamepath, 800, 424, TRUE, NULL);
	} else {
		landscape=FALSE;
		GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_scale(imagenamepath, 424, 800, TRUE, NULL);
		Rpixbuf = gdk_pixbuf_rotate_simple (pixbuf, GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
		g_object_unref(pixbuf);
	}
	image = gtk_image_new_from_pixbuf(Rpixbuf);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox),
                        image,
                        TRUE,
                        TRUE,
                        0);

    gtk_container_add (GTK_CONTAINER (win),
                       vbox);

	/* Get the system DBus connection */
	dbus_con = osso_get_sys_dbus_connection(osso_cont);

	/* Add the dbus callback that is called when device orientation changes */
	dbus_bus_add_match(dbus_con, MCE_MATCH_RULE, NULL);
	dbus_connection_add_filter(dbus_con, dbus_handle_mce_message, NULL, NULL);

    /* Sun Nov 01 16:29:54 2009 lcuk : adding focusin/out events to only set the timer when ontop */
    /* Sun Nov 01 16:29:54 2009 lcuk : thanks to RST38H in #maemo */
    g_signal_connect(GTK_WIDGET(win),"focus_in_event",G_CALLBACK(GTKFocus_In_Handler),osso_cont);
    g_signal_connect(GTK_WIDGET(win),"focus_out_event",G_CALLBACK(GTKFocus_Out_Handler),osso_cont);

    /* Stop listening for accelerometer when window is closed */
    g_signal_connect (G_OBJECT (win), "delete_event", G_CALLBACK(GTKFocus_Out_Handler),osso_cont);

    /* This call show the window and also add the window to the stack */
    gtk_widget_show_all (win);
    g_free(name);
    g_free(imagename);
    g_free(soundfile);
    g_free(imagenamepath);
    g_object_unref (Rpixbuf);
}

/*GtkTreeModel **/
void create_grid(GtkListStore *list_store, const gchar* themoodir)
{
	GtkTreeIter iter;
	gchar **str;
	GDir *dir;
	GError *error = NULL;
	const gchar* filename = NULL;

	/* Find all the images in the moobox directory */
	dir = g_dir_open (themoodir, 0, &error);
	if (error) {
	  g_printerr("ERROR: Cannot open moobox directory: %s\n", error->message);
	  g_clear_error(&error);
	  return;
	}

	while ( (filename = g_dir_read_name(dir)) ) {
		gchar *path = g_build_filename (themoodir, filename, NULL);

		/* Try to load the file as an image: */
		GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file_at_scale(path, 150, 150, TRUE, NULL);
		if (pixbuf) {
			str=g_strsplit(filename, ".", 2);
			gchar *soundfile=g_strconcat(str[0], ".mp3",NULL);
			gchar *soundfilepath=g_build_filename (themoodir, soundfile, NULL);
			if (g_file_test(soundfilepath, G_FILE_TEST_EXISTS)) {
				gtk_list_store_append(list_store, &iter);
				gtk_list_store_set(list_store, &iter, 0, str[0], 1, pixbuf, 2, themoodir, -1);
			}
			g_object_unref (pixbuf);
			g_strfreev (str);
			g_free(soundfile);
			g_free(soundfilepath);
		}
		g_free (path);
	}

	return;
}

void display_about(GtkWidget *parent)
{
	GtkWidget *dialog, *Label;
	gint response;


	dialog = gtk_dialog_new_with_buttons(
			"About",
			GTK_WINDOW(parent),
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_OK,
			GTK_RESPONSE_OK,
/*			GTK_STOCK_CANCEL,
			GTK_RESPONSE_CANCEL,
*/			NULL);

	Label = gtk_label_new ("");
	gtk_label_set_markup(GTK_LABEL(Label),"<span size=\"x-small\">Pictures are under Creative Commons and taken from http://animalphotos.info/.\n"
			"Sounds are from http://www.universal-soundbank.com/.\n"
			"You can add new pictures and sounds in the MooBox directory created in your documents folder (/home/user/MyDocs/MooBox)."
			" Pictures and sounds must have the same name, with jpg and mp3 extensions respectively.</span>");
	gtk_label_set_justify(GTK_LABEL(Label), GTK_JUSTIFY_LEFT);
	gtk_label_set_line_wrap(GTK_LABEL(Label), TRUE);
	gtk_container_add (GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), Label);
	gtk_widget_show_all (dialog);


	/* Wait until user finishes the dialog. */
	response = gtk_dialog_run(GTK_DIALOG(dialog));
	if (response == GTK_RESPONSE_OK)
	{
		/*write_ua(gtk_entry_get_text (UaString));*/
	}
	/* Free the dialog (and it's children) */
	gtk_widget_destroy(GTK_WIDGET(dialog));

}

int main( int argc, char* argv[] )
{
    /* Create needed variables */
    HildonProgram *program;
    HildonWindow *window;
    GtkWidget *table, *pannable_area;
    osso_context_t *osso_cont;
	osso_return_t ret;
	HildonAppMenu *menu;
	GtkTreeModel *treemodel;

	locale_init();

    osso_cont = osso_initialize(APP_NAME, APP_VER, TRUE, NULL);
	if (osso_cont == NULL)
    {
    	g_printerr("osso_initialize failed.\n");
    	exit (1);
    }

    /* Initialize the GTK. */
    hildon_gtk_init (&argc, &argv);
    /* Init gst */
    gst_init(&argc, &argv);

    /* Create the hildon program and setup the title */
    program = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("MooBox");

    /* Create HildonWindow and set it to HildonProgram */
    window = HILDON_WINDOW(hildon_stackable_window_new());
    hildon_program_add_window(program, window);

    pannable_area = GTK_WIDGET(hildon_pannable_area_new ());

    treemodel = gtk_list_store_new(3, G_TYPE_STRING, GDK_TYPE_PIXBUF, G_TYPE_STRING);
    create_grid(treemodel, MOODIR);
    create_grid(treemodel, USERMOODIR);
    table=GTK_WIDGET(hildon_gtk_icon_view_new_with_model(HILDON_UI_MODE_NORMAL,GTK_TREE_MODEL(treemodel)));

    gtk_container_add (GTK_CONTAINER(pannable_area), table);

    gtk_icon_view_set_text_column(GTK_ICON_VIEW(table),0);
    gtk_icon_view_set_pixbuf_column(GTK_ICON_VIEW(table),1);

    gtk_container_add ( GTK_CONTAINER (window), pannable_area);

    /* Quit program when window is closed. */
    g_signal_connect (G_OBJECT (window), "delete_event",
		      G_CALLBACK (gtk_main_quit), NULL);

    /* Quit program when window is otherwise destroyed. */
    g_signal_connect (G_OBJECT (window), "destroy",
		      G_CALLBACK (gtk_main_quit), NULL);

    /* When an item on the grid is selected, open the box window */
    g_signal_connect (table, "item-activated", G_CALLBACK (show_box_window), osso_cont);

    ret = osso_rpc_set_cb_f (osso_cont,
						   APP_SERVICE,
						   APP_METHOD,
						   APP_SERVICE,
						   dbus_callback, GTK_WIDGET( window ));
	if (ret != OSSO_OK) {
		g_printerr("osso_rpc_set_cb_f failed: %d.\n", ret);
		exit (1);
	}

	/* Create menu and add it to the window */
	menu= HILDON_APP_MENU (hildon_app_menu_new ());
	GtkWidget * button = gtk_button_new_with_label ("About");
	g_signal_connect_after (button, "clicked", G_CALLBACK (display_about), window);
	hildon_app_menu_append (menu, GTK_BUTTON (button));
	gtk_widget_show_all (GTK_WIDGET (menu));
	hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

    /* Begin the main application */
    gtk_widget_show_all ( GTK_WIDGET ( window ) );

    gtk_main();

    /* deinitialize osso */
    osso_deinitialize (osso_cont);

    if (soundfilepath!=NULL) g_free(soundfilepath);
    /* Exit */
    return 0;
}
