/** \file */

#include <qmetatype.h>
#include <vector>
#include <iostream>
#include <string>
#include <stdio.h>
#include <QHostAddress>
#include <QHostInfo>
#include <QBrush>
#include <QTextEdit>
#include <QTableWidgetSelectionRange>
#include <QComboBox>
#include <QDir>
#include <QDialog>
#include <QSlider>
#include <QRadioButton>

#include "cameraui.h"
#include "OverlayWidget.h"
#include "util.h"

using namespace std;

namespace Plat = FCam::N900;

// Activated by cameraUI's SIGNAL CropObtained (emitted in send() method). Begins processing the crop.
// img is the already-cropped full-color image. pt is the top left coordinate from the global image that img represents.
// correctVignetting is a boolean specifying whether to attempt vignetting correction
void CameraUI::ProcessCropAndSend(QImage img, QPoint topLeft, bool correctVignetting){
    // Now that the crop is obtained, can go ahead and set the bbView to load in the cropped image
    QPixmap pixmap = QPixmap::fromImage(img);
    croppedPixmap = pixmap;
    bbScenePixmapItem->setPixmap(pixmap);
    bbView->setSceneRect(pixmap.rect());



    // Correct vignetting if necessary
    if(correctVignetting){
        CorrectVignetting(&img, topLeft);
    }

    QImage croppedImageBW = ConvertToGrayScale(img);
    crop_filename = (QDir::homePath() + QString("/.mir-translator/MIRATS_cropped.jpg")).toAscii().data();
    croppedImageBW.save(crop_filename.c_str(), 0, 80);

    // Get the Amazon EC2 instance's IP and make the request to EC2
    QHostInfo ec2host = QHostInfo::fromName("mir.mhsueh.com");
    QHostAddress ec2addr = ec2host.addresses().first();
    //ocr_req = new OCRRequest("192.168.2.14", 20248, (char *) crop_filename.c_str());


    //The OCRRequest is instantiated here. Hardcoded line for local server. Use the above line for EC2 server.
    ocr_req = new OCRRequest(ec2addr.toString(), 20248, (char *) crop_filename.c_str());
    m_just_timed_out = false;
    //qDebug() << ec2addr.toString();
    connect(ocr_req, SIGNAL(TimedOutSignal(OCRRequest *)), this, SLOT(ShowTimedOutDiscDialog(OCRRequest *)));
    connect(ocr_req, SIGNAL(ErrorConnecting(OCRRequest *)), this, SLOT(ShowErrorConnectingDialog(OCRRequest*)));

    // Connect the ocr_req's SIGNAL OCR_Finished to the SLOT for cameraUI to set the OCR results
    connect(ocr_req, SIGNAL(OCR_finished(QString)), this, SLOT(SetOCRText(QString)));
    // Connect cameraUI's SIGNAL for finished setting the OCR results to the SLOT for ocr_req to make a translation request
    connect(this, SIGNAL(SetOCRText_Finished(QString, QString, QString)), ocr_req, SLOT(Make_Trans_Request(QString, QString, QString)));
    // Connect the ocr_req's SIGNAL for translation received to the cameraUI's SLOT for setting the translation text results
    connect(ocr_req, SIGNAL(TransFinished(QString)), this, SLOT(SetTranslationText(QString)));

    //Set the source language and then send the OCR request to the server using ocr_req
    ocr_req->SetLang(sourceCBox->currentText());
    ocr_req->MakeOCRRequest();
    connect(quitButton, SIGNAL(clicked()), ocr_req->server_socket, SLOT(disconnectFromHostImplementation()));
    connect(sourceCBox, SIGNAL(currentIndexChanged(int)), this, SLOT(CropChanged())); // Changing languages shoudl re-enable request button
    connect(destCBox, SIGNAL(currentIndexChanged(int)), this, SLOT(CropChanged()));
}

void CameraUI::QuitApp(){
    overlay->disable();
    ::close(overlay->overlay_fd);
    qApp->quit();
}

// Populates the language combo box cb
void CameraUI::PopulateLangCBox(QComboBox* cb){
    cb->clear();
    QList<QString> keys = langStrings.keys();
    qSort(keys);
    keys.removeAll(QString("English"));
    keys.insert(0, QString("English")); // Do this to put English at the front.
    QList<QString>::iterator it;
    for(it = keys.begin(); it != keys.end(); it++){
        cb->addItem(*it);
    }
}

// Initialize the mappings from the combo box text to Google API strings
void CameraUI::InitLangStrings(){
    langStrings["English"] = "en";
    langStrings["Japanese"] = "ja";
    langStrings["Chinese (tra)"] = "zh-TW";
    langStrings["Chinese (sim)"] = "zh-CN";
    langStrings["Korean"] = "ko";

    langStrings["Arabic"] = "ar";
    langStrings["Bulgarian"] = "bg";
    langStrings["Catalan"] = "ca";
    langStrings["Czech"] = "cs";
    langStrings["Danish"] = "da";
    langStrings["German"] = "de";
    langStrings["Greek"] = "el";

    langStrings["Finnish"] = "fi";
    langStrings["French"] = "fe";
    langStrings["Hebrew"] = "iw";
    langStrings["Hungarian"] = "hu";
    langStrings["Indonesian"] = "id";
    langStrings["Italian"] = "it";
    langStrings["Latvian"] = "lv";
    langStrings["Lituanian"] = "lt";
    langStrings["Dutch"] = "nl";
    langStrings["Norwegian"] = "no";
    langStrings["Polish"] = "pl";
    langStrings["Portuguese"] = "pt";
    langStrings["Romanian"] = "ro";
    langStrings["Russian"] = "ru";
    langStrings["Slovakian"] = "sk";
    langStrings["Slovenian"] = "sl";
    langStrings["Spanish"] = "es";
    langStrings["Serbian"] = "sr";
    langStrings["Swedish"] = "sv";
    langStrings["Filipino"] = "tl";
    langStrings["Thai"] = "th";
    langStrings["Turkish"] = "tr";
    langStrings["Ukrainian"] = "uk";
    langStrings["Vietnamese"] = "vi";
}

// Make a retranslation request
void CameraUI::Make_Retrans_Request(){
    retransEnabled = false;
    retransButton->setEnabled(false);
    retransButton->updateGeometry();
    statusLabel->setText("- Translating -");
    ocr_req->SetInputPath(crop_filename.c_str());
    ocr_req->Make_Retrans_Request(resultView->toPlainText(), langStrings[sourceCBox->currentText()], langStrings[destCBox->currentText()]);
}


/* Correct vignetting according to the calibrated model
offset specifies the offset of img's top-left corner relative to the ORIGINAL image's CENTER pixel */
void CameraUI::CorrectVignetting(QImage * img, QPoint offset){

    /* Debug stuff */
    QString debug_tif_filename;
    if(DEBUG){
        int existing_debug_tifs_count = 0;
        while(true){
            debug_tif_filename.sprintf("%03d", existing_debug_tifs_count);
            debug_tif_filename = QDir::homePath() + "/cs262a_debug/" + debug_tif_filename;
            if(!QFile::exists(debug_tif_filename + ".tif")){
                break;
            }else{
                existing_debug_tifs_count++;
            }
        }
        QImage no_correction_img(*img);
        no_correction_img = ConvertToGrayScale(no_correction_img);
        bool saved = no_correction_img.save(debug_tif_filename + ".tif");
    }
    /* End debug stuff */

    // Iterate through every pixel in img and apply the correction
    for(int x = 0; x < img->width(); x++){
        for(int y = 0; y < img->height(); y++){
            int manhattan_distance = abs(x + offset.x()) + abs(y + offset.y());
            float scale_by = pow((float)manhattan_distance, 2) * 0.0000002 + 0.00007 * manhattan_distance - 0.0483;
            QColor oldPixelColor = QColor::fromRgb(img->pixel(x, y));

            // Don't change the pixel color if it's darker than the original. Clip pixel value at 255.
            int new_value = oldPixelColor.value() * (1 + scale_by);
            if(new_value < oldPixelColor.value()){
                new_value = oldPixelColor.value();
            }else if(new_value > 255){
                new_value = 255;
            }
            // Scale up to the new V (HSV)
            QColor newPixelColor = QColor::fromHsv(oldPixelColor.hue(), oldPixelColor.saturation(), new_value);
            img->setPixel(x, y, newPixelColor.rgb());
        }
    }

    /* Debug stuff */
    if(DEBUG){
        QImage corrected_img(*img);
        corrected_img = ConvertToGrayScale(corrected_img);
        bool saved = corrected_img.save(debug_tif_filename + "_corrected.tif");
    }
    /* End Debug stuff */

}

// Convert the img to grayscale, return a new img
QImage CameraUI::ConvertToGrayScale(QImage &img) {
    // Get a grayscale color table
    QVector<QRgb> colorTable(256);
    for(int i = 0; i < 256; i++){
        colorTable[i] = QColor(i,i,i).rgb();
    }

    QImage rtn_img = QImage(img.size(), QImage::Format_Indexed8);
    rtn_img.setColorTable(colorTable);

    for(int x = 0; x < img.width(); x++){
        for(int y = 0; y < img.height(); y++){
            int gray = qGray(img.pixel(x,y));
            rtn_img.setPixel(x,y,gray);
        }
    }
    return rtn_img;
}

// Delete the source photo file
void CameraUI::DeleteSrcPhoto() {
    if (src_filename.length() > 0) {
        cout << "Deleting src photo: " << src_filename << endl;
        remove(src_filename.c_str());
        src_filename = "";
    }
}

// Delete the cropped photo file
void CameraUI::deleteCropPhoto() {
    if (crop_filename.length() > 0) {
        cout << "Deleting crop photo: " << src_filename << endl;
        remove(crop_filename.c_str());
        crop_filename = "";
    }
}

// Clear the translated and OCRd results
void CameraUI::clearResults() {
    translatedText = "";
    OCRdText = "";
}

// This slot is called after SIGNAL TransFinished from OCRReq is emitted.
void CameraUI::SetTranslationText(QString text){
    bTranslationComplete = true;
    textButton->setEnabled(true);
    statusLabel->setText("- Done -");
    translatedText = text;
    translateView->setText(text);
    viewText();
}

void CameraUI::SetResegText(QString text){
    // Need to record the indexes in which to correct the offset info

    for(int pos = 0; pos < text.length();){
        ResultChar result_char;
        result_char.src_text_start_pos = pos;

        // First check if the next character is a white space variant
        if(text[pos] == QChar(' ')){
            pos += 1;
        }else{

            QString dummy;
            // Not a white space, so we begin the parsing by getting out the confidence level
            //uint8_t conf = OCRdText.mid(pos, 3).toInt();
            pos += 3;

            // Now get out the bounding box
            int left = text.mid(pos, 4).toInt() + ResegOffsetX;
            text.replace(pos, 4, dummy.sprintf("%04d", left));
            pos += 4;

            int top = text.mid(pos, 4).toInt() + ResegOffsetY;
            text.replace(pos, 4, dummy.sprintf("%04d", top));
            pos += 4;

            int right = text.mid(pos, 4).toInt() + ResegOffsetX;
            text.replace(pos, 4, dummy.sprintf("%04d", right));
            pos += 4;

            int bottom = text.mid(pos, 4).toInt() + ResegOffsetY;
            text.replace(pos, 4, dummy.sprintf("%04d", bottom));
            pos += 4;

            // Now create the alternative candidate list for the character we are processing
            for(int i = 0; i < num_alts; i++){
                pos += 1;
            }
        }
    }

    // Insert into the raw, unparsed OCRdText
    int start = m_result_chars[resultView->textCursor().selectionStart()].src_text_start_pos;
    int n = m_result_chars[resultView->textCursor().selectionEnd() - 1].src_text_end_pos - start;
    OCRdText.replace(start, n, text);
    ParseOCRdText();
    updateResultView();
    // qDebug() << text;
}

// This slot is called after SIGNAL OCR_finished from OCRReq is emitted
void CameraUI::SetOCRText(QString text){
    if(text == ""){
        text = " "; // Needed to prevent crashing
        ShowNoResultsDialog();
    }

    // Reset the alternative candidates table, set the OCRdText to whatever the server returned
    OCRdText = text;
    retransEnabled = false;

    // Need to disconnect the cursor changed signal before we mess with the result text
    QObject::disconnect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));
    resultView->clear();

    // OCRdText now contains the OCR text returned from the server (in the "unparsed" format)

    /* Parse the results */
    ParseOCRdText();
    updateResultView();
    /* Results are now parsed and contained in the result text box */
    //OCRdText = resultView->toPlainText(); // OCRdText now contains the PARSED version of the text
    QObject::connect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange())); // Reconnect the signal since we are no longer messing with the result text box
    // Emit a signal to indicate readiness for translation. Provide the desired language selections as well.
    emit SetOCRText_Finished(resultView->toPlainText(), langStrings[sourceCBox->currentText()], langStrings[destCBox->currentText()]);
}

void CameraUI::ParseOCRdText(){
    m_result_chars.clear();
    OCRdText = OCRdText.replace('\n', ' ').replace('\r', ' ');
    for(int pos = 0; pos < OCRdText.length();){
        ResultChar result_char;
        result_char.src_text_start_pos = pos;

        // First check if the next character is a white space variant
        if(OCRdText[pos] == QChar(' ')){
            result_char.alts.append(OCRdText[pos]);
            pos += 1;
        }else{

            // Not a white space, so we begin the parsing by getting out the confidence level
            //uint8_t conf = OCRdText.mid(pos, 3).toInt();
            result_char.confidence = OCRdText.mid(pos, 3).toInt();
            pos += 3;

            // Now get out the bounding box
            result_char.bounding_box.left = OCRdText.mid(pos, 4).toInt();
            pos += 4;
            result_char.bounding_box.top = OCRdText.mid(pos, 4).toInt();
            pos += 4;
            result_char.bounding_box.right = OCRdText.mid(pos, 4).toInt();
            pos += 4;
            result_char.bounding_box.bottom = OCRdText.mid(pos, 4).toInt();
            pos += 4;

            // Now create the alternative candidate list for the character we are processing
            for(int i = 0; i < num_alts; i++){
                result_char.alts.append(OCRdText[pos]);
                pos += 1;
            }
        }
        result_char.src_text_end_pos = pos;
        result_char.selected_alt = 0;
        m_result_chars.append(result_char);
    }
}

// Called when then send button is pressed
void CameraUI::send() {
    langMode = sourceCBox->currentText();

    // "Change" the send button to the result button
    sendButton->hide();
    alt_table->hide();
    textButton->show();
    textButton->setEnabled(false);
    statusLabel->show();
    statusLabel->setText("- Translating -");
    qApp->processEvents();

    // We will need to discard any existing OCRReqs because we will be creating a new one
    if(ocr_req != 0){
        ocr_req->deleteLater();
    }

    // Get out the cropped rectangle
    QRectF cropRect = cropItem->cropRect();
    QImage src_img = QImage(src_filename.c_str());

    // Need to interpolate the area to crop out from full-size image
    qreal tl_x = (cropRect.topLeft().x() / viewScene->width()) * src_img.width();
    qreal tl_y = (cropRect.topLeft().y() / viewScene->height()) * src_img.height();
    qreal crop_width = (cropRect.width() / viewScene->width()) * src_img.width();
    qreal crop_height = (cropRect.height() / viewScene->height()) * src_img.height();
    QImage croppedImage = src_img.copy(tl_x, tl_y, crop_width, crop_height);

    QPoint src_center = src_img.rect().center();

    emit CropObtained(croppedImage, QPoint(tl_x - src_center.x(), tl_y - src_center.y()), true);
}

// Handles the cursor inside the result view text box changing positions. Needs to select the "closest" new character to considered for alternatives
// Activated by resultView's SIGNAL cursorPositionChanged()
void CameraUI::HandleResultCursorChange(){

    // Get the current cursor
    QTextCursor cursor = resultView->textCursor();
    // Disconnect the cursor changed signal to avoid a feedback loop back to this slot!
    QObject::disconnect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));

    // Don't process alternative selection if the user is trying to select a group of text
    if(cursor.selectionStart() != cursor.selectionEnd()){
        int pos = cursor.position();
        if(cursor.position() > oldTextCursor.selectionEnd()){
            cursor.setPosition(oldTextCursor.selectionStart());
            cursor.setPosition(pos, QTextCursor::KeepAnchor);
            //qDebug("old start:%d, old end:%d", oldTextCursor.selectionStart(), oldTextCursor.selectionEnd());
        }else if(cursor.position() < oldTextCursor.selectionStart()){
            cursor.setPosition(oldTextCursor.selectionEnd());
            cursor.setPosition(pos, QTextCursor::KeepAnchor);
        }
        resultView->setTextCursor(cursor);
        alt_table->setRangeSelected(QTableWidgetSelectionRange(0,0,0,num_alts - 1), false);
        for(int i = 0; i < num_alts; i++){
            alt_table->item(0, i)->setText("");
        }
    }else{

        // Set the background of the highlighted text to GREY (different from the default)
        cursor.movePosition(QTextCursor::Left, QTextCursor::KeepAnchor);
        //QTextCharFormat highlightedTextFormat;
        //highlightedTextFormat.setBackground(Qt::gray);
        resultView->setTextCursor(cursor);

        // Now time to populate the set of alternative characters available for the selected character
        int text_pos = cursor.selectionStart();
        for(int i = 0; i < num_alts; i++){
            if(cursor.selectedText() == " " || cursor.selectedText() == "\n" || cursor.selectedText() == QString(QChar(QChar::LineSeparator))
                    || cursor.selectedText() == QString(QChar(0x2029)) || cursor.selectedText() == ""){
                alt_table->item(0, i)->setText(" ");
            }else{
                alt_table->item(0, i)->setText(QString(m_result_chars[text_pos].alts[i]));
            }
        }

        // Disconnect the cellActivated signal to avoid feedback loop with trying to handle a replacement alternative chosen
        QObject::disconnect(alt_table, SIGNAL(cellActivated(int,int)), this, SLOT(ChangeChoiceToAlt(int, int)));

        // Clear all selection(s) in the alternative table
        alt_table->setRangeSelected(QTableWidgetSelectionRange(0,0,0,num_alts - 1), false);

        // Determine which cell to select for the alt table
        //alt_table->setRangeSelected(QTableWidgetSelectionRange(0,alt_table_selections[text_pos],0,alt_table_selections[text_pos]), true);
        alt_table->setRangeSelected(QTableWidgetSelectionRange(0,m_result_chars[text_pos].selected_alt,0,m_result_chars[text_pos].selected_alt), true);

        AdjustAltTableSize();
        oldTextCursor = cursor;

        // Reconnect the signals
        QObject::connect(alt_table, SIGNAL(cellActivated(int,int)), this, SLOT(ChangeChoiceToAlt(int, int)));
    }

    // Find the bounding box over ALL selected characters
    BoundingBoxCoords maxBoundingBox = GetSelectionBB();
    bbSceneRectItem->setRect(maxBoundingBox.left, maxBoundingBox.top,
                                         maxBoundingBox.right - maxBoundingBox.left, maxBoundingBox.bottom - maxBoundingBox.top);
    bbView->fitInView(maxBoundingBox.left, maxBoundingBox.top,
                      maxBoundingBox.right - maxBoundingBox.left, maxBoundingBox.bottom - maxBoundingBox.top, Qt::KeepAspectRatio);
    bbView->setTransformationAnchor(QGraphicsView::AnchorViewCenter);
    bbView->scale(0.6, 0.6);

    QObject::connect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));
}

// Just refreshes the size of the alternative selection table
void CameraUI::AdjustAltTableSize(){
    alt_table->resizeColumnsToContents();
    alt_table->resizeRowsToContents();

    int total_width = 0;
    for(int i = 0; i < alt_table->columnCount(); i++){
        total_width += alt_table->columnWidth(i);
    }

    alt_table->setFixedWidth(total_width);
    alt_table->setRowHeight(0, 40);
    alt_table->setFixedHeight(40);
    alt_table->updateGeometry();
}

// Enables the retranslation button
void CameraUI::EnableRetranslate(){
    retransEnabled = true;
    retransButton->setEnabled(true);
}

void CameraUI::viewText() {
    quitButton->show();
    statusLabel->show();
    bbView->show(); // Show the bb image
    overlay->hide();
    imageView->hide();
    sendButton->hide();
    textButton->hide();
    resultLabel->show();
    translateView->show();
    translateLabel->show();
    photoButton->show();
    resultView->show();
    cameraButton->hide();
    retransButton->show();
    retransButton->setEnabled(retransEnabled);
    alt_table->show();

    candidatesLabel->show();
    sourceLangLabel->hide();
    destLangLabel->hide();
    destCBox->hide();
    sourceCBox->hide();
    manualCorrectionLabel->show();

    HandleResultCursorChange();

    translateView->setText(translatedText);

    resultView->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    resultView->viewport()->updateGeometry();
    translateView->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    translateView->viewport()->updateGeometry();

    QObject::connect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));
    QObject::connect(resultView, SIGNAL(textChanged()), this, SLOT(EnableRetranslate()));
}


void CameraUI::viewCamera() {

    /*Delete all crop / src / results upon entering this view*/
    DeleteSrcPhoto();
    deleteCropPhoto();
    clearResults();

    candidatesLabel->hide();
    bbView->hide();
    photoButton->hide();
    alt_table->hide();
    quitButton->show();
    textButton->hide();
    cameraButton->hide();
    sendButton->hide();
    resultView->hide();
    imageView->hide();
    sourceCBox->hide();
    destCBox->hide();
    statusLabel->hide();
    busyView->hide();
    resultLabel->hide();
    translateView->hide();
    translateLabel->hide();
    overlay->show();
    retransButton->hide();
    sourceLangLabel->hide();
    destLangLabel->hide();
    manualCorrectionLabel->hide();

    //QObject::disconnect(resultView, SIGNAL(textChanged()), this, SLOT(EnableRetranslate()));
}

void CameraUI::viewBusyScreen(){
    bbView->hide();
    photoButton->hide();
    quitButton->hide();
    textButton->hide();
    cameraButton->hide();
    sendButton->hide();
    resultView->hide();
    imageView->hide();
    sourceCBox->hide();
    statusLabel->hide();
    overlay->hide();
    resultLabel->hide();
    translateView->hide();
    translateLabel->hide();
    busyView->show();
    candidatesLabel->hide();
    sourceLangLabel->hide();
    destLangLabel->hide();
    manualCorrectionLabel->hide();
}

// Slot activated by cameraThread SIGNAL image_saved, signaling that its done saving the src image from the camera
void CameraUI::SetFileNameAndViewImage(QString fname){
    src_filename = fname.toStdString();
    ViewPhoto();
}

// A SLOT that is SIGNALED by cropItem's SIGNAL CropChanged() when the selection region changes
void CameraUI::CropChanged(){
    deleteCropPhoto();
    translatedText = "";
    OCRdText = "";
    statusLabel->setText("Select area to translate");
    textButton->hide();
    sendButton->show();
}

// Activated when the QTableWidget for the alternative table SIGNALS that a new cell was activated
void CameraUI::ChangeChoiceToAlt(int r, int c){
    if(resultView->textCursor().selectionStart() != resultView->textCursor().selectionEnd() - 1){
        return; // Dont change alternatives unless only one char is selected
    }
    QTextCursor cursor = resultView->textCursor();

    // Record which cell was selected
    m_result_chars[cursor.selectionStart()].selected_alt = c;

    // If a blank cell was selected, ignore it
    if(alt_table->item(r,c)->text() == " "){
        return;
    }

    updateResultView();

}

void CameraUI::ViewPhoto() {
    if (src_filename.length() == 0){
        return;
    }

    bbView->hide();
    photoButton->hide();
    quitButton->show();
    resultView->hide();
    alt_table->hide();
    sourceCBox->show();
    destCBox->show();
    busyView->hide();
    resultLabel->hide();
    translateView->hide();
    translateLabel->hide();
    imageView->show();
    cameraButton->show();
    candidatesLabel->hide();
    statusLabel->show();
    sourceLangLabel->show();
    destLangLabel->show();
    manualCorrectionLabel->hide();
    if(OCRdText.isEmpty()){
        statusLabel->setText("Select area to translate");
    }else{
        statusLabel->setText("- Done -");
    }
    retransButton->hide();
    overlay->hide();

    // Need to set the correct button states in case we are returning to the photo view (e.g., not from the camera screen)
    if(translatedText.length() != 0 || OCRdText.length() != 0){
        textButton->show();
        sendButton->hide();
        statusLabel->setText("- Done -");
    }else if(m_just_timed_out){
        textButton->hide();
        sendButton->show();
        statusLabel->setText("Select area to translate");
        m_just_timed_out = false;
    }else{
        // We are entering the photo view fresh off a snap from the camera screen
        textButton->hide();
        sendButton->show();
        cropItem->_cropRect = QRectF(20, 20, 100, 70);
        statusLabel->setText("Select area to translate");
    }

    // Implicitly shared pixmap so dont need to worry about stack frame going away
    QImage image(src_filename.c_str());
    QPixmap pixmap = QPixmap::fromImage(image);
    viewScenePixmapItem->setPixmap(pixmap.scaled(QSize(
                                                     (int)viewScene->width(), (int)viewScene->height())));
    QObject::disconnect(resultView, SIGNAL(textChanged()), this, SLOT(EnableRetranslate()));
}

void CameraUI::StartAdaptiveThresholder(){

    QRectF resegmentBB = resegmentCropItem->cropRect();
    QImage resegmentCropImage = croppedPixmap.toImage();
    resegmentCropImage = resegmentCropImage.copy(resegmentBB.toRect());

        at->Init(resegmentCropImage);
        UpdateAdaptiveThresholder();
}

void CameraUI::ResetAdaptiveThresholder(){
    thresholdNoneRadio->click();
    ResetAdaptiveThresholderImage();
}

void CameraUI::ResetAdaptiveThresholderImage(){
    thresholderSlider->setSliderPosition(thresholderSlider->minimum());
    bbScenePixmapItem->setPixmap(croppedPixmap);
    if(!thresholdNoneRadio->isChecked()){
        UpdateAdaptiveThresholder();
    }
}

void CameraUI::UpdateAdaptiveThresholder(){
    float val = ((float) thresholderSlider->value() - thresholderSlider->minimum()) / (thresholderSlider->maximum() - thresholderSlider->minimum());

    // TEST FOR OVERLAY PIXMAP
        QPixmap retVal = QPixmap(bbScenePixmapItem->pixmap().size());
        QPainter p(&retVal);

        QImage thresholded;
        QRectF resegmentBB = resegmentCropItem->cropRect();

        if(thresholdGlobalRadio->isChecked()){

            QImage resegmentCropImage = croppedPixmap.toImage();
            resegmentCropImage = resegmentCropImage.copy(resegmentBB.toRect());
            thresholded = GloballyThreshold(resegmentCropImage, val*255);
        }else{

            qDebug() << val;
        thresholded = at->AdaptivelyThreshold((min(resegmentBB.width(), resegmentBB.height()))*val - 1, 0.9);
        }

        p.drawPixmap(bbScenePixmapItem->pixmap().rect(), bbScenePixmapItem->pixmap());
        p.drawPixmap(resegmentCropItem->cropRect().toRect(), QPixmap::fromImage(thresholded));
        bbScenePixmapItem->setPixmap(retVal);
        // END TEST FOR OVERLAY PIXMAP
}

void CameraUI::StartResegmentDialog(){

    // Hopefully no bad memory issues here:
    if(resegmentDialog != 0){
        resegmentDialog->deleteLater();
    }
    resegmentDialog = new QDialog();
    resegmentDialog->setWindowTitle("Manual Correction");
    QHBoxLayout resegmentDialogLayout;
    QVBoxLayout resegmentButtons;
    QHBoxLayout thresholdRadioButtonLayout;
    QGraphicsView resegmentGraphicsView;
    resegmentDialog->setModal(true);
    resegmentDialog->setLayout(&resegmentDialogLayout);
    resegmentDialogLayout.addWidget(&resegmentGraphicsView);
    QPushButton resegmentButton("Retry");
    resegmentButtons.addWidget(&resegmentButton);
    QLabel manualThresholdingLabel("Manual Thresholding Mode:");
    resegmentButtons.addWidget(&manualThresholdingLabel);
    resegmentButtons.addLayout(&thresholdRadioButtonLayout);
    resegmentDialogLayout.addLayout(&resegmentButtons);

    QObject::connect(&resegmentButton, SIGNAL(clicked()), this, SLOT(SendResegmentReq()));

    resegmentButtons.addWidget(thresholderSlider);
    thresholdRadioButtonLayout.addWidget(thresholdNoneRadio);
    thresholdRadioButtonLayout.addWidget(thresholdGlobalRadio);
    thresholdRadioButtonLayout.addWidget(thresholdAdaptiveRadio);
    resegmentButton.setFixedWidth(150);

    thresholderSlider->show();
    thresholderSlider->setDisabled(true);
    resegmentButtons.addStretch(1);
    thresholdNoneRadio->setChecked(true);

    // Get the CROPPED image, display it, and attach crop reticles over it
    // For convenience, we will reuse the bbScene, since its the same, cropped image we need. Just need to remember to remove the reticles when done

    resegmentGraphicsView.setScene(bbScene);
    resegmentGraphicsView.setSceneRect(bbScenePixmapItem->pixmap().rect());
    resegmentCropItem = new THGraphicsCropItem(bbScenePixmapItem);
    QObject::connect(resegmentCropItem, SIGNAL(CropChanged()), this, SLOT(ResetAdaptiveThresholder()));

    // Find the bounding box over ALL selected characters
    BoundingBoxCoords maxBoundingBox = GetSelectionBB();
    resegmentCropItem->setCropRect(QRectF(maxBoundingBox.left, maxBoundingBox.top,
                                         maxBoundingBox.right - maxBoundingBox.left, maxBoundingBox.bottom - maxBoundingBox.top));

    // Center the view on the center of the bounding region
    resegmentGraphicsView.centerOn((maxBoundingBox.left + maxBoundingBox.right) / 2, (maxBoundingBox.top + maxBoundingBox.bottom) / 2);
    //resegmentGraphicsView.fitInView(bbScenePixmapItem->pixmap().rect(), Qt::KeepAspectRatio);

    // Remove bounding box rect until after we are done resegmenting the image
    bbScene->removeItem(bbSceneRectItem);
    resegmentDialog->exec();
    resegmentCropItem->deleteLater();
    bbScene->addItem(bbSceneRectItem);
    ResetAdaptiveThresholder();
}

void CameraUI::ShowTimedOutDiscDialog(OCRRequest * req){
    // Don't need to delete here. Will be deleted on new send request.
    ocr_req->blockSignals(true);
    QDialog * timedOutDiscDialog = new QDialog();
    timedOutDiscDialog->setWindowTitle("Timed Out");
    timedOutDiscDialog->setModal(true);
    QLabel l("Request Timed Out. Please check your connection or try translating a smaller region.", timedOutDiscDialog);
    l.setWordWrap(true);
    l.show();
    timedOutDiscDialog->exec();
    timedOutDiscDialog->deleteLater();
    m_just_timed_out = true;
    ViewPhoto();
}

void CameraUI::ShowNoResultsDialog(){
    // Don't need to delete here. Will be deleted on new send request.
    ocr_req->blockSignals(true);
    QDialog * noResultsDialog = new QDialog();
    noResultsDialog->setWindowTitle("No Recognized Characters");
    noResultsDialog->setModal(true);
    QLabel l("No characters recognized. Please try a different image.", noResultsDialog);
    l.setWordWrap(true);
    l.show();
    noResultsDialog->exec();
    noResultsDialog->deleteLater();
    m_just_timed_out = true;
    ViewPhoto();
}

void CameraUI::ShowErrorConnectingDialog(OCRRequest *req){
    // Don't need to delete here. Will be deleted on new send request.
    ocr_req->blockSignals(true);
    QDialog * errorConnectingDialog = new QDialog();
    errorConnectingDialog->setWindowTitle("Connection Error");
    errorConnectingDialog->setModal(true);
    QLabel l("Error connecting to the server. Please check your connection.", errorConnectingDialog);
    l.setWordWrap(true);
    l.show();
    errorConnectingDialog->exec();
    errorConnectingDialog->deleteLater();
    m_just_timed_out = true;
    ViewPhoto();
}

BoundingBoxCoords CameraUI::GetSelectionBB(){
    QTextCursor cursor = resultView->textCursor();
    BoundingBoxCoords maxBoundingBox(bbScene->width(), bbScene->height(), 0, 0);
    for (int i = cursor.selectionStart(); i < cursor.selectionEnd(); i++){
        BoundingBoxCoords bbCoords = m_result_chars[i].bounding_box;
        if(bbCoords.bottom == 0 && bbCoords.left == 0 && bbCoords.right == 0 && bbCoords.top == 0){
            continue;
        }
        maxBoundingBox.bottom = max(maxBoundingBox.bottom, bbCoords.bottom);
        maxBoundingBox.top = min(maxBoundingBox.top, bbCoords.top);
        maxBoundingBox.left = min(maxBoundingBox.left, bbCoords.left);
        maxBoundingBox.right = max(maxBoundingBox.right, bbCoords.right);
    }
    return maxBoundingBox;
}

void CameraUI::SendResegmentReq(){
    QRectF resegmentBB = resegmentCropItem->cropRect();
    //QImage resegmentCropImage(crop_filename.c_str());
    QImage resegmentCropImage = bbScenePixmapItem->pixmap().toImage();
    resegmentCropImage = resegmentCropImage.copy(resegmentBB.toRect());
    QString reseg_filename = QDir::homePath() + QString("/.mir-translator/MIRATS_reseg.jpg");
    resegmentCropImage.save(reseg_filename);

    if(resegment_ocr_req != 0){
        resegment_ocr_req->deleteLater();
    }

    // Get the Amazon EC2 instance's IP and make the request to EC2
    QHostInfo ec2host = QHostInfo::fromName("mir.mhsueh.com");
    QHostAddress ec2addr = ec2host.addresses().first();

    //resegment_ocr_req = new OCRRequest("192.168.2.14", 20248, reseg_filename);
    resegment_ocr_req = new OCRRequest(ec2addr.toString(), 20248, reseg_filename);

    connect(resegment_ocr_req, SIGNAL(OCR_finished(QString)), this, SLOT(SetResegText(QString)));
    connect(resegment_ocr_req, SIGNAL(OCR_finished(QString)), resegmentDialog, SLOT(accept()));
    resegment_ocr_req->SetLang(sourceCBox->currentText());

    ResegOffsetX = resegmentCropItem->cropRect().left();
    ResegOffsetY = resegmentCropItem->cropRect().top();

    resegment_ocr_req->MakeOCRRequest();
    connect(quitButton, SIGNAL(clicked()), resegment_ocr_req->server_socket, SLOT(disconnectFromHostImplementation()));
}

void CameraUI::EnableThresholdSlider(){
    thresholderSlider->setDisabled(false);
}

void CameraUI::DisableThresholdSlider(){
    thresholderSlider->setDisabled(true);
}
