#ifndef CAMERAUI_H
#define CAMERAUI_H

#include <QThread>
#include <string>
#include <iostream>
#include <QTextEdit>
#include <QGraphicsView>
#include <QPushButton>
#include <QLabel>
#include <QComboBox>
#include "THGraphicsCropItem.h"
#include "ocrrequest.h"
#include <FCam/N900.h>
#include <QList>
#include <QTableWidget>
#include "BoundingBoxQGraphicsView.h"
#include <QHBoxLayout>
#include <QRadioButton>
#include <QButtonGroup>
#include <QSlider>

class OverlayWidget;
class InfoWidget;

using namespace std;

const int num_alts = 4;
const bool DEBUG = false;

class CameraUI : public QObject {
    Q_OBJECT;

public:
    CameraUI(OverlayWidget *o, QTextEdit *rv, QLabel *rl, QTextEdit *tv, QLabel *tl, QGraphicsView *iv, QGraphicsView *bv,
             QPushButton *sendB, QPushButton *cameraB, QPushButton *photoB,
             QPushButton * textB, QPushButton *quitB, QComboBox *srcCB, QComboBox *destCB, QLabel *statusL,
             QTableWidget * altT, QPushButton * retransB, BoundingBoxQGraphicsView *bbv, QLabel *candL, QLabel * srcLangL,
             QLabel * destLangL, QLabel * manL, QObject *parent = NULL) : QObject(parent) {
        overlay = o;
        resultView = rv;
        imageView = iv;
        busyView = bv;
        bbView = bbv;
        keepGoing = true;
        src_filename = "";
        crop_filename = "";
        sendButton = sendB;
        cameraButton = cameraB;
        textButton = textB;
        quitButton = quitB;
        sourceCBox = srcCB;
        destCBox = destCB;
        statusLabel = statusL;
        photoButton = photoB;
        resultLabel = rl;
        translateView = tv;
        translateLabel = tl;
        ocr_req = 0;
        alt_table = altT;
        retransButton = retransB;
        retransEnabled = false;
        resegment_ocr_req = 0;
        candidatesLabel = candL;
        sourceLangLabel = srcLangL;
        destLangLabel = destLangL;
        manualCorrectionLabel = manL;

        //Used for photo viewing & cropping
        viewScene = new QGraphicsScene(QRectF(0, 0, imageView->geometry().width(), imageView->geometry().height()), 0);
        viewScenePixmapItem = new QGraphicsPixmapItem;
        viewScene->addItem(viewScenePixmapItem);
        imageView->setScene(viewScene);
        cropItem = new THGraphicsCropItem(viewScenePixmapItem);
        connect(cropItem, SIGNAL(CropChanged()), this, SLOT(CropChanged()));

        // Used for displaying the bb contents
        bbScene = new QGraphicsScene();
        bbScenePixmapItem = new QGraphicsPixmapItem;
        bbScene->addItem(bbScenePixmapItem);
        bbSceneRectItem = new QGraphicsRectItem(0, 0 ,0, 0); // Initialize a "nothing" rectangle at first RECT TAKEN OUT REPLACED WITH THCROPITEM
        bbScene->addItem(bbSceneRectItem);
        bbView->setScene(bbScene);

        translatedText = "No translation request made yet.";
        bTranslationComplete = false;

        connect(this, SIGNAL(ImageIsSaving()), this, SLOT(viewBusyScreen()));
        connect(this, SIGNAL(ImageDoneSaving()), this, SLOT(ViewPhoto()));

        connect(sourceCBox, SIGNAL(currentIndexChanged(int)), this, SLOT(EnableRetranslate()));
        connect(destCBox, SIGNAL(currentIndexChanged(int)), this, SLOT(EnableRetranslate()));

        InitLangStrings();
        PopulateLangCBox(sourceCBox);
        PopulateLangCBox(destCBox);

        thresholdRadioButtonGroup = new QButtonGroup();
        thresholdNoneRadio = new QRadioButton("None");
        thresholdGlobalRadio = new QRadioButton("Global");
        thresholdAdaptiveRadio = new QRadioButton("Adaptive");
        thresholdRadioButtonGroup->addButton(thresholdNoneRadio);
        thresholdRadioButtonGroup->addButton(thresholdGlobalRadio);
        thresholdRadioButtonGroup->addButton(thresholdAdaptiveRadio);
        thresholdNoneRadio->setFixedWidth(80);
        thresholdGlobalRadio->setFixedWidth(90);
        thresholdAdaptiveRadio->setFixedWidth(110);
        thresholderSlider = new QSlider(Qt::Horizontal);
        thresholderSlider->setFixedWidth(280);
        at = new AdaptiveThresholder();

        QObject::connect(thresholdNoneRadio, SIGNAL(clicked()), this, SLOT(DisableThresholdSlider()));
        QObject::connect(thresholdNoneRadio, SIGNAL(clicked()), this, SLOT(ResetAdaptiveThresholderImage()));
        QObject::connect(thresholdGlobalRadio, SIGNAL(clicked()), this, SLOT(EnableThresholdSlider()));
        QObject::connect(thresholdGlobalRadio, SIGNAL(clicked()), this, SLOT(ResetAdaptiveThresholderImage()));
        QObject::connect(thresholdAdaptiveRadio, SIGNAL(clicked()), this, SLOT(EnableThresholdSlider()));
        QObject::connect(thresholdAdaptiveRadio, SIGNAL(clicked()), this, SLOT(ResetAdaptiveThresholderImage()));


        QObject::connect(thresholdAdaptiveRadio, SIGNAL(clicked()), this, SLOT(StartAdaptiveThresholder()));

        QObject::connect(thresholderSlider, SIGNAL(sliderMoved(int)), this, SLOT(UpdateAdaptiveThresholder()));

        // Connect the retranslate button to cameraUI's SLOT for Make_Retrans_Request
        connect(retransButton, SIGNAL(clicked()), this, SLOT(Make_Retrans_Request()));

        resegmentDialog = 0;
    }



private:
    string src_filename;
    string crop_filename;
    void AdjustAltTableSize();
    bool keepGoing;
    OverlayWidget *overlay;
    QTextEdit *resultView;
    QGraphicsView *imageView;
    QGraphicsView *busyView;
    BoundingBoxQGraphicsView *bbView;
    QPushButton *sendButton;
    QPushButton *cameraButton;
    QPushButton *textButton;
    QPushButton *quitButton;    
    QPushButton *photoButton;
    QComboBox *sourceCBox;
    QComboBox *destCBox;
    QHash<QString, QString> langStrings;
    OCRRequest *ocr_req;
    QPixmap croppedPixmap;
    bool m_just_timed_out;

    QLabel *resultLabel;
    QTextEdit *translateView;
    QLabel *translateLabel;
    QLabel *statusLabel;
    QLabel *candidatesLabel;
    QLabel *sourceLangLabel;
    QLabel *destLangLabel;
    QLabel *manualCorrectionLabel;

    //Used for photo viewing & cropping
    QGraphicsScene *viewScene;
    QGraphicsPixmapItem *viewScenePixmapItem;
    THGraphicsCropItem *cropItem;

    //Used for bb alternative contents
    QGraphicsScene *bbScene;
    QGraphicsPixmapItem *bbScenePixmapItem;
    QGraphicsRectItem *bbSceneRectItem;

    // Resegmentation
    // The client used for requesting translation/OCR services
    OCRRequest *resegment_ocr_req;
    THGraphicsCropItem * resegmentCropItem;
    QDialog * resegmentDialog;
    int ResegOffsetX;
    int ResegOffsetY;

    // Thresholding controls
    QRadioButton * thresholdNoneRadio;
    QRadioButton * thresholdGlobalRadio;
    QRadioButton * thresholdAdaptiveRadio;
    QButtonGroup * thresholdRadioButtonGroup;
    QSlider * thresholderSlider;
    AdaptiveThresholder * at;

    QString translatedText;
    QString OCRdText;
    QString langMode;
    bool bTranslationComplete;
    QList<ResultChar> m_result_chars;

    //Alts processing
    QTableWidget * alt_table;
    QPushButton * retransButton;
    bool retransEnabled;
    QTextCursor oldTextCursor;

    void updateResultView(){
        int pos = resultView->textCursor().selectionEnd();

        // Need to disconnect the cursor changed signal before we mess with the result text
        QObject::disconnect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));

        resultView->clear();
        QTextCursor cursor = resultView->textCursor();
        QTextCharFormat red_text;
        QTextCharFormat black_text;
        QBrush red_brush(Qt::red);
        QBrush black_brush(Qt::black);
        red_text.setForeground(red_brush);
        black_text.setForeground(black_brush);
        red_text.setFontPointSize(28);
        black_text.setFontPointSize(28);

        // Insert the char in the color that is appropriate for the confidence level
        for(int i = 0; i < m_result_chars.size(); i++){
            ResultChar rc = m_result_chars[i];
            if(rc.confidence < 85){
                cursor.insertText(rc.alts[rc.selected_alt], red_text);
            }else{
                cursor.insertText(rc.alts[rc.selected_alt], black_text);
            }
        }

        // Reconnect signals
        QObject::connect(resultView, SIGNAL(cursorPositionChanged()), this, SLOT(HandleResultCursorChange()));
        cursor.setPosition(pos);
        resultView->setTextCursor(cursor);
    }

    BoundingBoxCoords GetSelectionBB();

    void ParseOCRdText();

public slots:
    void stop() {
        exit(0);
    }
    void DeleteSrcPhoto();
    void deleteCropPhoto();
    void clearResults();
    void send();
    void viewText();
    void HandleResultCursorChange();
    void viewCamera();
    void ViewPhoto();
    void CorrectVignetting(QImage * img, QPoint offset);
    QImage ConvertToGrayScale(QImage & img);
    void SetTranslationText(QString text);
    void SetOCRText(QString text);
    void SetResegText(QString text);
    void viewBusyScreen();
    void ProcessCropAndSend(QImage img, QPoint topLeft, bool correctVignetting);
    void SetFileNameAndViewImage(QString fname);
    void CropChanged();
    void ChangeChoiceToAlt(int r, int c);
    void EnableRetranslate();
    void Make_Retrans_Request();
    void InitLangStrings();
    void PopulateLangCBox(QComboBox*);
    void StartResegmentDialog();
    void SendResegmentReq();
    //void StartGlobalThresholder();
    void StartAdaptiveThresholder();
    void UpdateAdaptiveThresholder();
    void ResetAdaptiveThresholder();
    void ResetAdaptiveThresholderImage();
    void EnableThresholdSlider();
    void DisableThresholdSlider();
    void ShowTimedOutDiscDialog(OCRRequest * req);
    void ShowErrorConnectingDialog(OCRRequest *req);


signals:
    void ImageIsSaving();
    void ImageDoneSaving();
    void CropObtained(QImage srcimg, QPoint offset, bool correctVignetting);
    void SetOCRText_Finished(QString, QString, QString);
};

#endif // CAMERAUI_H
