/** \file */

#include <QApplication>
#include <QPushButton>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QSlider>
#include <QTextEdit>
#include <QImage>
#include <QGraphicsView>
#include <QLabel>
#include <QScrollBar>
#include <QString>
#include <QListWidget>
#include <QComboBox>
#include <QStandardItemModel>
#include <QStandardItem>
#include <QListView>
#include <QTableWidget>
#include <QHeaderView>
#include <vector>

#include "OverlayWidget.h"
#include "CameraThread.h"
#include "cameraui.h"
#include "BoundingBoxQGraphicsView.h"

/***********************************************************/
/* MIRATS Application
   Sets up the Qt application. Creates buttons, layouts, camera
   thread, UI. Starts the application */
/***********************************************************/

int main(int argc, char **argv) {
    QApplication app(argc, argv);

    // Make the main window and a layout for it
    QWidget *window = new QWidget;
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);

    // Make an overlay for displaying viewfinder frames
    OverlayWidget *overlay = new OverlayWidget(window);
    layout->addWidget(overlay);

    // Image view for viewing and cropping the captured image
    QGraphicsView *imageView = new QGraphicsView();
    imageView->setFixedSize(640, 480);
    imageView->hide();
    layout->addWidget(imageView);

    // Result View for viewing the results
    QLabel *resultLabel = new QLabel("Selected Text:");
    QLabel *translatedLabel = new QLabel("Translated Text:");

    QLabel *candidatesLabel = new QLabel("Candidates:");
    QLabel *srcLangLabel = new QLabel("From:");
    QLabel *destLangLabel = new QLabel("To:");
    QLabel *manualCorrectionLabel = new QLabel("Click to Fix:");

    QTextEdit *resultView = new QTextEdit();
    resultView->setReadOnly(true);
    QTextEdit *translatedView = new QTextEdit();
    translatedView->setReadOnly(true);
    translatedView->setFontPointSize(28);
    QVBoxLayout *resultLayout = new QVBoxLayout;
    resultLayout->addWidget(resultLabel);
    resultLayout->addWidget(resultView);
    resultLayout->addWidget(translatedLabel);
    resultLayout->addWidget(translatedView);
    layout->addLayout(resultLayout);
    resultView->hide();

    // Busy View used between snapping the photo and displaying it
    QGraphicsView *busyView = new QGraphicsView();
    busyView->setFixedSize(640, 480);
    busyView->hide();
    layout->addWidget(busyView);
    QGraphicsScene busyScene;
    busyScene.addText("Please Wait...");
    busyView->setScene(&busyScene);

    // Make some buttons on the right
    QVBoxLayout *buttonLayout = new QVBoxLayout;
    layout->addLayout(buttonLayout);
    QPushButton *quitButton = new QPushButton("Close");
    QPushButton *cameraButton = new QPushButton("Back");
    QPushButton *retranslateButton = new QPushButton("Retranslate");
    QPushButton *sendButton = new QPushButton("Send");
    QPushButton *textButton = new QPushButton("Result");
    QPushButton *photoButton = new QPushButton("Back");
    retranslateButton->setFixedSize(150, 40);
    quitButton->setFixedSize(150, 40);
    sendButton->setFixedSize(150, 40);
    textButton->setFixedSize(150, 40);
    cameraButton->setFixedSize(150, 40);
    photoButton->setFixedSize(150,40);

    // Create Language selection buttons
    QHBoxLayout * langButtonsLayout = new QHBoxLayout();
    QComboBox *sourceLangComboBox = new QComboBox();
    QComboBox *destLangComboBox = new QComboBox();
    /*
    langButtonsLayout->addWidget(sourceLangComboBox);
    langButtonsLayout->addWidget(destLangComboBox);
    sourceLangComboBox->setFixedWidth(75);
    destLangComboBox->setFixedWidth(75);
*/
    sourceLangComboBox->setFixedSize(150,40);
    destLangComboBox->setFixedSize(150,40);

    // Create Status Label
    QLabel *statusLabel = new QLabel();
    statusLabel->setFixedWidth(150);
    statusLabel->setWordWrap(true);
    statusLabel->setAlignment(Qt::AlignCenter);

    //Create the alternate candidtate table choices box
    QTableWidget * altTable = new QTableWidget(1, num_alts);
    for(int i = 0; i < num_alts; i++){
        QTableWidgetItem * table_item = new QTableWidgetItem("");
        altTable->setItem(0, i, table_item);
    }
    altTable->verticalHeader()->setVisible(false);
    altTable->horizontalHeader()->setVisible(false);
    altTable->setSelectionMode(QAbstractItemView::SingleSelection);
    altTable->setShowGrid(false);

    // Create bounding box graphics view widget
    BoundingBoxQGraphicsView *bbView = new BoundingBoxQGraphicsView();
    bbView->setFixedSize(150, 150);
    bbView->hide();

    // Add buttons to the button layout
    buttonLayout->addWidget(quitButton);
    buttonLayout->addWidget(statusLabel);
    buttonLayout->addWidget(candidatesLabel);
    buttonLayout->addWidget(altTable, 0, Qt::AlignHCenter);
    QSpacerItem space(150, 20);
    buttonLayout->addSpacerItem(&space);
    buttonLayout->addWidget(manualCorrectionLabel);
    buttonLayout->addWidget(bbView);
    buttonLayout->addStretch(1);
    //buttonLayout->addLayout(langButtonsLayout);
    buttonLayout->addWidget(srcLangLabel);
    buttonLayout->addWidget(sourceLangComboBox);
    buttonLayout->addWidget(destLangLabel);
    buttonLayout->addWidget(destLangComboBox);

    buttonLayout->addWidget(textButton);
    buttonLayout->addWidget(sendButton);
    buttonLayout->addWidget(retranslateButton);
        buttonLayout->addWidget(photoButton);
    buttonLayout->addWidget(cameraButton);

    window->setLayout(layout);

    // Create a camera UI for controlling the UI / window
    CameraUI *cameraUI = new CameraUI(overlay, resultView, resultLabel, translatedView, translatedLabel, imageView, busyView,
                                      sendButton, cameraButton, photoButton, textButton, quitButton, sourceLangComboBox, destLangComboBox, statusLabel, altTable,
                                      retranslateButton, bbView, candidatesLabel, srcLangLabel, destLangLabel, manualCorrectionLabel, window);

    // Make a thread that controls the camera
    CameraThread *cameraThread = new CameraThread(overlay);

    // Connect signals
    QObject::connect(altTable, SIGNAL(cellActivated(int,int)), cameraUI, SLOT(ChangeChoiceToAlt(int, int)));
    QObject::connect(cameraThread, SIGNAL(image_saving(QString)), cameraUI, SLOT(viewBusyScreen()));
    QObject::connect(cameraThread, SIGNAL(image_saved(QString)), cameraUI, SLOT(SetFileNameAndViewImage(QString)));
    QObject::connect(cameraUI, SIGNAL(CropObtained(QImage, QPoint, bool)), cameraUI, SLOT(ProcessCropAndSend(QImage, QPoint, bool)));
    QObject::connect(bbView, SIGNAL(clicked()), cameraUI, SLOT(StartResegmentDialog()));

    // Hook up the buttons
    QObject::connect(quitButton, SIGNAL(clicked()), 
                     cameraThread, SLOT(stop()));
    QObject::connect(photoButton, SIGNAL(clicked()),
                     cameraUI, SLOT(ViewPhoto()));
    QObject::connect(sendButton, SIGNAL(clicked()),
                     cameraUI, SLOT(send()));
    QObject::connect(textButton, SIGNAL(clicked()),
                     cameraUI, SLOT(viewText()));
    QObject::connect(cameraButton, SIGNAL(clicked()),
                     cameraUI, SLOT(viewCamera()));
    QObject::connect(quitButton, SIGNAL(clicked()),
                     cameraUI, SLOT(deleteCropPhoto()));
    QObject::connect(quitButton, SIGNAL(clicked()),
                     cameraUI, SLOT(DeleteSrcPhoto()));

    // Once the camera thread stops, quit the app
    QObject::connect(cameraThread, SIGNAL(finished()),
                     &app, SLOT(quit()));


    // Show the app full screen
    window->showFullScreen();

    // Launch the camera thread
    cameraThread->start();

    // Start the app with the camera view
    cameraUI->viewCamera();

    // Enter the QT main event loop
    return app.exec();
}
