/*
 * Copyright (c) 2005 Jukka Zitting <jz@yukatan.fi>
 * Copyright (c) 2005,2007 Piotr Pokora <piotr.pokora@infoglob.pl>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include "midgard_schema.h"
#include "query_builder.h"
#include "query_constraint.h"
#include "group_constraint.h"
#include "query_order.h"
#include "midgard_object_class.h"
#include "schema.h"
#include "midgard_core_query_builder.h"
#include "midgard_error.h"
#include "midgard_core_object.h"
#include "midgard_core_query.h"
#include "midgard_datatypes.h"
#include "midgard_reflection_property.h"
#include "midgard_object.h"
#include "midgard_metadata.h"
#include "midgard_sitegroup.h"
#include "midgard_user.h"
#include "midgard_core_object_class.h"
#include "midgard_dbobject.h"
#include "midgard_timestamp.h"
#include "midgard_core_metadata.h"

/* Internal prototypes , I am not sure if should be included in API */
gchar *midgard_query_builder_get_object_select(MidgardQueryBuilder *builder, guint select_type);

static gboolean __type_is_valid(GType type)
{
	GType _type = type;
	if(!_type)
		return FALSE;

	if(type == MIDGARD_TYPE_DBOBJECT 
			|| type == MIDGARD_TYPE_OBJECT)
		return TRUE;

	/* check MIDGARD_TYPE_OBJECT */
	if(g_type_parent(_type) != MIDGARD_TYPE_DBOBJECT) {
		
		/* if fails, try MIDGARD_TYPE_DBOBJECT */
		if(g_type_parent(_type) != MIDGARD_TYPE_OBJECT) {
			
			return FALSE;
		
		} else {

			return TRUE;
		}
	}

	return TRUE;
}

MidgardQueryBuilder *midgard_query_builder_new(
        MidgardConnection *mgd, const gchar *classname)
{
        g_assert(mgd != NULL);
        g_assert(classname != NULL);

	GType class_type = g_type_from_name(classname);

	if(!__type_is_valid(class_type)) {

		g_warning("Can not initialize Midgard Query Builder for '%s'. It's not registered GType system class", classname);
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INVALID_OBJECT);
		return NULL;
	}

        MidgardQueryBuilder *builder = 
		g_object_new(MIDGARD_TYPE_QUERY_BUILDER, NULL);
	
	builder->priv = midgard_query_builder_private_new();

        builder->priv->mgd = mgd;
        builder->priv->type = g_type_from_name(classname);
        builder->priv->lang = -1;
	builder->priv->unset_lang = FALSE;
	builder->priv->include_deleted = FALSE;
        builder->priv->error = 0;	
	
	MidgardDBObjectClass *klass = 
		(MidgardDBObjectClass*) g_type_class_peek(class_type);

	if (klass->dbpriv == NULL) {
		
		g_warning("Given %s class has no storage definitions", g_type_name(class_type));
		g_object_unref(builder);
		return NULL;
	}

	builder->priv->schema = klass->dbpriv->storage_data;
       	
	gchar **tables = g_strsplit(midgard_core_class_get_tables(MIDGARD_DBOBJECT_CLASS(klass)), ",", 0);
	guint i = 0;
	while(tables[i] != NULL) {	
		midgard_core_qb_add_table(builder, 
				(const gchar *)g_strchug(tables[i]));
		i++;
	}
	g_strfreev(tables);

        builder->priv->offset = 0;
        builder->priv->limit = G_MAXUINT;
	builder->priv->sitegrouped = TRUE;

        if (builder->priv->type && builder->priv->schema) {
                return builder;
        } else {
		g_object_unref(builder);
                return NULL;
        }
}

void midgard_query_builder_free(MidgardQueryBuilder *builder) 
{
	g_assert(builder != NULL);

	midgard_query_builder_private_free(builder->priv);
}

gboolean midgard_query_builder_add_constraint(
        MidgardQueryBuilder *builder,
        const gchar *name, const gchar *op, const GValue *value) 
{
	g_assert(builder);
        g_assert(name);
        g_assert(op);
        g_assert(value);
     
	MidgardQueryConstraint *constraint = midgard_query_constraint_new();
	
	if(!midgard_query_constraint_add_operator(constraint, op))
		return FALSE;
		
	if(!midgard_core_qb_parse_property(builder, &constraint, name))
		return FALSE;

	if(!midgard_query_constraint_add_value(constraint, value))
		return FALSE;
	
	midgard_query_constraint_build_condition(constraint);

	/* FIXME, table should be stored per every constraint, order, whatever */
	midgard_core_qb_add_table(builder, constraint->priv->prop_left->table);
	
	if(builder->priv->grouping_ref > 0) {

		MidgardGroupConstraint *group = 
			(MidgardGroupConstraint *)builder->priv->group_constraint;
		midgard_group_constraint_add_constraint(group, constraint);
		return TRUE;
	
	} else {

		midgard_core_qb_add_constraint(builder, constraint);
	}

	midgard_core_qb_add_constraint(builder, constraint);

	return TRUE;
}

gboolean midgard_query_builder_add_constraint_with_property(
		MidgardQueryBuilder *builder, const gchar *property_a,
		const gchar *op, const gchar *property_b)
{
	g_assert(builder != NULL);
	g_assert(property_a != NULL);
	g_assert(op != NULL);
	g_assert(property_b != NULL);

	MidgardQueryConstraint *constraint = midgard_query_constraint_new();

	if(!midgard_core_qb_parse_property(builder, &constraint, property_a))
		return FALSE;
	constraint->priv->current = constraint->priv->prop_right;
	if(!midgard_core_qb_parse_property(builder, &constraint, property_b))
		return FALSE;

	constraint->priv->condition_operator = g_strdup(op);

	midgard_query_constraint_build_condition(constraint);

	midgard_core_qb_add_table(builder, constraint->priv->prop_left->table);
	midgard_core_qb_add_table(builder, constraint->priv->prop_right->table);

	if(builder->priv->grouping_ref > 0) {
		
		MidgardGroupConstraint *group = builder->priv->group_constraint;
		midgard_group_constraint_add_constraint(group, constraint);
		return TRUE;
	
	} else {
		
		midgard_core_qb_add_constraint(builder, constraint);
	}
	
	return TRUE;
}

gboolean midgard_query_builder_begin_group(
                MidgardQueryBuilder *builder, const gchar *type) 
{
	g_assert(builder != NULL);
	
	const gchar *group_op = type;

	if(group_op == NULL)
		group_op = "OR";

        MidgardGroupConstraint *group = 
		midgard_group_constraint_new(group_op);

	if(!group)
		return FALSE;

	/* Check if this is nested grouping */
	if(builder->priv->grouping_ref > 0) {
		
		MidgardGroupConstraint *pgroup = builder->priv->group_constraint;
		pgroup->nested = g_slist_prepend(pgroup->nested, group);
		group->parent_group = pgroup;
	
	} else {
		
		builder->priv->groups =
			g_slist_prepend(builder->priv->groups, group);
	}
	
	builder->priv->group_constraint = group;
	builder->priv->grouping_ref++;
	builder->priv->is_grouping = TRUE;

	return TRUE;
}

gboolean midgard_query_builder_end_group(MidgardQueryBuilder *builder) 
{
	g_assert(builder != NULL);

	if(builder->priv->grouping_ref == 0) {
		
		g_warning("Missed 'begin_group' method");
		return FALSE;
	}

	MidgardGroupConstraint *group = NULL;
	
	if(builder->priv->group_constraint)
		group = MIDGARD_GROUP_CONSTRAINT(builder->priv->group_constraint);
	
	if(group && group->parent_group) {
	
		builder->priv->group_constraint = group->parent_group;

	} else {
		
		builder->priv->group_constraint = NULL;
	}

	builder->priv->grouping_ref--;

	return TRUE;
}

gboolean midgard_query_builder_add_order(
        MidgardQueryBuilder *builder, const gchar *name, const gchar *dir)
{
        g_assert(builder != NULL);
        g_assert(name != NULL);

	MidgardQueryOrder *order = 
		midgard_core_query_order_new(builder, name, dir);

	if (order) {
		
		midgard_core_qb_add_order(builder, order);
		return TRUE;
	
	} else {

		g_warning("Skipping a ordering constraint specification");
		return FALSE;
	}
}

void midgard_query_builder_set_limit(
        MidgardQueryBuilder *builder, guint limit)
{
	g_assert(builder != NULL);

        builder->priv->limit = limit;
}

void midgard_query_builder_set_offset(
        MidgardQueryBuilder *builder, guint offset)
{
        g_assert(builder != NULL);

        builder->priv->offset = offset;
}

void midgard_query_builder_set_lang(MidgardQueryBuilder *builder, gint lang)
{
	g_assert(builder != NULL);
	
	builder->priv->lang = lang;
}

void midgard_query_builder_unset_languages(MidgardQueryBuilder *builder)
{
	g_assert(builder);
	
	builder->priv->unset_lang = TRUE;
}

gint _compare_ml_guid(gconstpointer a, gconstpointer b){

        if(g_ascii_strcasecmp(a,b) == 0) return 0;
        return 1;
}

static GList *midgard_query_builder_execute_or_count(
	MidgardQueryBuilder *builder, MidgardTypeHolder *holder, guint select_type)
{
        g_assert(builder != NULL);

	if(builder->priv->grouping_ref > 0) {
		
		g_warning("Incorrect constraint grouping. Missed 'end_group'?");
		return NULL;
	}

	MidgardUser *user = midgard_connection_get_user(builder->priv->mgd);

	if(builder->priv->type == MIDGARD_TYPE_SITEGROUP) {
		
		if(!midgard_user_is_root(user)) {
			
			MIDGARD_ERRNO_SET(builder->priv->mgd, MGD_ERR_ACCESS_DENIED);
			g_warning("Type incompatible with Midgard Query Builder");
			return NULL;
		}
	}

	if(builder->priv->type == MIDGARD_TYPE_USER) {

		if(midgard_user_is_user(user)) {

			if(user->dbpriv->guid == NULL) {
				
				MIDGARD_ERRNO_SET(builder->priv->mgd, MGD_ERR_ACCESS_DENIED);
				g_warning("Type incompatible with Midgard Query Builder");
				return NULL;
			}
			
			GValue gval = {0, };
			g_value_init(&gval, G_TYPE_STRING);
			g_value_set_string(&gval, user->dbpriv->guid);

			midgard_query_builder_add_constraint(builder, 
					"guid", "=", &gval);		
			
		}
	} 
        
        GList *list = midgard_core_qb_set_object_from_query(builder, select_type, NULL);
        if(list == NULL){
		if(holder)
			 holder->elements = 0;
                return NULL; 
        }
        
       	if(holder)
		holder->elements = g_list_length(list);

	return list;	 
}

GObject **midgard_query_builder_execute(
	MidgardQueryBuilder *builder, guint *n_objects)
{
	g_assert(builder != NULL);
	g_assert(n_objects != NULL);

	*n_objects = 0;

	guint i = 0;
	GList *list = NULL;
	MIDGARD_ERRNO_SET(builder->priv->mgd, MGD_ERR_OK);

	MidgardTypeHolder *holder = g_new(MidgardTypeHolder, 1);

        list = midgard_query_builder_execute_or_count(builder, holder, MQB_SELECT_OBJECT);
	
	if(list == NULL) {
		
		g_free(holder);
		return NULL;
	}

	*n_objects = holder->elements;
	MgdObject **objects = g_new(MgdObject *, holder->elements+1);	

	for( ; list; list = list->next){
		objects[i] = list->data;
		i++;
	}	
	objects[i] = NULL; /* Terminate by NULL */

	g_list_free(list);
	g_free(holder);

	return (GObject **)objects;	
}

guint midgard_query_builder_count(MidgardQueryBuilder *builder)
{
	g_assert(builder != NULL);
	MIDGARD_ERRNO_SET(builder->priv->mgd, MGD_ERR_OK);
	
	GList *list =
		midgard_query_builder_execute_or_count(builder, NULL, MQB_SELECT_GUID);
	
	if(!list)
		return 0;

	MidgardTypeHolder *holder = (MidgardTypeHolder *)list->data;

	if(!holder) {
		g_list_free(list);
		return 0;
	}

	guint elements = holder->elements;
	
	g_free(holder);
	g_list_free(list);
	
	return elements;
}

GList *midgard_query_builder_get_guid(
		MidgardQueryBuilder *builder)
{
	g_assert(builder != NULL);
	
	GList *list = 
		midgard_query_builder_execute_or_count(builder, NULL, MQB_SELECT_GUID);
	
	return list;	
}


gchar *midgard_query_builder_get_object_select(
        MidgardQueryBuilder *builder, guint select_type){

        g_assert(builder != NULL);
        g_assert(builder->priv->type);
	
	MidgardObjectClass *klass = (MidgardObjectClass*) g_type_class_peek(builder->priv->type);
	const gchar *table = midgard_core_class_get_table(MIDGARD_DBOBJECT_CLASS(klass));

        if (!table){
                g_warning("Object '%s' has no table or storage defined!", 
                                g_type_name(builder->priv->type));
                return NULL;
        }
		
        GString *select = g_string_new("");

	/* We are getting only guids */
	if(select_type == MQB_SELECT_GUID){
		
		g_string_append_printf(select, "COUNT(%s.guid) ", table);
		return g_string_free(select, FALSE);
	}
	
        /* guid , sitegroup hardcoded ( inherited from MgdObjectClass )
         * metadata properties hardcoded ( defined in midgard_metadata )
         */ 
	
	if (MIDGARD_DBOBJECT_CLASS(klass)->dbpriv->has_metadata) {

		g_string_append_printf(select, 
				"%s.guid, %s.sitegroup, "
				"%s.metadata_creator, "
				"%s.metadata_created, "
				"%s.metadata_revisor, "
				"%s.metadata_revised, "
				"%s.metadata_revision, "
				"%s.metadata_locker, "
				"%s.metadata_locked, "
				"%s.metadata_approver, "
				"%s.metadata_approved, "
				"%s.metadata_authors, "
				"%s.metadata_owner, "
				"%s.metadata_schedule_start, "
				"%s.metadata_schedule_end, "
				"%s.metadata_hidden, "
				"%s.metadata_nav_noentry, "
				"%s.metadata_size, "
				"%s.metadata_published, "
				"%s.metadata_exported, "
				"%s.metadata_imported, "
				"%s.metadata_deleted, "
				"%s.metadata_score, "
				"%s.metadata_islocked, "
				"%s.metadata_isapproved "
				", ", 
			table, table, table, table, table, table, table, table, table,
			table, table, table, table, table, table, table, table, table,
			table, table, table, table, table, table, table); 
		/* TODO, Set select which suits better */
	}
        
	if(builder->priv->schema->query->select_full != NULL) {
		g_string_append_printf(select, " %s", 
				builder->priv->schema->query->select_full);
	}

        return g_string_free(select, FALSE);        
}

#define _SET_METADATA_UINT(__prop, __value) \
	if(G_VALUE_HOLDS_UINT(__value)) { \
		__prop = \
			g_value_get_uint(__value); \
	} \
	if(G_VALUE_HOLDS_INT(__value)) { \
		__prop = \
			(guint)g_value_get_int(__value); \
	}  

#define _SET_METADATA_TIMESTAMP(__pname) { \
	g_value_init(&mt, MGD_TYPE_TIMESTAMP); \
	g_value_transform(gvalue, &mt); \
	midgard_core_metadata_set_##__pname(mdata, &mt); \
	g_value_unset(&mt); }

static gboolean __mqb_set_metadata(MidgardMetadata *mdata, GdaDataModel *model, gint i)
{
	guint col = 1;
	const GValue *gvalue;	
	GValue mt = {0, };

	/* creator */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->creator);
	mdata->priv->creator = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->creator =
			g_value_dup_string(gvalue);
	}

	/* created */	
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(created);

	/* revisor */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->revisor);
	mdata->priv->revisor = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->revisor = 
			g_value_dup_string(gvalue);
	}
		
	/* revised */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(revised);

	/* revision */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->revision = 0;
	_SET_METADATA_UINT(mdata->priv->revision, gvalue);
		
	/* locker */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->locker);
	mdata->priv->locker = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->locker = 
			g_value_dup_string(gvalue);
	}
		
	/* locked */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(locked);

	/* approver */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->approver);
	mdata->priv->approver = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->approver = 
			g_value_dup_string(gvalue);
	}

	/* approved */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(approved);

	/* authors */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->authors);
	mdata->priv->authors = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->authors = 
			g_value_dup_string(gvalue);
	}
		
	/* owner */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_free(mdata->priv->owner);
	mdata->priv->owner = NULL;
	if(G_VALUE_HOLDS_STRING (gvalue)) {
		mdata->priv->owner = 
			g_value_dup_string(gvalue);
	}

	/* schedule_start */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_value_init(&mt, MGD_TYPE_TIMESTAMP); 
	g_value_transform(gvalue, &mt); 
	mdata->priv->schedule_start = g_value_dup_boxed(&mt);
	g_value_unset(&mt);

	/* schedule_end */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_value_init(&mt, MGD_TYPE_TIMESTAMP); 
	g_value_transform(gvalue, &mt); 
	mdata->priv->schedule_end = g_value_dup_boxed(&mt);
	g_value_unset(&mt);

	/* hidden */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->hidden = FALSE;
	if(G_VALUE_HOLDS_BOOLEAN(gvalue)) {
		mdata->priv->hidden = 
			g_value_get_boolean(gvalue);
	}

	/* nav_noentry */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->nav_noentry = FALSE;
	if(G_VALUE_HOLDS_BOOLEAN(gvalue)) {
		mdata->priv->nav_noentry = 
			g_value_get_boolean(gvalue);
	}

	/* size */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->size = 0;
	if(G_VALUE_HOLDS_INT(gvalue)) {
		mdata->priv->size = (guint) g_value_get_int(gvalue);
	}	

	/* published */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	g_value_init(&mt, MGD_TYPE_TIMESTAMP); 
	g_value_transform(gvalue, &mt); 
	mdata->priv->published = g_value_dup_boxed(&mt);
	g_value_unset(&mt);

	/* exported */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(exported);

	/* imported */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	_SET_METADATA_TIMESTAMP(imported);

	/* deleted */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->deleted = FALSE;
	if(G_VALUE_HOLDS_BOOLEAN(gvalue)) {
		mdata->priv->deleted = 
			g_value_get_boolean(gvalue);
	}
		
	/* score */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	mdata->priv->score = 0;
	if(G_VALUE_HOLDS_INT(gvalue)) {
		mdata->priv->score = 
			g_value_get_int(gvalue);
	}

	/* islocked */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	if(G_VALUE_HOLDS_BOOLEAN(gvalue)) {
		mdata->priv->is_locked = 
			g_value_get_boolean(gvalue);
	}

	/* isapproved */
	gvalue = midgard_data_model_get_value_at(model, ++col, i);
	if(G_VALUE_HOLDS_BOOLEAN(gvalue)) {
		mdata->priv->is_approved = 
			g_value_get_boolean(gvalue);
	}

	col = 1;
}

GList *midgard_core_qb_set_object_from_query(MidgardQueryBuilder *builder, guint select_type, MgdObject *nobject)
{
        g_assert(builder != NULL);

        MgdObject *object;
        guint ret_rows, ret_fields;
      
	gchar *sql = midgard_core_qb_get_sql(
			builder, select_type, 
			midgard_query_builder_get_object_select(builder, select_type));

	GdaDataModel *model = 
		midgard_core_query_get_model(
				builder->priv->mgd, sql);
	g_free(sql);

	if(!model) 
		return NULL;	

	gint rows, columns;
	const GValue *gvalue = NULL;

	ret_rows = gda_data_model_get_n_rows(model);
	ret_fields =  gda_data_model_get_n_columns(model);

	/* records found , allocate as many objects as many returned rows */ 
	GList *list = NULL;

	if(ret_rows == 0) {
		g_object_unref(model);
		return list;
	}

	/* We count guids only */
	if(select_type == MQB_SELECT_GUID) {
		
		gvalue = midgard_data_model_get_value_at(model, 0, 0);	
		MidgardTypeHolder *holder = g_new(MidgardTypeHolder, 1);
		if (G_IS_VALUE(gvalue) && G_VALUE_HOLDS(gvalue, G_TYPE_INT64))
			holder->elements = (guint)g_value_get_int64((GValue*)gvalue);
		else 
			holder->elements = 0;
		list = g_list_append(list, holder);	
		g_object_unref(model);
		return list;		
	}

	/* Get every row */
	for (rows = 0; rows < ret_rows; rows++) {
	
		if(nobject)
			object = nobject;
		else
			object = g_object_new(builder->priv->type, NULL);

		MIDGARD_DBOBJECT(object)->dbpriv->mgd = builder->priv->mgd;
		
		MidgardDBObjectClass *dbklass = 
			MIDGARD_DBOBJECT_GET_CLASS(object);
		
		if(dbklass->dbpriv->set_from_sql != NULL) {
			
			dbklass->dbpriv->set_from_sql(MIDGARD_DBOBJECT(object), 
					model, rows); 

		} else {
			
			/* we have guid, sitegroup and metadata columns first */
			for (columns = 25; columns < ret_fields; columns++) {	
				
				gvalue = midgard_data_model_get_value_at(model, columns, rows);
				const gchar *coltitle =
					gda_data_model_get_column_title(model, columns);
				GParamSpec *pspec =
					g_object_class_find_property(G_OBJECT_GET_CLASS(object), coltitle);
				
				if(G_IS_VALUE(gvalue)) {

					guint puint = 0;
					gint pint = 0;

					if (!pspec) {
						
						g_warning("Failed to found (unregistered) %s property (%s class)", 
								coltitle, G_OBJECT_TYPE_NAME(object));
						continue;
					}

					switch(pspec->value_type) {

						case G_TYPE_INT:
							MIDGARD_GET_INT_FROM_VALUE(pint, gvalue);
							g_object_set(G_OBJECT(object), coltitle, pint, NULL);
							break;

						case G_TYPE_UINT:
							MIDGARD_GET_UINT_FROM_VALUE(puint, gvalue);
							g_object_set(G_OBJECT(object), coltitle, puint, NULL);
							break;

						default:
							g_object_set_property(G_OBJECT(object), coltitle, gvalue);
							break;
					}
				
				} else if (gda_value_is_null(gvalue)) {			

					switch (pspec->value_type) {

						case G_TYPE_INT:
						case G_TYPE_UINT:
							g_object_set(G_OBJECT(object), coltitle, 0, NULL);
							break;

						case G_TYPE_STRING:
							g_object_set(G_OBJECT(object), coltitle, "", NULL);
							break;

						default:
							g_warning("Found (%s) not a value at %d.%d (%s)", 
									g_type_name(pspec->value_type), 
									columns, rows, 
									gda_data_model_get_column_title(model, columns));
							break;
					}
				}
			}	
		}

		/* Set sitegroup and guid */
		const gchar *guid;
		guint sg = 0;
		gvalue = midgard_data_model_get_value_at(model, 0, rows);
		if(G_IS_VALUE(gvalue) && G_VALUE_HOLDS_STRING(gvalue)){
			guid = g_value_get_string(gvalue);   
			g_free((gchar *)MIDGARD_DBOBJECT(object)->dbpriv->guid);
			MIDGARD_DBOBJECT(object)->dbpriv->guid = g_strdup(guid);
		}
		gvalue = midgard_data_model_get_value_at(model, 1, rows);
		if(G_VALUE_HOLDS_UINT(gvalue)) 
			sg = g_value_get_uint(gvalue);  
		if(G_VALUE_HOLDS_INT(gvalue))
			sg = (guint) g_value_get_int(gvalue);

		MIDGARD_DBOBJECT(object)->dbpriv->sg = sg;
		
		/* Set metadata */   
		GParamSpec *pspec = 
			g_object_class_find_property(G_OBJECT_GET_CLASS(object), "metadata");

		if(pspec){

			MidgardMetadata *mdata = object->metadata;	
			if(mdata)
				__mqb_set_metadata(mdata, model, rows);
		}

		list = g_list_append(list, G_OBJECT(object));                		
	}

	g_object_unref(model);
	return list;	
}

gboolean midgard_query_builder_join(
		MidgardQueryBuilder *builder, const gchar *prop, 
		const gchar *jobject, const gchar *jprop)
{
	g_assert(prop != NULL);
	g_assert(jobject != NULL);
	g_assert(jprop != NULL);

	g_warning("FIXME");
}

 
/* Returns type name of the type which is currently used by Query Builder. */
const gchar *midgard_query_builder_get_type_name(
		MidgardQueryBuilder *builder)
{
	g_assert(builder != NULL);
	return g_type_name(builder->priv->type);
}

void midgard_query_builder_include_deleted(MidgardQueryBuilder *builder)
{
	g_assert(builder);

	builder->priv->include_deleted = TRUE;
}

/* GOBJECT ROUTINES*/

static void _midgard_query_builder_finalize(GObject *object)
{
	midgard_query_builder_free(
			(MidgardQueryBuilder *)object);
}

static void _midgard_query_builder_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	MidgardQueryBuilderClass *klass = MIDGARD_QUERY_BUILDER_CLASS (g_class);
	
	gobject_class->finalize = _midgard_query_builder_finalize;
	klass->set_lang = midgard_query_builder_set_lang;
	klass->unset_languages = midgard_query_builder_unset_languages;
	klass->add_constraint = midgard_query_builder_add_constraint;
	klass->add_constraint_with_property = 
		midgard_query_builder_add_constraint_with_property;
	klass->begin_group = midgard_query_builder_begin_group;
	klass->end_group = midgard_query_builder_end_group;
	klass->add_order = midgard_query_builder_add_order;
	klass->set_limit = midgard_query_builder_set_limit;
	klass->set_offset = midgard_query_builder_set_offset;
	klass->include_deleted = midgard_query_builder_include_deleted;
	klass->execute = midgard_query_builder_execute;
	klass->count = midgard_query_builder_count;
}

static void _midgard_query_builder_instance_init(
		GTypeInstance *instance, gpointer g_class)
{

}

/* Registers the type as  a fundamental GType unless already registered. */
GType midgard_query_builder_get_type(void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardQueryBuilderClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_query_builder_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardQueryBuilder),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_query_builder_instance_init /* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT,
				"midgard_query_builder",
				&info, 0);
	}
	return type;
}

