/* 
 * Copyright (C) 2007, 2008 Piotr Pokora <piotrek.pokora@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "midgard_connection.h"
#include "midgard_sitegroup.h"
#include "midgard_metadata.h"
#include "midgard_error.h"
#include "midgard_core_query.h"
#include "midgard_core_object.h"
#include "midgard_core_object_class.h"
#include "midgard_user.h"
#include "guid.h"
#include "query_builder.h"
#include "midgard_dbobject.h"
#include "midgard_datatypes.h"

/* Sitegroup object properties */
enum {
	MIDGARD_SITEGROUP_GUID = 1,
	MIDGARD_SITEGROUP_NAME,
	MIDGARD_SITEGROUP_REALM,
	MIDGARD_SITEGROUP_ADMINGROUP,
	MIDGARD_SITEGROUP_PUBLIC,
	MIDGARD_SITEGROUP_ID,
	MIDGARD_SITEGROUP_METADATA,
	MIDGARD_SITEGROUP_ADMINID
};

#define SITEGROUP_TABLE "sitegroup"

/**
 * midgard_sitegroup_new:
 * @mgd: #MidgardConnection instance
 * @value: GValue which holds sitegroup identifier
 *
 * Given @value must be of type G_TYPE_STRING or G_TYPE_INT. 
 * Might be set to NULL if "empty" instance must be created.
 *
 * Cases to return %NULL:
 * <itemizedlist>
 * <listitem><para>
 * @value doesn't hold string or int
 * </para></listitem>
 * <listitem><para>
 * @value holds valid type but object doesn't exists in database ( MGD_ERR_NOT_EXISTS )
 * </para></listitem>
 * <listitem><para>
 * more than one object identified by particular id or name returned ( MGD_ERR_INTERNAL ) 
 * </para></listitem>
 * </itemizedlist> 
 *
 * Returns: New #MidgardSitegroup object or NULL on failure
 */ 
MidgardSitegroup *midgard_sitegroup_new(MidgardConnection *mgd, const GValue *value) 
{
	g_return_val_if_fail(mgd != NULL, NULL);

	if (G_VALUE_HOLDS_STRING(value) || G_VALUE_HOLDS_INT(value) || value == NULL) {
		
		/* This is OK */
	
	} else {
		
		g_warning("Expected GValue of type string or int");
		return NULL;
	}

	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);

	MidgardSitegroup *self;

	if(value == NULL) {

		self = (MidgardSitegroup *)g_object_new(MIDGARD_TYPE_SITEGROUP, NULL);	
		midgard_core_object_class_set_midgard(mgd, G_OBJECT(self));
		self->dbpriv->guid = midgard_guid_new(mgd);
		
		return self;
	}

	MidgardQueryBuilder *builder =
		midgard_query_builder_new(mgd, "midgard_sitegroup");
	
	if(!builder)
		return NULL;

	/* sitegroup's name constraint */
	if (G_VALUE_HOLDS_STRING(value)) {
	
		midgard_query_builder_add_constraint(builder, "name", "=", (const GValue *)&value);

	/* sitegroup id constraint */
	} else if (G_VALUE_HOLDS_INT(value) || G_VALUE_HOLDS_UINT(value)) {

		midgard_query_builder_add_constraint(builder, "id", "=", (const GValue *)&value);
	}

	guint n_objects;	
	GObject **objects = midgard_query_builder_execute(builder, &n_objects);
	g_object_unref(builder);	

	if(!objects) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_NOT_EXISTS);
		return NULL;
	}
	
	guint i;
	if(n_objects > 1) {
		
		for(i = 0; i < n_objects ; i++) {
			g_object_unref(objects[i]);			
		}

		g_free(objects);
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		g_warning("Database inconsistency. Found %d sitegroups with the same identifier", n_objects);
		return NULL;
	}

	self = MIDGARD_SITEGROUP(objects[0]);
	g_free(objects);

	return self;
}

/**
 * midgard_sitegroup_list:
 * @mgd: #MidgardConnection instance
 *
 * If method is called by normal user or by anonymous, all public sitegroup's name are returned. 
 * For root user, every sitegroup name is returned.
 *
 * Returns: Newly allocated (NULL terminated) array of strings or %NULL on failure. 
 * g_strfreev should be used to free returned array.
 */
gchar **midgard_sitegroup_list(MidgardConnection *mgd)
{
	g_return_val_if_fail(mgd != NULL, NULL);

	MidgardUser *user = midgard_connection_get_user(mgd);
	const gchar *query;

	if (user && midgard_user_is_root(user)) {
	
		query = "SELECT name from sitegroup";
	
	} else {
		
		query = "SELECT name from sitegroup WHERE public=1";
	}

	GdaDataModel *model =
		midgard_core_query_get_model(mgd, query);

	if(!model)
		return NULL;

	const GValue *value = NULL;
	gint rows = gda_data_model_get_n_rows(model);
	guint i;

	gchar **names = g_new(gchar*, rows+1);

	for (i = 0; i < rows; i++) {
		value = midgard_data_model_get_value_at(model, 0, i);

		if(G_IS_VALUE(value)) {
			if(G_VALUE_HOLDS_STRING (value)) {
				names[i] = g_value_dup_string(value);
			}
		}
	}

	names[i+1] = NULL;

	return names;
}

/**
 * midgard_sitegroup_create:
 * @self: #MidgardSitegroup self instance
 *
 * Creates new record for the given #MidgardSitegroup object.
 * Only Midgard root is able to create sitegroup.
 * Also root user is responsible to set name property before creating.
 *
 * Setting public property is optional. By default it's set to %TRUE.
 * Internal #MidgardConnection sitegroup cache is refreshed when object is created.
 *
 * Cases to return %FALSE:
 * <itemizedlist>
 * <listitem><para>
 * Logged in user is not a root (MGD_ERR_ACCESS_DENIED)
 * </para></listitem>
 * <listitem><para>
 * Given #MidgardSitegroup name is empty or %NULL (MGD_ERR_INVALID_NAME)
 * </para></listitem>
 * <listitem><para>
 * #MidgardSitegroup object already exists with given name (MGD_ERR_OBJECT_NAME_EXISTS) 
 * </para></listitem>
 * </itemizedlist> 
 *
 * Returns %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_sitegroup_create(MidgardSitegroup *self)
{
	g_return_val_if_fail(self != NULL, FALSE);

	MidgardConnection *mgd = self->dbpriv->mgd;
	MidgardUser *user = MIDGARD_USER(mgd->priv->user);
	
	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);
	
	if(!user || !midgard_user_is_root(user)) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_ACCESS_DENIED);
		return FALSE;
	}

	if(self->name == NULL) {
		midgard_set_error(mgd,
				MGD_GENERIC_ERROR,
				MGD_ERR_INVALID_NAME,
				" Can not create sitegroup with empty name");
		return FALSE;
	}

	GList *cols = NULL;
	GList *values = NULL;
	guint n_prop = 0, i;
	GValue pval = {0, };

	MidgardQueryBuilder *builder = 
		midgard_query_builder_new(mgd, "midgard_sitegroup");
	
	if(!builder) 
		return FALSE;		

	GValue val = {0, };

	midgard_query_builder_begin_group(builder, "OR");
	
	/* sitegroup's name constraint */
	g_value_init(&val, G_TYPE_STRING);
	g_value_set_string(&val, self->name);
	midgard_query_builder_add_constraint(builder, 
			"name", "=", &val);
	g_value_unset(&val);

	/* empty admin group guid constraint */
	g_value_init(&val, G_TYPE_STRING);
	g_value_set_string(&val, "");
	midgard_query_builder_add_constraint(builder,
			"group", "=", &val);
	g_value_unset(&val);

	midgard_query_builder_end_group(builder);

	guint n_objects;	
	GObject **objects = midgard_query_builder_execute(builder, &n_objects);
	g_object_unref(builder);
	
	if(objects) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_NAME_EXISTS);	
		/* We never should get more than 1 object */
		g_object_unref(objects[0]);
		g_free(objects);
		return FALSE;
	}

	GParamSpec **pspecs = 
		g_object_class_list_properties(G_OBJECT_GET_CLASS(self), &n_prop);
	
	for(i = 0; i < n_prop; i++) {
		
		if(g_str_equal(pspecs[i]->name, "id")
				|| g_str_equal(pspecs[i]->name, "metadata")) 
			continue;
		
		g_value_init(&pval, pspecs[i]->value_type);

		g_object_get_property(G_OBJECT(self), pspecs[i]->name, &pval);

		GValue *dval = g_new0(GValue, 1);
		g_value_init(dval, pspecs[i]->value_type);
		g_value_copy((const GValue*) &pval, dval);
		values = g_list_prepend(values, (gpointer) dval);

		if(g_str_equal(pspecs[i]->name, "group")) {
			
			cols = g_list_prepend(cols, "admingroup_guid");

		} else if(g_str_equal(pspecs[i]->name, "adminid")) {

			cols = g_list_prepend(cols, "admingroup");

		} else {

			cols = g_list_prepend(cols, pspecs[i]->name);
		}

		g_value_unset(&pval);
	}

#ifdef HAVE_LIBGDA_4
	guint inserted = 
		midgard_core_query_insert_records(mgd, "sitegroup", 
				cols, values,
				GDA_SQL_STATEMENT_INSERT, NULL);
#else
	guint inserted = 
		midgard_core_query_insert_records(mgd, "sitegroup", 
				cols, values,
				GDA_QUERY_TYPE_INSERT, NULL);
#endif

	g_free(pspecs);
	g_list_free(cols);

	GList *l; 
	for(l = values; l != NULL; l = l->next){
		g_value_unset((GValue *) l->data);
		g_free((GValue *) l->data);
	}
	g_list_free(values);

	if(inserted == 0) {

		/* Refresh sitegroup cache list */
		id_list_free(mgd->priv->cache->sg_ids);
		GData *gdlist = mgd->priv->cache->sg_datalist;
		_midgard_get_sg_and_ml_rds(self->dbpriv->mgd,
				"id, name", "sitegroup", &gdlist);

		return TRUE;
	}

	return FALSE;
}

/**
 * midgard_sitegroup_update:
 * @self: #MidgardSitegroup self instance
 *
 * Updates #MidgardSitegroup record.
 * Only Midgard root is able to update sitegroup.
 * 
 * Cases to return %FALSE:
 * <itemizedlist>
 * <listitem><para>
 * Logged in user is not a root (MGD_ERR_ACCESS_DENIED)
 * </para></listitem>
 * <listitem><para>
 * Given #MidgardSitegroup name is empty or %NULL (MGD_ERR_INVALID_NAME)
 * </para></listitem>
 * <listitem><para>
 * #MidgardSitegroup object already exists with given name and with different guid (MGD_ERR_OBJECT_NAME_EXISTS) 
 * </para></listitem>
 * </itemizedlist> 
 *
 * Returns %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_sitegroup_update(MidgardSitegroup *self)
{
	g_return_val_if_fail(self != NULL, FALSE);

	MidgardConnection *mgd = self->dbpriv->mgd;

	g_assert(mgd != NULL);

	MidgardUser *user = midgard_connection_get_user(mgd);
	
	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);
	
	if(!user || !midgard_user_is_root(user)) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_ACCESS_DENIED);
		return FALSE;
	}
	
	if(self->name == NULL) {
		midgard_set_error(mgd,
				MGD_GENERIC_ERROR,
				MGD_ERR_INVALID_NAME,
				" Can not update sitegroup with empty name");
		return FALSE;
	}

	/* Check for duplicates */
	GValue val = {0, };
	MidgardQueryBuilder *builder =
		midgard_query_builder_new(mgd, "midgard_sitegroup");
	
	if(!builder)
		return FALSE;
	
	g_value_init(&val, G_TYPE_STRING);
	g_value_set_string(&val, self->name);
	midgard_query_builder_add_constraint(builder,
			"name", "=", &val);
	g_value_unset(&val);

	guint n_objects; 
	GObject **objects = midgard_query_builder_execute(builder, &n_objects);
	g_object_unref(builder);
	
	if(objects) {

		MidgardSitegroup *esg = MIDGARD_SITEGROUP(objects[0]);
		if(!g_str_equal(esg->dbpriv->guid, 
					self->dbpriv->guid)) {
			MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_NAME_EXISTS);
			/* We never should get more than 1 object */
			g_object_unref(objects[0]);
			g_free(objects);
			return FALSE;
		}

		g_free(objects);
	}


	/* Update sitegroup */
	GString *sql = g_string_new("UPDATE sitegroup SET ");

	g_string_append_printf(sql, 
			"name = '%s', realm = '%s', public = %d, "
			"admingroup = %d, admingroup_guid = '%s' "
			"WHERE guid = '%s'" ,
			self->name, self->realm, self->ispublic, 
			self->adminid, self->group,
			self->dbpriv->guid);

	gint rv =  midgard_core_query_execute(mgd, sql->str, FALSE);
	g_string_free(sql, TRUE);

	if(rv == -1)
		return FALSE;

	/* Refresh sitegroup and language cache */
	id_list_replace(mgd->priv->cache->sg_ids, 
			self->priv->initial_name,
			self->name, self->id);

	if(rv == -2) {
		g_warning("Provider didn't return number of updated rows. Sitegroup record might be not updated");
		return TRUE;
	}

	return TRUE;
}

/**
 * midgard_sitegroup_delete:
 * @self: #MidgardSitegroup self instance
 *
 * Not yet implemented.
 */ 
gboolean midgard_sitegroup_delete(MidgardSitegroup *self)
{
	g_warning("Not implemented");
	return FALSE;
}


/* GOBJECT ROUTINES */

static void
__midgard_sitegroup_set_property (GObject *object, guint property_id,
		const GValue *value, GParamSpec *pspec)
{
	MidgardSitegroup *self = (MidgardSitegroup *) object;
	
	switch (property_id) {
		
		case MIDGARD_SITEGROUP_GUID:
			g_warning("Setting guid property forbidden");
			break;
			
		case MIDGARD_SITEGROUP_NAME:
			if(self->priv->initial_name != NULL)
				g_free(self->priv->initial_name);
			self->priv->initial_name = g_strdup(self->name);
			g_free((gchar *)self->name);
			self->name = g_value_dup_string(value);
			break;

		case MIDGARD_SITEGROUP_REALM:
			g_free((gchar *)self->realm);
			self->realm = g_value_dup_string(value);
			break;
			
		case MIDGARD_SITEGROUP_ADMINGROUP:
			g_free((gchar *)self->group);
			self->group = g_value_dup_string(value);
			break;
			
		case MIDGARD_SITEGROUP_PUBLIC:
			self->ispublic = g_value_get_boolean(value);
			break;
			
		case MIDGARD_SITEGROUP_ID:
			g_warning("Setting ID property forbidden");
			break;

		case MIDGARD_SITEGROUP_METADATA:
			self->metadata = g_value_get_object(value);
			break;
			
		case MIDGARD_SITEGROUP_ADMINID:
			self->adminid = g_value_get_uint(value);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object,property_id,pspec);
			break;
	}
}

static void
__midgard_sitegroup_get_property (GObject *object, guint property_id,
		GValue *value, GParamSpec *pspec)
{
	MidgardSitegroup *self = (MidgardSitegroup *) object;
	
	switch (property_id) {

		case MIDGARD_SITEGROUP_GUID:
			g_value_set_string(value, self->dbpriv->guid);
			break;
			
		case MIDGARD_SITEGROUP_NAME:	
			g_value_set_string(value, self->name);
			break;

		case MIDGARD_SITEGROUP_REALM:
			g_value_set_string(value, self->realm);
			break;

		case MIDGARD_SITEGROUP_ADMINGROUP:
			g_value_set_string(value, self->group);
			break;

		case MIDGARD_SITEGROUP_PUBLIC:
			g_value_set_boolean(value, self->ispublic);
			break;

		case MIDGARD_SITEGROUP_ID:
			g_value_set_uint(value, self->id);
			break;

		case MIDGARD_SITEGROUP_METADATA:
			g_value_set_object(value, self->metadata);
			break;

		case MIDGARD_SITEGROUP_ADMINID:
			g_value_set_uint(value, self->adminid);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(self, property_id, pspec);
			break;
	}
}

static void __midgard_sitegroup_finalize(GObject *object)
{
	g_assert(object != NULL);
	
	MidgardSitegroup *self = (MidgardSitegroup *) object;

	g_free((gchar *) self->name);
	g_free((gchar *) self->realm);
	g_free((gchar *) self->group);
	
	g_object_unref(self->metadata);

	if(self->priv->initial_name != NULL)
		g_free(self->priv->initial_name);

	self->priv->initial_name = NULL;
	
	if(self->priv != NULL)
		g_free(self->priv);

	self->priv = NULL;
}

static GObjectClass *__parent_class= NULL;

static GObject *
__midgard_sitegroup_constructor (GType type,
		guint n_construct_properties,
		GObjectConstructParam *construct_properties)
{
	GObject *object = (GObject *)
		G_OBJECT_CLASS (__parent_class)->constructor (type,
				n_construct_properties,
				construct_properties);

	return G_OBJECT(object);
}

static void
__midgard_sitegroup_dispose (GObject *object)
{
	__parent_class->dispose (object);
}

#define __SG_SELECT_FULL "guid, id, name, realm, public, admingroup_guid AS admingroupguid, admingroup AS adminid"

static gboolean __set_from_sql(MidgardDBObject *object, GdaDataModel *model, gint row)
{
	g_assert(object != NULL);
	g_assert(model != NULL);
	guint i = 0;
	gboolean public = FALSE;

	MidgardSitegroup *self = MIDGARD_SITEGROUP(object);	
	const GValue *value = NULL;

	/* guid */
	value = midgard_data_model_get_value_at(model, 0, row);
	if (!G_VALUE_HOLDS_STRING(value)) 
	{
		g_warning("Data model doesn't hold guid of string type");
	} 
	else {
		g_free((gchar *)self->dbpriv->guid);
		self->dbpriv->guid = g_value_dup_string(value);
	}

	/* id */
	value = midgard_data_model_get_value_at(model, 1, row);
	MIDGARD_GET_UINT_FROM_VALUE(i, value);
	self->id = i;

	/* name */
	value = midgard_data_model_get_value_at(model, 2, row);
	if (!G_VALUE_HOLDS_STRING(value)) 
	{
		g_warning("Data model doesn't hold name of string type");
	} 
	else {	
		g_free((gchar *)self->name);
		self->name = g_value_dup_string(value);
	}

	/* realm */
	value = midgard_data_model_get_value_at(model, 3, row);
	if (!G_VALUE_HOLDS_STRING(value)) 
	{
		g_warning("Data model doesn't hold realm of string type");
	} 
	else {
		g_free((gchar *)self->realm);
		self->realm = g_value_dup_string(value);
	}

	/* public */
	value = midgard_data_model_get_value_at(model, 4, row);
	MIDGARD_GET_BOOLEAN_FROM_VALUE(public, value);
	self->ispublic = public;

	/* admingroupguid */
	value = midgard_data_model_get_value_at(model, 5, row);
	if (!G_VALUE_HOLDS_STRING(value)) 
	{
		g_warning("Data model doesn't hold admingroup guid of string type");
	} 
	else {
		g_free((gchar *)self->group);
		self->group = g_value_dup_string(value);
	}

	/* adminid */
	value = midgard_data_model_get_value_at(model, 6, row);
	MIDGARD_GET_UINT_FROM_VALUE(i, value);
	self->adminid = i;
}


static void _midgard_sitegroup_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	MidgardSitegroupClass *klass = MIDGARD_SITEGROUP_CLASS (g_class);
	__parent_class = g_type_class_peek_parent (g_class);

	gobject_class->constructor = __midgard_sitegroup_constructor;
	gobject_class->dispose = __midgard_sitegroup_dispose;
	gobject_class->finalize = __midgard_sitegroup_finalize;
	gobject_class->set_property = __midgard_sitegroup_set_property;
	gobject_class->get_property = __midgard_sitegroup_get_property;

	MgdSchemaPropertyAttr *prop_attr;
	MgdSchemaTypeAttr *type_attr = _mgd_schema_type_attr_new();
	
	GParamSpec *pspec;
	
	/* GUID */
	pspec = g_param_spec_string ("guid",
			"Object's guid",
			"",
			"",
			G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_GUID,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_GUID;
	prop_attr->field = g_strdup("guid");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", SITEGROUP_TABLE, "guid", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"guid"), prop_attr);

	/* NAME */
	pspec = g_param_spec_string ("name",
			"Sitegroup's name",
			"Unique name",
			"",
			G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_NAME,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_STRING;
	prop_attr->field = g_strdup("name");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "name", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"name"), prop_attr);
	
	/* REALM */
	pspec = g_param_spec_string ("realm",
			"Sitegroup's realm",
			"",
			"Midgard",
			G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_REALM,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_STRING;
	prop_attr->field = g_strdup("realm");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "realm", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"realm"), prop_attr);

	/* GROUP */	
	pspec = g_param_spec_string ("admingroupguid",
			"Sitegroup's admin group",
			"",
			"",
			G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_ADMINGROUP,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_GUID;
	prop_attr->field = g_strdup("admingroup_guid");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "admingroup_guid", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"admingroupguid"), prop_attr);

	/* PUBLIC */
	pspec = g_param_spec_boolean("public",
			"Sitegroup's public info",
			"",
			TRUE, G_PARAM_READWRITE);
	/**
	 * MidgardSitegroup:public:
	 *
	 * Defines whether sitegroup is public or not.
	 * Non public sitegroup's context can not be set in anonymous mode.
	 */ 
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_PUBLIC,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_BOOLEAN;
	prop_attr->field = g_strdup("public");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "public", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"public"), prop_attr);

	/* ID */
	pspec = g_param_spec_uint ("id",
			"Sitegroup's ID",
			"",
			0, G_MAXUINT32, 0, G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_ID,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_UINT;
	prop_attr->field = g_strdup("id");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "id", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"id"), prop_attr);

	/* METADATA */
	/* We do not want metadata. For legacy at least. 
	pspec = g_param_spec_object ("metadata", 
			"",
			"Property with Midgard metadata object",
			G_TYPE_OBJECT, G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_METADATA,
			pspec);
	*/
	
	/* LEGACY ADMIN ID */
	pspec = g_param_spec_uint ("adminid",
			"Legacy ",
			"Legacy admingroup id",
			0, G_MAXUINT32, 0, G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			MIDGARD_SITEGROUP_ID,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_UINT;
	prop_attr->field = g_strdup("admingroup");
	prop_attr->table = g_strdup(SITEGROUP_TABLE);
	prop_attr->tablefield = g_strjoin(".", 
			SITEGROUP_TABLE, "adminid", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"adminid"), prop_attr);

	/* Set storage data */
	type_attr->table = g_strdup(SITEGROUP_TABLE);
	type_attr->tables = g_strdup(SITEGROUP_TABLE);
	
	/* This must be replaced with GDA classes */
	MgdSchemaTypeQuery *_query = _mgd_schema_type_query_new();
	_query->select_full = g_strdup(__SG_SELECT_FULL);
	type_attr->query = _query;

	/* Initialize private member */
	klass->dbpriv = g_new(MidgardDBObjectPrivate, 1);
	klass->dbpriv->storage_data = type_attr;
	klass->dbpriv->has_metadata = FALSE;
	klass->dbpriv->set_from_sql = __set_from_sql;
}

static void _midgard_sitegroup_instance_init(
		GTypeInstance *instance, gpointer g_class)
{
	MidgardSitegroup *self = (MidgardSitegroup *) instance;

	self->name = NULL;
	self->realm = NULL;
	self->group = NULL;
	self->ispublic = FALSE;
	self->id = 0;
	self->adminid = 0;
	self->metadata = g_object_new(MIDGARD_TYPE_METADATA, NULL);

	self->priv = g_new(MidgardSitegroupPrivate, 1);
	self->priv->initial_name = NULL;
}

GType midgard_sitegroup_get_type(void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardSitegroupClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_sitegroup_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardSitegroup),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_sitegroup_instance_init/* instance_init */
		};
		type = g_type_register_static (MIDGARD_TYPE_DBOBJECT,
				"midgard_sitegroup",
				&info, 0);
	}
	return type;
}



