/* 
 * Copyright (C) 2006, 2008 Piotr Pokora <piotrek.pokora@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <midgard_request_config.h>
#include <midgard_object.h>

/**
 * midgard_request_config_new:
 * @uri: unique uri which identifies request
 *
 * @uri can not be explicitly set to NULL. 
 * Use empty string (not NULL) if there's not need for unique uri.
 * 
 * Returns: new MidgardRequestConfig or %NULL otherwise.
 */ 
extern MidgardRequestConfig *midgard_request_config_new(const gchar *uri) 
{
	if(uri == NULL) {
		g_warning("Can not set NULL uri for request");
		return NULL;
	}

	MidgardRequestConfig *self;
	self = g_object_new(MIDGARD_TYPE_REQUEST_CONFIG, NULL);
	self->uri = g_strdup(uri);

	return self;
}

/**
 * midgard_request_config_set_host:
 * @self: #MidgardRequestConfig instance
 * @host: #GObject host
 *
 * Host can be set only once. 
 * Warning message is thrown if there's attempt to change host.
 * #MidgardRequestConfig only holds a pointer to the host.
 * It doesn't allocate or free its memory.
 *
 * Returns: %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_host(MidgardRequestConfig *self, GObject *host)
{
	g_assert(self != NULL);

	if(self->host != NULL) {

		g_warning("Request host is already set");
		return FALSE;
	}

	self->host = host;

	return TRUE;
}

/**
 * midgard_request_config_get_host:
 * @self: #MidgardRequestConfig instance
 *
 * Returns: GObject of midgard_host type. %NULL on failure.
 */ 
GObject *midgard_request_config_get_host(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->host;
}

/**
 * midgard_request_config_set_style:
 * @self: #MidgardRequestConfig instance
 * @style: #GObject style
 *
 * Style can be set only once. 
 * Warning message is thrown if there's attempt to change style.
 * #MidgardRequestConfig only holds a pointer to the style.
 * It doesn't allocate or free its memory.
 *
 * Returns: %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_style(MidgardRequestConfig *self, GObject *style)
{
	g_assert(self != NULL);

	if(self->style != NULL) {

		g_warning("Request style is already set");
		return FALSE;
	}

	self->style = style;

	return TRUE;
}

/**
 * midgard_request_config_get_style:
 * @self: MidgardRequestConfig instance
 *
 * Returns: GObject of midgard_style type. %NULL on failure.
 */ 
GObject *midgard_request_config_get_style(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->style;
}

/**
 * midgard_request_config_set_page:
 * @self: #MidgardRequestConfig instance
 * @page: #GObject page
 *
 * Page can be set only once. 
 * Warning message is thrown if there's attempt to change page.
 * #MidgardRequestConfig only holds a pointer to the page.
 * It doesn't allocate or free its memory.
 *
 * Returns: %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_page(MidgardRequestConfig *self, GObject *page)
{
	g_assert(self != NULL);

	if(self->page != NULL) {

		g_warning("Request page is already set");
		return FALSE;
	}

	self->page = page;

	return TRUE;
}

/**
 * midgard_request_config_get_page:
 * @self: MidgardRequestConfig instance
 *
 * Returns: GObject of midgard_page type. %NULL on failure.
 */
GObject *midgard_request_config_get_page(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->page;
}

/**
 * midgard_request_config_set_pages:
 * @self: #MidgardRequestConfig instance
 * @pages: array of GObject pages
 *
 * Pages can be set only once per #MidgardRequestConfig instance. 
 * Warning message is thrown if there's attempt to change pages array.
 *
 * Given pages array is owned by #MidgardRequestConfig and should not be freed.
 *
 * Returns: %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_pages(MidgardRequestConfig *self, GObject **pages)
{
	g_assert(self != NULL);

	if(self->pages != NULL) {

		g_warning("Request pages array is already set");
		return FALSE;
	}

	self->pages = (const GObject**) pages;

	return TRUE;
}

/**
 * midgard_request_config_get_pages:
 * @self: MidgardRequestConfig instance
 *
 * Returned array is owned by #MidgardRequestConfig, and shouldn't be freed.
 *
 * Returns: NULL terminated array of GObject objects.
 */ 
const GObject **midgard_request_config_get_pages(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->pages;
}

/**
 * midgard_request_config_set_auth:
 * @self: #MidgardRequestConfig instance
 * @auth: authentication switch
 *
 * A toggle to switch if request requires authentication.
 * It can be set as many times as needed.
 *
 * Returns %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_auth(MidgardRequestConfig *self, gboolean auth)
{
	g_assert(self != NULL);

	self->auth_required = auth;
}

/**
 * midgard_request_config_get_auth:
 * @self: MidgardRequestConfig instance
 *
 * Returns: %TRUE if authentication is required, %FALSE otherwise
 */ 
gboolean midgard_request_config_get_auth(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->auth_required;
}

/**
 * midgard_request_config_set_argv:
 * @self: #MidgardRequestConfig instance
 * @argv: array of strings
 *
 * Arguments' array can be set only once. 
 * Warning message is thrown if there's attempt to change array.
 *
 * Returns: %TRUE on success, %FALSE otherwise
 */ 
gboolean midgard_request_config_set_argv(MidgardRequestConfig *self, GValueArray *argv)
{
	g_assert(self != NULL);

	if(self->argv != NULL) {

		g_warning("Request argv array is already set");
		return FALSE;
	}

	self->argv = argv;

	self->argc = argv->n_values;

	return TRUE;

}

/** 
 * midgard_request_config_get_argv:
 * @self: #MidgardRequestConfig instance
 *
 * Returns argv array or %NULL
 */
GValueArray *midgard_request_config_get_argv(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->argv;
}

/**
 * midgard_request_config_set_argc:
 * @self: #MidgardRequestConfig instance
 * @argc: number of arguments
 *
 * Sets number of request's arguments. It can be set only once per instance.
 *
 * Returns: %TRUE if argc number has been set, %FALSE otherwise
 */
gboolean midgard_request_config_set_argc(MidgardRequestConfig *self, guint argc)
{
	g_assert(self != NULL);

	if (self->argc > -1) {

		g_warning("Number of request's arguments is already set");
		return FALSE;
	}

	self->argc = argc;

	return TRUE;
}

/**
 * midgard_request_config_get_argc:
 * @self: #MidgardRequestConfig instance
 *
 * Returns number of request's arguments
 */
guint midgard_request_config_get_argc(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->argc;
}

/**
 * midgard_request_config_get_uri:
 * @self: #MidgardRequestConfig instance
 *
 * Returns: uri, which #MidgardRequestConfig has been initialized for. Or %NULL.
 */
const gchar *midgard_request_config_get_uri(MidgardRequestConfig *self)
{
	g_assert(self != NULL);

	return self->uri;
}

/* GOBJECT ROUTINES */

static void _midgard_request_config_instance_init(
		GTypeInstance *instance, gpointer g_class)
{
	MidgardRequestConfig *self = (MidgardRequestConfig *) instance;

	self->host = NULL;
	self->page = NULL;
	self->pages = NULL;
	self->style = NULL;	
	self->uri = NULL;	
	self->auth_required = FALSE;

	self->argv = NULL;
	self->argc = -1;
}

static void _midgard_request_config_finalize(GObject *object)
{
	MidgardRequestConfig *self = (MidgardRequestConfig *) object;
	
	if(self->host && G_IS_OBJECT(self->host))
		g_object_unref(self->host);
	self->host = NULL;

	/* self->page is only a pointer to an object inserted
	 * to self->pages array. Do not free it */

	if(self->style)
		g_object_unref(self->style);
	self->style = NULL;
	
	g_free((gchar *)self->uri);

	GObject **objects = (GObject **) self->pages;

	guint i = 0;
	if (objects != NULL) {

		while (objects[i] != NULL) {

			/* This might be unsafe. Language bindings should unref objects in garbaga collection */
			/*
			if (objects[i] != NULL) {
				
				g_object_unref(objects[i]);
			} */
			i++;
		}

		g_free(self->pages);
	}

	GValueArray *array = self->argv;
	if(array != NULL)
		g_value_array_free(array);
	self->argv = NULL;
}

static void _midgard_request_config_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);	
	gobject_class->finalize = _midgard_request_config_finalize;
}

extern GType midgard_request_config_get_type(void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardRequestConfigClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_request_config_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardRequestConfig),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_request_config_instance_init /* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT,
				"midgard_request_config",
				&info, 0);
	}
	return type;
}
