/* 
 * Copyright (C) 2006 Piotr Pokora <piotrek.pokora@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "midgard_replicator.h"
#include "midgard_core_object.h"
#include "midgard_core_object_class.h"
#include "midgard_object.h"
#include <sys/stat.h>
#include "midgard_blob.h"
#include "midgard_timestamp.h"
#include "midgard_error.h"
#include "midgard_object_class.h"
#include "midgard_user.h"
#include "midgard_core_query.h"
#include "guid.h"

#define MR_SET_LANG(__mgdc, __lang) \
	midgard_connection_set_lang(__mgdc, __lang); 

#define MR_SET_DLANG(__mgdc, __lang) \
	midgard_connection_set_default_lang(__mgdc, __lang); 

#define MR_SET_ALL_LANG(__mgdc, __lang) \
	midgard_connection_set_lang(__mgdc, __lang); \
	midgard_connection_set_default_lang(__mgdc, __lang);

#define __mr_dbus_send(_obj, _action) \
	gchar *_dbus_path = g_strconcat("/replication/", \
			G_OBJECT_TYPE_NAME(G_OBJECT(_obj)), \
			"/", _action, NULL); \
	midgard_core_dbus_send_serialized_object(_obj, _dbus_path); \
        g_free(_dbus_path);

struct _MidgardReplicatorPrivate 
{ 
	MgdObject *object;
	MidgardObjectClass *klass;
	GType type;
	const gchar *typename;	
	MidgardConnection *mgd;
};

/* Creates new MidgardReplicator instance for the given Midgard Object. */
MidgardReplicator *midgard_replicator_new(MgdObject *object)
{
	g_assert(object != NULL);

	MidgardReplicator *self = 
		g_object_new(MIDGARD_TYPE_REPLICATOR, NULL);

	if(object) {
		
		g_assert(object->dbpriv->mgd != NULL);

		if(MIDGARD_IS_OBJECT(object)){
			self->priv->object = object;
			self->priv->klass = 
				MIDGARD_OBJECT_GET_CLASS(object);
			self->priv->type = 
				G_OBJECT_TYPE(G_OBJECT(object));
			self->priv->typename = 
				G_OBJECT_TYPE_NAME(G_OBJECT(object));
			self->priv->mgd = object->dbpriv->mgd;
		} else {
			g_warning("Replicator constructor: object is not Midgard Object");
			g_object_unref(self);
			return NULL;
		}
	
	} else {

		/* self->priv->mgd = mgd; */
	}

	return self;
}

/* Returns serialized objects as xml content. */ 
gchar *midgard_replicator_serialize(MidgardReplicator *self, GObject *object) 
{
	if(!self) {
		
		if(G_IS_OBJECT(object))
			return midgard_core_object_to_xml(object);
		else 
			g_warning("Object is not base GObject");
	
	} else {

		if(object)
			g_warning("Ignoring object argument. Another object already assigned with replicator");

		return midgard_core_object_to_xml(object);
	}

	return NULL;
}

/* Export object identified by the given guid. */
gboolean midgard_replicator_export(	MidgardReplicator *self, 
					MidgardDBObject *object) 
{
	if(self && object){
		g_warning("midgard_replicator already initialized with %s:%s",
				self->priv->typename, object->dbpriv->guid);
		return FALSE;
	}

	MidgardDBObject *_object;
	MidgardConnection *mgd = NULL;
	const gchar *guid = NULL;

	if(!object)
		_object = MIDGARD_DBOBJECT(self->priv->object);
	else 
		_object = object;

	if(_object){

		mgd = MGD_OBJECT_CNC(_object);
		guid = MGD_OBJECT_GUID(_object);

		if(guid == NULL){
			midgard_set_error(mgd,
					MGD_GENERIC_ERROR,
					MGD_ERR_INVALID_PROPERTY_VALUE,
					"Empty guid value! ");
			g_warning("%s", mgd->errstr);
			g_clear_error(&mgd->err);
			return FALSE;
		}
	}
	
	g_signal_emit(object, MIDGARD_OBJECT_GET_CLASS(_object)->signal_action_export, 0);
	
	/* FIXME, invoke class method here */
	if(MIDGARD_IS_OBJECT(_object))
		return _midgard_object_update(MIDGARD_OBJECT(_object), OBJECT_UPDATE_EXPORTED);
}

/* Marks as exported the object which is identified by the given guid. */
gboolean midgard_replicator_export_by_guid (MidgardConnection *mgd, const gchar *guid)
{
	if(guid == NULL) {
		g_warning("Can not identify object by NULL guid");
		MIDGARD_ERRNO_SET(mgd,
				MGD_ERR_INVALID_PROPERTY_VALUE);
		return FALSE;
	}

	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);
	GString *sql = g_string_new("SELECT ");
	g_string_append_printf(sql,
			"typename, object_action FROM repligard "
			"WHERE guid = '%s' ",
			guid);
	
	MidgardUser *user = MIDGARD_USER(mgd->priv->user);
	if(!midgard_user_is_root(user)){
		g_string_append_printf(sql,
				" AND sitegroup = %d",
				midgard_connection_get_sitegroup_id(mgd));
	}

	GdaDataModel *model =
		midgard_core_query_get_model(mgd, sql->str);
	guint rows = gda_data_model_get_n_rows(model);
	g_string_free(sql, TRUE);
	
	if(rows == 0) {
		
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_NOT_EXISTS);
		g_object_unref(model);
		return FALSE;
	}
	
	/* TODO, this is somehow fatal, we should handle such cases */
	if(midgard_user_is_root(user)
			&& (rows > 1)) {
		
		g_object_unref(model);
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_SITEGROUP_VIOLATION);
		return FALSE;
	}

	const GValue *value = midgard_data_model_get_value_at(model, 0, 0);
	const gchar *classname = g_value_get_string(value);
	value = midgard_data_model_get_value_at(model, 0, 1);
	guint action = MGD_OBJECT_ACTION_NONE;
	MIDGARD_GET_UINT_FROM_VALUE(action, value);

	MidgardObjectClass *klass = 
		MIDGARD_OBJECT_GET_CLASS_BY_NAME(classname);

	if(klass == NULL) {
	
		g_warning("Failed to get class pointer for '%s', an object identified with '%s'", classname, guid);
		g_object_unref(model);
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		return FALSE;
	}

	g_object_unref(model);

	const gchar *table = 
		midgard_core_class_get_table(MIDGARD_DBOBJECT_CLASS(klass)); 
	GValue *tval = NULL;
	gchar *timeupdated;
	gint qr;

	switch(action) {
		
		case MGD_OBJECT_ACTION_PURGE:
			MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_PURGED);
			return FALSE;
			break;

		default:
			sql = g_string_new("UPDATE ");
			g_string_append_printf(sql,
					"%s SET ",
					table);

			tval = midgard_timestamp_new_current();
			timeupdated = midgard_timestamp_get_string(tval);
			
			g_string_append_printf(sql,
					"metadata_exported='%s' "
					"WHERE guid = '%s' AND sitegroup = %d", 
					timeupdated,
					guid, 
					midgard_connection_get_sitegroup_id(mgd));

			qr = midgard_core_query_execute(mgd, sql->str, TRUE);
			g_string_free(sql, TRUE);

			g_value_unset(tval);
			g_free(timeupdated);

			if (qr == 0) {
				MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
				return FALSE;
			}

			return TRUE;

			break;
	}

	return FALSE;
}

/* Export purged objects. */ 
gchar *midgard_replicator_export_purged(	MidgardReplicator *self, 
						MidgardObjectClass *klass, 
						MidgardConnection *mgd,
						const gchar *startdate, 
						const gchar *enddate)
{
	if(self && klass){
		g_warning("midgard_replicator already initialized with %s",
				self->priv->typename);
		return NULL;
	}

	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);

	MidgardConnection *_mgd;
	MidgardObjectClass *_klass;

	if(self == NULL){
		_klass = klass;
		_mgd = mgd;
	} else {
		_klass = self->priv->klass;
		_mgd = self->priv->mgd;
	}

	GString *sql = g_string_new(" SELECT guid, object_action_date ");
	g_string_append_printf(sql, " FROM repligard where typename = '%s' "
			" AND object_action = %d",
			G_OBJECT_CLASS_NAME(_klass),
			MGD_OBJECT_ACTION_PURGE);

	MidgardUser *user = MIDGARD_USER(_mgd->priv->user);
        if(!midgard_user_is_root(user)){
		g_string_append_printf(sql, " AND sitegroup = %d ",
				midgard_connection_get_sitegroup_id(_mgd));
	}
	
	if(startdate != NULL){
		g_string_append_printf(sql, " AND object_action_date > '%s' ",
				startdate);
	}

	if(enddate != NULL) {
		g_string_append_printf(sql, " AND object_action_date < '%s' ",
				enddate);
	}

	GdaDataModel *model =
		midgard_core_query_get_model(mgd, sql->str);
	guint rows = gda_data_model_get_n_rows(model);
	g_string_free(sql, TRUE);

	if(!model) 
		return NULL;

	if(rows == 0) {
		
		if(model) g_object_unref(model);
		return NULL;
	}

	xmlNode *object_node;
	xmlDoc *doc = 
		midgard_core_object_create_xml_doc();
	xmlNode *root_node = 
		xmlDocGetRootElement(doc);
	
	guint i;
	for(i = 0; i < rows; i++) {
		
		gboolean free_converted = FALSE;
		const gchar *purged = NULL;
		GValue strv = {0, };

		const GValue *value = 
			midgard_data_model_get_value_at(model, 0, i);
		const gchar *guid = g_value_get_string(value);
		
		value = midgard_data_model_get_value_at(model, 1, i);
		if(G_IS_VALUE(value) && G_VALUE_TYPE(value) == GDA_TYPE_TIMESTAMP) {
			
			g_value_init(&strv, G_TYPE_STRING);
			free_converted = TRUE;
			if(g_value_transform((const GValue *)value, &strv)) 
				purged = g_value_get_string(&strv);
	
		} else {

			purged = g_value_get_string(value);
		}
			
		object_node =
			xmlNewNode(NULL, BAD_CAST G_OBJECT_CLASS_NAME(_klass));
		xmlNewProp(object_node, BAD_CAST "purge", BAD_CAST "yes");
		xmlNewProp(object_node, BAD_CAST "guid", BAD_CAST guid);
		xmlNewProp(object_node, BAD_CAST "purged", BAD_CAST purged);
		xmlAddChild(root_node, object_node);

		if(free_converted)
			g_value_unset(&strv);
	}

	g_object_unref(model);
	xmlChar *buf;
	gint size;
	xmlDocDumpFormatMemoryEnc(doc, &buf, &size, "UTF-8", 1);
	xmlFreeDoc(doc);
	xmlCleanupParser();
	
	return (gchar*) buf;
}

/* Serialize midgard_blob binary data */
gchar *midgard_replicator_serialize_blob(MidgardReplicator *self,
					MgdObject *object)
{
	MidgardConnection *_mgd;

	if(self == NULL)
		_mgd = object->dbpriv->mgd;
	else
		_mgd = self->priv->mgd;
	
	MidgardBlob *blob = 
		midgard_blob_new(object, NULL);

	if(!blob)
		return NULL;
	
	gsize bytes_read = 0;
	gchar *content = midgard_blob_read_content(blob, &bytes_read);
	
	if(!content) {
		g_object_unref(blob);
		return NULL;
	}

	gchar *encoded =
		g_base64_encode((const guchar *)content, bytes_read);
	g_free(content);

	xmlDoc *doc = midgard_core_object_create_xml_doc();
	xmlNode *root_node =
		xmlDocGetRootElement(doc);
	xmlNode *blob_node = 
		xmlNewTextChild(root_node, NULL,
				(const xmlChar*)
				"midgard_blob",
				BAD_CAST encoded);
	xmlNewProp(blob_node, BAD_CAST "guid",
			BAD_CAST object->dbpriv->guid);

	g_free(encoded);

	xmlChar *buf;
	gint size;
	xmlDocDumpFormatMemoryEnc(doc, &buf, &size, "UTF-8", 1);
	xmlFreeDoc(doc);
	xmlCleanupParser();
	
	return (gchar*) buf;
}

gchar *midgard_replicator_export_blob(MidgardReplicator *self,
		MgdObject *object)
{
	return midgard_replicator_serialize_blob(self, object);
}

/*  Creates object instance from the given xml string. */
GObject **midgard_replicator_unserialize(	MidgardReplicator *self, 
						MidgardConnection *mgd, 
						const gchar *xml,
						gboolean force)
{
	MidgardConnection *_mgd;
	if(self == NULL)
		_mgd = mgd;
	else 
		_mgd = self->priv->mgd;

	return midgard_core_object_from_xml(_mgd, xml, force); 
}

/*  Imports object  to database. */
gboolean midgard_replicator_import_object (	MidgardReplicator *self, 
						MidgardDBObject *object, gboolean force)
{	
	if(self == NULL)
		g_assert(object != NULL);

	if(self && object){
		g_warning("midgard_replicator already initialized with %s:%s",
				self->priv->typename, object->dbpriv->guid);
		return FALSE;
	}

	if(object->dbpriv->guid == NULL 
			||(object->dbpriv->guid != NULL 
				&& object->dbpriv->guid == '\0')) {
			
		midgard_set_error(object->dbpriv->mgd,
				MGD_GENERIC_ERROR,
				MGD_ERR_INVALID_PROPERTY_VALUE,
				" Empty guid ");
		g_warning("%s", object->dbpriv->mgd->err->message);
		g_clear_error(&object->dbpriv->mgd->err);

		return FALSE;
	}

	GType class_type = G_OBJECT_TYPE(object);
	if(!class_type || g_type_parent(class_type) != MIDGARD_TYPE_OBJECT ) {
		
		g_warning("Can not import object with invalid type");
		
		/* FIXME, there might be midgard object or not midgard one
		 * we should check if object is MgdDBObjectClass derived */
		/* MIDGARD_ERRNO_SET(mgd, MGD_ERR_INVALID_OBJECT); */
		return FALSE;
	}

	MIDGARD_ERRNO_SET(object->dbpriv->mgd, MGD_ERR_OK);
	g_signal_emit(object, MIDGARD_OBJECT_GET_CLASS(object)->signal_action_import, 0);

	MidgardConnection *mgd;
	if(self) {
		object = MIDGARD_DBOBJECT(self->priv->object);
		mgd = self->priv->mgd;
	} else {
		mgd = MGD_OBJECT_CNC(object);
	}
	
	/* Get object from database */
	MidgardQueryBuilder *builder = 
		midgard_query_builder_new(mgd, 
				G_OBJECT_TYPE_NAME(object));
	if(!builder)
		return FALSE;

	/* Set internal language */
	gint init_lang, object_lang, dbobject_lang;
	init_lang = midgard_connection_get_lang_id(object->dbpriv->mgd, NULL);
	const gchar *init_dlang_str = 
		midgard_connection_get_default_lang(object->dbpriv->mgd);
	const gchar *init_lang_str = midgard_connection_get_lang(object->dbpriv->mgd);
	gboolean multilang = FALSE;
	gboolean ret_val;
	const gchar *_lang;

	MidgardObjectClass *klass =
		MIDGARD_OBJECT_GET_CLASS(object);
	if(midgard_object_class_is_multilang(klass)) {
		multilang = TRUE;
		g_object_get(G_OBJECT(object), "lang", &object_lang, NULL);
		_lang = midgard_connection_get_lang_from_id(object->dbpriv->mgd, object_lang);
		midgard_connection_set_lang(object->dbpriv->mgd, _lang);
		//mgd_set_default_lang(object->dbpriv->mgd, object_lang);
	}

	GValue pval = {0, };
	/* Add guid constraint */
	g_value_init(&pval,G_TYPE_STRING);
	g_object_get_property(G_OBJECT(object), "guid", &pval);
	midgard_query_builder_add_constraint(builder,
			"guid",
			"=", &pval);
	g_value_unset(&pval);
	
	/* Get db object which is "elder" than imported one */
	/*
	g_value_init(&pval,G_TYPE_STRING);
	g_object_get_property(G_OBJECT(object->metadata), "revised", &pval);
	midgard_query_builder_add_constraint(builder, 
			"metadata.revised", "<", &pval);
	g_value_unset(&pval);
	*/

	/* Unset languages if we have multilang object */
	if(multilang) 
		midgard_query_builder_unset_languages(builder);

	/* Get deleted or undeleted object */
	midgard_query_builder_include_deleted(builder);

	guint n_objects;
	GObject **_dbobject = 
		midgard_query_builder_execute(builder, &n_objects);
	MgdObject *dbobject;

	if(_dbobject && multilang){
		g_object_get(G_OBJECT(_dbobject[0]), "lang", &dbobject_lang, NULL);
		g_object_get(G_OBJECT(object), "lang", &object_lang, NULL);
		_lang = midgard_connection_get_lang_from_id(object->dbpriv->mgd, object_lang);
		midgard_connection_set_lang(object->dbpriv->mgd, _lang);
	}

	if(!_dbobject){
	
		g_object_unref(G_OBJECT(builder));

		dbobject = midgard_object_class_get_object_by_guid(
				mgd, object->dbpriv->guid);
		
		/* Error is already set by get_by_guid.
		 * This is not thread safe */
		if(dbobject) {
			if(multilang) {
				MR_SET_LANG(object->dbpriv->mgd, init_lang_str);
				MR_SET_DLANG(object->dbpriv->mgd, init_dlang_str);
			}
			return FALSE;	
		}

		if((mgd->errnum == MGD_ERR_OBJECT_PURGED)
			&& force) {

			/* we need to delete repligard entry here 
			 * In any other case we need to change core's API a bit
			 * and make create method more complicated with 
			 * additional needless cases */

			GString *sql = g_string_new("DELETE from repligard WHERE ");
			g_string_append_printf(sql,
					"typename = '%s' AND guid = '%s' "
					"AND sitegroup = %d",
					G_OBJECT_TYPE_NAME(object),
					object->dbpriv->guid,
					object->dbpriv->sg);
			midgard_core_query_execute(object->dbpriv->mgd, sql->str, TRUE);
			g_string_free(sql, TRUE);
			
			ret_val =  _midgard_object_create(MIDGARD_OBJECT(object), 
					object->dbpriv->guid, 
					OBJECT_UPDATE_IMPORTED);		

			if(multilang) {
				MR_SET_LANG(object->dbpriv->mgd, init_lang_str);
				MR_SET_DLANG(object->dbpriv->mgd, init_dlang_str);
			}

			return ret_val;
		}

		if(mgd->errnum == MGD_ERR_NOT_EXISTS) {

			ret_val =  _midgard_object_create(MIDGARD_OBJECT(object),
					MGD_OBJECT_GUID(object), 
					OBJECT_UPDATE_IMPORTED);

			if(multilang) {
				midgard_connection_set_lang(object->dbpriv->mgd,
						init_lang_str);
				midgard_connection_set_default_lang(object->dbpriv->mgd,
						init_lang_str);
			}

			return ret_val;
		}

	} else {

		gchar *updated, *dbupdated;
		dbobject = (MgdObject *)_dbobject[0];

		/* Compare revised datetimes. We must know if imported 
		 * object is newer than that one which exists in database */
		g_object_get(G_OBJECT(MIDGARD_OBJECT(object)->metadata),
				"revised", &updated, NULL);
		g_object_get(G_OBJECT(MIDGARD_OBJECT(dbobject)->metadata),
				"revised", &dbupdated, NULL);
		/* We can use g_ascii_strcasecmp as it type cast every single 
		 * pointer to integer and unsigned char returning substract result */		
		gint datecmp;
		
		if(force || (multilang
					&& (dbobject_lang != object_lang))) {

			datecmp = -1;
		
		} else {
			
			datecmp = g_ascii_strcasecmp(
					(const gchar *)dbupdated, 
					(const gchar *)updated);
		}

		g_free(updated);
		g_free(dbupdated);

		gboolean deleted;
		gboolean ret;

		if(datecmp > 0 || datecmp == 0) {
			
			/* Database object is more recent or exactly the same */
			g_object_unref(builder);
			g_object_unref(dbobject);	
			MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_IMPORTED);
	
			if(multilang) {
				midgard_connection_set_lang(object->dbpriv->mgd, 
						init_lang_str);
				midgard_connection_set_default_lang(object->dbpriv->mgd,
						init_lang_str);
			}

			return FALSE;

		} else if (datecmp < 0) {
			
			/* Database object is elder */	
			
			/* DELETE */
			g_object_get(G_OBJECT(MIDGARD_OBJECT(object)->metadata), 
					"deleted", &deleted, NULL);
			/* Imported object is marked as deleted , so 
			 * * we delete object from database */
			if(deleted){
				g_object_unref(builder);
				ret = midgard_object_delete(dbobject);
				g_object_unref(dbobject);
	
				if(multilang) {
					midgard_connection_set_lang(object->dbpriv->mgd,
							init_lang_str);
					midgard_connection_set_default_lang(
							object->dbpriv->mgd, init_lang_str);
				}

				return ret;
			}

			/* UPDATE */
		
			/* Check if dbobject is deleted */
			g_object_get(G_OBJECT(dbobject->metadata),
					"deleted", &deleted, NULL);

			guint undelete;
			g_object_get(G_OBJECT(MIDGARD_OBJECT(object)->metadata), 
					"deleted", &undelete, NULL);

			if((deleted && !undelete)) {
				ret = midgard_object_undelete(mgd, 
						dbobject->dbpriv->guid);
				goto _update_object;
			}

			if(deleted && !force) {
				
				MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_DELETED);
				g_object_unref(dbobject);

				if(multilang) {
					midgard_connection_set_lang(
							object->dbpriv->mgd, init_lang_str);
					midgard_connection_set_default_lang(
							object->dbpriv->mgd, init_lang_str);
				}

				return FALSE;
			}

			_update_object:
			g_object_unref(builder);

			ret = _midgard_object_update(MIDGARD_OBJECT(object), OBJECT_UPDATE_IMPORTED);
			g_object_unref(dbobject);

			if(multilang) {
				midgard_connection_set_lang(
						object->dbpriv->mgd, init_lang_str);
				midgard_connection_set_default_lang(
						object->dbpriv->mgd, init_lang_str);
			}

			return ret;
		}		
	}
	return FALSE;
}


static gboolean __import_blob_from_xml(	MidgardConnection *mgd,
					xmlDoc *doc, 
					xmlNode *node)
{
	gchar *content;
	struct stat statbuf;
	const gchar *guid = 
		(const gchar *)xmlGetProp(node, BAD_CAST "guid");
	
	if(!guid) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		g_warning("Object's guid is empty. Can not import blob file.");
		g_free((gchar *)guid);
		xmlFreeDoc(doc);
		return FALSE;
	}

	if(!midgard_is_guid((const gchar *)guid)) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		g_warning("'%s' is not a valid guid", guid);
		g_free((gchar *)guid);
		xmlFreeDoc(doc);
		return FALSE;
	}
	
	MgdObject *object = 
		midgard_object_class_get_object_by_guid(mgd, guid);
	
	/* TODO , Add more error messages to inform about object state. 
	 * One is already set by midgard_object_class_get_object_by_guid */
	if(!object) {
		g_free((gchar *)guid);
		xmlFreeDoc(doc);
		return FALSE;
	}

	/* FIXME, define even macro to get blobdir ( core level only ) */
	gchar *blobdir = object->dbpriv->mgd->priv->config->blobdir;

	if (!blobdir || (*blobdir != '/')
			|| (stat(blobdir, &statbuf) != 0)
			|| !S_ISDIR(statbuf.st_mode)) {
		g_warning("Blobs directory is not set");
		g_free((gchar *)guid);
		xmlFreeDoc(doc);
		return FALSE;
	}

	gchar *location;
	gchar *blobpath = NULL;
	g_object_get(G_OBJECT(object), "location", &location, NULL);
	if(strlen(location) > 1) {
		blobpath = g_strconcat(
				blobdir,
				"/",
				location,
				NULL);
	}

	/* TODO, Find the way to get content and not its copy */
	/* node->content doesn't seem to hold it */
	content = (gchar *)xmlNodeGetContent(node);
	
	gsize content_length =
		(gsize) strlen(content);
	guchar *decoded =
		g_base64_decode(content, &content_length);
	g_free(content);
	
	FILE *fp = fopen(blobpath, "w+");
	fwrite(decoded, sizeof(char),
			content_length, fp);
	
	fclose(fp);	
	g_free(decoded);
	g_free((gchar *)guid);
	xmlFreeDoc(doc);

	return TRUE;
}

void midgard_replicator_import_from_xml(	MidgardReplicator *self,
						MidgardConnection *mgd,
						const gchar *xml, 
						gboolean force)
{
	MidgardConnection *_mgd;
	
	if(self	== NULL)
		_mgd = mgd;
	else 
		_mgd = self->priv->mgd;
	
	xmlDoc *doc = NULL;
	xmlNode *root_node = NULL;
	const gchar *init_lang_str = 
		midgard_connection_get_lang(_mgd);
	const gchar *init_dlang_str = 
		midgard_connection_get_default_lang(_mgd);
	midgard_core_object_get_xml_doc(_mgd, xml, &doc, &root_node);
	
	if(doc == NULL || root_node == NULL)
		return;

	xmlNodePtr child = _get_type_node(root_node->children);
	if(!child) {
		g_warning("Can not get midgard type name from the given xml");
		xmlFreeDoc(doc);
		return;
	}

	GType object_type = g_type_from_name((const gchar *)child->name);
	
	if(object_type == MIDGARD_TYPE_BLOB) {
		/* it will destroy xmlDoc */
		__import_blob_from_xml(_mgd, doc, child);
		return;
	}

	xmlChar *attr, *guid_attr;
	MgdObject *dbobject;
	guint langid = 0;

	for(; child; child = _get_type_node(child->next)) {
		
		attr = xmlGetProp(child, BAD_CAST "purge");
		guid_attr = xmlGetProp(child, BAD_CAST "guid");

		if(g_str_equal(attr, "yes")) {
			
			dbobject = midgard_object_class_get_object_by_guid(
					_mgd, (const gchar *)guid_attr);

			if(dbobject || 
					( !dbobject && 
					 (mgd->errnum == MGD_ERR_OBJECT_DELETED)
					 )) {
				midgard_object_purge(dbobject);
				if(dbobject)
					g_object_unref(dbobject);
				xmlFree(attr);
				xmlFree(guid_attr);
				continue;
			}
		}

		xmlFree(attr);

		MgdObject *object =
			midgard_object_new(_mgd, (const gchar *)child->name, NULL);
		if(!object) {
			g_warning("Can not create %s instance", child->name);
			xmlFreeDoc(doc);
			xmlFree(attr);
			xmlFree(guid_attr);
			continue;
		}

		if (guid_attr) {
			object->dbpriv->guid = (const gchar *)g_strdup((gchar *)guid_attr);
		}

		if(!_nodes2object(G_OBJECT(object), child->children, force)) {
			xmlFree(guid_attr);
			g_object_unref(object);
			continue;
		}		
		
		MidgardObjectClass *klass = 
			MIDGARD_OBJECT_GET_CLASS(object);
		if(midgard_object_class_is_multilang(klass)){
			
			gboolean langnz = TRUE;
			attr =  xmlGetProp(child, BAD_CAST "lang");

			if((!attr) || (attr && g_str_equal(attr, "")))
				langnz = FALSE;
			
			if(langnz) {
							
				langid = midgard_connection_get_lang_id(
						_mgd, (const gchar *)attr);
				MR_SET_ALL_LANG(_mgd, (const gchar *)attr);
				g_object_set(object, "lang", langid, NULL);
			}
		}

		/* TODO */
		/* Add lang property ! */
		
		if(!midgard_replicator_import_object(NULL, MIDGARD_DBOBJECT(object), force)){
			xmlFree(guid_attr);
			g_object_unref(object);
			midgard_connection_set_lang(_mgd, init_lang_str);
			MR_SET_LANG(_mgd, init_lang_str);
			MR_SET_DLANG(_mgd, init_dlang_str);
			continue;
		} else {
			xmlFree(guid_attr);
			g_object_unref(object);
			MR_SET_LANG(_mgd, init_lang_str);
			MR_SET_DLANG(_mgd, init_dlang_str);
		}
	}
	xmlFreeDoc(doc);
}

/* GOBJECT ROUTINES */

static void _midgard_replicator_instance_init(
		GTypeInstance *instance, gpointer g_class)
{
	MidgardReplicator *self = (MidgardReplicator *) instance;
	self->priv = g_new(MidgardReplicatorPrivate, 1);

	self->priv->object = NULL;
	self->priv->klass = NULL;
	self->priv->type = 0;
	self->priv->typename = NULL;
}

static void _midgard_replicator_finalize(GObject *object)
{
	g_assert(object != NULL);
	
	MidgardReplicator *self = (MidgardReplicator *) object;

	g_free(self->priv);
}

static void _midgard_replicator_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	/* MidgardReplicatorClass *klass = MIDGARD_REPLICATOR_CLASS (g_class); */
	
	gobject_class->finalize = _midgard_replicator_finalize;
}

/* Returns MidgardReplicator type. */
GType midgard_replicator_get_type(void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardReplicatorClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_replicator_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardReplicator),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_replicator_instance_init /* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT,
				"midgard_replicator",
				&info, 0);
	}
	return type;
}
