/* 
 * Copyright (C) 2008 Piotr Pokora <piotrek.pokora@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <midgard_http_host.h>
#include "midgard_core_object.h"
#include "midgard_core_query_builder.h"
#include "midgard_error.h"

static void __list_string_free(GList *lst)
{
	if(lst == NULL)
		return;
	
	GList *l;

	for ( l = lst; l != NULL; l = l->next ) {		
		if(l->data != NULL)
			g_free(l->data);
		l->data = NULL;
	}

	g_list_free(l);
	l = NULL;

	return;
}

static void __list_from_uri(MidgardHttpHost *self, const gchar *uri)
{
	g_assert(uri != NULL);
	
	GList *list = NULL;
	guint i = 1;

	gchar **s = g_strsplit(uri, "/", -1);

	/* This is '/' request */
	if(!s) {
		self->uri = g_strdup(uri);
		self->prefix = NULL;
		return;
	}

	/* prepend tokens , it's faster than append them */
	/* note that we start from 1 ( which is empty or null value before '/' ) */
	while(s[i] != NULL){
	
		/* We do not need empty string after the trailing slash,
		 * or arguments from url like /a//b/. */
		if(*s[i] != '\0') 
			list = g_list_prepend(list, g_strdup(s[i]));		
			
		i++;
	}
	
	g_strfreev(s);

	/* Reverse tokens list to get correct order */
	list = g_list_reverse(list);

	self->urilist = list;

	return;
}

MgdObject *__find_host(
		MidgardConnection *mgd, const gchar *name,
		guint port, const gchar *uri)
{
	g_assert(mgd != NULL);

	GValue gval = {0, }, pval = {0, };

	MidgardQueryBuilder *builder = 
		midgard_query_builder_new(mgd, "midgard_host");	

	/* We must find *any* sitegroup host */
	builder->priv->sitegrouped = FALSE;

	/* hostname */
	g_value_init(&gval,G_TYPE_STRING);
	g_value_set_string(&gval, name);
	midgard_query_builder_add_constraint(builder,
			"name",
			"=", &gval);
	g_value_unset(&gval);

	/* port */
	g_value_init(&gval,G_TYPE_VALUE_ARRAY);
	GValueArray *array =
		g_value_array_new(2);

	/* port 0 */
	g_value_init(&pval, G_TYPE_UINT);
	g_value_set_uint(&pval, 0);
	g_value_array_append(array, &pval);
	g_value_unset(&pval);

	/* port from uri */
	g_value_init(&pval, G_TYPE_UINT);
	g_value_set_uint(&pval, (guint) port);
	g_value_array_append(array, &pval);
	g_value_unset(&pval);

	g_value_take_boxed(&gval, array);
	midgard_query_builder_add_constraint(builder,
			"port",
			"IN", &gval);
	g_value_unset(&gval);

	/* online */
	g_value_init(&gval,G_TYPE_UINT);
	g_value_set_uint(&gval, 0);
	midgard_query_builder_add_constraint(builder,
			"online",
			"<>", &gval);
	g_value_unset(&gval);

	/* prefix */
	g_value_init(&gval,G_TYPE_VALUE_ARRAY);
	array = g_value_array_new(2);
	
	/* no prefix */
	g_value_init(&pval, G_TYPE_STRING);
	g_value_set_string(&pval, "");
	g_value_array_append(array, &pval);
	g_value_unset(&pval);

	/* prefix from uri */
	g_value_init(&pval,G_TYPE_STRING);
	g_value_set_string(&pval, uri);
	g_value_array_append(array, &pval);
	g_value_unset(&pval);

	g_value_take_boxed(&gval, array);
	midgard_query_builder_add_constraint(builder,
			"prefix",
			"IN", &gval);
	g_value_unset(&gval);

	/* Limit returned array to only one object */
	/* We get the best match */
	midgard_query_builder_set_limit(builder, 1);
	midgard_query_builder_add_order(builder, "prefix", "DESC");
	midgard_query_builder_add_order(builder, "port", "DESC");

	guint n_objects;
	GObject **ret_object = 
		midgard_query_builder_execute(builder, &n_objects);

	if(!ret_object){
		g_object_unref(builder);
		return NULL;
	}

	MgdObject *host = (MgdObject *) ret_object[0];
	
	g_object_unref(builder);
	g_free(ret_object);

	return host;	
}

/**
 * midgard_http_host_new:
 * @mgd: #MidgardConnection instance
 * @name: host's name
 * @port: port
 * @uri: request's uri
 *
 * Lookup for named host in midgard database.
 * This object is an opaque object which holds all midgard_host related data.
 * Style, root page, etc.
 *
 * host, style and root_page members are not freed when g_object_unref is called
 * for given #MidgardHttpHost. You must free these objects if no longer needed.
 *
 * Sitegroup and language context is set if #MidgardHttpHost is successfully initialized.
 *
 * Returns: new #MidgardHttpHost or %NULL otherwise.
 * #uri can not be explicitly set to NULL. Use empty string 
 * if there's not need for unique uri.
 */ 
extern MidgardHttpHost *midgard_http_host_new(
		MidgardConnection *mgd, const gchar *name,
		guint port, const gchar *uri) 
{
	g_assert(mgd != NULL);

	mgd_info("Creating new http_host (%s:%d%s).", name, port, uri);

	if(uri == NULL) {
		g_warning("Can not set NULL uri for request");
		return NULL;
	}

	MidgardHttpHost *self;
	self = g_object_new(MIDGARD_TYPE_HTTP_HOST, NULL);

	self->mgd = mgd;

	/* create list from uri */
	gchar *uri_copy = g_strdup(uri);
	__list_from_uri(self, uri_copy);	
	g_free(uri_copy);

	GList *list = self->urilist;

	gchar *prefix = NULL;
	if(list) {
		list = g_list_first(list);
		prefix = g_strconcat("/", list->data, NULL);
	} else {
		prefix = g_strdup("");
	}

	MgdObject *host = __find_host(mgd, name, port, prefix);

	if (!host) {
		mgd_info("Failed to find host (%s:%d%s).", name, port, prefix);
		g_object_unref(self);
		g_free(prefix);
		__list_string_free(list);
		return NULL;
	}

	gchar *hprefix;
	gchar *hname;
	guint hport;
	g_object_get(G_OBJECT(host), 
			"prefix", &hprefix, 
			"name", &hname ,
			"port", &hport, NULL);

	if (hprefix == NULL) {
		hprefix = g_strdup("");
	}

	mgd_info("Found host (%s:%d%s).", hname, hport, hprefix);

	/* IMPORTANT! We must set sitegroup context NOW */
	/* We can ignore sitegroup context, but then we should initialize all 
	 * required data much more later, which in practice means that we must 
	 * write the same solution per language */
	const gchar *pname = 
		midgard_connection_get_sitegroup_from_id(mgd, host->dbpriv->sg);
	midgard_connection_set_sitegroup(mgd, pname);

	/* set language context */
	guint propid = 0;
	pname = NULL;
	g_object_get(G_OBJECT(host), "lang", &propid, NULL);
	if(propid > 0) {
		pname = midgard_connection_get_lang_from_id(mgd, propid);
		midgard_connection_set_lang(mgd, pname);
	}

	/* set name and port */
	self->name = g_strdup(name);
	self->port = port;

	guint pint = 0;

	/* set prefix */	
	if(g_str_equal(prefix, hprefix)) {
		self->prefix = prefix;
	} else {
		self->prefix = g_strdup("");
		g_free(prefix);
	}

	g_free(hprefix);
	g_free(hname);

	/* set uri */
	GList *ulist;	
	GString *ustr = g_string_new("");
	for(ulist = list; ulist != NULL; ulist = ulist->next){	
		g_string_append_printf(ustr, "/%s", (gchar *) ulist->data);
	}
	self->uri = ustr->str;
	g_string_free(ustr, FALSE);

	/* set host */
	self->host = host;

	/* set root_page */
	g_object_get(G_OBJECT(host), "root", &pint, NULL);
	if (pint > 0) {	
		GValue pval = {0, };
		g_value_init(&pval, G_TYPE_UINT);
		g_value_set_uint(&pval, pint);
		MgdObject *page = midgard_object_new(mgd, "midgard_page", &pval);
		g_value_unset(&pval);

		self->root_page = page;
	}

	/* set argv */
	self->argv = list;

	return self;
}

static void __set_auth(MidgardRequestConfig *rcfg, gchar *authinfo)
{
	if(authinfo != NULL && *authinfo != '\0') {  
		if(g_strstr_len((const gchar *)authinfo, strlen(authinfo), "auth")) {  
			mgd_info("Setting required authentication flag");
			midgard_request_config_set_auth(rcfg, TRUE);
		}
	}

	if(authinfo != NULL) {
		g_free(authinfo);
		authinfo = NULL;
	}
}

/**
 * midgard_http_host_get_request_config:
 * @self: #MidgardHttpHost instance
 *
 * This method initialize all request's configuration data.
 * It's host, style, pages, argv, argc, etc.
 * You do not have to explicitly set sitegroup or language context, as those
 * are set when #MidgardHttpHost is being initialized.
 *
 * Returns: #MidgardHttpHost object or %NULL on failure.
 */ 
MidgardRequestConfig *midgard_http_host_get_request_config(MidgardHttpHost *self)
{
	g_assert(self != NULL);
	g_assert(self->mgd != NULL);
	
	if(!self->root_page) {
		g_warning("Root page not set for current host");
		return NULL;
	}

	if(!self->host) {
		g_warning("Midgard host not set");
		return NULL;
	}
	
	MidgardRequestConfig *rcfg = midgard_request_config_new(self->uri);
	
	if(!rcfg) {
		g_warning("Can not initialize midgard_request_config for http request");
		return NULL;
	}

	/* Check if host requires athentication */
	/* TODO, FIXME, add auth boolean property to host and pages */
	gchar *authinfo = NULL;
	guint styleid = 0, styleset = 0;
	g_object_get(G_OBJECT(self->host), 
			"info", &authinfo, "style", &styleid, NULL);
	
	if(styleid > 0) styleset = styleid;
	styleid = 0;
	
	if(authinfo != NULL) {
		if(g_str_equal(authinfo, "auth"))
			midgard_request_config_set_auth(rcfg, TRUE);
		g_free(authinfo);
		authinfo = NULL;
	}

	GList *list = NULL;
	MidgardConnection *mgd = self->mgd;

	/* Build pages array and decrease urilist */
	GList *pages = NULL;
	pages = g_list_prepend(pages, self->root_page);
	
	guint up_id = 0;
	g_object_get(G_OBJECT(self->root_page), 
			"id", &up_id, "info", &authinfo, "style", &styleid, NULL);
	__set_auth(rcfg, authinfo);

	if(styleid > 0) styleset = styleid;
	styleid = 0;
	
	MidgardQueryBuilder *builder;
	GObject **objects = NULL, *last_page;
	
	last_page = G_OBJECT(self->root_page);
	list = NULL;
	GValue pval = {0, };	

	/* Uri "iterator". Keep urilist list. Do not copy its elements to improve performance. */
	list = self->urilist;
	if(g_list_length(list) < 1)
		goto _SET_PAGES;
	do {
		builder = midgard_query_builder_new(mgd, "midgard_page");
		g_value_init(&pval, G_TYPE_UINT);
		g_value_set_uint(&pval, up_id);
		midgard_query_builder_add_constraint(builder, "up", "=", &pval);
		g_value_unset(&pval);
		
		g_value_init(&pval, G_TYPE_STRING);
		g_value_set_string(&pval, list->data);
		midgard_query_builder_add_constraint(builder, "name", "=", &pval);
		g_value_unset(&pval);
		
		midgard_query_builder_set_limit(builder, 1);
		
		guint n_objects;
		objects = midgard_query_builder_execute(builder, &n_objects);
		
		g_object_unref(builder);

		if(objects) {
			
			if(MIDGARD_IS_OBJECT(objects[0])) {
				
				pages = g_list_prepend(pages, objects[0]);
				last_page = objects[0];
				if(list->data)
					g_free(list->data);
				list = g_list_remove(list, (gconstpointer) list->data);
				g_object_get(G_OBJECT(objects[0]), 
						"id", &up_id, 
						"info", &authinfo, 
						"style", &styleid, NULL);
				__set_auth(rcfg, authinfo);

				if(styleid > 0) styleset = styleid;
				styleid = 0;
				
			} else {
				
				g_warning("Tried to get page, but it doesn't seem to be midgard object");
			}
			 
			if(objects != NULL)
				g_free(objects);
			
			objects = NULL;
		
		} else {

			break;
		}

	} while(list != NULL);

_SET_PAGES:
	/* Set pages array */
	rcfg->pages = (const GObject **) g_new(GObject *, g_list_length(pages)+1);
	
	guint pi = 0;
	for(pages = g_list_reverse(pages); pages != NULL; pages = pages->next) {
		
		rcfg->pages[pi] = G_OBJECT(pages->data);
		pi++;
	}

	rcfg->pages[pi] = NULL;

	g_list_free(pages);

	self->urilist = g_list_first(list);
	guint listl = g_list_length(self->urilist);

	if(listl == 1) {
		mgd_info("Trying to get attachment object ( FIXME )");
		/* TODO, try to get attachment and set listl=0 */
	}

	if(listl > 0) {
		
		GValueArray *array = g_value_array_new(listl);
		
		for(list = g_list_first(self->urilist); list != NULL; list = list->next) {
			
			GValue *strval = g_new0(GValue, 1);
			g_value_init((GValue *)strval, G_TYPE_STRING);
			g_value_set_string((GValue *)strval, (gchar *)list->data);
			array = g_value_array_append(array, (const GValue *)strval);
			g_value_unset(strval);
			g_free(strval);
			g_free(list->data);
		}
		
		midgard_request_config_set_argv(rcfg, array);
	}

	self->urilist = g_list_first(list);

	/*set style */
	if(styleset > 0) {

		GValue sval = {0, };
		g_value_init(&sval, G_TYPE_UINT);
		g_value_set_uint(&sval, styleset);
		MgdObject *style = midgard_object_new(mgd, "midgard_style", &sval);
		g_value_unset(&sval);
	
		midgard_request_config_set_style(rcfg, G_OBJECT(style));
	}

	midgard_request_config_set_host(rcfg, G_OBJECT(self->host));
	midgard_request_config_set_page(rcfg, last_page);	

	return rcfg;
}

/* GOBJECT ROUTINES */

static void _midgard_http_host_instance_init(
		GTypeInstance *instance, gpointer g_class)
{
	MidgardHttpHost *self = (MidgardHttpHost *) instance;
	
	self->host = NULL;
	self->root_page = NULL;
	self->style = NULL;
	self->name = NULL;
	self->prefix = NULL;
	self->uri = NULL;
	self->port = 0;
	self->auth_required = FALSE;

	self->possible_prefix = NULL;
	self->mgd = NULL;
	self->urilist = NULL;
}

static void _midgard_http_host_finalize(GObject *object)
{
	MidgardHttpHost *self = (MidgardHttpHost *) object;
	
	self->host = NULL;
	self->style = NULL;

	if(self->prefix)
		g_free((gchar *)self->prefix);
	self->prefix = NULL;

	if(self->name)
		g_free((gchar *)self->name);
	self->name = NULL;
	
	if(self->uri)
		g_free((gchar *)self->uri);
	self->uri = NULL;

	if(self->possible_prefix)
		g_free((gchar *)self->possible_prefix);
	self->possible_prefix = NULL;

	if(self->urilist) 
		__list_string_free(self->urilist);
}

static void _midgard_http_host_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);

	gobject_class->set_property = NULL;
	gobject_class->get_property = NULL;
	gobject_class->finalize = _midgard_http_host_finalize;
}

extern GType midgard_http_host_get_type(void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardHttpHostClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_http_host_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardHttpHost),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_http_host_instance_init /* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT,
				"midgard_http_host",
				&info, 0);
	}
	return type;
}
