/* Midgard schema , records and objects definition.
   
  Copyright (C) 2004,2005,2006,2007,2008 Piotr Pokora <piotrek.pokora@gmail.com>
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as published
  by the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <config.h>
#include "midgard_schema.h"
#include "midgard_object.h"
#include <libxml/parser.h>
#include <libxml/tree.h>
#include "midgard_datatypes.h"
#include "midgard_type.h"
#include "schema.h"
#include "midgard_object_class.h"
#include "midgard_core_object.h"
#include "midgard_core_object_class.h"
#include "midgard_reflection_property.h"
#include <libxml/parserInternals.h>
#include "guid.h"

/* TODO tune header files , no need to include string.h while we need to include midgard.h in almost every case */

/* Global schema_trash used when MgdSchema is destroyed 
 * Trash must be global as we share pointers between different schemas.
 * Especially Midgard core object's MgdSchemaTypes are shared.
 * */
static GSList *schema_trash = NULL;

/* Global schema file path, it's a pointer to filename being parsed.
 * Only for warning purposes */
static const gchar *parsed_schema = NULL;

MgdSchemaPropertyAttr *_mgd_schema_property_attr_new(){
	
	MgdSchemaPropertyAttr *prop = g_new(MgdSchemaPropertyAttr, 1); /* FIXME, LEAK */
	prop->gtype = G_TYPE_NONE;
	prop->type = NULL;
	prop->default_value = NULL;
	prop->dbtype = NULL;
	prop->field = NULL;
	prop->dbindex = FALSE;
	prop->table = NULL;
	prop->tablefield = NULL;
	prop->upfield = NULL;
	prop->parentfield = NULL;
	prop->primaryfield = NULL;
	prop->is_multilang = FALSE;
	prop->link = NULL;
	prop->link_target = NULL;
	prop->is_primary = FALSE;
	prop->is_reversed = FALSE;
	prop->is_link = FALSE;
	prop->is_linked = FALSE;
	prop->description = NULL;
	
	return prop; 
}

void _mgd_schema_property_attr_free(MgdSchemaPropertyAttr *prop)
{
	g_assert(prop != NULL);

	g_free((gchar *)prop->type);
	g_free((gchar *)prop->dbtype);	
	g_free((gchar *)prop->field);
	g_free((gchar *)prop->table);
	g_free((gchar *)prop->tablefield);
	g_free((gchar *)prop->upfield);
	g_free((gchar *)prop->parentfield);
	g_free((gchar *)prop->primaryfield);
	g_free((gchar *)prop->link);
	g_free((gchar *)prop->link_target);

	g_free(prop);
}

void _destroy_property_hash(gpointer key, gpointer value, gpointer userdata)
{
	MgdSchemaPropertyAttr *prop = (MgdSchemaPropertyAttr *) value;
	gchar *name = (gchar *) key;
	
	if(prop)
		_mgd_schema_property_attr_free(prop);

	if(name)
		g_free(name);
}

MgdSchemaTypeQuery *_mgd_schema_type_query_new()
{
	MgdSchemaTypeQuery *query = g_new(MgdSchemaTypeQuery, 1); /* FIXME , LEAK */
	query->select = NULL;
	query->select_full = NULL;
	query->from = NULL;
	query->where = NULL;
	query->table = NULL;
	query->link = NULL;
	query->tables = g_strdup("");
	query->primary = NULL;
	query->parentfield = NULL;
	query->upfield = NULL;
	query->use_lang = FALSE;

	return query;
}

void _mgd_schema_type_query_free(MgdSchemaTypeQuery *query)
{
	g_assert(query != NULL);
	
	g_free((gchar *)query->select);
	g_free((gchar *)query->select_full);
	/* pointer to type->table, g_free((gchar *)query->from); */
	g_free((gchar *)query->where);
	g_free((gchar *)query->table);
	g_free((gchar *)query->primary);
	g_free((gchar *)query->link);
	g_free((gchar *)query->tables);
	g_free((gchar *)query->parentfield);
	g_free((gchar *)query->upfield);
	

	g_free(query);
}

void _destroy_query_hash(gpointer key, gpointer value, gpointer userdata)
{
	MgdSchemaTypeQuery *query = (MgdSchemaTypeQuery *) value;
	gchar *name = (gchar *) key;
	
	if(query)
		_mgd_schema_type_query_free(query);

	if(name)
		g_free(name);
}

MgdSchemaTypeAttr *_mgd_schema_type_attr_new()
{
	MgdSchemaTypeAttr *type = g_new(MgdSchemaTypeAttr, 1);
	type->base_index = 0;
	type->num_properties = 0;
	type->class_nprop = 0;
	type->params = NULL;
	type->properties = NULL;
	type->table = NULL;
	type->parent = NULL;
	type->primary = NULL;
	type->property_up = NULL;
	type->property_parent = NULL;				
	type->property_count = 0;
	type->use_lang = FALSE;
	type->tables = NULL;
	type->tableshash = g_hash_table_new(g_str_hash, g_str_equal);
	type->prophash = g_hash_table_new(g_str_hash, g_str_equal);
	type->query = _mgd_schema_type_query_new();
	type->childs = NULL;
	type->unique_name = NULL;
	
	return type;
}

void _mgd_schema_type_attr_free(MgdSchemaTypeAttr *type)
{
	g_assert(type != NULL);

	g_free((gchar *)type->table);
	g_free((gchar *)type->tables);
	g_free((gchar *)type->parent);
	g_free((gchar *)type->primary);
	g_free((gchar *)type->property_up);
	g_free((gchar *)type->property_parent);

	g_hash_table_foreach(type->tableshash, _destroy_query_hash, NULL);
	g_hash_table_destroy(type->tableshash);	
	
	g_hash_table_foreach(type->prophash, _destroy_property_hash, NULL);
	g_hash_table_destroy(type->prophash);

	_mgd_schema_type_query_free(type->query);

	g_free(type->params);
	g_free(type->properties);
	g_free((gchar *)type->unique_name);
	
	g_free(type);
}

/* return type's struct  or NULL if type is not in schema */
MgdSchemaTypeAttr *
midgard_schema_lookup_type (MidgardSchema *schema, gchar *name)
{
	g_assert(schema != NULL);
	g_assert(name != NULL);

	return (MgdSchemaTypeAttr *) g_hash_table_lookup(schema->types, name);	
}


/**
 * \ingroup mgdschema
 * \internal
 * Lookup for property attributes 
 *
 * @param schema MgdSchema used by application
 * @param type Name of type registered in schema
 * @param propname Name of property registered for type
 *
 * @return hash table which contains property's attributes
 */ 
GHashTable *
midgard_schema_lookup_type_member(MidgardSchema *schema, const gchar *typename, gchar *propname)
{
	g_assert(schema != NULL);
	g_assert(typename != NULL);
	g_assert(propname != NULL);
	
	MgdSchemaTypeAttr *sts;
	
	sts = midgard_schema_lookup_type(schema, (gchar *)typename);
	
	if (sts != NULL) 
		return g_hash_table_lookup(sts->prophash, propname);

	return NULL;
}

static void __warn_msg(xmlNode * node, const gchar *msg)
{
	gchar *prop = (gchar *)xmlGetProp(node, (const xmlChar *)"name");
	g_warning("%s ( %s:%s, %s on line %ld )", msg, node->name, prop,
			parsed_schema, xmlGetLineNo(node));
	g_free(prop);
}

/* Define which element names should be addedd to schema. 
 * We change them only here , and in xml file itself.
 */
static const gchar *mgd_complextype[] = { "type", "property", "Schema", "description", "include", NULL };
static const gchar *mgd_attribute[] = { 
	"name", 
	"table", 
	"parent", 
	"parentfield", 
	"type", 
	"link", 
	"upfield", 
	"field", 
	"multilang", 
	"reverse", 
	"primaryfield", 
	"dbtype", 
	"index", 
	"default", 
	"unique",
	NULL };

static const gchar *schema_allowed_types[] = {
	"integer",
	"unsigned integer",
	"float",
	"double",
	"string",
	"text",
	"guid",
	"longtext",
	"bool",
	"boolean",
	"datetime",
	NULL
};


/* RESERVED type names */
static const gchar *rtypes[] = { 
	"midgard_sitegroup", 
	"midgard_user",
	NULL
};

/* RESERVED table names */
static const gchar *rtables[] = {
	"sitegroup",
	"repligard",
	"midgard_user",
	NULL
};

/* RESERVED column names */
static const gchar *rcolumns[] = {
	"metadata_",
	"sitegroup",
	"group", /* This is MySQL reserved word */
	NULL
};

static gboolean strv_contains(const char **strv, const xmlChar *str) {
        g_assert(strv != NULL);
        g_assert(str != NULL);
        while (*strv != NULL) {
                if (g_str_equal(*strv++, str)) {
                        return TRUE;
                }
        }
        return FALSE;
}

static void check_metadata_column(const xmlChar *column){
	
	if(g_str_has_prefix((const gchar *)column, "metadata_"))
		g_critical("Column names prefixed with 'metadata_' are reserved ones");
}

static void check_property_name(const xmlChar *name, const gchar *typename){
	
	g_assert(name != NULL);

	if(g_strrstr_len((const gchar *)name, strlen((const gchar *)name), "_") != NULL)
		g_critical("Type '%s', property '%s'. Underscore not allowed in property names!",
				typename, name);

	if(g_str_equal(name, "sitegroup"))
		g_critical("Type '%s'.Property name 'sitegroup' is reserved!",
				typename);

	if(g_str_equal(name, "guid"))
		g_critical("Type '%s'.Property name 'guid' is reserved!",
				typename);		
}

/* Get all type attributes */
static void
_get_type_attributes(xmlNode * node, MgdSchemaTypeAttr *type_attr)
{
	xmlAttr *attr;
	xmlChar *attrval;

	if (node != NULL){

		attr = node->properties;
		attr = attr->next; /* avoid getting already added */

		while (attr != NULL){

			if(!strv_contains(mgd_attribute, attr->name)){
				g_warning("Wrong attribute '%s' in '%s' on line %ld",
						attr->name, parsed_schema, xmlGetLineNo(node));	
			}	
			attrval = xmlNodeListGetString (node->doc, attr->children, 1);
					
			if(g_str_equal(attr->name, "table")) {
				/* Check if table name is reserved one */
				if (strv_contains(rtables, attrval)) {
					g_critical("'%s' is reserved table name",
							attrval);
				}								
				type_attr->table = g_strdup((const gchar *)attrval);							}

			if(g_str_equal(attr->name, "parent"))
				type_attr->parent = g_strdup((gchar *)attrval);
	
			xmlFree(attrval);
			attr = attr->next;			
		}
	}
}

static void midgard_core_schema_get_property_type(xmlNode *node, 
		MgdSchemaTypeAttr *type_attr, MgdSchemaPropertyAttr *prop_attr)
{
	gboolean typeisset = FALSE;
	xmlAttr *attr;
	xmlChar *attrval;

	if (node != NULL) {
		
		attr = node->properties;
		attr = attr->next;
		
		while (attr != NULL) {

			if(g_str_equal(attr->name, "type")) {

				attrval = xmlNodeListGetString (node->doc, attr->children, 1);

				if (strv_contains(schema_allowed_types, attrval)) {
					
					typeisset = TRUE;
					
					prop_attr->type = g_strdup((const gchar*)attrval);
			
					if(g_str_equal(attrval, "string"))
						prop_attr->gtype = MGD_TYPE_STRING;

					if(g_str_equal(attrval, "integer"))
						prop_attr->gtype = MGD_TYPE_INT;

					if(g_str_equal(attrval, "unsigned integer"))
						prop_attr->gtype = MGD_TYPE_UINT;

					if(g_str_equal(attrval, "float"))
						prop_attr->gtype = MGD_TYPE_FLOAT;
					
					/* FIXME, change to MGD_TYPE_DOUBLE once mgdschema supports it */
					if(g_str_equal(attrval, "double"))
						prop_attr->gtype = MGD_TYPE_FLOAT;

					if(g_str_equal(attrval, "boolean"))
						prop_attr->gtype = MGD_TYPE_BOOLEAN;

					if(g_str_equal(attrval, "bool"))
						prop_attr->gtype = MGD_TYPE_BOOLEAN;
				
					if(g_str_equal(attrval, "datetime"))
						prop_attr->gtype = MGD_TYPE_TIMESTAMP;
				
					if(g_str_equal(attrval, "longtext"))
						prop_attr->gtype = MGD_TYPE_LONGTEXT;
				
					if(g_str_equal(attrval, "text"))
						prop_attr->gtype = MGD_TYPE_LONGTEXT;				

					if(g_str_equal(attrval, "guid"))
						prop_attr->gtype = MGD_TYPE_GUID;
				
				} else {

					prop_attr->type = g_strdup("string");
					prop_attr->gtype = MGD_TYPE_STRING;
					g_debug("Setting string type for declared %s type", attrval);
				}

				xmlFree(attrval);
			}

			attr = attr->next;
		}
	}
	
	return;
}

static void midgard_core_schema_get_default_value(xmlNode *node, 
		MgdSchemaTypeAttr *type_attr, MgdSchemaPropertyAttr *prop_attr)
{	
	xmlAttr *attr;
	xmlChar *attrval;

	if (node != NULL) {
		
		attr = node->properties;
		attr = attr->next;
		
		while (attr != NULL) {

			if(g_str_equal(attr->name, "default")) {

				attrval = xmlNodeListGetString (node->doc, attr->children, 1);
				prop_attr->default_value = g_new0(GValue, 1);

				if (prop_attr->gtype == MGD_TYPE_STRING
						|| prop_attr->gtype == MGD_TYPE_LONGTEXT) {

					g_value_init(prop_attr->default_value, G_TYPE_STRING);
					g_value_set_string(prop_attr->default_value, (const gchar *)attrval);

				} else if (prop_attr->gtype == MGD_TYPE_INT) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default integer value from empty string");
						goto STEP_FORWARD;
					}

					g_value_init(prop_attr->default_value, G_TYPE_INT);
					g_value_set_int(prop_attr->default_value, (gint) atoi((const gchar *)attrval));

				} else if (prop_attr->gtype == MGD_TYPE_UINT) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default integer value from empty string");
						goto STEP_FORWARD;
					}

					g_value_init(prop_attr->default_value, G_TYPE_UINT);
					g_value_set_uint(prop_attr->default_value, (guint) atoi((const gchar *)attrval));


				} else if (prop_attr->gtype == MGD_TYPE_FLOAT) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default float value from empty string");
						goto STEP_FORWARD;
					}

					g_value_init(prop_attr->default_value, G_TYPE_FLOAT);
					g_value_set_float(prop_attr->default_value, (gfloat) g_ascii_strtod((const gchar *)attrval, NULL));

				} else if (prop_attr->gtype == MGD_TYPE_TIMESTAMP) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default timestamp value from empty string");
						goto STEP_FORWARD;
					}

					GValue strval = {0, };
					g_value_init(&strval, G_TYPE_STRING);
					g_value_set_string(&strval, (const gchar *)attrval); 

					g_value_init(prop_attr->default_value, MIDGARD_TYPE_TIMESTAMP);
					g_value_transform(&strval, prop_attr->default_value);
					g_value_unset(&strval);

				} else if (prop_attr->gtype == MGD_TYPE_BOOLEAN) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default boolean value from empty string");
						goto STEP_FORWARD;
					}

					g_value_init(prop_attr->default_value, G_TYPE_BOOLEAN);
					g_value_set_boolean(prop_attr->default_value, (gint) atoi((const gchar *)attrval));
	

				} else if (prop_attr->gtype == MGD_TYPE_GUID) {

					if (attrval == NULL || *attrval == '\0') {
						__warn_msg(node, "Can not set default guid value from empty string");
						goto STEP_FORWARD;
					}

					if (!midgard_is_guid((const gchar *)attrval)) {

						__warn_msg(node, "Given string (default value) is not guid");
						goto STEP_FORWARD;
					}

					g_value_init(prop_attr->default_value, G_TYPE_STRING);
					g_value_set_string(prop_attr->default_value, (const gchar *)attrval);

				} else {

					__warn_msg(node, "Do not know how to set default value for unknown type");
				}
					
				xmlFree(attrval);
			}
STEP_FORWARD:
			attr = attr->next;
		}
	}
	
	return;
}

/* Check if property is unique_name */
static void
midgard_core_schema_get_unique_name(xmlNode * node, MgdSchemaTypeAttr *type_attr, MgdSchemaPropertyAttr *prop_attr)
{
	xmlAttr *attr;
	xmlChar *attrval;

	if (node != NULL){

		attr = node->properties;
		attr = attr->next; /* avoid getting already added */

		while (attr != NULL){

			if (!strv_contains(mgd_attribute, attr->name)){
				g_warning("Wrong attribute '%s' in '%s' on line %ld",
						attr->name, parsed_schema, xmlGetLineNo(node));	
			}	
					
			if (g_str_equal(attr->name, "unique")) {

				attrval = xmlNodeListGetString (node->doc, attr->children, 1);

				if (g_str_equal(attrval, "yes")) {
					
					if (prop_attr->gtype != MGD_TYPE_STRING) 
						__warn_msg(node, "Unique name must be declared as string type");

					gchar *propname = (gchar *)xmlGetProp(node, (const xmlChar *)"name");
					type_attr->unique_name = propname;
				}

				xmlFree(attrval);
			}
	
			attr = attr->next;			
		}
	}
}

/* Get property attributes */
static void 
_get_property_attributes(xmlNode * node, 
		MgdSchemaTypeAttr *type_attr, MgdSchemaPropertyAttr *prop_attr)
{
	xmlAttr *attr;
	xmlChar *attrval;

	if (node != NULL){

		attr = node->properties;
		attr = attr->next;

		while (attr != NULL){

			attrval = xmlNodeListGetString (node->doc, attr->children, 1);
		
			if(!strv_contains(mgd_attribute, attr->name)){
				g_warning("Wrong attribute '%s' in '%s' on line %ld",
						attr->name, parsed_schema, xmlGetLineNo(node));
			}
			
			if(g_str_equal(attr->name, "dbtype"))
				prop_attr->dbtype = g_strdup((gchar *)attrval);
			
			if(g_str_equal(attr->name, "field")){
				/* Check if column name is reserved one */
				check_metadata_column(attrval);
				if (strv_contains(rcolumns, attrval)) {
					g_critical("'%s' is reserved column name",
							attrval);
				}								
				prop_attr->field = g_strdup((gchar *)attrval);
			}
		
			if(g_str_equal(attr->name, "index")) {
				
				if(g_str_equal(attrval, "yes"))
					prop_attr->dbindex = TRUE;
			}

			if(g_str_equal(attr->name, "table")){
				/* Check if table name is reserved one */
				if (strv_contains(rtables, attrval)) {
					g_critical("'%s' is reserved table name",
							attrval);
				}
				prop_attr->table = g_strdup((gchar *)attrval);
			}

			if(g_str_equal(attr->name, "upfield")){
				check_metadata_column(attrval);
				/* Check if column name is reserved one */
				if (strv_contains(rcolumns, attrval)) {
					g_critical("'%s' is reserved column name",
							attrval);
				}				
				if(!type_attr->property_up){
					xmlChar *tmpattr = 
						xmlGetProp (node, (const xmlChar *)"name");
					type_attr->property_up = g_strdup((gchar *)tmpattr);
					type_attr->query->upfield = g_strdup((gchar *)attrval);
					if(!midgard_core_object_prop_up_is_valid(prop_attr->gtype)) {
						__warn_msg(node, "Invalid type for up property.");
						g_error("Wrong schema attribute");
					}
					xmlFree(tmpattr);
				} else {
					g_warning("upfield redefined!");
				}
				prop_attr->upfield = g_strdup((gchar *)attrval);			
			}

			if(g_str_equal(attr->name, "parentfield")){
				check_metadata_column(attrval);
				/* Check if column name is reserved one */
				if (strv_contains(rcolumns, attrval)) {
					g_critical("'%s' is reserved column name",
							attrval);
				}
				if(!type_attr->property_parent){
					xmlChar *tmpattr = xmlGetProp (node, (const xmlChar *)"name");
					type_attr->property_parent = g_strdup((gchar *)tmpattr);
					type_attr->query->parentfield = g_strdup((gchar *)attrval);
					if(!midgard_core_object_prop_parent_is_valid(prop_attr->gtype)) {
						__warn_msg(node, "Invalid type for parent property.");
						g_error("Wrong schema attribute");
					}
					xmlFree(tmpattr);
				} else {                         
					g_warning("parentfield redefined!");
				}
				prop_attr->parentfield = g_strdup((gchar *)attrval);
			}			

			if(g_str_equal(attr->name, "multilang")) {
				if(g_str_equal(attrval, "yes")) {
					prop_attr->is_multilang = TRUE;	
					type_attr->use_lang = TRUE;
				}
			}

			if(g_str_equal(attr->name, "primaryfield")) {
				check_metadata_column(attrval);
				/* Check if column name is reserved one */
				if (strv_contains(rcolumns, attrval)) {
					g_critical("'%s' is reserved column name",
							attrval);
				}
				prop_attr->primaryfield = g_strdup((gchar *)attrval);
				prop_attr->is_primary = TRUE;
				xmlChar *tmpattr = xmlGetProp (node, (const xmlChar *)"name");
				type_attr->primary = g_strdup((gchar *)tmpattr);
				type_attr->query->primary = g_strdup((gchar *)attrval);
				xmlFree(tmpattr);
			}

			if(g_str_equal(attr->name, "reverse")) {
				if(g_str_equal(attrval, "yes"))
					prop_attr->is_reversed = TRUE;	
			}
			
			if(g_str_equal(attr->name, "link")) {

				if(!midgard_core_object_prop_link_is_valid(prop_attr->gtype)) {
					__warn_msg(node, "Invalid type for link property.");
					g_error("Wrong schema attribute");
				}

				gchar **link = g_strsplit((gchar *)attrval, ":", -1);
				prop_attr->link = g_strdup((gchar *)link[0]);
				prop_attr->is_link = TRUE;
				if(link[1])
					prop_attr->link_target = 
						g_strdup((gchar *)link[1]);
				else 
					prop_attr->link_target =
						g_strdup("guid");

				g_strfreev(link);
			}

			xmlFree(attrval);
			attr = attr->next;								
		}
	}
}

/* Parse the whole file and make tree with nodes and properties.
 * Create huge hash of hashes, according to names which are defined
 * in ondef and attrdef chars. The only one hardcoded node's name is 
 * "name". I do not think it should force anyone to hardcode more things.
 * The rest is automagic :)
 */
static void
_get_element_names (xmlNode * curn , MgdSchemaTypeAttr *type_attr, MidgardSchema *schema)
{
	xmlNode *obj = NULL;
	xmlChar *nv = NULL, *dparent;
	MgdSchemaTypeAttr *duplicate;
	MgdSchemaPropertyAttr *prop_attr = NULL;
	gchar *tmpstr;

	for (obj = curn; obj; obj = obj->next){
		
		if (obj->type == XML_ELEMENT_NODE){
			/* Get nodes with "name" name. Check if obj->name was defined */
			nv = xmlGetProp (obj, (const xmlChar *)"name"); 
			
			if(!strv_contains(mgd_complextype, obj->name)){
				g_warning("Wrong node name '%s' in '%s' on line %ld",
						obj->name, parsed_schema, xmlGetLineNo(obj));
			}

			if(g_str_equal(obj->name, "type")) {

				/* Check if type name is reserved one */
				if (strv_contains(rtypes, nv)) {
					g_critical("'%s' is reserved type name",
							nv);

				}
				
				if (g_str_equal(obj->name, "type")) {
					duplicate = midgard_schema_lookup_type(
							schema, (gchar *)nv);
					
					if (duplicate != NULL) {
						g_error("%s:%s already added to schema!",
								obj->name, nv);
					} else {
						type_attr = _mgd_schema_type_attr_new();
						g_hash_table_insert(schema->types,
								g_strdup((gchar *)nv), type_attr);
						_get_type_attributes (obj, type_attr);
					}
				} 
			}

			if (g_str_equal(obj->name, "property")){
				
				dparent = xmlGetProp(obj->parent,
						(const xmlChar *)"name");
				check_property_name(nv, (const gchar *)dparent);
				/* g_debug("\t Property: %s ", nv); */ 
				prop_attr = g_hash_table_lookup(
						type_attr->prophash, (gchar *)nv);
				
				if (prop_attr != NULL) {
					g_warning("Property '%s' already added to %s",
							nv ,dparent);
				} else {
					/* FIXME
					 * one prop_attr seems to be assigned nowhere */
					prop_attr = _mgd_schema_property_attr_new();
					midgard_core_schema_get_property_type(obj, type_attr, prop_attr);
					midgard_core_schema_get_default_value(obj, type_attr, prop_attr);
					midgard_core_schema_get_unique_name(obj, type_attr, prop_attr);
					_get_property_attributes(obj, type_attr, prop_attr);
					if(prop_attr->is_primary 
							&& prop_attr->gtype != MGD_TYPE_UINT){
						tmpstr = (gchar *)xmlGetProp(obj->parent,
								(const xmlChar *)"name");
						g_message(" %s - type for primaryfield not defined"
								" as 'unsigned integer'."
								" Forcing uint type"
								, tmpstr);
						g_free(tmpstr);
					}
					g_hash_table_insert(type_attr->prophash, 
							g_strdup((gchar *)nv), prop_attr);
				}					
				xmlFree(dparent);				
			}

			if(g_str_equal(obj->name, "description")){

				tmpstr = (gchar *)xmlGetProp(obj->parent, (const xmlChar *)"name");
				prop_attr = g_hash_table_lookup(type_attr->prophash, tmpstr);
				
				if(prop_attr == NULL) {
					__warn_msg(curn, "Found empty property attribute");             
					g_error("Found description for empty property");
				}	
			
				xmlParserCtxtPtr parser = xmlNewParserCtxt();
				gchar *value = (gchar *)xmlNodeGetContent(obj);
				
				xmlChar *decoded =
					xmlStringDecodeEntities(parser,
							(const xmlChar *) value,
							XML_SUBSTITUTE_REF,
							0, 0, 0);

				prop_attr->description = g_strdup((gchar *)decoded);
				g_free(decoded);
				xmlFreeParserCtxt(parser);
				g_free(value);
				g_free(tmpstr);
			}

			xmlFree(nv);
			if (obj->children != NULL)
				_get_element_names (obj->children, type_attr, schema);	 
		}
	}
}

/* Start processing type's properties data. 
 * This function is called from __get_tdata_foreach 
 */
void __get_pdata_foreach(gpointer key, gpointer value, gpointer user_data)
{
	MgdSchemaTypeAttr *type_attr  = (MgdSchemaTypeAttr *) user_data;  
	MgdSchemaPropertyAttr *prop_attr = (MgdSchemaPropertyAttr *) value;
	const gchar *table, *tables, *primary, *parentname, *ext_table = NULL;
	gchar *tmp_sql = NULL , *nick = "";
	MgdSchemaTypeQuery *tquery;
	/* type_attr->num_properties is used ( and is mandatory! ) 
	 * when we register new object types and set class properties */
	guint n = ++type_attr->num_properties;;
	const gchar *pname = (gchar *) key;
	const gchar *fname = NULL, *upfield = NULL, *parentfield = NULL; 
	gchar *tmpstr = NULL;
	GParamSpec **params = type_attr->params;
	GString *tmp_gstring_sql, *table_sql;
	
	/* Set default string type if not set */
	if(!prop_attr->type) {
		prop_attr->type = g_strdup("string");
		prop_attr->gtype = MGD_TYPE_STRING;
	}		
	
	table = prop_attr->table;
	tables = type_attr->tables;
	parentname = table;

	/* TODO
	 * Set ACL data, create sql queries, add additional data
	 * All data required is present at this moment.
	 */
	
	/* Search for upfield and parentfield definition.
	 * We will use them ( if found ) for nick ( table.field ) definition 
	 * without need to define field attribute.  */
	
	upfield = prop_attr->upfield;
	parentfield = prop_attr->parentfield;
	primary = prop_attr->primaryfield;

	if(table == NULL)
		table = type_attr->table;

	ext_table = table;
	
	/* "external" tables for type */
	
	/* Create GHashTable for all tables described in type's schema definition */
	if (ext_table != NULL){
		
		/* Check if table key already exists , and create one if not */
		tquery = g_hash_table_lookup(type_attr->tableshash, ext_table);
		if (tquery == NULL){
			tquery = _mgd_schema_type_query_new();
			g_hash_table_insert(type_attr->tableshash, g_strdup(ext_table), tquery);      
		}
		
		table_sql = g_string_new("");
		if(tquery->select)
			g_string_append(table_sql, tquery->select);
		
		if (prop_attr->is_multilang)
			tquery->use_lang = TRUE;
		
		tquery->from = (gchar *)ext_table;
		fname = prop_attr->field;

		prop_attr->table = g_strdup(ext_table); 

		if (fname != NULL) {
			tmpstr = g_strjoin(".", ext_table, fname,NULL);
		} else {
			tmpstr = g_strjoin(".", ext_table, pname,NULL);	
		}
		prop_attr->tablefield = g_strdup(tmpstr); 
		//nick = g_strdup(tmpstr); /* FIXME, LEAK */
		
		tmp_gstring_sql = g_string_new(" ");		
		if(prop_attr->gtype == MGD_TYPE_TIMESTAMP) {			
			g_string_append_printf(tmp_gstring_sql,
					"NULLIF(%s,'0000-00-00 00:00:00') AS %s",
					tmpstr, pname);
		} else {
			g_string_append_printf(tmp_gstring_sql,
					"%s AS %s",
					tmpstr, pname);
		}
		g_free(tmpstr);
		tmpstr = g_string_free(tmp_gstring_sql, FALSE);
		/* Avoid duplicated coma */
		if(tquery->select)
			g_string_append(table_sql, ", ");
		g_string_append(table_sql, tmpstr);
		g_free(tmpstr);
		g_free(tquery->select);
		tquery->select = g_string_free(table_sql, FALSE);
		
		g_hash_table_insert(type_attr->tableshash, (gchar *)ext_table, tquery);
		/* "external" tables end */
	} 
	
	table = type_attr->table;
	
	if (ext_table == NULL) {
		tmpstr = NULL;
		type_attr->query->from = (gchar *)table;
		fname = prop_attr->field;
		if ( (fname == NULL ) && ((upfield == NULL) && (parentfield == NULL)) 
				&& (primary == NULL)){
			tmpstr = g_strjoin(".", table, pname,NULL);
		} else if((fname != NULL) && ((parentfield == NULL) || (upfield == NULL))) {
			tmpstr = g_strjoin(".", table, fname,NULL);
		}
		if(tmpstr){
			table_sql = g_string_new("");
			if(type_attr->query->select)
				g_string_append(table_sql, type_attr->query->select);				
			if(prop_attr->tablefield != NULL)
				g_free((gchar *)prop_attr->tablefield);
			prop_attr->tablefield = g_strdup(tmpstr);
			tmp_gstring_sql = g_string_new(" ");
			if(prop_attr->gtype == MGD_TYPE_TIMESTAMP) {
				g_string_append_printf(tmp_gstring_sql,
						"NULLIF(%s,'0000-00-00 00:00:00') AS %s",
						tmpstr, pname);
			} else {
				g_string_append_printf(tmp_gstring_sql,
						"%s AS %s",
						tmpstr, pname);
			}
			g_free(tmpstr);
			tmpstr = g_string_free(tmp_gstring_sql, FALSE);
			/* Do not prepend coma if type select is empty */
			if(type_attr->query->select)
				g_string_append(table_sql, ",");
			g_string_append(table_sql, tmpstr);	
			g_free(tmpstr);
			g_free(type_attr->query->select);
			type_attr->query->select = g_string_free(table_sql, FALSE);
		}
	}
	/* FIXME , this needs to be refactored for real property2field mapping */
	if(!type_attr->query->primary)
		type_attr->query->primary =  g_strdup("guid");

	if(!type_attr->primary){
		type_attr->primary = g_strdup("guid");
	}	

	if(primary != NULL) {
		nick = g_strjoin(".", table, primary, NULL);
		if(prop_attr->tablefield != NULL)
			g_free((gchar *)prop_attr->tablefield);
		prop_attr->tablefield = g_strjoin(".", table, primary, NULL);
		tmpstr = g_strconcat(nick, " AS ", pname, NULL);
		tmp_sql = g_strjoin(",", tmpstr, type_attr->query->select,NULL);
		if(type_attr->query->select)
			g_free(type_attr->query->select);
		type_attr->query->select = g_strdup(tmp_sql);
		g_free(tmpstr);
		g_free(tmp_sql);                       
		g_free(nick);
	}

	/* Set field which is used as up.
	 * At the same time we define property_up which is keeps such data  */ 
	if(upfield != NULL) {
		nick = g_strjoin(".", table, upfield, NULL);
		if(prop_attr->tablefield != NULL)
			g_free((gchar *)prop_attr->tablefield);
		prop_attr->tablefield = g_strjoin(".", table, upfield, NULL);
		tmpstr = g_strconcat(nick, " AS ", pname, NULL);
		tmp_sql = g_strjoin(",", tmpstr, type_attr->query->select,NULL);
		if(type_attr->query->select)
			g_free(type_attr->query->select);
		type_attr->query->select = g_strdup(tmp_sql);
		g_free(tmpstr);
		g_free(tmp_sql);
		g_free(nick);
	}
	
	/* Set parentfield and property_parent */
	if(parentfield != NULL) {    
		nick = g_strjoin(".", table, parentfield, NULL);
		if(prop_attr->tablefield != NULL)
			g_free((gchar *)prop_attr->tablefield);
		prop_attr->tablefield = g_strjoin(".", table, parentfield, NULL);
		tmpstr = g_strconcat(nick, " AS ", pname, NULL);
		tmp_sql = g_strjoin(",", tmpstr, type_attr->query->select,NULL);
		if(type_attr->query->select)
			g_free(type_attr->query->select);
		type_attr->query->select = g_strdup(tmp_sql);
		g_free(tmpstr);
		g_free(tmp_sql);
		g_free(nick);
	}

	if(prop_attr->field == NULL)
		prop_attr->field = g_strdup(pname);

	/* Force G_TYPE_UINT type for primaryfield */
	if((prop_attr->is_primary) && (prop_attr->gtype != G_TYPE_UINT))
		prop_attr->gtype = G_TYPE_UINT;
			
	nick = NULL;
	/* Create param_specs for object's properties */

	if(prop_attr->description == NULL)
		prop_attr->description = g_strdup("");

	GType ptype = prop_attr->gtype;
		
	if (ptype == MGD_TYPE_STRING) {

		params[n-1] = g_param_spec_string(
				pname, nick, prop_attr->description,
				"",  G_PARAM_READWRITE);

	} else if (ptype == MGD_TYPE_TIMESTAMP) {
		
		params[n-1] = g_param_spec_boxed(
				pname, nick, prop_attr->description,
				MGD_TYPE_TIMESTAMP, G_PARAM_READWRITE);

	} else if (ptype == MGD_TYPE_UINT) {
		
		params[n-1] = g_param_spec_uint(
				pname, nick, prop_attr->description,
				0, G_MAXUINT32, 0, G_PARAM_READWRITE);
	
	} else if (ptype == MGD_TYPE_INT) {
		
		params[n-1] = g_param_spec_int(
				pname, nick, prop_attr->description,
				G_MININT32, G_MAXINT32, 0, G_PARAM_READWRITE);

	} else if (ptype == MGD_TYPE_FLOAT) {
		
		params[n-1] = g_param_spec_float(
				pname, nick, prop_attr->description,
				-G_MAXFLOAT, G_MAXFLOAT, 0, G_PARAM_READWRITE);
	
	} else if (ptype == MGD_TYPE_BOOLEAN) {
		
		params[n-1] = g_param_spec_boolean(
				pname, nick, prop_attr->description,
				FALSE, G_PARAM_READWRITE);
	
	} else {
		
		params[n-1] = g_param_spec_string(
				pname, nick, prop_attr->description,
				"", G_PARAM_READWRITE);				
	} 

	g_free(prop_attr->description);
}

/* Append all tables' select to type's full_select */
void _set_object_full_select(gpointer key, gpointer value, gpointer userdata)
{
	GString *fullselect = (GString *) userdata;
	MgdSchemaTypeQuery *query = (MgdSchemaTypeQuery *) value;
	
	g_string_append_printf(fullselect, 
			",%s", query->select);
}


/* Get types' data. Type's name and its values.
 * We can not start creating GParamSpec while we parse xml file.
 * This is internally used with _register_types function. At this 
 * moment we can count all properties of type and call sizeof
 * with correct values
 */ 

typedef struct {
	GString *string;
	guint elts;
}_str_cont;

static void __build_tables_string(
		gpointer key, gpointer value, gpointer user_data)
{
	_str_cont *_cont  = (_str_cont *) user_data;
	
	if(_cont->elts == 0)
		g_string_append_printf(_cont->string, "%s", (gchar *) key);
	else 
		g_string_append_printf(_cont->string, ", %s", (gchar *) key);
	
	_cont->elts++;
}

void __get_tdata_foreach(gpointer key, gpointer value, gpointer user_data)
{
	GType new_type;
	MgdSchemaTypeAttr *type_attr = (MgdSchemaTypeAttr *) value;
	guint np;		
	
	if(g_type_from_name(key))
		return;
	
	np = g_hash_table_size(type_attr->prophash);

	type_attr->params = g_malloc(sizeof(GParamSpec*)*(np+1)); 

	if (np > 0) {
		g_hash_table_foreach(type_attr->prophash, 
				__get_pdata_foreach, type_attr);

		GString *_tables = g_string_new("");
		_str_cont *cont = g_new(_str_cont, 1);
		cont->string = _tables;
		cont->elts = 0;
		
		g_hash_table_foreach(type_attr->tableshash,
				__build_tables_string, cont);
		g_free(cont);

		type_attr->tables = g_string_free(_tables, FALSE);
		
		/* add NULL as last value, 
		 * we will use ud.param as GParamSpec for object's properties. */ 
		
		type_attr->params[np] = NULL;
		
		/* Define tree management fields and tables */
		//type_attr->parent = typename;
		
		/* Set NULL to child_cname, we will fill this value when all types
		 * are already registered ( _schema_postconfig ).  */ 
		type_attr->childs = NULL;
		
		GString *fullselect = g_string_new("");
		g_string_append(fullselect , type_attr->query->select );
		g_hash_table_foreach(type_attr->tableshash,  
				_set_object_full_select, fullselect);

		type_attr->query->select_full = g_string_free(fullselect, FALSE);

		/* Register type , and initialize class. We can not add properties while 
		 * class is registered and we can not initialize class with properties later.
		 * Or rather "we should not" do this later. */ 
		if (type_attr->params != NULL) {

			new_type = midgard_type_register(key, type_attr, MIDGARD_TYPE_OBJECT);
			if (new_type) {
				/* Initialize objects so we have properties 
				 * private data assigned before "real" application 
				 * start */
				GObject *foo = g_object_new(new_type, NULL);
				/* Set number of properties.
				 * This way we gain performance for instance_init call */
				GParamSpec **pspecs = 
					g_object_class_list_properties(
							G_OBJECT_GET_CLASS(G_OBJECT(foo)), 
							&type_attr->class_nprop);
				g_free(pspecs);
				g_object_unref(foo); 
			}
		}
	
	} else {
		g_warning("Type %s has less than 1 property!", (gchar *)key);
	}
}

/* We CAN NOT define some data during __get_tdata_foreach.
 * parent and childs relation ( for example ) must be done AFTER
 * all types are registered and all types internal structures are
 * already initialized and defined
 */ 
void __postconfig_schema_foreach(gpointer key, gpointer value, gpointer user_data)
{
	MgdSchemaTypeAttr *type_attr = (MgdSchemaTypeAttr *) value, *parenttype;
	MidgardSchema *schema = (MidgardSchema *) user_data;
	gchar *typename = (gchar *) key;
	const gchar *parentname;

	parentname = type_attr->parent;
	
	if (parentname  != NULL ){

		/* validate tree parent class */
		MidgardObjectClass *pklass = MIDGARD_OBJECT_GET_CLASS_BY_NAME(parentname);
		if(pklass == NULL) {
			
			g_critical("Parent %s for %s class is not registered in GType system",
					parentname, typename);
		}

		/* validate parent property */
		MidgardObjectClass *klass = MIDGARD_OBJECT_GET_CLASS_BY_NAME(typename);
		const gchar *parent_property = midgard_object_class_get_property_parent(klass);
		if(parent_property == NULL) {
			
			g_critical("Parent property missed for %s class. %s declared as tree parent class",
					typename, parentname);
		}

		/* Set child type name for parent's one */
		/* PP: I use g_type_from_name(typename)) as gslist->data because
		 * typename ( even typecasted to char ) doesn't work.
		 * If you think this is wrong and have better solution , 
		 * feel free to change this code. */
		
		/* WARNING!! , all types which has parenttype defined will be appended to 
		 * this list. It doesn't matter if they are "registered" in such schema.
		 * Every GObjectClass has only one corresponding C structure , so it is impossible
		 * to initialize new MgdSchemaTypeAttr per every classname 
		 * in every schema and keep them  separated for every GObjectClass , 
		 * as we define only one GType.
		 * It may be resolved by expensive hash lookups for every new object instance.
		 */

		if ((parenttype = midgard_schema_lookup_type(schema, (gchar *)parentname)) != NULL){
			/* g_debug("type %s, parent %s", typename, parentname); */
			if (!g_slist_find(parenttype->childs, 
						(gpointer)g_type_from_name(typename))) {
				parenttype->childs = 
					g_slist_append(parenttype->childs, 
							(gpointer)g_type_from_name(typename));
			}			
		}
	}
}

/**
 * \internal
 * The namespace of the Midgard schema files.
 */
static const char *NAMESPACE = "http://www.midgard-project.org/repligard/1.4";

/* Forward declarations used by read_schema_path() */
static void read_schema_directory(GHashTable *files, const char *directory);
static void read_schema_file(GHashTable *files, const char *file);

/**
 * \internal
 * Reads the schema file or directory from the given path to the given schema
 * hash table. A warning is logged if the path does not exist.
 */
static void read_schema_path(GHashTable *files, const char *path) {
        g_assert(files != NULL);
        g_assert(path != NULL);
        if (g_file_test(path, G_FILE_TEST_IS_DIR)) {
                read_schema_directory(files, path);
        } else if (g_file_test(path, G_FILE_TEST_EXISTS)) {
                read_schema_file(files, path);
        } else {
                g_warning("Schema path %s not found", path);
        }
}

/**
 * \internal
 * Reads the schema file from the given path to the given schema hash table.
 * The file is parsed as an XML file and the resulting xmlDocPtr is inserted
 * to the schema hash table with the file path as the key. Both the XML
 * document and the file path are newly allocated and must be freed when
 * the hash table is destroyed. Logs a warning and returns if the file
 * has already been included in the hash table or if it cannot be parsed.
 * Any includes within the parsed file are processed recursively.
 */
static void read_schema_file(GHashTable *files, const char *file) {
        g_assert(files != NULL);
        g_assert(file != NULL);

        /* Guard against duplicate loading of schema files */
        if (g_hash_table_lookup(files, file) != NULL) {
                g_warning("Skipping already seen schema file %s", file);
                return;
        }

        /* Parse this schema file */
        /* g_debug("Reading schema file %s", file); */
        xmlDocPtr doc = xmlParseFile(file);
        if (doc == NULL) {
                g_warning("Skipping malformed schema file %s", file);
                return;
        }
        xmlNodePtr root = xmlDocGetRootElement(doc);
        if (root == NULL
            || root->ns == NULL
            || !g_str_equal(root->ns->href, NAMESPACE)
            || !g_str_equal(root->name, "Schema")) {
                g_warning("Skipping invalid schema file %s", file);
                xmlFreeDoc(doc);
                return;
        }

        /* Add the schema file to the hash table */
        g_hash_table_insert(files, g_strdup(file), doc);

        /* Read all included schema files */
        xmlNodePtr node = root->children;
        while (node != NULL) {
                xmlChar *attr = xmlGetNoNsProp(node, (xmlChar *) "name");
                if (node->type == XML_ELEMENT_NODE
                    && node->ns != NULL
                    && g_str_equal(node->ns->href, NAMESPACE)
                    && g_str_equal(node->name, "include")
                    && attr != NULL) {
                        GError *error = NULL;
                        gchar *name = g_filename_from_utf8(
                                (const gchar *) attr, -1, NULL, NULL, &error);
                        if (name == NULL) {
                                g_warning("g_filename_from_utf8: %s", error->message);
                                g_error_free(error);
                        } else if  (g_path_is_absolute(name)) {
                                read_schema_path(files, name);
                                g_free(name);
                        } else {
                                gchar *dir = g_path_get_dirname(file);
                                gchar *path = g_build_filename(dir, name, NULL);
                                read_schema_path(files, path);
                                g_free(path);
                                g_free(dir);
                                g_free(name);
                        }
                }
                if (attr != NULL) {
                        xmlFree(attr);
                }
                node = node->next;
        }
	/* xmlFreeDoc(doc); */ /* wtf? */	
}

/**
 * \internal
 * Reads the schema directory from the given path to the given schema
 * hash table. Ignores all hidden files and subdirectories, and recurses
 * into normal subdirectories. All files with the suffix ".xml" are read
 * and inserted into the schema hash table.
 */
static void read_schema_directory(GHashTable *files, const char *directory) {
        g_assert(files != NULL);
        g_assert(directory != NULL);

        GError *error = NULL;
        GDir *dir = g_dir_open(directory, 0, &error);
        if (dir != NULL) {
                const gchar *file = g_dir_read_name(dir);
                while (file != NULL) {
                        gchar *path = g_build_filename(directory, file, NULL);
                        if (g_str_has_prefix(file, ".")) {
                                /* skip hidden files and directories */
                        } else if (g_str_has_prefix(file, "#")) {
                                /* skip backup files and directories */
                        } else if (!g_file_test(path, G_FILE_TEST_IS_DIR)) {
                                /* recurse into subdirectories */
                                read_schema_directory(files, path);
                        } else if (g_str_has_suffix(file, ".xml")) {
                                /* read xml file, guaranteed to exist */
                                read_schema_file(files, path);
                        }
                        g_free(path);
                        file = g_dir_read_name(dir);
                }
                g_dir_close(dir);
        } else {
                g_warning("g_dir_open: %s", error->message);
                g_error_free(error);
        }
}

/**
 * \internal
 * Reads all the schema files identified by the given path and any includes
 * within the schema files. Returns the fiels as a newly allocated hash
 * table with the schema file paths as keys and the parsed XML documents
 * (xmlDocPtr) as values. The returned hash table contains content destroy
 * functions so the caller can just use g_hash_table_destroy() to release
 * all memory associated with the returned hash table.
 */
static GHashTable *read_schema_files(const char *path) {
        g_assert(path != NULL);
        GHashTable *files = g_hash_table_new_full(
                g_str_hash, g_str_equal, g_free, (GDestroyNotify) xmlFreeDoc);
        read_schema_path(files, path);
        return files;
}

static void parse_schema(gpointer key, gpointer value, gpointer user_data) {
        g_assert(key != NULL);
        g_assert(value != NULL);
        g_assert(user_data != NULL);
	parsed_schema = (const gchar *)key;

        MidgardSchema *schema = (MidgardSchema *) user_data;
        xmlDocPtr doc = (xmlDocPtr) value;
        xmlNodePtr root = xmlDocGetRootElement(doc);
        _get_element_names(root, NULL, schema);
}

/* API functions */

/* Initializes basic Midgard classes */
/**
 * midgard_schema_init:
 * @self: #MidgardSchema instance
 * @path: full path to a xml file with common classes
 *
 * Reads xml file which defines very basic and common classes.
 * By default it's `/usr/local/share/midgard/MgdObjects.xml` file.
 */
void midgard_schema_init(MidgardSchema *self, const gchar *path)
{	
	g_assert(self != NULL);

	if (path == NULL)
		path = MIDGARD_GLOBAL_SCHEMA;

	if (g_file_test(path, G_FILE_TEST_IS_DIR)) {

		g_warning("Common MgdObjects.xml path is a directory");
		return;
	}
	
	if (!g_file_test(path, G_FILE_TEST_EXISTS)) {

		g_warning("Common MgdObjects.xml file (%s) doesn't exist.", path);
		return;
	}

	midgard_schema_read_file(self, path);
}

/* Checks if classname is registsred for schema. */
/**
 * midgard_schema_type_exists:
 * @self: #MidgardSchema instance
 * @classname: #GObjectClass derived class name
 *
 * Returns: %TRUE if class is registered as #MidgardObjectClass derived, %FALSE otherwise
 */
gboolean midgard_schema_type_exists(MidgardSchema *self, const gchar *classname)
{
	g_assert(self != NULL);
	g_assert(classname != NULL);
	
	if(g_hash_table_lookup(self->types, classname))
		return TRUE;

	return FALSE;
}

/**
 * midgard_schema_read_file:
 * @self: #MidgardSchema instance
 * @filename: full path to a file
 *
 * Reads file at given path and initialize all #MidgardObjectClass derived classes defined
 * in this file. Also reads all files which are included in this file.
 * Such files are read and parsed when given file is already parsed and all classes
 * defined in given file are already registered in GType system.
 */
void midgard_schema_read_file(
		MidgardSchema *self, const gchar *filename) 
{
	MidgardSchema *schema = self;

	g_assert(schema != NULL);
        g_assert(filename != NULL);

        GHashTable *files = read_schema_files(filename);
        g_hash_table_foreach(files, parse_schema, schema);
        g_hash_table_destroy(files);

        /* register types */
        g_hash_table_foreach(schema->types, __get_tdata_foreach, schema);
    
        /* Do postconfig for every initialized schema */
        g_hash_table_foreach(schema->types, __postconfig_schema_foreach, schema);
}

static void __is_linked_valid(MidgardDBObjectClass *klass, const gchar *prop)
{
	MidgardReflectionProperty *smrp = NULL;
	MidgardReflectionProperty *dmrp = NULL;
	smrp = midgard_reflection_property_new(klass);
	const gchar *typename = G_OBJECT_CLASS_NAME(klass);

	if(!midgard_reflection_property_is_link(smrp, prop)) 
		return;

	MidgardDBObjectClass *pklass = midgard_reflection_property_get_link_class(smrp, prop);
	
	if(!pklass)		
		g_error("Can not get link class for %s.%s", typename, prop);
				
	const gchar *prop_link = midgard_reflection_property_get_link_target(smrp, prop);
	
	if(!prop_link) 
		g_error("Can not get link for %s.%s", typename, prop);

	/* We need to set is_linked for linked property. It can not be done when schema is parsed because
	 * linked class might be not registered in that phase */
	MgdSchemaPropertyAttr *prop_attr =
		g_hash_table_lookup(pklass->dbpriv->storage_data->prophash, prop_link);

	if (prop_attr == NULL) {
		
		g_warning("Couldn't find property attribute for %s.%s",
				G_OBJECT_CLASS_NAME(pklass), prop_link);
	} else {
		
		prop_attr->is_linked = TRUE;
	}

	GType src_type = midgard_reflection_property_get_midgard_type(smrp, prop);
	dmrp = midgard_reflection_property_new(MIDGARD_DBOBJECT_CLASS(pklass));
	GType dst_type = midgard_reflection_property_get_midgard_type(dmrp, prop_link);
	
	if(src_type != dst_type)
		g_error("Mismatched references' types: %s.%s type != %s.%s type",
				typename, prop, G_OBJECT_CLASS_NAME(pklass), prop_link);

	return;
}

static void __midgard_schema_validate()
{
	guint n_types, i;
	guint n_prop = 0;
	guint j = 0;
	const gchar *typename, *parentname;
	GType *all_types = g_type_children(MIDGARD_TYPE_OBJECT, &n_types);
	MidgardObjectClass *klass = NULL;
	MidgardObjectClass *pklass = NULL;
	
	for (i = 0; i < n_types; i++) {

		typename = g_type_name(all_types[i]);
		klass = MIDGARD_OBJECT_GET_CLASS_BY_NAME(typename);

		GParamSpec **pspecs = g_object_class_list_properties(G_OBJECT_CLASS(klass), &n_prop);

		if(n_prop == 0)
			continue;

		for(j = 0; j < n_prop; j++) {

			__is_linked_valid(MIDGARD_DBOBJECT_CLASS(klass), pspecs[j]->name);
		}

		g_free(pspecs);

		parentname = klass->dbpriv->storage_data->parent;

		if(parentname != NULL) {
			
			/* validate tree parent class */
			pklass = MIDGARD_OBJECT_GET_CLASS_BY_NAME(parentname);
		
			if(pklass == NULL) {
				
				g_error("Parent %s for %s class is not registered in GType system",
						parentname, typename);
			}
		
			/* validate parent property */
			const gchar *parent_property = midgard_object_class_get_property_parent(klass);
			
			if(parent_property == NULL) {
				
				g_error("Parent property missed for %s class. %s declared as tree parent class",
						typename, parentname);
			}

			/* validate parent property if exists */
			const gchar *prop = midgard_object_class_get_property_parent(klass);	
			MidgardReflectionProperty *smrp = NULL;
	
			if(prop) {
		
				smrp = midgard_reflection_property_new(MIDGARD_DBOBJECT_CLASS(klass));
				if(!midgard_reflection_property_is_link(smrp, prop)) 
					g_error("Parent property %s of class %s is not defined as link", prop, typename);
			
				__is_linked_valid(MIDGARD_DBOBJECT_CLASS(klass), prop);

				g_object_unref(smrp);	
			}
			
			/* validate up property link */
			prop = midgard_object_class_get_property_up(klass);
			
			if(prop) {
				
				smrp = midgard_reflection_property_new(MIDGARD_DBOBJECT_CLASS(klass));
				if(!midgard_reflection_property_is_link(smrp, prop)) 
					g_error("Up property %s of class %s is not defined as link", prop, typename);
			
				__is_linked_valid(MIDGARD_DBOBJECT_CLASS(klass), prop);

				g_object_unref(smrp);
			}
		}
	}

	g_free(all_types);
}

/**
 * midgard_schema_read_dir:
 * @self: #MidgardSchema instance
 * @dirname: a directory to read xml file from
 *
 * Read all files from given directory.
 * midgard_schema_read_file() is invoked for every valid xml MgdSchema file 
 * found in the given directory.
 */
gboolean midgard_schema_read_dir(
		MidgardSchema *self, const gchar *dirname)
{
	const gchar *fname = NULL;
	gchar *fpfname = NULL ;
	GDir *dir;
	gint visible = 0;

	MidgardSchema *gschema = self;

	dir = g_dir_open(dirname, 0, NULL);
	
	if (dir != NULL) {
		
		while ((fname = g_dir_read_name(dir)) != NULL) {

			visible = 1;
			fpfname = g_strconcat(dirname, "/", fname, NULL);
			
			/* Get files only with xml extension, 
			 * swap files with '~' suffix will be ignored */
			if (!g_file_test (fpfname, G_FILE_TEST_IS_DIR) 
					&& g_str_has_suffix(fname, ".xml")){
				
				/* Hide hidden files */
				if(g_str_has_prefix(fname, "."))
					visible = 0;
				
				/* Hide recovery files if such exist */
				if(g_str_has_prefix(fname, "#"))
					visible = 0;
				
				if ( visible == 0)
					g_warning("File %s ignored!", fpfname);
				
				if(visible == 1){  					
					/* FIXME, use fpath here */
					midgard_schema_read_file(gschema, fpfname);
				}
			}
			g_free(fpfname);
			
			/* Do not free ( or change ) fname. 
			 * Glib itself is responsible for data returned from g_dir_read_name */
		}

		/* validate */
		__midgard_schema_validate();

		g_dir_close(dir);
		return TRUE;
	}
	return FALSE;
}

/* SCHEMA DESTRUCTORS */

/* Free type names, collect all MgdSchemaTypeAttr pointers */
static void _get_schema_trash(gpointer key, gpointer val, gpointer userdata)
{
	MgdSchemaTypeAttr *stype = (MgdSchemaTypeAttr *) val;
	
	/* Collect types data pointers*/
	if(!g_slist_find(schema_trash, stype)) {
		/* g_debug("Adding %s type to trash", (gchar *)key); */ 
		schema_trash = g_slist_append(schema_trash, stype);
	}
}

/* GOBJECT AND CLASS */

/* Create new object instance */
static void 
_schema_instance_init(GTypeInstance *instance, gpointer g_class)
{
	MidgardSchema *self = (MidgardSchema *) instance;
	self->types = g_hash_table_new(g_str_hash, g_str_equal);
}

/* Finalize  */
static void _midgard_schema_finalize(GObject *object)
{
	g_assert(object != NULL); /* just in case */
	MidgardSchema *self = (MidgardSchema *) object;	

	g_hash_table_foreach(self->types, _get_schema_trash, NULL);
	g_hash_table_destroy(self->types);
	
	for( ; schema_trash ; schema_trash = schema_trash->next){
		_mgd_schema_type_attr_free((MgdSchemaTypeAttr *) schema_trash->data);
	}
	g_slist_free(schema_trash);					
}

/* Initialize class */
static void _midgard_schema_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	MidgardSchemaClass *klass = MIDGARD_SCHEMA_CLASS (g_class);
	
	gobject_class->set_property = NULL;
	gobject_class->get_property = NULL;
	gobject_class->finalize = _midgard_schema_finalize;
	
	klass->init = midgard_schema_init;
	klass->read_dir = midgard_schema_read_dir;
	klass->read_file = midgard_schema_read_file;
	klass->type_exists = midgard_schema_type_exists;
}

/* Register MidgardSchema type */
GType
midgard_schema_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardSchemaClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_schema_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardSchema),
			0,              /* n_preallocs */
			_schema_instance_init    /* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT,
				"midgard_schema",
				&info, 0);
	}
	return type;
}
