/* 
 * Copyright (C) 2007 Piotr Pokora <piotrek.pokora@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include <libgda/libgda.h>
/* stddef included due to bug in openssl package.
 * In theory 0.9.8g fix this, but on some systems stddef is not included.
 * Remove this include when stddef is included in openssl dev headers. */
#include <stddef.h>
#include "midgard_md5.h"
#include "midgard_core_query.h"
#include "midgard_core_object.h"
#include "midgard_dbobject.h"
#include "midgard_core_query.h"
#include "midgard_user.h"
#include "midgard_error.h"
#include "midgard_replicator.h"

#if HAVE_CRYPT_H
# include <crypt.h>
#else
# define _XOPEN_SOURCE
# include <unistd.h>
#endif

#ifdef HAVE_SECURITY_PAM_APPL_H
#include <security/pam_appl.h>
#endif
/* Mac OS X puts PAM headers into /usr/include/pam, not /usr/include/security */
#ifdef HAVE_PAM_PAM_APPL_H
#include <pam/pam_appl.h>
#endif

struct _MidgardUserPrivate {
	MgdObject *person;
	guint user_type;
	gchar *login;
	gchar *password;
	gboolean active;
	guint hashtype;
	gboolean fetched;
};

#define MIDGARD_USER_TYPE_NONE 0
#define MIDGARD_USER_TYPE_USER 1
#define MIDGARD_USER_TYPE_ADMIN 2
#define MIDGARD_USER_TYPE_ROOT 3

#define MIDGARD_USER_TABLE "midgard_user"

/* MidgardUser object properties */
enum {
	MIDGARD_USER_GUID = 1,
	MIDGARD_USER_LOGIN, 
	MIDGARD_USER_PASS, 
	MIDGARD_USER_ACTIVE,
	MIDGARD_USER_SG,
	MIDGARD_USER_HASH
};

/* PAM AUTH DATA */
typedef struct {
	const gchar *username;
	const gchar *password;
} midgard_auth_pam_appdata;

static void __initialize_user(MidgardUser *self)
{
	g_assert(self != NULL);
	self->priv = g_new(MidgardUserPrivate, 1);
	self->priv->person = NULL;
	self->priv->user_type = 0;
	self->priv->fetched = FALSE;
	self->priv->login = NULL;
	self->priv->password = NULL;
	self->priv->active = FALSE;
	self->priv->hashtype = 0;	
}

#ifdef HAVE_LIBPAM

static gint _midgard_pam_conv(int num_msg, const struct pam_message **msg, 
		struct pam_response **resp, void *appdata_ptr)
{
	midgard_auth_pam_appdata *appdata = (midgard_auth_pam_appdata*) appdata_ptr;
	int i = 0, j = 0;

	if (num_msg && msg && *msg && resp && appdata_ptr) {
		*resp = malloc(sizeof(struct pam_response)*num_msg);
		if (*resp) {
			/* Process conversation and fill in results */
			for(; i < num_msg; i++) {
				(*resp)[i].resp_retcode = 0;
				(*resp)[i].resp = NULL;
				switch((*msg)[i].msg_style) {
					
					case PAM_PROMPT_ECHO_ON: /* username */
						(*resp)[i].resp = strdup(appdata->username);
						break;
					
					case PAM_PROMPT_ECHO_OFF: /* password */
						(*resp)[i].resp = strdup(appdata->password);
						break;
				}

				/* If there was an error during strdup(), 
				 * clean up already allocated  structures and return PAM_CONV_ERR */
				if (!(*resp)[i].resp) {
					for(j = i; j >= 0 ; j--) {
						if ((*resp)[j].resp)
							free((*resp)[j].resp);
					}
					free(*resp);
					*resp = NULL;
					g_debug("Return PAM_CONV_ERROR due to strdup() failure");
					return PAM_CONV_ERR;
				}
			}
			g_debug("Return PAM_SUCCESS");
			return PAM_SUCCESS;
		}
	}
	g_debug("Return PAM_CONV_ERROR");
	return PAM_CONV_ERR;
}

#endif /* HAVE_LIBPAM */

/* PAM AUTH DATA END */

static gchar *__string2md5hash(const gchar *str) 
{
	g_assert(str != NULL);

	guchar mdbuf[16];

	MIDGARD_MD5_CTX md5;
	MIDGARD_MD5_Init(&md5);
	MIDGARD_MD5_Update(&md5, (unsigned char *) str, strlen(str));
	MIDGARD_MD5_Final(mdbuf, &md5);
	int i;
	
	GString *gstr = g_string_new("");
	
	for (i = 0; i < 16; i++) {
		g_string_append_printf(gstr, "%02x", mdbuf[i]);	
	}

	return g_string_free(gstr, FALSE);
}

/* Creates new MidgardUser object for the given Midgard Object. */
MidgardUser *midgard_user_new(MgdObject *person)
{
	MidgardConnection *mgd = NULL;

	if(person && !MIDGARD_IS_OBJECT(person)) {
		g_warning("Expected MidgardObject type");
		return NULL;
	}

	if(person && !person->dbpriv->guid) {
		g_warning("Midgard user initialized for person with NULL guid!");
		return NULL;
	}

	MidgardUser *self =
		g_object_new(MIDGARD_TYPE_USER, NULL);

	if(person) {

		mgd = person->dbpriv->mgd;

		/* TODO, Remove when legacy auth support is removed */
		if(!midgard_core_table_exists(mgd, "midgard_user")) {

			self->dbpriv->mgd = mgd;
			self->dbpriv->guid = g_strdup(person->dbpriv->guid);
			self->dbpriv->sg = person->dbpriv->sg;
			self->priv->person = MIDGARD_OBJECT(person);
			mgd->priv->user = G_OBJECT(self);

			return self;
		}

		MidgardQueryBuilder *builder = 
			midgard_query_builder_new(mgd, "midgard_user");
		GValue val = {0, };
		/* GUID */
		g_value_init(&val, G_TYPE_STRING);
		g_value_set_string(&val, person->dbpriv->guid);
		midgard_query_builder_add_constraint(builder, 
				"guid", "=", &val);
		g_value_unset(&val);
		/* SITEGROUP */
		g_value_init(&val, G_TYPE_UINT);
		g_value_set_uint(&val, person->dbpriv->sg);
		midgard_query_builder_add_constraint(builder, 
				"sitegroup", "=", &val);
		g_value_unset(&val);
	
		guint n_objects;	
		GObject **objects = 
			midgard_query_builder_execute(builder, &n_objects);
		g_object_unref(builder);

		g_object_unref(self);

		if(!objects)
			return NULL;

		return MIDGARD_USER(objects[0]);
	}

	return self;
}

MidgardUser *__legacy_auth(MidgardConnection *mgd, 
		const gchar *name, const gchar *password, const gchar *sitegroup, gboolean trusted_auth)
{
	MgdObject *person = NULL; 
	MidgardQueryBuilder *builder = 
		midgard_query_builder_new(mgd, "midgard_person");
	GdaDataModel *admin_model = NULL;
	guint user_type = MIDGARD_USER_TYPE_NONE;

	if(!builder)
		return NULL;

	/* We set sitegroup context, if sitegroup name is passed.
	 * Then we reset sitegroup context if auth fails */
	const gchar *current_sg = midgard_connection_get_sitegroup(mgd);
	if(sitegroup != NULL) {
		if(!midgard_connection_set_sitegroup(mgd, sitegroup))
			return NULL;	
	}

	guint sg_id = midgard_connection_get_sitegroup_id(mgd);

	GString *sql = g_string_new("SELECT id, password ");
	g_string_append_printf(sql, "FROM person "
			"WHERE username = '%s' AND sitegroup = %d "
			"AND metadata_deleted=0 ",
			name, sg_id);

	GdaDataModel *model = midgard_core_query_get_model(mgd, sql->str);
	g_string_free(sql, TRUE);
	
	if(!model) {
		
		midgard_connection_set_sitegroup(mgd, current_sg);
		return NULL;
	}

	const GValue *value =
		midgard_data_model_get_value_at (model, 0, 0);
	guint person_id = 0;
	MIDGARD_GET_UINT_FROM_VALUE(person_id, value);
	value = midgard_data_model_get_value_at (model, 1, 0);

	if (trusted_auth)
		goto PASSWORD_CHECK_DONE;

	GValue passval = {0, };
	g_value_init(&passval, G_TYPE_STRING);

	gchar *passwd;

	if(GDA_VALUE_HOLDS_BINARY(value)) {
		
		passwd = gda_binary_to_string(gda_value_get_binary(value), 0);

	} else {
		
		if(!G_VALUE_HOLDS_STRING(value))
			g_warning("Expected string password. Got '%s'.", G_VALUE_TYPE_NAME(value));

		passwd = g_value_dup_string(value);
	}

	if(passwd == NULL || g_str_equal(passwd, "")) {
		
		g_warning("Can not handle null or empty passwords");
		if(passwd != NULL)
			g_free(passwd);
		goto FREE_AND_RETURN_NULL;
	}

	const gchar *cipher;	
	const gchar *npass = NULL;

	if (passwd[0] == '*' && passwd[1] == '*') {
		
		npass = passwd+2;
		cipher = password;
		
	} else {
		
		npass = passwd;
		cipher = (const char *)crypt(password, passwd);
	}
	
	if(!g_str_equal(cipher, npass)) {
		
		midgard_connection_set_sitegroup(mgd, current_sg);
		g_object_unref(model);
		g_free(passwd);  
		return NULL;
	}

	g_free(passwd);

PASSWORD_CHECK_DONE:

	if((sg_id == 0 && sitegroup == NULL)
			|| ((*sitegroup == '\0') && sg_id == 0)) {
		
		user_type = MIDGARD_USER_TYPE_ROOT;
	
	} else {
		
		/* Password match, so we check group membership */
		sql = g_string_new("SELECT member.gid from member, grp, sitegroup ");
		g_string_append_printf(sql, "WHERE sitegroup.id = %d AND "
				"member.uid = %d AND sitegroup.admingroup = grp.id "
				"AND grp.sitegroup = sitegroup.id ", 
				sg_id, person_id);
		
		admin_model = midgard_core_query_get_model(mgd, sql->str);
		g_string_free(sql, TRUE);

		if(admin_model) {
			
			value =
				midgard_data_model_get_value_at (model, 0, 0);
			guint gid = 0;
			MIDGARD_GET_UINT_FROM_VALUE(gid, value);
			if(gid > 0)
				user_type = MIDGARD_USER_TYPE_ADMIN;
		}
	}

	GValue pval = {0, };
	g_value_init(&pval, G_TYPE_UINT);
	g_value_set_uint(&pval, person_id);
	/* it shouild resolve duplicate problems */
	person = midgard_object_new(mgd, "midgard_person", &pval);
	g_value_unset(&pval);

	if(!person) {
		midgard_connection_set_sitegroup(mgd, current_sg);
		goto FREE_AND_RETURN_NULL;
	}
	
	if(model)
		g_object_unref(model);
	if(admin_model)
		g_object_unref(admin_model);
	
	MidgardUser *user = midgard_user_new(NULL);
	user->priv->person = MIDGARD_OBJECT(person);
	user->priv->user_type = user_type;
	user->dbpriv->guid = g_strdup(person->dbpriv->guid);	
	user->dbpriv->sg = MIDGARD_OBJECT(person)->dbpriv->sg;
	user->dbpriv->mgd = mgd;
	
	mgd->priv->user = G_OBJECT(user);
	mgd->priv->person = G_OBJECT(person);
	g_signal_emit_by_name(mgd, "auth_changed");

	return user;

FREE_AND_RETURN_NULL:

	if(model)
		g_object_unref(model);
	if(admin_model)
		g_object_unref(admin_model);
	return NULL;
}

/* Authenticate Midgard User. */
MidgardUser *midgard_user_auth(MidgardConnection *mgd, 
		const gchar *name, const gchar *password, 
		const gchar *sitegroup, gboolean trusted)
{
	g_return_val_if_fail(mgd != NULL, NULL);

	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);

	/* Empty login name. Not allowed */
	if(!name) {
		g_warning("Can not authenticate user with empty login");
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INVALID_NAME);
		return NULL;
	}

	/* Password and trusted is not set */
	if(!trusted && !password) {
		g_warning("Can not authenticate user with empty password");
		return NULL;
	}

	/* Trusted auth not allowed for SG0 users */
	if(!sitegroup && trusted) {
		g_warning("Can not authenticate SG0 user using \"trusted\" type");
		return NULL;
	}

	/* TODO, remove this once we drop legacy auth support */
	if(!midgard_core_table_exists(mgd, "midgard_user"))
		return __legacy_auth(mgd, name, password, sitegroup, trusted);

	GString *sql = g_string_new("SELECT ");

	/* We need only hashed password and hash type if sitegruop is NULL */
	if(!sitegroup) {
	
		g_string_append(sql, "guid AS user_guid, "
				"hashed, hashtype, user_type ");
		g_string_append(sql, "FROM midgard_user WHERE ");
		g_string_append_printf(sql,
				"login='%s' AND active=1 " 
				"AND midgard_user.sitegroup=0",
				name);
	} else {		

		g_string_append(sql, "midgard_user.guid AS user_guid, "
				"hashed, hashtype, user_type, "
				"sitegroup.guid AS sg_guid ");
		g_string_append(sql, "FROM midgard_user, sitegroup WHERE ");
		g_string_append_printf(sql, 
				"login='%s' AND active=1 "
				"AND midgard_user.sitegroup=sitegroup.id "
				"AND sitegroup.name='%s'",
				name, sitegroup);
	}

	GdaDataModel *model = 
		midgard_core_query_get_model(mgd, sql->str);

	g_string_free(sql, TRUE);

	/* There is no midgard_user record, so do legacy fallback */
	if(!model)
		return __legacy_auth(mgd, name, password, sitegroup, trusted);

	gint rows = gda_data_model_get_n_rows(model);
	GValue idval = {0, };
	const gchar *user_guid = NULL;
	const gchar *hashed = NULL;
	guint hash_type = 0;
	guint user_type = 0;
	const GValue *gvalue;
	gchar *passhash = NULL;

	if(rows == 0) {
		g_object_unref(model);
		g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "No %s user found", name);
		return NULL;
	}

	/* This is fatal error */
	if(rows > 1) {
		g_object_unref(model);
		g_critical("More than one record found for %s user", name);
		return NULL;
	}

	gvalue = midgard_data_model_get_value_at(model, 0, 0);
	if(G_VALUE_HOLDS_STRING(gvalue)) {
		user_guid = g_value_get_string(gvalue);
	}

	gvalue = midgard_data_model_get_value_at(model, 1, 0);
	if(G_VALUE_HOLDS_STRING(gvalue)) {
		hashed = g_value_get_string(gvalue);
	}

	gvalue = midgard_data_model_get_value_at(model, 2, 0);
	if(G_VALUE_HOLDS_INT(gvalue)) {
		hash_type = g_value_get_int(gvalue);
	}

	gvalue = midgard_data_model_get_value_at(model, 3, 0);
	if(G_VALUE_HOLDS_INT(gvalue)) {
		user_type = g_value_get_int(gvalue);
	}

	//g_object_unref(model);

	/* We get a model, first we check SG0 case. No trusted or PAM */
	if(!sitegroup) {
		
		/* This is fatal error */
		if(user_type != MIDGARD_USER_TYPE_ROOT) {
			g_object_unref(model);			
			g_critical("SG0 user is not root type user! There is either bug in midgard_user::password or someone edited database manually");
			return NULL;
		}

		switch(hash_type) {
			
			case MIDGARD_USER_HASH_MD5:
				passhash = __string2md5hash(password);
				break;

			case MIDGARD_USER_HASH_PLAIN:
				passhash = g_strdup(password);
				break;

			case MIDGARD_USER_HASH_SHA1:
				g_warning("SHA1 not yet supported");	
				return NULL;
				break;

			case MIDGARD_USER_HASH_PAM:
				/* TODO */
				return NULL;
				break;

			default:
				g_warning("Unsupported authentication type");
				return NULL;
		}

		if(g_str_equal(hashed, passhash)) {
			g_free(passhash);
			goto do_auth_with_person;
		}

		g_free(passhash);
		/* SET AUTH FAILURE error */
		return NULL;	
	} 

	/* SGn user auth. First we get user , next we check membership */
	
	/* This is fatal error */
	if(user_type == MIDGARD_USER_TYPE_ROOT) {
		g_error("%s sitegroup user is root type user! There is either bug in midgard_user::password or someone edited database manually", sitegroup);
	}

	if(mgd->priv->config->authtype == MIDGARD_AUTHTYPE_TRUST)
		goto do_auth_with_person;

	if(mgd->priv->config->authtype == MIDGARD_AUTHTYPE_PAM)
		goto do_auth_pam;

	switch(hash_type) {

		case MIDGARD_USER_HASH_MD5:
			passhash = __string2md5hash(password);
			break;
		
		case MIDGARD_USER_HASH_PLAIN:
			passhash = g_strdup(password);
			break;
		
		case MIDGARD_USER_HASH_SHA1:
			g_warning("SHA1 not yet supported");	
			return NULL;
			break;
	
		case MIDGARD_USER_HASH_PAM:
			goto do_auth_pam;
			break;

		default:
			g_warning("Unsupported authentication type");	
			return NULL;
	}

	if(g_str_equal(hashed, passhash)) {
		g_free(passhash);

		/* TODO, membership check here */


		goto do_auth_with_person;
	}

	g_free(passhash);
	
	/* SET AUTH FAILURE error */
	return NULL;	

	do_auth_pam:
	g_debug("Trying PAM auth");
#ifdef HAVE_LIBPAM
	midgard_auth_pam_appdata *appdata = 
		g_new(midgard_auth_pam_appdata, 1);
		appdata->username = name;
		appdata->password = password;
	
	struct pam_conv pconv = {
		_midgard_pam_conv,
		&appdata
	};

	pam_handle_t *phandle = NULL;
	int result = pam_start(mgd->priv->config->pamfile, name, &pconv, &phandle);
	g_debug("PAM start result: %d", result);
	
	if (result == PAM_SUCCESS) 
		result = pam_authenticate(phandle, 0);
	g_debug("PAM authenticate result: %d", result);
	
	result = pam_end(phandle, result);
	g_debug("PAM end result: %d", result);
	
	if(result == PAM_SUCCESS) {
		
		goto do_auth_with_person;
	
	} else {

		return NULL;
	}
#else /* HAVE_LIBPAM */
	g_warning("PAM not compiled in");
	return NULL;
#endif /* HAVE_LIBPAM */
	
	/* Initialize midgard_user object with person */
	do_auth_with_person: 
	g_value_init(&idval, G_TYPE_STRING);
	g_value_set_string(&idval, user_guid);
	MgdObject *pobject =
		midgard_object_new(mgd,	"midgard_person", &idval);
	g_value_unset(&idval);

	if(!pobject) {
		g_error("No midgard_person object for user %s", name);
		g_object_unref(model);
		return NULL;
	}

	mgd->priv->person = G_OBJECT(pobject);

	MidgardUser *user = midgard_user_new(NULL);

	if(!user) {
		/* FIXME, we should deallocate resources here */
		g_critical("Couldn't create new user instance");
		return NULL;
	}
	user->priv->person = MIDGARD_OBJECT(pobject);
	user->priv->user_type = user_type;
	user->dbpriv->guid = g_strdup(user_guid);
	user->dbpriv->sg = MIDGARD_OBJECT(pobject)->dbpriv->sg;
	
	mgd->priv->user = G_OBJECT(user);	

	g_signal_emit_by_name(mgd, "auth_changed");

	return user;	
}

/* Check if user is a user */
gboolean midgard_user_is_user(MidgardUser *self)
{
	if(!self || !self->priv)
		return FALSE;

	if(self->priv->user_type == MIDGARD_USER_TYPE_USER)
		return TRUE;

	return FALSE;
}

/* Check if user is an admin */
gboolean midgard_user_is_admin(MidgardUser *self)
{
	if(!self || !self->priv)
		return FALSE;

	if(self->priv->user_type == MIDGARD_USER_TYPE_ADMIN)
		return TRUE;
	
	return FALSE;
}

/* Check if user is root */
gboolean midgard_user_is_root(MidgardUser *self)
{
	if(!self || !self->priv)
		return FALSE;

	if(self->priv->user_type == MIDGARD_USER_TYPE_ROOT)
		return TRUE;

	return FALSE;
}

/* Set active flag */
gboolean midgard_user_set_active(MidgardUser *self, gboolean flag)
{
	g_assert(self != NULL);

	if(!self->dbpriv->mgd) {
		g_warning("Can not set activity for user without person assigned");
		return FALSE;
	}

	MidgardConnection *mgd = self->dbpriv->mgd;
	MidgardUser *user = MIDGARD_USER(mgd->priv->user);

	if(!user) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_ACCESS_DENIED);
		return FALSE;
	}
	
	if(midgard_user_is_user(user)) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_ACCESS_DENIED);
		return FALSE;
	}

	guint active = 0;
	if(flag)
		active = 1;

	GString *sql = g_string_new("UPDATE midgard_user SET ");
	g_string_append_printf(sql, 
			"active = %d WHERE guid = '%s' "
			"AND sitegroup = %d", 
			active,
			self->dbpriv->guid, 
			self->dbpriv->sg);
	
	gint rv =  midgard_core_query_execute(mgd, sql->str, FALSE);
	g_string_free(sql, TRUE);

	if(rv == -1)
		return FALSE;

	self->priv->active = flag;

	if(rv == -2) {
		g_warning("Provider didn't return number of updated rows. Sitegroup record might be not updated");
		return TRUE;
	}
	
	return TRUE;
}

static gboolean __set_legacy_password(MidgardUser *self, const gchar *login, 
		const gchar *password, guint hashtype) 
{
	if(self->dbpriv->guid == NULL)
		return FALSE;

	GValue gval = {0, };
	g_value_init(&gval, G_TYPE_STRING);
	g_value_set_string(&gval, self->dbpriv->guid);
	MgdObject *person = 
		midgard_object_new(self->dbpriv->mgd, "midgard_person", &gval);
	g_value_unset(&gval);

	if(!person)
		return FALSE;
	
	GString *sql = g_string_new("UPDATE person SET ");
	g_string_append_printf(sql, "username = '%s', password = ", login);
	
	if(hashtype == MIDGARD_USER_HASH_LEGACY) {
		
		g_string_append_printf(sql, "Encrypt('%s') ", password);

	} else {

		g_string_append_printf(sql, "'**%s' ", password);
	}

	g_string_append_printf(sql, "WHERE guid = '%s' AND sitegroup = %d ",
			self->dbpriv->guid, self->dbpriv->sg);

	gint up = midgard_core_query_execute(self->dbpriv->mgd, sql->str, TRUE);
	g_string_free(sql, TRUE);

	if(up > 0) {

		midgard_replicator_export(NULL, MIDGARD_DBOBJECT(person));
		return TRUE;
	} 

	return FALSE;	
}

/* Sets user's password. */
gboolean midgard_user_password(MidgardUser *self, const gchar *login,
				const gchar *password, guint hashtype)
{
	g_assert(self != NULL);

	MidgardConnection *mgd = self->dbpriv->mgd;
	MIDGARD_ERRNO_SET(mgd, MGD_ERR_OK);

	if(self->priv->person == NULL) {
		g_warning("Can not create user account. Person object is not assigned");
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		return FALSE;
	}
	
	if(self->dbpriv->guid == NULL) {
		g_warning("Can not change user account. Person object assigned with NULL guid");
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_INTERNAL);
		return FALSE;
	}

	/* MidgardUser *user = MIDGARD_USER(mgd->priv->user); */
	MidgardUser *user = self;

	gint anon = 3;
	
	if(!midgard_user_is_root(user))
		anon--;

	if(!midgard_user_is_admin(user))
		anon--;

	if(!midgard_user_is_user(user)) {	
		anon--;
	} else {
		if(!g_str_equal(self->dbpriv->guid, 
					user->dbpriv->guid))
			anon--;
	}

	if(anon < 1) {
		MIDGARD_ERRNO_SET(mgd, MGD_ERR_ACCESS_DENIED);
		return FALSE;
	}

	/* Login can not be empty */
	if(login == NULL || g_str_equal(login, "")) {
		
		 midgard_set_error(mgd,
				 MGD_GENERIC_ERROR,
				 MGD_ERR_INVALID_NAME,
				 " Can not set empty login");
		 return FALSE;
	}

	/* Password can not be empty, unless we use PAM */
	if((password == NULL || g_str_equal(password, ""))
			&& (hashtype != MIDGARD_USER_HASH_PAM)) {
		
		midgard_set_error(mgd,
				MGD_GENERIC_ERROR,
				MGD_ERR_INVALID_NAME,
				" Can not set empty password");
		return FALSE;
	}

	if(hashtype == MIDGARD_USER_HASH_LEGACY
			|| hashtype == MIDGARD_USER_HASH_LEGACY_PLAIN) {
		
		return __set_legacy_password(self, login, password, hashtype);
	}

	if(password == NULL && hashtype == MIDGARD_USER_HASH_PAM)
		password = "";

	gchar *new_password;

	switch(hashtype) {
		
		case MIDGARD_USER_HASH_MD5:
			new_password = __string2md5hash(password);
			break;
			
		case MIDGARD_USER_HASH_PLAIN:
			new_password = g_strdup(password);
			break;

		case MIDGARD_USER_HASH_SHA1:
			g_warning("SHA1 not yet supported");
			return FALSE;
			break;

		case MIDGARD_USER_HASH_PAM:
			new_password = g_strdup(password);
			break;

		default:
			g_warning("Unsupported authentication type");
			return FALSE;
	}

	MidgardQueryBuilder *builder;	
	GValue val = {0, };
	gint rv;
	GString *sql;

	/* Let's check if person's record already exists */
	builder = midgard_query_builder_new(mgd, "midgard_user");
	g_value_init(&val, G_TYPE_STRING);
	g_value_set_string(&val, login);
	midgard_query_builder_add_constraint(builder, 
			"login", "=", &val);
	g_value_unset(&val);
	g_value_init(&val, G_TYPE_UINT);
	g_value_set_uint(&val, self->dbpriv->sg);
	midgard_query_builder_add_constraint(builder,
			"sitegroup", "=", &val);
	g_value_unset(&val);

	guint count = midgard_query_builder_count(builder);

	if(count == 0)
		goto create_midgard_user;

	/* Let's check if duplicate exists. */
	builder = midgard_query_builder_new(mgd, "midgard_user");
	g_value_init(&val, G_TYPE_STRING);
	g_value_set_string(&val, login);
	midgard_query_builder_add_constraint(builder, 
			"login", "=", &val);
	g_value_unset(&val);
	g_value_init(&val, G_TYPE_UINT);
	g_value_set_uint(&val, self->dbpriv->sg);
	midgard_query_builder_add_constraint(builder, 
			"sitegroup", "=", &val);
	g_value_unset(&val);

	guint n_objects;
	GObject **objects = 
		midgard_query_builder_execute(builder, &n_objects);
	g_object_unref(builder);

	if(objects && (!g_str_equal(MIDGARD_DBOBJECT(objects[0])->dbpriv->guid, 
				   self->dbpriv->guid))) {

		MidgardUser *euser = MIDGARD_USER(objects[0]);
		if(!g_str_equal(euser->dbpriv->guid, 
					self->dbpriv->guid)) {
			g_object_unref(objects[0]);
			g_free(objects);
			MIDGARD_ERRNO_SET(mgd, MGD_ERR_OBJECT_NAME_EXISTS);
			return FALSE;
		}

		g_free(objects);
	
	} else {

		sql = g_string_new("UPDATE midgard_user SET ");
		g_string_append_printf(sql, 
				"login = '%s', hashed = '%s', "
				"hashtype = %d WHERE "
				"guid = '%s' AND sitegroup = %d",
				login, new_password, hashtype, 
				self->dbpriv->guid, self->dbpriv->sg);

		rv =  midgard_core_query_execute(mgd, sql->str, FALSE);
		g_string_free(sql, TRUE);
		
		if(rv == -1)
			return FALSE;
		
		if(rv == -2) {
			g_warning("Provider didn't return number of updated rows. midgard_user record might be not updated");
			return TRUE;
		}
		
		return TRUE;
	}
	
	create_midgard_user:
	sql = g_string_new("INSERT INTO midgard_user ");
	g_string_append_printf(sql, 
			"(guid, login, hashed, hashtype, sitegroup, "
			"active, user_type) VALUES ('%s', '%s', '%s', "
			"%d, %d, %d, %d )", 
			self->dbpriv->guid, login, new_password, 
			hashtype, self->dbpriv->sg, 0, MIDGARD_USER_TYPE_USER);
	
	rv = midgard_core_query_execute(mgd, sql->str, TRUE);
	g_string_free(sql, TRUE);
	
	if(rv == -1)
		return FALSE;
	
	if(rv == -2) {
		g_warning("Provider didn't return number of updated rows. midgard_user record might be not updated");
		return TRUE;
	}
	
	return TRUE;
}

static gboolean __set_from_sql(MidgardDBObject *object, 
		GdaDataModel *model, gint row)
{
	MidgardUser *self = MIDGARD_USER(object);
	MidgardConnection *mgd = self->dbpriv->mgd;

	__initialize_user(self);
	self->dbpriv->mgd = mgd;

	/* 'at_col_name' acceptable since we do not use this frequently */
	const GValue *value;
	
#ifdef HAVE_LIBGDA_4
	value = midgard_data_model_get_value_at(model, gda_data_model_get_column_index (model, "login"), row);
#else
	value = gda_data_model_get_value_at_col_name(model, "login", row);
#endif
	self->priv->login = g_value_dup_string(value);

	MidgardUser *user = MIDGARD_USER(mgd->priv->user);

	if(midgard_user_is_root(user) 
			|| midgard_user_is_admin(user)) {

		/* ACTIVE */
#ifdef HAVE_LIBGDA_4
		value = midgard_data_model_get_value_at(model, gda_data_model_get_column_index(model, "active"), row);
#else
		value = gda_data_model_get_value_at_col_name(model, "active", row);
#endif
		if(G_VALUE_TYPE(value) != G_TYPE_BOOLEAN) {
			GValue nval = {0, };
			g_value_init(&nval, G_TYPE_BOOLEAN);
			if(g_value_transform(value, &nval)) {
				self->priv->active = g_value_get_boolean(&nval);
			} else {
				g_warning("Couldn't transform active property value. Report a bug!");
			}
			g_value_unset(&nval);
		} else {
			self->priv->active = g_value_get_boolean(value);
		}

		/* PASSWORD */
#ifdef HAVE_LIBGDA_4
		value = midgard_data_model_get_value_at(model, gda_data_model_get_column_index(model, "password"), row);
#else
		value = gda_data_model_get_value_at_col_name(model, "password", row);
#endif
		self->priv->password = g_value_dup_string(value);
	}
}

/* Get person */
MgdObject *midgard_user_get_person(MidgardUser *self)
{
	g_assert(self != NULL);

	return self->priv->person;
}


/* GOBJECT ROUTINES */

static void _midgard_user_finalize(GObject *object)
{
	g_assert(object != NULL);
	
	MidgardUser *self = (MidgardUser *) object;

	g_free(self->priv);
	self->priv = NULL;
}

static void
_midgard_user_get_property (GObject *object, guint property_id,
		GValue *value, GParamSpec *pspec)
{
	MidgardUser *self = (MidgardUser *) object;
	
	switch (property_id) {
		
		case MIDGARD_USER_GUID:
			g_value_set_string(value, self->dbpriv->guid);
			break;

		case MIDGARD_USER_LOGIN:
			g_value_set_string(value, self->priv->login);
			break;

		case MIDGARD_USER_PASS:
			g_value_set_string(value, self->priv->password);
			break;

		case MIDGARD_USER_ACTIVE:
			g_value_set_boolean(value, self->priv->active);	
			break;

		case MIDGARD_USER_SG:
			g_value_set_uint(value, self->dbpriv->sg);
			break;

		case MIDGARD_USER_HASH:
			g_value_set_uint(value, self->priv->hashtype);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(self, property_id, pspec);
			break;
	}
}

static GObjectClass *__parent_class= NULL;

static GObject *
__midgard_user_constructor (GType type,
		guint n_construct_properties,
		GObjectConstructParam *construct_properties)
{
	GObject *object = (GObject *)
		G_OBJECT_CLASS (__parent_class)->constructor (type,
				n_construct_properties,
				construct_properties);
	return G_OBJECT(object);
}

static void
__midgard_user_dispose (GObject *object)
{
	__parent_class->dispose (object);
}

static void _midgard_user_class_init(
		gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	MidgardUserClass *klass = MIDGARD_USER_CLASS (g_class);
	 __parent_class = g_type_class_peek_parent (g_class);
	 
	gobject_class->constructor = __midgard_user_constructor;
	gobject_class->dispose = __midgard_user_dispose;
	gobject_class->finalize = _midgard_user_finalize;
	gobject_class->get_property = _midgard_user_get_property;
	
	MgdSchemaPropertyAttr *prop_attr;
	MgdSchemaTypeAttr *type_attr = _mgd_schema_type_attr_new();

	GParamSpec *pspec;

	 /* GUID */
	pspec = g_param_spec_string ("guid",
			"midgard_person object's guid",
			"",
			"",
			G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_GUID,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_GUID;
	prop_attr->field = g_strdup("guid");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "guid", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"guid"), prop_attr);

	/* LOGIN */
	pspec = g_param_spec_string ("login",
			"midgard_user's login",
			"",
			"",
			G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_LOGIN,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_STRING;
	prop_attr->field = g_strdup("login");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "login", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"login"), prop_attr);

	 /* PASSWORD */
	pspec = g_param_spec_string ("password",
			"midgard_user's password",
			"",
			"",
			G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_PASS,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_STRING;
	prop_attr->field = g_strdup("hashed");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "hashed", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"password"), prop_attr);
	
	/* ACTIVE */
	pspec = g_param_spec_boolean ("active",
			"midgard_user's active info",
			"",
			FALSE, G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_ACTIVE,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_BOOLEAN;
	prop_attr->field = g_strdup("active");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "active", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"active"), prop_attr);

	/* SITEGROUP */
	pspec = g_param_spec_uint ("sitegroup",
			"midgard_user's sitegroup",
			"",
			0, G_MAXUINT32, 0, G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_SG,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_UINT;
	prop_attr->field = g_strdup("sitegroup");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "sitegroup", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"sitegroup"), prop_attr);

	/* HASHTYPE */
	pspec = g_param_spec_uint ("hashtype",
			"midgard_user's hash type",
			"",
			0, G_MAXUINT32, 0, G_PARAM_READABLE);
	g_object_class_install_property (gobject_class,
			MIDGARD_USER_HASH,
			pspec);
	prop_attr = _mgd_schema_property_attr_new();
	prop_attr->gtype = MGD_TYPE_UINT;
	prop_attr->field = g_strdup("hashtype");
	prop_attr->table = g_strdup(MIDGARD_USER_TABLE);
	prop_attr->tablefield = g_strjoin(".", MIDGARD_USER_TABLE, "hashtype", NULL);
	g_hash_table_insert(type_attr->prophash,
			g_strdup((gchar *)"hashtype"), prop_attr);

	/* This must be replaced with GDA classes */
	MgdSchemaTypeQuery *_query = _mgd_schema_type_query_new();   
	_query->select_full = "login, hashed AS password, hashtype, active";
	type_attr->query = _query;

	klass->dbpriv = g_new(MidgardDBObjectPrivate, 1);
	klass->dbpriv->storage_data = type_attr;
	klass->dbpriv->storage_data->table = g_strdup(MIDGARD_USER_TABLE);
	klass->dbpriv->storage_data->tables = g_strdup(MIDGARD_USER_TABLE);
	klass->dbpriv->has_metadata = FALSE;

	klass->dbpriv->set_from_sql = __set_from_sql;
}

static void _midgard_user_instance_init(
		GTypeInstance *instance, gpointer g_class)
{
	MidgardUser *self = MIDGARD_USER(instance);
	
	__initialize_user(self);
}

/* Register midgard_user type */
GType midgard_user_get_type(void) 
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (MidgardUserClass),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) _midgard_user_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof (MidgardUser),
			0,              /* n_preallocs */
			(GInstanceInitFunc) _midgard_user_instance_init/* instance_init */
		};
		type = g_type_register_static (MIDGARD_TYPE_DBOBJECT,
				"midgard_user",
				&info, 0);
	}
	
	return type;
}
