/*
 *  Microfeed - Backend for accessing feed-based services (utils)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedmain.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdio.h>
#include <stdint.h>

static void error_occured(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	if (error_name) {
		printf("Failed: %s %s (%s): %s (%s)\n", publisher, uri, uid, error_name, error_message);
	} else {
		printf("Succeeded: %s %s (%s)\n", publisher, uri, uid);
	}
}

static void feed_subscribed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	MicrofeedMain* microfeed_main;

	microfeed_main = (MicrofeedMain*)user_data;

	if (error_name) {
		printf("Subscribing failed: %s %s: %s (%s)\n", publisher, uri, error_name, error_message);
	} else {

		printf("Subscribed: %s %s\n", publisher, uri);
	
		microfeed_subscriber_republish_items(subscriber, publisher, uri, NULL, NULL, 20, NULL, NULL);
	}
}


static void feed_update_started(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	printf("Update started: %s %s\n", publisher, uri);
}

static void feed_update_ended(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	printf("Update ended: %s %s\n", publisher, uri);
}

static void feed_republishing_started(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	printf("Republishing started: %s %s\n", publisher, uri);
}

static void feed_republishing_ended(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data) {
	printf("Republishing ended: %s %s\n", publisher, uri);
}

static void item_added(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;
	
	printf("Added: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
	for (iterator = microfeed_item_iterate_properties(item, NULL);
	     microfeed_item_iterator_get(iterator, &key, &value);
	     microfeed_item_iterator_next(iterator)) {
		printf("    %s: %s\n", key, value);
	}
	if ((value = microfeed_item_get_property(item, "content.image"))) {
		microfeed_subscriber_send_item_data(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, value, NULL, NULL);
	}
	if ((value = microfeed_item_get_property(item, "sender.image"))) {
		microfeed_subscriber_send_item_data(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, value, NULL, NULL);
	}
	
	microfeed_item_iterator_free(iterator);
}

static void item_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;

	printf("Changed: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
	for (iterator = microfeed_item_iterate_properties(item, NULL);
	     microfeed_item_iterator_get(iterator, &key, &value);
	     microfeed_item_iterator_next(iterator)) {
		printf("    %s: %s\n", key, value);
	}
	microfeed_item_iterator_free(iterator);
}

static void item_republished(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;

	printf("Republished: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
	for (iterator = microfeed_item_iterate_properties(item, NULL);
	     microfeed_item_iterator_get(iterator, &key, &value);
	     microfeed_item_iterator_next(iterator)) {
		printf("    %s: %s\n", key, value);
	}
	microfeed_item_iterator_free(iterator);
}

static void item_removed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, void* user_data) {
	printf("Removed: %s %s %s\n", publisher, uri, uid);
}

static void item_status_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const MicrofeedItemStatus status, void* user_data) {
	printf("Status changed: %s %s %s %u\n", publisher, uri, uid, (unsigned int)status);
}

static void item_data_received(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t length, void* user_data) {
	printf("Data: %s %s %s (length: %u)\n", publisher, uri, uid, length);
}

static MicrofeedSubscriberCallbacks callbacks = {
	error_occured,
	feed_update_started,
	feed_update_ended,
	feed_republishing_started,
	feed_republishing_ended,
	item_added,
	item_changed,
	item_republished,
	item_removed,
	item_status_changed,
	item_data_received
};

int main(int argc, char** argv) {
	MicrofeedMain* microfeed_main;
	MicrofeedSubscriber* subscriber;
	int i;
	
	microfeed_thread_init();

	if (argc < 2 || (argc > 2 && argc % 2 != 1)) {
		fprintf(stderr, "Usage: %s <publisher_identifier> | (<publisher_identifier> <uri>)...\n", argv[0]);
		
		return 1;
	}
	
	microfeed_main = microfeed_main_new();
	subscriber = microfeed_subscriber_new("org.microfeed.Utils.Subscriber", "/org/microfeed/Utils/Subscriber", microfeed_main_get_dbus_connection(microfeed_main));
	if (argc == 2) {
		microfeed_subscriber_subscribe_feed(subscriber, argv[1], MICROFEED_FEED_URI_OVERVIEW, &callbacks, feed_subscribed, microfeed_main);
	} else {
		for (i = 1; i + 1 < argc; i += 2) {
			microfeed_subscriber_subscribe_feed(subscriber, argv[i], argv[i + 1], &callbacks, feed_subscribed, microfeed_main);
		}	
	}
	microfeed_main_loop(microfeed_main);

	return 0;
}
