
#include "config.h"

#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedmisc.h>

#include "configuration.h"

typedef struct {
	GtkWidget* dialog;
	GtkWidget* edit_button;
	GtkWidget* delete_button;
} PublishersDialog;


static void publisher_destroyed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	GtkWidget* dialog;

	if (error_name) {
		dialog = gtk_message_dialog_new(*((GtkWindow**)user_data), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Could not delete %s: %s", publisher, (error_message ? error_message : error_name));
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		if (*((GtkDialog**)user_data)) {
			gtk_dialog_response(*((GtkDialog**)user_data), GTK_RESPONSE_REJECT);
		}
	} else {
		if (*((GtkDialog**)user_data)) {
			gtk_dialog_response(*((GtkDialog**)user_data), GTK_RESPONSE_ACCEPT);
		}
	}
}

static void show_delete_dialog(MicrofeedSubscriber* subscriber, GtkWindow* parent, const char* publisher) {
	GtkWidget* dialog;
	GtkWidget* info_dialog;

	dialog = gtk_message_dialog_new(parent, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, "Do you readlly want to delete the publisher called %s?", publisher);
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_YES) {
		if (microfeed_subscriber_destroy_publisher(subscriber, publisher, publisher_destroyed, &info_dialog)) {
			info_dialog = gtk_message_dialog_new(GTK_WINDOW(dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_CANCEL, "Deleting publisher... Wait!");
			gtk_dialog_run(GTK_DIALOG(info_dialog));
			gtk_widget_destroy(info_dialog);
		}
	}
	gtk_widget_destroy(dialog);
}

static void on_row_activated(GtkTreeView* tree_view, GtkTreePath* path, GtkTreeViewColumn* column, gpointer user_data) {
	PublishersDialog* publishers_dialog;
	
	publishers_dialog = (PublishersDialog*)user_data;
	gtk_dialog_response(GTK_DIALOG(publishers_dialog->dialog), 2);
}

static void on_selection_changed(GtkTreeSelection* selection, gpointer user_data) {
	PublishersDialog* publishers_dialog;
	gboolean selected;

	publishers_dialog = (PublishersDialog*)user_data;
	selected = gtk_tree_selection_get_selected(selection, NULL, NULL);
	gtk_widget_set_sensitive(publishers_dialog->edit_button, selected);
	gtk_widget_set_sensitive(publishers_dialog->delete_button, selected);
}

static void update_list(MicrofeedConfiguration* configuration, GtkListStore* list_store) {
	const char** publishers;
	GtkTreeIter iter;

	gtk_list_store_clear(list_store);
	microfeed_configuration_invalidate(configuration);
	for (publishers = microfeed_configuration_get_publishers(configuration); publishers && *publishers; publishers++) {
		gtk_list_store_append(list_store, &iter);
		gtk_list_store_set(list_store, &iter, 0, *publishers, -1);
	}
}

void show_publishers_dialog(MicrofeedSubscriber* subscriber, GtkWindow* parent) {
	MicrofeedConfiguration* configuration;
	PublishersDialog* publishers_dialog;
	GtkWidget* button;
	GtkListStore* list_store;
	GtkWidget* tree_view;
	GtkWidget* scrolled_window;
	GtkTreeIter iter;
	GtkCellRenderer* renderer;
	GtkTreeViewColumn* column;
	GtkTreeSelection* selection;
	int response;
	char* publisher;
	GValue value = { 0 };

	configuration = microfeed_subscriber_get_configuration(subscriber);
	
	publishers_dialog = microfeed_memory_allocate(PublishersDialog);
	publishers_dialog->dialog = gtk_dialog_new_with_buttons("Configure Microfeed Publishers", parent, 0, NULL);
	button = gtk_button_new_from_stock(GTK_STOCK_NEW);
	gtk_dialog_add_action_widget(GTK_DIALOG(publishers_dialog->dialog), button, 1);
	publishers_dialog->edit_button = gtk_button_new_from_stock(GTK_STOCK_EDIT);
	gtk_widget_set_sensitive(publishers_dialog->edit_button, FALSE);
	gtk_dialog_add_action_widget(GTK_DIALOG(publishers_dialog->dialog), publishers_dialog->edit_button, 2);
	publishers_dialog->delete_button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	gtk_widget_set_sensitive(publishers_dialog->delete_button, FALSE);
	gtk_dialog_add_action_widget(GTK_DIALOG(publishers_dialog->dialog), publishers_dialog->delete_button, 3);
#ifndef MAEMO_CHANGES
	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_dialog_add_action_widget(GTK_DIALOG(publishers_dialog->dialog), button, GTK_RESPONSE_CLOSE);
#endif	

	list_store = gtk_list_store_new(1, G_TYPE_STRING);
	update_list(configuration, list_store);
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store));
	gtk_widget_set_size_request(tree_view, 400, 150);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree_view), FALSE);
	g_signal_connect(tree_view, "row-activated", G_CALLBACK(on_row_activated), publishers_dialog);
	g_object_unref(G_OBJECT(list_store));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Publisher", renderer, "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW(tree_view), column);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
 	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_IN);
 	gtk_container_add(GTK_CONTAINER(scrolled_window), tree_view);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(publishers_dialog->dialog)->vbox), scrolled_window);
			
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
	g_signal_connect(selection, "changed", G_CALLBACK(on_selection_changed), publishers_dialog);
			
	gtk_widget_show_all(publishers_dialog->dialog);
	
	while ((response = gtk_dialog_run(GTK_DIALOG(publishers_dialog->dialog))) != GTK_RESPONSE_DELETE_EVENT && response != GTK_RESPONSE_CLOSE) {
		if (response == 1) {
			if ((publisher = show_new_dialog(subscriber, GTK_WINDOW(publishers_dialog->dialog)))) {
				update_list(configuration, list_store);
				show_edit_dialog(subscriber, GTK_WINDOW(publishers_dialog->dialog), publisher);	
				free(publisher);
			}
		} else if (response == 2) {
			if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
				gtk_tree_model_get_value(GTK_TREE_MODEL(list_store), &iter, 0, &value);
				show_edit_dialog(subscriber, GTK_WINDOW(publishers_dialog->dialog), g_value_get_string(&value));
				g_value_unset(&value);
			}
		} else if (response == 3) {
			if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
				gtk_tree_model_get_value(GTK_TREE_MODEL(list_store), &iter, 0, &value);
				show_delete_dialog(subscriber, GTK_WINDOW(publishers_dialog->dialog), g_value_get_string(&value));
				g_value_unset(&value);
				update_list(configuration, list_store);
				
			}
		}
	}
	gtk_widget_destroy(publishers_dialog->dialog);
}
