/*
 *  Microfeed - Backend for accessing feed-based services (unstable providers)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmain.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-common/microfeedobject.h>

#include "twitter.h"

#include <string.h>
#include <stdio.h>

static MicrofeedPublisher* instantiate_publisher(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, void* user_data);
static MicrofeedFeed* instantiate_feed(MicrofeedPublisher* publisher, const char* uri, int create, void* user_data);
static void no_more_publishers(MicrofeedProvider* publisher, void* user_data);
static void initialize_settings(MicrofeedPublisher* publisher, void* user_data);
static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data);
static void* add_timeout(MicrofeedProvider* provider, unsigned long int interval, MicrofeedProviderTimeoutHandler handler, void* data, void* user_data);
static void remove_timeout(MicrofeedProvider* provider, void* timeout_implementation, void* user_data);
static MicrofeedError* download_image(MicrofeedPublisher* publisher, const char* url, void** data, size_t* length, void* user_data);
static MicrofeedError* update_comments_feed(MicrofeedFeed* feed, int user_initiated, void* user_data);

static MicrofeedProviderCallbacks provider_callbacks = {
	.instantiate_publisher = instantiate_publisher,
	.no_more_publishers = no_more_publishers,
	.add_timeout = add_timeout,
	.remove_timeout = remove_timeout
};

static MicrofeedPublisherCallbacks publisher_callbacks = {
	.instantiate_feed = instantiate_feed,
	.initialize_settings = initialize_settings,
	.update_setting = update_setting,
	.store_data = download_image
};

static MicrofeedFeedCallbacks comments_feed_callbacks = {
	.update = update_comments_feed
};

int main(void) {
	MicrofeedMain* microfeed_main;
	MicrofeedProvider* provider;
	MicrofeedWeakReference* weak_reference;
	
	//dbus_connection_set_change_sigpipe(FALSE);
	dbus_threads_init_default();
	microfeed_thread_init();
	microfeed_http_init(1);

	microfeed_main = microfeed_main_new();
	provider = microfeed_provider_new(PROVIDER_ID_TWITTER, microfeed_main_get_dbus_connection(microfeed_main), &provider_callbacks, microfeed_main);
	weak_reference = microfeed_object_get_weak_reference(provider, MicrofeedProvider);

	do {
		microfeed_main_loop(microfeed_main);
		
		microfeed_object_unref(provider, MicrofeedProvider);
		provider = microfeed_weak_reference_get_object(weak_reference, MicrofeedProvider);
	} while (provider);

	microfeed_weak_reference_unref(weak_reference);

	return 0;
}

static void no_more_publishers(MicrofeedProvider* provider, void* user_data) {
	MicrofeedMain* microfeed_main;
	
	microfeed_main = (MicrofeedMain*)user_data;
	
	microfeed_main_exit(microfeed_main);
}

static MicrofeedPublisher* instantiate_publisher(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, void* user_data) {
	MicrofeedConfiguration* configuration = NULL;
	MicrofeedPublisher* publisher;
	
	if (!directory) {
		configuration = microfeed_configuration_new();
		directory = microfeed_configuration_get_default_publisher_directory(configuration);
	}
	
	publisher = microfeed_publisher_new(provider, publisher_identifier, directory, &publisher_callbacks, user_data);

	if (configuration) {
		microfeed_configuration_free(configuration);
	}

	return publisher;
}

static MicrofeedFeed* instantiate_feed(MicrofeedPublisher* publisher, const char* uri, int create, void* user_data) {
	MicrofeedFeed* feed = NULL;
	MicrofeedMain* microfeed_main;
	size_t length;
	char* url;
	MicrofeedFeed* master_feed;
	
	microfeed_main = (MicrofeedMain*)user_data;

	if (!strcmp(uri, MICROFEED_FEED_URI_OVERVIEW)) {
		url = strdup("/statuses/friends_timeline.json");
		feed = microfeed_feed_new(publisher, uri, "Overview", MICROFEED_FEED_PERMISSION_ADD, 1, &overview_feed_callbacks, url);
		microfeed_feed_add_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE);
		microfeed_feed_add_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_IMAGE);
		microfeed_feed_add_virtual_feed_property(feed, ".twitter.id", MICROFEED_FEED_URI_USER_PREFIX, 1);
		microfeed_feed_add_virtual_feed_property(feed, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM, MICROFEED_FEED_URI_COMMENTS_PREFIX, 1);
		microfeed_feed_add_virtual_feed_property(feed, NULL, MICROFEED_FEED_URI_COMMENTS_PREFIX, 0);
	} else if (!strcmp(uri, MICROFEED_FEED_URI_CONTACTS)) {
		feed = microfeed_feed_new(publisher, uri, "Contacts", MICROFEED_FEED_PERMISSION_ADD, 1, &contacts_callbacks, NULL);
		microfeed_feed_add_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE);
	} else if ((length = microfeed_util_string_starts_with(uri, MICROFEED_FEED_URI_USER_PREFIX))) {
		url = microfeed_util_string_concatenate("/statuses/user_timeline.json?user_id=", uri + length, NULL);
		master_feed = microfeed_publisher_get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_OVERVIEW, 1);
		feed = microfeed_feed_new_virtual(master_feed, uri, uri + length, MICROFEED_FEED_PERMISSION_NONE, create, &feed_callbacks, url);
		microfeed_object_unref(master_feed, MicrofeedFeed);
	} else if ((length = microfeed_util_string_starts_with(uri, MICROFEED_FEED_URI_COMMENTS_PREFIX))) {
		master_feed = microfeed_publisher_get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_OVERVIEW, 1);
		feed = microfeed_feed_new_virtual(master_feed, uri, uri + length, MICROFEED_FEED_PERMISSION_NONE, create, &comments_feed_callbacks, NULL);
		microfeed_object_unref(master_feed, MicrofeedFeed);
	}

	return feed;	
}

static void initialize_settings(MicrofeedPublisher* publisher, void* user_data) {
	microfeed_publisher_add_setting(publisher, "service.url", "URL", "", "string", "256", "http://twitter.com/");
	microfeed_publisher_add_setting(publisher, "authentication.login", "Login", "", "string", "128", "");
	microfeed_publisher_add_setting(publisher, "authentication.password", "Password", "", "password", "128", "");
}

static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data) {
	return 1;
}

static MicrofeedError* download_image(MicrofeedPublisher* publisher, const char* url, void** data, size_t* length, void* user_data) {
	char* buffer;
	MicrofeedError* error = NULL;
	MicrofeedHttp* http;

	http = microfeed_http_new();
	if ((buffer = (void*)microfeed_http_get_data(http, url, length))) {
		*data = malloc(*length);
		memcpy(*data, buffer, *length);
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not download image");
	}
	microfeed_http_free(http);
	
	return error;	
}

typedef struct {
	unsigned long int interval;
	MicrofeedProviderTimeoutHandler handler;
	void* data;
	MicrofeedTimeout* timeout;
} TimeoutData;

static void timeout_callback(MicrofeedMain* microfeed_main, void* user_data) {
	TimeoutData* timeout_data;
	
	timeout_data = (TimeoutData*)user_data;
	if (timeout_data->handler(timeout_data->data)) {
		timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, timeout_data->interval, timeout_callback, user_data);
	} else {
		microfeed_memory_free(timeout_data);
	}
}

static void* add_timeout(MicrofeedProvider* provider, unsigned long int interval, MicrofeedProviderTimeoutHandler handler, void* data, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;

	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = microfeed_memory_allocate(TimeoutData);
	timeout_data->interval = interval;
	timeout_data->handler = handler;
	timeout_data->data = data;
	timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, interval, timeout_callback, timeout_data);

	return timeout_data;
}

static void remove_timeout(MicrofeedProvider* provider, void* timeout_implementation, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;
	
	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = (TimeoutData*)timeout_implementation;

	microfeed_main_remove_timeout(microfeed_main, timeout_data->timeout);	
	microfeed_memory_free(timeout_data);
}

static MicrofeedError* update_comments_feed(MicrofeedFeed* feed, int user_initiated, void* user_data) {
	MicrofeedError* error = NULL;
	const char* uri;
	size_t length;
	char* site;
	char* url;
	MicrofeedHttp* http;
	int rate_limit;
	time_t delta = 0;
	MicrofeedJson* status;
	const char* id;
	const char* created_at;
	time_t timestamp;
	MicrofeedItem* item;
	const char* cs;
	char* s;
	
	uri = microfeed_feed_get_uri(feed);
	if ((length = microfeed_util_string_starts_with(uri, MICROFEED_FEED_URI_COMMENTS_PREFIX))) {
		site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
		url = microfeed_util_string_concatenate(site, "/statuses/show/", uri + length, ".json", NULL);
		free(site);

		http = microfeed_http_new();
		do_authentication(microfeed_feed_get_publisher(feed), http);
		if ((rate_limit = get_rate_limit(microfeed_feed_get_publisher(feed), http)) != -1 && rate_limit < 1) {
			error = microfeed_error_new(MICROFEED_ERROR_FROM_SERVICE, "Rate limit exceeded."); /* TODO: Change to MICROFEED_ERROR_SERVICE_RATE_LIMIT_EXCEEDED */
		} else if ((status = microfeed_http_get_json(http, url))) {
			if ((id = microfeed_json_get_as_string(status, "id"))) {
				if (microfeed_http_get_server_time(http)) {
					delta = microfeed_http_get_server_time(http) - microfeed_http_get_reply_start_time(http);
				}

			/* TODO: ? */
			delta = 0;
			
				if ((created_at = microfeed_json_get_string(status, "created_at"))) {
					timestamp = microfeed_http_parse_date(http, created_at) - delta;
				} else {
					timestamp = time(NULL) - delta;
				}
				item = microfeed_item_new_with_status(id, timestamp, MICROFEED_ITEM_STATUS_NEW);
				microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT, microfeed_json_get_string(status, "text"));
				microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NICK, microfeed_json_get_string_by_path(status, "user", "screen_name", NULL));
				if ((cs = microfeed_json_get_as_string_by_path(status, "user", "id", NULL))) {
					microfeed_item_set_property(item, ".twitter.id", cs);						
					s = microfeed_util_string_concatenate(MICROFEED_FEED_URI_USER_PREFIX, cs, NULL);
					microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_FEED, s);
					free(s);
				}
				microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, microfeed_json_get_string_by_path(status, "user", "profile_image_url", NULL));
				if (!microfeed_json_is_null(status, "in_reply_to_status_id")) {
					microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_ITEM, microfeed_json_get_as_string(status, "in_reply_to_status_id"));
				}

				microfeed_feed_replace_item(feed, item);
				microfeed_item_free(item);
			}
			microfeed_json_free(status);
		}
		microfeed_http_free(http);
		free(url);
	}
	
	return error;
}
