
#define _XOPEN_SOURCE
#define _BSD_SOURCE
#define _GNU_SOURCE

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-provider/microfeedjson.h>

#include "twitter.h"

#include <stdio.h>
#include <string.h>

static void destroy_feed(MicrofeedFeed* feed, void* user_data);
static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data);
static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data);
static MicrofeedError* mark_item(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data);

MicrofeedFeedCallbacks overview_feed_callbacks = {
	destroy_feed,
	update_feed,
	modify_item,
	NULL, /* download_item_data */
	mark_item
};

MicrofeedFeedCallbacks feed_callbacks = {
	destroy_feed,
	update_feed,
	NULL, /* modify_feed_item */
	NULL, /* download_item_data */
	mark_item
};

static void destroy_feed(MicrofeedFeed* feed, void* user_data) {
	char* url;

	url = (char*)user_data;
	free(url);
}

static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data) {
	MicrofeedError* error = NULL;
	char* site;
	const char* path;
	char* url;
	MicrofeedHttp* http;
	time_t delta = 0;
	MicrofeedJson* json;
	MicrofeedFeedIterator* iterator;
	unsigned int i;
	MicrofeedJson* status;
	const char* id;
	const char* created_at;
	time_t timestamp;
	MicrofeedItem* item;
	const char* cs;
	char* s;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	path = (const char*)user_data;
	url = microfeed_util_string_concatenate(site, path, NULL);
	free(site);

	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if ((json = microfeed_http_get_json(http, url))) {
		if (microfeed_http_get_server_time(http)) {
			delta = microfeed_http_get_server_time(http) - microfeed_http_get_reply_start_time(http);
		}
		if (microfeed_json_get_type(json, NULL) == MICROFEED_JSON_TYPE_ARRAY) {
			iterator = microfeed_feed_iterate_timeline(feed, 0, 1);
			for (i = 0; i < microfeed_json_get_size(json); i++) {
				if ((status = microfeed_json_get_object_by_index(json, i)) &&
				    (id = microfeed_json_get_as_string(status, "id"))) {
					if (microfeed_feed_iterator_jump_and_remove_previous_items(iterator, id)) {
						microfeed_feed_iterator_next(iterator);
					} else {
						if ((created_at = microfeed_json_get_string(status, "created_at"))) {
							timestamp = microfeed_http_parse_date(http, created_at) - delta;
						} else {
							timestamp = time(NULL) - delta;
						}
						item = microfeed_item_new(id, timestamp);
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT, microfeed_json_get_string(status, "text"));
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NICK, microfeed_json_get_string_by_path(status, "user", "screen_name", NULL));
						if ((cs = microfeed_json_get_string_by_path(status, "user", "id", NULL))) {							
							s = microfeed_util_string_concatenate(MICROFEED_FEED_URI_USER_PREFIX, cs, NULL);
							microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_URI, s);
							free(s);
						}
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, microfeed_json_get_string_by_path(status, "user", "profile_image_url", NULL));
						microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_UID, microfeed_json_get_string(status, "in_reply_to_status_id"));
						if ((cs = microfeed_json_get_string(status, "in_reply_to_user_id"))) {
							s = microfeed_util_string_concatenate(MICROFEED_FEED_URI_USER_PREFIX, cs, NULL);
							microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_URI, s);
							free(s);
						}
						microfeed_feed_replace_item(feed, item);
						microfeed_item_free(item);
					}
				} else {
					error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply (expected JSON object with id in array).");
				}
			}
			microfeed_feed_iterator_free(iterator);
		} else if (!(error = check_if_error(json))) {
			error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply (expected JSON array).");
		}
		microfeed_json_free(json);
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not get the feed.");
	}

	free(url);

	return error;
}

static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data) {
	const char* text;
	const char* referred;
	char* post_data;
	char* site;
	char* url;
	MicrofeedHttp* http;
	MicrofeedJson* json;
	
	if (existing_item) {

		return microfeed_error_new(MICROFEED_ERROR_CANNOT_MODIFY_ITEM, "Cannot modify an existing item.");
	}
	if (!(text = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_TEXT))) {

		return microfeed_error_new(MICROFEED_ERROR_MISSING_PROPERTY, "Expected a content.text property.");
	}
	
	if ((referred = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_REFERRED_UID))) {
		post_data = microfeed_util_string_concatenate("status=", text, "&in_reply_to_status_id=", referred, NULL);
	} else {
		post_data = microfeed_util_string_concatenate("status=", text, NULL);		
	}

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	url = microfeed_util_string_concatenate(site, "/statuses/update.json", NULL);
	free(site);
	
	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if (!(json = microfeed_http_post_json(http, url, post_data))) {

		return microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply when updating status.");
	}
	
	microfeed_json_free(json);
	free(url);
	free(post_data);
	
	return NULL;
}

static MicrofeedError* mark_item(MicrofeedFeed* feed, const char* uid, int mark_status, void* user_data) {
	char* site;
	char* url;
	MicrofeedHttp* http;
	MicrofeedJson* json;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	if (mark_status) {
		url = microfeed_util_string_concatenate(site, "/favorites/create/", uid, ".json", NULL);
	} else {
		url = microfeed_util_string_concatenate(site, "/favorites/destroy/", uid, ".json", NULL);
	}
	free(site);

	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	if (!(json = microfeed_http_post_json(http, url, ""))) {

		return microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply when changing favorite status.");
	}
	
	microfeed_json_free(json);
	free(url);
	
	return NULL;
}
