/*
 *  message blocker
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>
#include <hildon/hildon.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <string.h>
#include <locale.h>
#include <gconf/gconf-client.h>
#include <libmcclient/mc-account-manager.h>
#include <telepathy-glib/dbus.h>
#include <libosso-abook/osso-abook-mc-account-model.h>
#include <libosso-abook/osso-abook-mc-account-selector.h>

#define GC_ROOT "/apps/maemo/message-blocker"
#define BLOCK_LIST_GCONF_PATH GC_ROOT
#define ACCOUNTS_BLOCK_LIST_GCONF_PATH GC_ROOT "/accounts"
#define BLOCKER_ACTIVE_GCONF_PATH GC_ROOT "/active"
#define DELETE_MSG_GCONF_PATH GC_ROOT "/delete-messages"

static void
_add_block_entry(GtkButton* button, gpointer user_data)
{
  HildonTouchSelector* selector = HILDON_TOUCH_SELECTOR(user_data);
  GtkWidget* dialog = gtk_dialog_new_with_buttons("add",
						  NULL,
						  GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
						  dgettext("hildon-libs", "wdgt_bd_save"), GTK_RESPONSE_ACCEPT,
						  NULL);
  GtkWidget* entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  hildon_gtk_entry_set_input_mode(GTK_ENTRY(entry),
				  HILDON_GTK_INPUT_MODE_FULL);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), entry);
  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    if(gtk_entry_get_text_length(GTK_ENTRY(entry)) > 0)
    {
      hildon_touch_selector_append_text(selector, 
					gtk_entry_get_text(GTK_ENTRY(entry)));
    }
  }

  gtk_widget_destroy(dialog);
}

static void
_remove_block_entry(GtkButton* button, gpointer user_data)
{
  HildonTouchSelector* selector = HILDON_TOUCH_SELECTOR(user_data);
  GList* selected_rows = hildon_touch_selector_get_selected_rows(selector,
								 0);
  GList* selected = g_list_last(selected_rows);
  GtkListStore* store = GTK_LIST_STORE(hildon_touch_selector_get_model(selector, 0));
  
  while(selected)
  {
    GtkTreeIter iter;
    GtkTreePath* path = selected->data;
    gtk_tree_model_get_iter(GTK_TREE_MODEL(store), &iter, path);
    gtk_list_store_remove(store, &iter);
    selected = g_list_previous(selected);
  }
  g_list_foreach(selected_rows, (GFunc)gtk_tree_path_free, NULL);
  g_list_free(selected);
}

static void
_save_block_list(HildonTouchSelector* selector, const gchar* account_name)
{
  GError* error = NULL;
  GSList* block_list = NULL;
  GConfClient* client = gconf_client_get_default();
  g_assert(GCONF_IS_CLIENT(client));

  GtkTreeModel* tree = hildon_touch_selector_get_model(selector, 0);
  GtkTreeIter iter;

  if(gtk_tree_model_get_iter_first(tree, &iter))
  {
    do
    {
      gchar* entry = NULL;
      gtk_tree_model_get(tree, &iter,
			 0, &entry, -1);
      block_list = g_slist_append(block_list, entry);
    }while(gtk_tree_model_iter_next(tree, &iter));
  }

  gchar* conf_key = g_strdup_printf("%s/%s/block-list", ACCOUNTS_BLOCK_LIST_GCONF_PATH, account_name);

  gconf_client_set_list(client,
			conf_key,
			GCONF_VALUE_STRING,
			block_list,
			&error);
  g_free(conf_key);
  if(error)
  {
    g_error_free(error);
    error = NULL;
  }
  g_slist_foreach(block_list, (GFunc)g_free, NULL);
  g_slist_free(block_list);
  g_object_unref(client);
}

static gboolean
_read_blocker_active_setting()
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return TRUE;
  gboolean active = gconf_client_get_bool(client, BLOCKER_ACTIVE_GCONF_PATH, NULL);
  g_object_unref(client);
  return active;
}

static gboolean
_read_delete_msg_setting()
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return TRUE;
  gboolean delete_msg = gconf_client_get_bool(client, DELETE_MSG_GCONF_PATH, NULL);
  g_object_unref(client);
  return delete_msg;
}

static void
_store_blocker_active_setting(HildonCheckButton *button,
			      gpointer user_data)
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return;
  gconf_client_set_bool(client, BLOCKER_ACTIVE_GCONF_PATH, hildon_check_button_get_active(button), NULL);
  g_object_unref(client);
}

static void
_store_delete_msg_setting(HildonCheckButton *button,
			  gpointer user_data)
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return;
  gconf_client_set_bool(client, DELETE_MSG_GCONF_PATH, hildon_check_button_get_active(button), NULL);
  g_object_unref(client);
}

static void
_read_block_list(HildonTouchSelector* selector, const gchar* account_name)
{
  GError* error = NULL;
  GConfClient* client = gconf_client_get_default();
  g_assert(GCONF_IS_CLIENT(client));
  gchar* conf_key = g_strdup_printf("%s/%s/block-list", ACCOUNTS_BLOCK_LIST_GCONF_PATH, account_name);
  GSList* block_list = gconf_client_get_list(client, conf_key, GCONF_VALUE_STRING, &error);
  if(error)
  {
    g_error_free(error);
    error = NULL;
  }

  if(block_list != NULL)
  {
    GSList* block_entry = block_list;
    while(block_entry)
    {
      gchar* block_id = (gchar*)block_entry->data;
      hildon_touch_selector_append_text(selector, block_id);
      g_free(block_id);
      block_entry = block_entry->next;
    }
    g_slist_free(block_list);
  }
  g_object_unref(client);
}

static void
_enable_disable_remove_button(HildonTouchSelector *widget,
			      gint column,
			      gpointer user_data)
{
  GtkWidget* button = GTK_WIDGET(user_data);
  GtkTreeIter iter;
  gchar* selection = hildon_touch_selector_get_current_text(widget);
  if(selection == NULL || strlen(selection) <= 2)
    gtk_widget_set_sensitive(button, FALSE);
  else
    gtk_widget_set_sensitive(button, TRUE);
  g_free(selection);
}

void
_show_edit_block_list_dialog(const gchar* account_name)
{
  GtkWidget* add_button;
  GtkWidget* remove_button;
  GtkWidget* block_list_selector;
  
  GtkWidget* dialog = gtk_dialog_new_with_buttons("Settings",
						  NULL,//GTK_WINDOW(user_data),
						  GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
						  dgettext("hildon-libs", "wdgt_bd_save"), GTK_RESPONSE_ACCEPT,
						  NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 350);

  add_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  remove_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* add_image = gtk_image_new_from_file("/usr/share/pixmaps/message_blocker/add.png");
  GtkWidget* minus_image = gtk_image_new_from_file("/usr/share/pixmaps/message_blocker/minus.png");
  gtk_button_set_image(GTK_BUTTON(add_button),
  		       add_image);
  gtk_button_set_image(GTK_BUTTON(remove_button),
		       minus_image);

  gtk_widget_set_sensitive(remove_button, FALSE);

  block_list_selector = hildon_touch_selector_new_text();
  hildon_touch_selector_set_hildon_ui_mode(HILDON_TOUCH_SELECTOR(block_list_selector),
					   HILDON_UI_MODE_EDIT);
  hildon_touch_selector_set_column_selection_mode(HILDON_TOUCH_SELECTOR(block_list_selector),
						  HILDON_TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE);
  _read_block_list(HILDON_TOUCH_SELECTOR(block_list_selector), account_name);
  
  g_signal_connect(add_button, "clicked", G_CALLBACK(_add_block_entry), block_list_selector);
  g_signal_connect(remove_button, "clicked", G_CALLBACK(_remove_block_entry), block_list_selector);
  g_signal_connect(block_list_selector, "changed", G_CALLBACK(_enable_disable_remove_button), remove_button);
  GtkWidget* button_box = gtk_hbox_new(TRUE, 3);
  GtkWidget* main_box = gtk_vbox_new(FALSE, 0);

  gtk_box_pack_start(GTK_BOX(button_box), add_button, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(button_box), remove_button, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(main_box), block_list_selector, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(main_box), button_box, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), main_box);
  gtk_widget_show_all(dialog);

  guint response = gtk_dialog_run(GTK_DIALOG(dialog));
  if(response == GTK_RESPONSE_ACCEPT)
  {
    _save_block_list(HILDON_TOUCH_SELECTOR(block_list_selector), account_name);
  }
  gtk_widget_destroy(dialog);
}

const gchar*
_get_secondary_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "SecondaryTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_active_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "ActiveTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

const gchar*
_get_default_text_color()
{
  static gchar buf[40] = {0};
  if(buf[0] == '\0')
  {
    GdkColor color;
    GtkStyle *style = gtk_rc_get_style_by_paths(gtk_settings_get_default(),
						NULL, NULL, GTK_TYPE_LABEL);
    if(gtk_style_lookup_color(style, "DefaultTextColor", &color))
      sprintf(buf, "#%02x%02x%02x", color.red / 256, color.green / 256, color.blue / 256);
  }
  return buf;
}

static const gchar*
_get_account_markup()
{
  static gchar* markup_template = NULL;
  if(markup_template == NULL)
  {
    markup_template = g_strdup_printf("<span color=\"%s\">%%s</span>\n"
				      "<span color=\"%s\" size=\"x-small\">%%s</span>",
				      _get_default_text_color(),
				      _get_secondary_text_color());
  }
  return markup_template;
}

static void
_fill_account_list_store(McAccountManager* account_manager,
			 const GError* error,
			 gpointer user_data,
			 GObject* weak_object)
{
  GtkListStore* list_store = (GtkListStore*)user_data;

  const gchar* const * account_list = mc_account_manager_get_valid_accounts(account_manager);
  int i;
  GtkTreeIter iter;
  for(i = 0; account_list[i];++i)
  {
    McAccount* account = mc_account_manager_get_account(account_manager, account_list[i]);
    McProfile* profile = mc_profile_lookup(mc_account_compat_get_profile(account));
    GdkPixbuf* pixbuf = NULL;

    const gchar* nick_name = mc_account_get_nickname(account);
    const gchar* display_name = mc_profile_get_display_name(profile);
    gchar* account_name = NULL;

    if(g_strcmp0(account->name, "ring/tel/ring") == 0)
    {
      gtk_list_store_prepend(list_store, &iter);
      account_name = g_strdup(dgettext("osso-addressbook", "addr_bd_send_card_sms"));
      pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					"general_sms",
					HILDON_ICON_PIXEL_SIZE_FINGER,
					GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    else
    {
      gtk_list_store_append(list_store, &iter);
      pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					mc_profile_get_icon_name(profile),
					HILDON_ICON_PIXEL_SIZE_FINGER,
					GTK_ICON_LOOKUP_NO_SVG, NULL);
      account_name = g_markup_printf_escaped(_get_account_markup(), display_name, nick_name);
    }

    gtk_list_store_set(list_store, &iter,
		       0, pixbuf,
		       1, account_name,
		       2, account,
		       -1);
    g_object_unref(pixbuf);
    g_object_unref(profile);
  }
}

static void
_open_account_block_list(GtkTreeView* tree_view, 
			 GtkTreePath* path, 
			 GtkTreeViewColumn* column,
			 gpointer data)
{
  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  GtkTreeIter iter;
  GtkTreeModel* tree_model = gtk_tree_view_get_model(tree_view);
  if(gtk_tree_model_get_iter(tree_model,
			     &iter,
			     path))
  {
    McAccount* account = NULL;
    gtk_tree_model_get(tree_model, &iter,
		       2, &account,
		       -1);
    gchar* name = gconf_escape_key(account->name, -1);
    _show_edit_block_list_dialog(name);
    g_free(name);
  }
}

osso_return_t execute(osso_context_t* osso, gpointer user_data, gboolean user_activated)
{
  GtkWidget* dialog = gtk_dialog_new_with_buttons("Settings",
						  GTK_WINDOW(user_data),
						  GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
						  dgettext("hildon-libs", "wdgt_bd_done"), GTK_RESPONSE_ACCEPT,
						  NULL);
  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 350);
  GtkListStore* account_list_store = gtk_list_store_new(3, GDK_TYPE_PIXBUF, G_TYPE_STRING, MC_TYPE_ACCOUNT);
  GtkWidget* account_list = hildon_gtk_tree_view_new(HILDON_UI_MODE_NORMAL);
  GtkTreeViewColumn* icon_column = gtk_tree_view_column_new();
  GtkTreeViewColumn* protocol_column = gtk_tree_view_column_new();
  GtkCellRenderer* icon_renderer = gtk_cell_renderer_pixbuf_new();
  GtkCellRenderer* protocol_name_renderer = gtk_cell_renderer_text_new();
  GtkWidget* active_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* delete_msg_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(active_check_button), "Enable Message Blocker");
  gtk_button_set_label(GTK_BUTTON(delete_msg_check_button), "Delete Blocked Messages");
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(active_check_button), 
				 _read_blocker_active_setting());
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(delete_msg_check_button), 
				 _read_delete_msg_setting());
  g_signal_connect(active_check_button, "toggled", G_CALLBACK(_store_blocker_active_setting), NULL);
  g_signal_connect(delete_msg_check_button, "toggled", G_CALLBACK(_store_delete_msg_setting), NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(account_list),
			      icon_column);
  gtk_tree_view_append_column(GTK_TREE_VIEW(account_list),
			      protocol_column);

  gtk_tree_view_column_pack_start(icon_column, icon_renderer, FALSE);
  gtk_tree_view_column_pack_start(protocol_column, protocol_name_renderer, TRUE);  

  gtk_tree_view_column_set_attributes(icon_column,
				      icon_renderer,
				      "pixbuf", 0,
				      NULL);
  gtk_tree_view_column_set_attributes(protocol_column,
				      protocol_name_renderer,
				      "markup", 1,
				      NULL);


  gtk_tree_view_set_model(GTK_TREE_VIEW(account_list),
			  GTK_TREE_MODEL(account_list_store));
  g_signal_connect(account_list, "row-activated", G_CALLBACK(_open_account_block_list), NULL);
  TpDBusDaemon *dbus = tp_dbus_daemon_dup(NULL);
  McAccountManager* account_manager = mc_account_manager_new(dbus);
  mc_account_manager_call_when_ready_with_accounts(account_manager,
						   _fill_account_list_store,
						   account_list_store,
						   NULL, NULL, 
						   MC_IFACE_QUARK_ACCOUNT, MC_IFACE_QUARK_ACCOUNT_INTERFACE_COMPAT,0);
  GtkWidget* pan = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pan), account_list);
  GtkWidget* content = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(content), pan, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(content), active_check_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(content), delete_msg_check_button, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), content);
  gtk_widget_show_all(dialog);
  guint response = gtk_dialog_run(GTK_DIALOG(dialog));
  if(response == GTK_RESPONSE_ACCEPT)
  {
  }
  gtk_widget_destroy(dialog);
  g_object_unref(dbus);
  if(account_manager)
    g_object_unref(account_manager);
  return OSSO_OK;
}

osso_return_t save_state(osso_context_t* osso, gpointer user_data)
{
  return OSSO_OK;
}
