/*
 *  message blocker
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <glib.h>
#include <string.h>
#include "message_blocker_approver.h"

#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/gtypes.h>
#include <telepathy-glib/channel-dispatch-operation.h>
#include <telepathy-glib/svc-generic.h>
#include <telepathy-glib/svc-client.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/defs.h>
#include <telepathy-glib/handle.h>
#include <telepathy-glib/connection.h>
#include <telepathy-glib/channel.h>



static void approver_iface_init(gpointer, gpointer);

#define MESSAGE_BLOCKER_APPROVER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), TYPE_MESSAGE_BLOCKER_APPROVER, MessageBlockerApproverPrivate))

struct _MessageBlockerApproverPrivate
{
  GHashTable* account_block_list;
  GPtrArray* filter;
  gboolean active;
};

typedef struct _claim_channel_data_t
{
  MessageBlockerApprover* self;
  TpChannelDispatchOperation* dispatch_operation;
  TpConnection* connection;
  TpChannel* channel;
} claim_channel_data_t;

G_DEFINE_TYPE_WITH_CODE (MessageBlockerApprover, message_blocker_approver, G_TYPE_OBJECT,
			G_IMPLEMENT_INTERFACE(TP_TYPE_SVC_DBUS_PROPERTIES,
					      tp_dbus_properties_mixin_iface_init);
			G_IMPLEMENT_INTERFACE(TP_TYPE_SVC_CLIENT, NULL);
			G_IMPLEMENT_INTERFACE(TP_TYPE_SVC_CLIENT_APPROVER, approver_iface_init);
			);

static const char* client_interfaces[] = 
  {
    TP_IFACE_CLIENT_APPROVER, 
    NULL
  };

enum
{
  PROP_0,
  PROP_INTERFACES,
  PROP_CHANNEL_FILTER
};

void 
message_blocker_approver_set_active(MessageBlockerApprover* self, gboolean active)
{
  self->priv->active = active;
}

void 
message_blocker_approver_set_block_list(MessageBlockerApprover* self, const gchar* account_name, GSList* block_list)
{
  g_return_if_fail(IS_MESSAGE_BLOCKER_APPROVER(self));
  gpointer value = g_hash_table_lookup(self->priv->account_block_list, account_name);
  g_hash_table_insert(self->priv->account_block_list,
		      g_strdup(account_name),
		      block_list);
}

gboolean
block_from_account(MessageBlockerApprover* self, const gchar* account_name, const gchar* targetID)
{
  gpointer block_list = g_hash_table_lookup(self->priv->account_block_list,
					    account_name);
  if(block_list)
  {
    GSList* bentry = block_list;
    while(bentry)
    {
      gchar* block_id = (gchar*)bentry->data;
      bentry = bentry->next;
    }
    GSList* list = g_slist_find_custom(block_list,
				       targetID,
				       (GCompareFunc)g_strcmp0);
    if(list)
      return TRUE;
  }
  return FALSE;
}

static void 
_close_channel_cb(TpChannel *proxy,
		  const GError *error,
		  gpointer user_data,
		  GObject *weak_object)
{
  claim_channel_data_t* claim_channel_data = (claim_channel_data_t*)user_data;
  if(error)
  {
    g_warning("close error %s\n", error->message);
  }
  g_object_unref(claim_channel_data->dispatch_operation);
  g_object_unref(claim_channel_data->channel);
  g_object_unref(claim_channel_data->connection);
  g_free(claim_channel_data);
}

static void
_pending_messages_cb(TpChannel *proxy,
		     const GPtrArray *messages_list,
		     const GError *error,
		     gpointer user_data,
		     GObject *weak_object)
{
  claim_channel_data_t* claim_channel_data = (claim_channel_data_t*)user_data;
  if(error)
  {
    g_warning("error on list pending %s\n", error->message);
  }
  /* else */
  /* { */
  /*   int i; */
  /*   for (i = 0; i < messages_list->len; i++) { */
  /*     GValueArray    *message_struct; */
  /*     const gchar    *message_body; */
  /*     guint           message_id; */
  /*     guint           timestamp; */
  /*     guint           from_handle; */
  /*     guint           message_type; */
  /*     guint           message_flags; */
  /*     message_struct = g_ptr_array_index(messages_list, i); */
  /*     message_id = g_value_get_uint(g_value_array_get_nth (message_struct, 0)); */
  /*     timestamp = g_value_get_uint(g_value_array_get_nth (message_struct, 1)); */
  /*     from_handle = g_value_get_uint(g_value_array_get_nth (message_struct, 2)); */
  /*     message_type = g_value_get_uint(g_value_array_get_nth (message_struct, 3)); */
  /*     message_flags = g_value_get_uint(g_value_array_get_nth (message_struct, 4)); */
  /*     message_body = g_value_get_string(g_value_array_get_nth (message_struct, 5)); */
  /*   } */
  /* }     */
  tp_cli_channel_call_close(claim_channel_data->channel,
			    -1,
			    _close_channel_cb, user_data, NULL, NULL);
}

static void
_claim_channel_cb(TpChannelDispatchOperation* proxy,
		  const GError* error,
		  gpointer user_data,
		  GObject* weak_object)
{
  claim_channel_data_t* claim_channel_data = (claim_channel_data_t*)user_data;
  tp_cli_channel_type_text_call_list_pending_messages(claim_channel_data->channel, 
						      -1,
						      TRUE,
						      _pending_messages_cb,
						      claim_channel_data,
						      NULL, NULL);
}

static void
_claim_channel(MessageBlockerApprover* self,
	       const gchar* dispatch_operation,
	       const gchar* connection_object_path,
	       const gchar* channel_object_path)
{
  GError* error = NULL;
  claim_channel_data_t* claim_channel_data = g_new0(claim_channel_data_t, 1);
  TpDBusDaemon* tpdbus = tp_dbus_daemon_dup(NULL);
  TpChannelDispatchOperation* tpdispatch_operation = 
    tp_channel_dispatch_operation_new(tpdbus, dispatch_operation, NULL, NULL);
    
  TpConnection* tpconnection = tp_connection_new(tpdbus, NULL, connection_object_path, &error);
  if(error)
  {
    g_warning("error creating connection %s\n", error->message);
    g_error_free(error);
    error = NULL;
  }
  TpChannel* tpchannel = tp_channel_new(tpconnection,
					channel_object_path,
					"org.freedesktop.Telepathy.Channel.Type.Text",
					TP_UNKNOWN_HANDLE_TYPE,
					0, &error);

  claim_channel_data->self = self;
  claim_channel_data->dispatch_operation = tpdispatch_operation;
  claim_channel_data->connection = tpconnection;  
  claim_channel_data->channel = tpchannel;
  
  tp_cli_channel_dispatch_operation_call_claim(tpdispatch_operation,
					       -1,
					       _claim_channel_cb,
					       claim_channel_data,
					       NULL, NULL);
}

static void
message_blocker_approver_add_dispatch_operation(TpSvcClientApprover* self,
						const GPtrArray* channels,
						const char *dispatch_op,
						GHashTable *observer_info,
						DBusGMethodInvocation* context)
{
  if(MESSAGE_BLOCKER_APPROVER(self)->priv->active)
  {
    GValue* connection_value = g_hash_table_lookup(observer_info, 
						   "org.freedesktop.Telepathy.ChannelDispatchOperation.Connection");
    GValue* account_value = g_hash_table_lookup(observer_info, 
						"org.freedesktop.Telepathy.ChannelDispatchOperation.Account");
  
    const gchar* connection_object_path = g_value_get_boxed(connection_value);
    const gchar* account_object_path = g_value_get_boxed(account_value);
    gchar* account_name = NULL;
    char *channel_object_path = NULL;
    const gchar* targetID = NULL;
    if(g_str_has_prefix(account_object_path, TP_ACCOUNT_OBJECT_PATH_BASE))
      account_name = g_strdup(account_object_path + strlen(TP_ACCOUNT_OBJECT_PATH_BASE));
    int i;
    for (i = 0; i < channels->len; i++)
    {
      GValueArray *channel = g_ptr_array_index (channels, i);
      const gchar* channel_object_pathx = g_value_get_boxed(g_value_array_get_nth (channel, 0));
      channel_object_path = g_strdup(channel_object_pathx);
      GHashTable *map = g_value_get_boxed(g_value_array_get_nth (channel, 1));
      targetID = tp_asv_get_string(map, TP_IFACE_CHANNEL ".TargetID");
    }
    gboolean block = FALSE;
    if(targetID != NULL)
      block = block_from_account(MESSAGE_BLOCKER_APPROVER(self), account_name, targetID);
    g_free(account_name);
    if(block)
    {
      _claim_channel(MESSAGE_BLOCKER_APPROVER(self), dispatch_op, connection_object_path, channel_object_path);
    }
  }
  tp_svc_client_approver_return_from_add_dispatch_operation(context);
}

static void
_block_list_free(GSList* block_list)
{
  GSList* block_entry = block_list;
  while(block_entry)
  {
    gchar* block_id = (gchar*)block_entry->data;
    g_free(block_id);
    block_entry = block_entry->next;
  }
  g_slist_free(block_list);
}

static void
message_blocker_approver_get_property(GObject* self,
				      guint property_id,
				      GValue* value,
				      GParamSpec* pspec)
{
  switch (property_id)
    {
    case PROP_INTERFACES:
      g_value_set_boxed (value, client_interfaces);
      break;

    case PROP_CHANNEL_FILTER:
      g_value_set_boxed (value, MESSAGE_BLOCKER_APPROVER(self)->priv->filter);

      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, property_id, pspec);
      break;
    }
}

static void
message_blocker_approver_finalize(GObject* object)
{
  MessageBlockerApprover* self = MESSAGE_BLOCKER_APPROVER(object);
  g_boxed_free(TP_ARRAY_TYPE_CHANNEL_CLASS_LIST, self->priv->filter);
  g_object_unref(self->priv->account_block_list);
  G_OBJECT_CLASS(message_blocker_approver_parent_class)->finalize(object);
}

static void
message_blocker_approver_class_init(MessageBlockerApproverClass* klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS(klass);
  object_class->get_property = message_blocker_approver_get_property;
  object_class->finalize = message_blocker_approver_finalize;
  g_type_class_add_private(klass, sizeof(MessageBlockerApproverPrivate));
  
  static TpDBusPropertiesMixinPropImpl client_props[] = {
    {"Interfaces", "interfaces", NULL},
    {NULL}
  };
  
  static TpDBusPropertiesMixinPropImpl client_approver_props[] = {
        { "ApproverChannelFilter", "channel-filter", NULL },
        { NULL }
  };

  static TpDBusPropertiesMixinIfaceImpl prop_interfaces[] = {
    {TP_IFACE_CLIENT,
     tp_dbus_properties_mixin_getter_gobject_properties,
     NULL,
     client_props
    },
    {TP_IFACE_CLIENT_APPROVER,
     tp_dbus_properties_mixin_getter_gobject_properties,
     NULL,
     client_approver_props
    },
    {NULL}
  };
  
  g_object_class_install_property(object_class, PROP_INTERFACES,
				  g_param_spec_boxed("interfaces",
						     "Interfaces",
						     "Available D-Bus Interfaces",
						     G_TYPE_STRV,
						     G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));
  
  g_object_class_install_property(object_class, PROP_CHANNEL_FILTER,
				  g_param_spec_boxed("channel-filter",
						     "Channel Filter",
						     "Filter for channels we want to approve",
						     TP_ARRAY_TYPE_CHANNEL_CLASS_LIST,
						     G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));
  
  klass->dbus_props_class.interfaces = prop_interfaces;
  tp_dbus_properties_mixin_class_init(object_class,
				      G_STRUCT_OFFSET(MessageBlockerApproverClass, dbus_props_class));

}

static void
message_blocker_approver_init(MessageBlockerApprover* self)
{
  self->priv = MESSAGE_BLOCKER_APPROVER_GET_PRIVATE(self);
  self->priv->account_block_list = g_hash_table_new_full(g_str_hash, g_str_equal,
							 g_free,
							 (GDestroyNotify)_block_list_free);
  self->priv->active = TRUE;
  self->priv->filter = g_ptr_array_new();
  GHashTable* asv = tp_asv_new(TP_IFACE_CHANNEL ".ChannelType",
			       G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
			       TP_IFACE_CHANNEL ".TargetHandleType",
			       G_TYPE_INT, TP_HANDLE_TYPE_CONTACT,
			       NULL);
  g_ptr_array_add(self->priv->filter, asv);
}


static void
approver_iface_init(gpointer g_iface, gpointer iface_data)
{
  TpSvcClientApproverClass* klass = (TpSvcClientApproverClass*)g_iface;
#define IMPLEMENT(x) tp_svc_client_approver_implement_##x (klass, \
    message_blocker_approver_##x)
  IMPLEMENT (add_dispatch_operation);
#undef IMPLEMENT
}

MessageBlockerApprover*
message_blocker_approver_new(void)
{
  return g_object_new(TYPE_MESSAGE_BLOCKER_APPROVER, NULL);
}
