/*
 *  Copyright 2011 Ruediger Gad
 *
 *  This file is part of MeePasswords.
 *
 *  MeePasswords is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MeePasswords is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MeePasswords.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "entrystorage.h"

#ifdef MEEGO_EDITION_HARMATTAN
#include <aegis_crypto.h>
#else
#include <QDir>
#include <QFile>
#endif

#include <QTextStream>

EntryStorage::EntryStorage(QObject *parent) :
    QObject(parent)
{
#ifdef MEEGO_EDITION_HARMATTAN
    aegisStorage = NULL;
#endif

    qDebug("Initializing QCA...");
    initializer = new QCA::Initializer();
    qDebug("QCA initialized.");

    QCA::scanForPlugins();
    qDebug("QCA Plugin Diagnostics Context: %s", QCA::pluginDiagnosticText().toUtf8().constData());

    QStringList capabilities;
    capabilities = QCA::supportedFeatures();
    qDebug("QCA supports: %s", capabilities.join(",").toUtf8().constData());

    key = NULL;
}

EntryStorage::~EntryStorage(){
#ifdef MEEGO_EDITION_HARMATTAN
    if(aegisStorage != NULL){
        delete aegisStorage;
    }
#endif
    if(key != NULL){
        delete key;
    }
    if(initializer != NULL){
        delete initializer;
    }
}

bool EntryStorage::equalsStoredPassword(QString password){
    return (key != NULL && *key == QCA::SymmetricKey(QByteArray(password.toUtf8().constData())));
}

void EntryStorage::loadAndDecryptData(QString password, EntryListModel *model){
    QByteArray encryptedData;

    setPassword(password);

#ifdef MEEGO_EDITION_HARMATTAN
    void *raw_data;
    size_t length;

    if(! aegisStorage->contains_file(ENCRYPTED_FILE)){
        qDebug("New file... Creating empty list.");
        emit newFileOpened();
        return;
    }

    int ret = aegisStorage->get_file(ENCRYPTED_FILE, &raw_data, &length);
    if(ret != 0){
        QString msg;
        QTextStream(&msg) << "Error opening encrypted file from storage: " << ret;
        qErrnoWarning(msg.toUtf8().constData());
        emit operationFailed(msg);
        return;
    }else if(length == 0){
        qDebug("Empty file...");
        emit newFileOpened();
        return;
    }

    encryptedData = QByteArray((char *) raw_data, length);
    qDebug("Read %d bytes of data. Going to continue with decryption process.", length);
#else
    qDebug("Opening storage file for reading...");
    QString storagePath = QDir::homePath() + "/." + DEFAULT_STORAGE + ENCRYPTED_FILE;
    qDebug("Using file: %s", storagePath.toUtf8().constData());
    QFile storageFile(storagePath);

    if(! storageFile.exists()){
        qDebug("Seems we have a new file...");
        emit newFileOpened();
        return;
    }

    if(storageFile.open(QIODevice::ReadOnly)){
        encryptedData = storageFile.readAll();
        storageFile.close();
    }else{
        QString msg = "Failed to open storage file for reading...";
        qErrnoWarning(msg.toUtf8().constData());
        emit operationFailed(msg);
        return;
    }
    qDebug("File successfully read.");
    qDebug("Read %d bytes.", encryptedData.size());
#endif

    QCA::MemoryRegion tempInput(encryptedData);

    qDebug("Create cipher for decrypting.");
    QCA::Cipher cipher(CIPHER_TYPE, CIPHER_MODE, CIPHER_PADDING, QCA::Decode, *key);

    qDebug("Perform decryption.");
    QByteArray decrypted = cipher.process(tempInput).toByteArray();
    if(!cipher.ok()){
        QString msg = "Decryption failed.";
        qErrnoWarning(msg.toUtf8().constData());
        emit decryptionFailed();
        return;
    }
    qDebug("Successfully decrypted.");

    qDebug("Size of decrypted data: %d", decrypted.size());
    qDebug("Going to add entries to model...");
    model->addFromByteArray(decrypted);

    qDebug("Emitting decryptionSuccess() signal...");
    emit decryptionSuccess();
}

void EntryStorage::openStorage(){
#ifdef MEEGO_EDITION_HARMATTAN
    aegisStorage = new aegis::storage(DEFAULT_STORAGE, NULL, aegis::storage::vis_private, aegis::storage::prot_encrypted);

    if(aegisStorage->status() == aegis::storage::writable){
        qDebug("Success opening storage for writing.");

        if(aegisStorage->contains_file(ENCRYPTED_FILE)){
            emit storageOpenSuccess();
        }else{
            emit storageOpenSuccessNewPassword();
        }
    }else{
        QString msg;
        QTextStream(&msg) << "Failed to open storage for writing. Status is: " << aegisStorage->status();
        qErrnoWarning(msg.toUtf8().constData());

        delete aegisStorage;
        aegisStorage = NULL;

        emit operationFailed(msg);
        return;
    }
#else
    QDir home = QDir::home();
    home.mkdir(QString(".") + QString(DEFAULT_STORAGE));

    QString storagePath = home.absolutePath() + QString("/.") + QString(DEFAULT_STORAGE) + QString(ENCRYPTED_FILE);
    qDebug("Path to storage file: %s", storagePath.toUtf8().constData());
    QFile storageFile(storagePath);
    if(storageFile.exists()){
        emit storageOpenSuccess();
    }else{
        emit storageOpenSuccessNewPassword();
    }
    if(storageFile.isOpen()){
        storageFile.close();
    }
#endif
}

void EntryStorage::setPassword(QString password){
    if(key != NULL){
        delete key;
    }
    qDebug("Creating symmetric key.");
    key = new QCA::SymmetricKey(QByteArray(password.toUtf8().constData()));
}


void EntryStorage::storeModel(EntryListModel *model){
    qDebug("Storing model...");

    qDebug("Get data from model.");
    qDebug("Entries in model: %d", model->rowCount());
    QCA::MemoryRegion data(model->toByteArray());

    qDebug("Create cipher.");
    QCA::Cipher cipher(CIPHER_TYPE, CIPHER_MODE, CIPHER_PADDING, QCA::Encode, *key);

    qDebug("Encrypt data.");
    QByteArray encrypted = cipher.process(data).toByteArray();
    if(!cipher.ok()){
        QString msg = "Encryption failed.";
        qErrnoWarning(msg.toUtf8().constData());
        emit operationFailed(msg);
        return;
    }
    qDebug("Successfully encrypted data.");

#ifdef MEEGO_EDITION_HARMATTAN
    int ret = aegisStorage->put_file(ENCRYPTED_FILE, encrypted.data(), encrypted.size());
    qDebug("Writing file to storage returned: %d", ret);

    if(!aegisStorage->commit()){
        QString msg = "Error commiting changes to storage.";
        qErrnoWarning(msg.toUtf8().constData());
        emit operationFailed(msg);
        return;
    }
    qDebug("Changes successfully commited to storage.");
#else
    qDebug("Opening storage file for writing...");
    QFile storageFile(QDir::homePath() + QString("/.") + QString(DEFAULT_STORAGE) + QString(ENCRYPTED_FILE));
    if(storageFile.isOpen() || storageFile.open(QIODevice::ReadWrite)){
        storageFile.resize(0);
        int count = storageFile.write(encrypted);
        qDebug("%d bytes written.", count);
        storageFile.close();
    }else{
        QString msg = "Failed to open storage file for writing...";
        qErrnoWarning(msg.toUtf8().constData());
        emit operationFailed(msg);
        return;
    }
    qDebug("File successfully written.");
#endif
}
