/* User interface for syntax selection.

   Copyright (C) 2005, 2006 Leonard den Ottolander <leonard den ottolander nl>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/** \file
 *  \brief Source: user %interface for syntax %selection
 *  \author Leonard den Ottolander
 *  \date 2005, 2006
 */

#include <config.h>

#include <stdlib.h>
#include <sys/types.h>

#include "lib/global.h"
#include "src/wtools.h"

#include "edit-impl.h"
#include "edit-widget.h"

#define MAX_ENTRY_LEN 40
#define LIST_LINES 14
#define N_DFLT_ENTRIES 2

static int
pstrcmp (const void *p1, const void *p2)
{
    return strcmp (*(char **) p1, *(char **) p2);
}

static int
exec_edit_syntax_dialog (const char **names, const char *current_syntax)
{
    size_t i;

    Listbox *syntaxlist = create_listbox_window (LIST_LINES, MAX_ENTRY_LEN,
                                                 _("Choose syntax highlighting"), NULL);
    LISTBOX_APPEND_TEXT (syntaxlist, 'A', _("< Auto >"), NULL);
    LISTBOX_APPEND_TEXT (syntaxlist, 'R', _("< Reload Current Syntax >"), NULL);

    for (i = 0; names[i] != NULL; i++)
    {
        LISTBOX_APPEND_TEXT (syntaxlist, 0, names[i], NULL);
        if ((current_syntax != NULL) && (strcmp (names[i], current_syntax) == 0))
            listbox_select_entry (syntaxlist->list, i + N_DFLT_ENTRIES);
    }

    return run_listbox (syntaxlist);
}

void
edit_syntax_dialog (WEdit * edit, const char *current_syntax)
{
    char *old_syntax_type;
    int old_auto_syntax, syntax;
    char **names;
    gboolean force_reload = FALSE;
    size_t count;

    names = g_new0 (char *, 1);

    /* We fill the list of syntax files every time the editor is invoked.
       Instead we could save the list to a file and update it once the syntax
       file gets updated (either by testing or by explicit user command). */
    edit_load_syntax (NULL, &names, NULL);
    count = g_strv_length (names);
    qsort (names, count, sizeof (char *), pstrcmp);

    syntax = exec_edit_syntax_dialog ((const char **) names, current_syntax);
    if (syntax < 0)
    {
        g_strfreev (names);
        return;
    }

    old_auto_syntax = option_auto_syntax;
    old_syntax_type = g_strdup (current_syntax);

    switch (syntax)
    {
    case 0:                    /* auto syntax */
        option_auto_syntax = 1;
        break;
    case 1:                    /* reload current syntax */
        force_reload = TRUE;
        break;
    default:
        option_auto_syntax = 0;
        g_free (edit->syntax_type);
        edit->syntax_type = g_strdup (names[syntax - N_DFLT_ENTRIES]);
    }

    /* Load or unload syntax rules if the option has changed */
    if ((option_auto_syntax && !old_auto_syntax) || old_auto_syntax ||
        (old_syntax_type && edit->syntax_type &&
         (strcmp (old_syntax_type, edit->syntax_type) != 0)) || force_reload)
        edit_load_syntax (edit, NULL, edit->syntax_type);

    g_strfreev (names);
    g_free (old_syntax_type);
}
