//Maemo Barcode Reader and Interpreter (mbarcode or maemo-barcode)
//Copyright (C) 2010 Simon Pickering
//Copyright (C) 2010 Svenn-Arne Dragly
//
//Some source code obtained from other individuals/companies (not affiliated with the project), such as
//Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
//
//Various parts of barcode recognition and GStreamer manipulation code written by:
//      Timothy Terriberry
//      Adam Harwell
//      Jonas Hurrelmann
//
//Original GStreamer code based on the maemo-examples package:
//Copyright (C) 2007-2008 Nokia Corporation. All rights reserved.
//Copyright (C) 2006 INdT.
//@author Talita Menezes <talita.menezes@indt.org.br>
//@author Cidorvan Leite <cidorvan.leite@indt.org.br>
//@author Jami Pekkanen <jami.pekkanen@nokia.com>
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

// qt includes
#include <QWidget>
#include <QtGui/QPushButton>
#include <QMaemo5InformationBox>
#include <QTableView>
#include <QTableWidgetItem>
#include <QDebug>
#include <QFileDialog>
#include <QFileInfo>


// gstreamer includes
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>
// other includes
#include <stdio.h>
//local includes
#include "maemobarcodewindow.h"
#include "barcodedetector.h"
#include "videowidget.h"
#include "resultswindow.h"
#include "plugininterfaces.h"
#include "pluginaction.h"
#include "settingswindow.h"
#include "pluginsettings.h"
#include "recentdialog.h"
#include "pluginloaderthread.h"
#include <cmath>

extern QTime debugSpeedTimer;


const QDBusArgument & operator<<(QDBusArgument &arg, const Property &change)
{
        arg.beginStructure();
        arg << change.name << change.added << change.removed;
        arg.endStructure();
        return arg;
}
const QDBusArgument & operator>>(const QDBusArgument &arg, Property &change)
{
        arg.beginStructure();
        arg >> change.name >> change.added >> change.removed;
        arg.endStructure();
        return arg;
}

MaemoBarcodeWindow::MaemoBarcodeWindow(QWidget *parent)
    : QMainWindow(parent)
{
    qDebug() << "MainWindow::MainWindow(): Started MaemoBarcodeWindow";

    recentDialog = NULL;
    pluginsLoaded = false;

    imageBuffer = NULL;

    // first of all, disable the camera app:
    qDebug() << "MainWindow::MainWindow(): Killing the camera app, t=" << debugSpeedTimer.elapsed();
    QProcess process;
    process.execute("/usr/sbin/dsmetool -k /usr/bin/camera-ui");

    qDebug() << "MainWindow::MainWindow(): Setting window title, t=" << debugSpeedTimer.elapsed();
    setWindowTitle("Maemo Barcode Reader");

    // set up dbus
    qDebug() << "MainWindow::MainWindow(): Setting up dbus, t=" << debugSpeedTimer.elapsed();
    qDBusRegisterMetaType< Property >();
    qDBusRegisterMetaType< QList<Property> >();
    qDebug() << "MainWindow::MainWindow(): Setting up lensCoverPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_SHUTTER_STATE,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(lensCoverPropertyModified(int, QList<Property>)));
    qDebug() << "MainWindow::MainWindow(): Setting up focusButtonPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_FOCUS_BUTTON,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(focusButtonPropertyModified(int, QList<Property>)));
    qDebug() << "MainWindow::MainWindow(): Setting up cameraButtonPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_RELEASE_BUTTON,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(cameraButtonPropertyModified(int,QList<Property>)));

    // init the state of the lens cover
    qDebug() << "MainWindow::MainWindow(): Initing state of lens cover, t=" << debugSpeedTimer.elapsed();
    initLensCoverState();

    // make sure the events are not called to pause the pipeline unecessary in the beginning
    firstChangeEvent = true;
    secondChangeEvent = true;

    // makes sure we don't try to do anything with a pointer to nowhere (using null-checks)
    videoWidget = NULL;

    qDebug() << "MainWindow::MainWindow(): Creating settings, t=" << debugSpeedTimer.elapsed();
    settings = new QSettings("mbarcode", "mbarcode"); // load settings
    if(settings->value("settings_version", 1).toInt() < SETTINGS_VERSION) {
        settings->clear();
        settings->setValue("settings_version", SETTINGS_VERSION);
    }

    qDebug() << "MainWindow::MainWindow(): Initializing decoder, t=" << debugSpeedTimer.elapsed();
    decoder = new BarcodeDetector(); // create a decoder

    //qDebug() << "MainWindow::MainWindow(): Initializing results window, t=" << debugSpeedTimer.elapsed();
    //resultsWindow = new ResultsWindow(this); // create a resultswindow

    // set up the videoWidget
    qDebug() << "MainWindow::MainWindow(): Initializing video widget, t=" << debugSpeedTimer.elapsed();
    videoWidget = new VideoWidget(this, decoder);
    qDebug() << "MainWindow::MainWindow(): Setting video widget attributes, t=" << debugSpeedTimer.elapsed();
    videoWidget->setAttribute(Qt::WA_NativeWindow, true);
    videoWidget->setContinuousFocusTime((int)(settings->value("autofocus_delay",2.0).toFloat()*1000)); // pull from the settings
    qDebug() << "MainWindow::MainWindow(): Finished setting video widget attributes, t=" << debugSpeedTimer.elapsed();
    //    videoWidget->setMinimumSize(300, 200);

    ////videoWidget->setMinimumSize(640,400);

    // connect the signal from the barcode decoder
    qDebug() << "MainWindow::MainWindow(): Connecting signals, t=" << debugSpeedTimer.elapsed();
    connect(decoder, SIGNAL(imageAnalysed(QString, QString, QImage*)),
            this, SLOT(imageAnalysed(QString, QString, QImage*)));
    // connect signals to disable/enable the refocus button
    connect(videoWidget, SIGNAL(focusingValueChanged(bool)), this, SLOT(setFocusing(bool)));
    connect(videoWidget, SIGNAL(doSaveBuffer()), this, SLOT(saveBufferImage()));

    qDebug() << "MainWindow::MainWindow(): Initializing buttons, t=" << debugSpeedTimer.elapsed();
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true); // currently not working since libqt4-maemo5 is broken. Will work in PR1.2

    // if the lens cover is open, start scanning
    qDebug() << "MainWindow::MainWindow(): Checking lens cover state, t=" << debugSpeedTimer.elapsed();
    if (isLensCoverOpen()) {
        videoWidget->start(true); // start and keep running
    } else {
        // We don't need to do anything here;
        // wait until the lens cover is opened.
    }

    qDebug() << "MainWindow::MainWindow(): Setting central widget, t=" << debugSpeedTimer.elapsed();
    this->setCentralWidget(videoWidget);
    //qDebug() << "MainWindow::MainWindow(): starting pluginloader thread, t=" << debugSpeedTimer.elapsed();

    //pluginloaderthread = new PluginLoaderThread(this);
    //pluginloaderthread->start(QThread::LowPriority);

    qDebug() << "MainWindow::MainWindow(): Finished, t=" << debugSpeedTimer.elapsed();

    //QTimer::singleShot(2000, this, SLOT(loadPlugins())); // set 2s as a start delay time
}

MaemoBarcodeWindow::~MaemoBarcodeWindow() {
    // revive the camera app
    QProcess process;
    this->videoWidget->stopTimedScan();
    process.execute("/usr/sbin/dsmetool -t /usr/bin/camera-ui");

    // clean up any temp files
    QDir d(QDir::homePath() + "/.mbarcode/temp/");
    if(!d.exists(QDir::homePath() + "/.mbarcode/temp/"))
        return;

    foreach ( QString file, d.entryList( QDir::Files))
        QFile::remove(QFileInfo( d, file).absoluteFilePath());
}

void MaemoBarcodeWindow::showRecentDialog() {
    int result = getRecentDialog()->exec();
    this->videoWidget->stopTimedScan();
    if(result == QDialog::Accepted) {
        ensurePluginsLoaded();
        emit barcodeAnalysedSignal(getRecentDialog()->selectedBarcodeType(),
                                   getRecentDialog()->selectedBarcodeData());
        showResultsWindow(getRecentDialog()->selectedBarcodeData());
    }
}

void MaemoBarcodeWindow::sendBarcodeAnalysedSignal(QString barcodeType, QString barcodeData) {
    ensurePluginsLoaded();
    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)
}

void MaemoBarcodeWindow::lensCoverPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_SHUTTER_STATE,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    lensCoverOpen = !propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::lensCoverPropertyModified(): is" << lensCoverOpen;
    if(lensCoverOpen && isActiveWindow()) {
        videoWidget->start(false);
    } else {
        this->videoWidget->stopTimedScan();
        videoWidget->stop(false);
    }
}

void MaemoBarcodeWindow::focusButtonPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_FOCUS_BUTTON,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    bool buttonPressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::focusButtonPropertyModified(): is" << buttonPressed;
    if(buttonPressed) {
        videoWidget->refocus();
    }
}

void MaemoBarcodeWindow::cameraButtonPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_RELEASE_BUTTON,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    bool buttonPressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::cameraButtonPropertyModified(: is" << buttonPressed;
    if(this->continuousScanningMode==true){
        if(buttonPressed) {
            saveBufferImage();
        }
    }else{
        if(buttonPressed) {
            this->videoWidget->startTimedScan(this->videoWidget->timedScanInterval);
        }
    }
}

void MaemoBarcodeWindow::changeEvent(QEvent *e) {
    if(firstChangeEvent) {
        firstChangeEvent = false;
        return;
    }
    if(secondChangeEvent) {
        secondChangeEvent = false;
        return;
    }
    qDebug() << "MaemoBarcodeWindow::changeEvent(): Event caught" << e->type();
    if(e->type() == QEvent::ActivationChange && isActiveWindow()) {
        qDebug() << "MaemoBarcodeWindow::changeEvent(): Is active";
        if(videoWidget != NULL && isLensCoverOpen()) {
            videoWidget->start(false);
        }
    } else {
        qDebug() << "MaemoBarcodeWindow::changeEvent(): Is not active window";
        if(videoWidget != NULL && videoWidget->isPipelineReady() && isLensCoverOpen()) {
            this->videoWidget->stopTimedScan();
            videoWidget->stop(false);
        }
    }
}

RecentDialog* MaemoBarcodeWindow::getRecentDialog()
{
    if (!recentDialog)
        recentDialog = new RecentDialog(this);
    return recentDialog;
}

void MaemoBarcodeWindow::ensurePluginsLoaded()
{
    if (!pluginsLoaded)
    {
        QMaemo5InformationBox::information (this, "Loading plugins (first time only)...", 3000);
        loadPlugins();
        //pluginloaderthread->wait();
        QMaemo5InformationBox::information (this, "Loaded plugins", 1);

        pluginsLoaded = true; // set this now so that if the timer elapses it will return and we can call loadPlugins() here
        //loadPlugins();

    }
}

void MaemoBarcodeWindow::loadPlugins() {
    // TODO: Create a system which lets the user enable/disable plugins. These should not be loaded.

    if (pluginsLoaded) // if we were too late and ensurePluginsLoaded() was called...
        return;

    qDebug() << "loadPlugins(): Initializing results window, t=" << debugSpeedTimer.elapsed();
    resultsWindow = new ResultsWindow(this); // create a resultswindow

    qApp->processEvents();

    qDebug() << "loadPlugins(): Loading plugins..., t=" << debugSpeedTimer.elapsed();

    //    pluginsDir = QDir(qApp->applicationDirPath());.
    QDir pluginsDir = QDir("/usr/share/mbarcode/");
    QDir pluginsDirHome = QDir(QDir::homePath() + "/.config/mbarcode/plugins");

    // some path stuff derived from Nokia's plugin tutorials - not really necessary in a Maemo app, but we'll just leave them here
#if defined(Q_OS_WIN)
    if (pluginsDir.dirName().toLower() == "debug" || pluginsDir.dirName().toLower() == "release")
        pluginsDir.cdUp();
#elif defined(Q_OS_MAC)
    if (pluginsDir.dirName() == "MacOS") {
        pluginsDir.cdUp();
        pluginsDir.cdUp();
        pluginsDir.cdUp();
    }
#endif
    pluginsDir.cd("plugins");
    QStringList allPluginFiles;

    foreach (QString fileName, pluginsDir.entryList(QDir::Files)) { // go through all files in the plugin dir
        allPluginFiles << pluginsDir.absoluteFilePath(fileName);
    }

    qApp->processEvents();

    foreach (QString fileName, pluginsDirHome.entryList(QDir::Files)) { // go through all files in the plugin dir in home
        allPluginFiles << pluginsDirHome.absoluteFilePath(fileName);
    }

    qApp->processEvents();

    QList<QString> origPluginNames;
    //QList<int> origPluginPriorities;
    QList<bool> origPluginEnabled;
    //QList<bool> origPluginPresent;

    int numPlugins = settings->beginReadArray("plugins");
    qDebug() << "loadPlugins(): Loading plugin info, found " << numPlugins;

    //int max_pri=-1;
    //int pri;

    if(numPlugins>0){
        for (int i=0; i<numPlugins; ++i){
            settings->setArrayIndex(i);

            origPluginNames.append(settings->value("pluginName").toString());
//            pri = settings->value("pluginPriority").toInt(); // read the saved priority

//            while(origPluginPriorities.contains(pri)) // keep incrementing this value until we find a slot that is unused
//                pri++;

//            if(pri>max_pri)
//                max_pri = pri; // save for later when we add new plugins

//            origPluginPriorities.append(pri );
            origPluginEnabled.append(settings->value("pluginEnabled").toBool());
            //origPluginPresent.append(false); // start with false, then we will look later on

            //qDebug() << "loadPlugins(): Loading plugin settings data for item " << settings->value("pluginName").toString() << " " << pri << " " << settings->value("pluginEnabled").toBool();
            qDebug() << "loadPlugins(): Loading plugin settings data for item " << settings->value("pluginName").toString() << settings->value("pluginEnabled").toBool();
        }
    }
    settings->endArray();

    this->NumEnabledPlugins=0;

    foreach (QString fileName, allPluginFiles) { // go through all files in the plugin dir
        qDebug(qPrintable("Loading " + fileName));
        QPluginLoader loader(fileName); // create a loader for the shared library
        QObject *plugin = loader.instance();  // load it as a QObject
        if (plugin) { // if it is a plugin (this fails if virtual functions are not implemented correct)
            qDebug(qPrintable("Loaded successfully: " + fileName));
            pluginFileNames += fileName; // list all our plugin file names
            PluginInterface* interface = qobject_cast<PluginInterface*>(plugin); // cast it to our kind of interface
            qDebug(qPrintable("Interface name: " + interface->getName()));
            pluginInterfaces.append(interface); // add this interface to our list of interfaces
            interface->initInterface(this); // initialize the plugin
            foreach(PluginAction* pluginAction, interface->getPluginActions()) { // get the list of sink plugins in the interface

                QString pluginIdentifier(interface->getName() + ": " + pluginAction->getName());
                // this matches the string shown in the pluginsettings window - see the initialisation fn

                // see if we know about this plugin
                int ind = origPluginNames.indexOf(pluginIdentifier);
                if (ind>-1){ // for some reason this is never true
                    // yes we know about this one

                    //pluginAction->setPriority(origPluginPriorities.value(ind));
                    pluginAction->setEnabled(origPluginEnabled.value(ind));

                    qDebug() << "loadPlugins(): Found plugin listed in settings " << pluginIdentifier << pluginAction->getEnabled();

                }else{
                    // no we don't know about this one

                    //pluginAction->setPriority(++max_pri);
                    pluginAction->setEnabled(true);

                    //qDebug() << "loadPlugins(): Found plugin not listed in settings " << pluginIdentifier << " " << max_pri << " true";
                    qDebug() << "loadPlugins(): Found plugin not listed in settings " << pluginIdentifier << " true";
                }

                if (pluginAction->getEnabled())
                    this->NumEnabledPlugins++;

                pluginActions.append(pluginAction); // add them to our list
            }
            qApp->processEvents();
        } else {
            qWarning(qPrintable("loadPlugins(): Could not load: " + fileName + " due to error: " + loader.errorString()));
        }
    }

    //qSort(pluginActions.begin(), pluginActions.end()); // sort the array, still a bit worried about this

    // now make the priorities list contiguous
    //for(int i=0; i<pluginActions.size(); ++i){
    //    pluginActions.at(i)->setPriority(i);
    //}

    qDebug() << "Done loading plugins, t=" << debugSpeedTimer.elapsed();
    resultsWindow->setPluginActions(pluginActions); // let the resultwindow have all our plugins

    pluginsLoaded = true;
}


void MaemoBarcodeWindow::dataAnalysed(QString datatype, QString barcodeData, QHash<QString, QString> data) {
    // this is a slot for all data which may come from other plugins

    // let's just send this data back to the other plugins
    // later on we should store this in a database which other plugins may use as lookup later

    ensurePluginsLoaded();
    emit dataAnalysedSignal(datatype, barcodeData, data);
}

void MaemoBarcodeWindow::imageAnalysed(QString barcodeType, QString barcodeData, QImage *im) {
    this->videoWidget->stopTimedScan();
    videoWidget->stop(false); // cancel scanning first

    ensurePluginsLoaded();
    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)

    // display the output (this is done before anything is added to the tblTillRoll - avoids flickering)
    showResultsWindow(barcodeData);

    getRecentDialog()->imageAnalysed(barcodeType, barcodeData, im);
}

void MaemoBarcodeWindow::showResultsWindow(QString barcodeData) {
    ensurePluginsLoaded(); // not needed as we call it earlier, put it back anyway just in case
    this->videoWidget->stopTimedScan();
    videoWidget->stop(false);
    resultsWindow->show();
    resultsWindow->processBarcode(barcodeData);
}
void MaemoBarcodeWindow::initLensCoverState() {
    QFile file("/sys/devices/platform/gpio-switch/cam_shutter/state");
    if (!file.open(QIODevice::ReadOnly))
        return;
    QTextStream stream(&file);
    QString state = stream.readLine();
    lensCoverOpen = (state == "open");
}

bool MaemoBarcodeWindow::isLensCoverOpen()
{
    return lensCoverOpen;
}

void MaemoBarcodeWindow::setFocusing(bool enabled) {
    focusing = enabled;
    //    ui->btnRefocus->setEnabled(!enabled); // if focusing, disable the Refocus button
}

void MaemoBarcodeWindow::showSettingsWindow()
{
    this->videoWidget->stopTimedScan();
    videoWidget->stop(false);
    qDebug() << "MainWindow::showSettingsWindow(): Creating settings window";
    ensurePluginsLoaded();
    SettingsWindow *settingsWindow = new SettingsWindow(this);
    settingsWindow->show();
}

void MaemoBarcodeWindow::OpenAndScanImage()
{

    this->videoWidget->stopTimedScan();
    videoWidget->stop(false);

    QString fileName = QFileDialog::getOpenFileName(this, tr("Open Image"), "/home/user/MyDocs/.images/", tr("Image Files (*.png *.jpg *.bmp *.gif *.tif*)"));

    // check this exists
    QFileInfo qfi = QFileInfo(fileName);
    if(!qfi.exists())
        return;

    QImage img(qfi.absoluteFilePath());


    // see if we are too large for the thread buffers
    if (img.width()>MAX_VIDEO_WIDTH && img.height()>MAX_VIDEO_HEIGHT) // both too big, so see which one we should scale to bring both down to correct size
        if (float(img.width())/float(MAX_VIDEO_WIDTH) > float(img.height())/float(MAX_VIDEO_HEIGHT))
            img = img.scaledToWidth(MAX_VIDEO_WIDTH,Qt::SmoothTransformation);
        else
            img = img.scaledToWidth(MAX_VIDEO_HEIGHT,Qt::SmoothTransformation);
    else if (img.width()>MAX_VIDEO_WIDTH)
        img = img.scaledToWidth(MAX_VIDEO_WIDTH,Qt::SmoothTransformation);
    else if (img.height()>MAX_VIDEO_HEIGHT)
        img = img.scaledToHeight(MAX_VIDEO_HEIGHT,Qt::SmoothTransformation);

    // convert to an appropriate format
    //img = img.convertToFormat(QImage::Format_RGB888);
    img = img.convertToFormat(QImage::Format_RGB888);

    char fourcc[5] = "RGB3";
    int bpp = 24;
//    char fourcc[5] = "Y800";
//    int bpp = 8;



    int padding;

    // we'll pad the image to a 4byte multiple of width
    // this is necessary for the libdmtx code which converts the buffer into a QImage to change its format
    // it also means we can then look at the returned buffer more easily too!

    if(img.bytesPerLine() == bpp/8*img.width()){
        // we're already a round number, so we're fine
        padding = 0;
    }else{            
        padding=0;
        int remainingBytes = img.bytesPerLine()-bpp/8*img.width();
        while( ( padding*bpp/8 - remainingBytes) % 4)
            padding++; // iterate until we've added enough padding pixels to be divisible by 4 (bytes)

        qDebug() << "bytesPerLine=" << img.bytesPerLine() << "bpp=" << bpp << "remainingBytes=" << remainingBytes << "padding=" << padding << "old width" << img.width() << "new width=" << img.width()+padding;
    }
    int width = img.width()+padding; // this is the padded width



    // the line data in the QImage are aligned on 32bit boundaries
    // but the decoder expects a contiguous chunk of data
    // so we need to generate that
    if(imageBuffer)
        free(imageBuffer);

    imageBuffer = (uchar*)malloc(width*img.height()*bpp/8*sizeof(char));
    if(!imageBuffer)
        return; // memory allocation error


    uchar* pos = imageBuffer;
    for(int i=0; i<img.height(); ++i){
        memcpy(pos, img.constScanLine(i), img.width()*bpp/8); //img.bytesPerLine()
        pos = pos + img.width()*bpp/8;

        // add padding pixels
        for(int j =0; j<padding; ++j){ // iterate through the padding pixels
            for(int k=0;k<bpp/8; ++k){ // iterate though the components of the pixel
                *pos = 0; // give the pixel component a value
                pos++; // move to the next one
            }
        }
    }

//    qDebug() << "Image data w=" << img.width() << "height " << img.height() << "bytes per line " << img.bytesPerLine() << "bytecount " << img.byteCount();

//    // debugging, as we don't seem to be able to decode
//    QFile file("/home/user/mbarcode-image.raw");
//    file.open(QIODevice::WriteOnly);
//    file.write((const char*)(imageBuffer), img.height()*width*bpp/8);
//    file.close();

//    img.save("/home/user/mbarcode-image.png");



    int type = decoder->analyseImageSync((char *)imageBuffer, width, img.height(), fourcc, bpp, 1000, this, SLOT(analyseImageSyncCallBack()));

    free(imageBuffer);
    imageBuffer=0;
}

void MaemoBarcodeWindow::analyseImageSyncCallBack(){
    decoder->scanning_status=false; // stop decoder threads if they've not stopped already
    int result = decoder->getAnalysedImageResult();

    if(imageBuffer)
        free(imageBuffer);

    qDebug() << "In MaemoBarcodeWindow::analyseImageSyncCallBack()";

    if(result)
        QMaemo5InformationBox::information (this, "Barcode detected", 1000);
    else
        QMaemo5InformationBox::information (this, "No barcode detected", 1000);

}

//void MaemoBarcodeWindow::on_actionPlugins_triggered()
//{
//    videoWidget->stop(false);
//    PluginSettings* win = new PluginSettings(pluginInterfaces, this);
//    win->show();
//}

//void MaemoBarcodeWindow::on_actionTest_portrait_mode_triggered()
//{
//    if(settings->value("portrait_mode", false).toBool()) {
//        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
//        settings->setValue("portrait_mode", false);
//    } else {
//        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
//        settings->setValue("portrait_mode", true);
//    }
//}

void MaemoBarcodeWindow::saveBufferImage()
{
    this->videoWidget->stopTimedScan();
    videoWidget->stop(false);

    QImage *im = decoder->getLastImage();
    im->convertToFormat(QImage::Format_RGB32);

    // save this file
    // pop-up a save dialog
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save current image buffer to file"),
                                                    "/home/user/MyDocs/.images/barcode.jpg",
                                                    tr("JPEG Image (*.jpg)"));

    QFileInfo finfo(fileName);
    //if (finfo.exists())
    //    return; // should at least offer a new filename I suppose...

    // see if the extension survived....
    if(finfo.completeSuffix().isEmpty())
        fileName.append(".jpg");

    im->save(fileName, 0, 95);

}

void MaemoBarcodeWindow::keyPressEvent(QKeyEvent *event)
{
    qDebug() << "Inside MaemoBarcodeWindow keypress";

    if(event->key()==32){ // look for space
        videoWidget->refocus();

        if(this->continuousScanningMode==true){
            saveBufferImage();
        }else{
            this->videoWidget->startTimedScan(this->videoWidget->timedScanInterval + 2000);
            // give it 2 more seconds of scanning time to account for the fact that focusing must also occur
        }
    }

    event->ignore(); // let it continue to wherever it might want to go
}

//// Use text files instead, then an app can provide a text file with its DBus data when installed
//int MainWindow::register_external_plugin(QString appname, QString dbus_command_to_call, QString RequiredBarcodeTypes){
//    // we will be called by external programs over DBus to register them as being external plugins
//    // make a list somewhere
//
//}



