//Maemo Barcode Reader and Interpreter (mbarcode or maemo-barcode)
//Copyright (C) 2010 Simon Pickering
//Copyright (C) 2010 Svenn-Arne Dragly
//
//Some source code obtained from other individuals/companies (not affiliated with the project), such as
//Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
//
//Various parts of barcode recognition and GStreamer manipulation code written by:
//      Timothy Terriberry
//      Adam Harwell
//      Jonas Hurrelmann
//
//Original GStreamer code based on the maemo-examples package:
//Copyright (C) 2007-2008 Nokia Corporation. All rights reserved.
//Copyright (C) 2006 INdT.
//@author Talita Menezes <talita.menezes@indt.org.br>
//@author Cidorvan Leite <cidorvan.leite@indt.org.br>
//@author Jami Pekkanen <jami.pekkanen@nokia.com>
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

// qt includes
#include <QWidget>
#include <QtGui/QPushButton>
#include <QMaemo5InformationBox>
#include <QTableView>
#include <QTableWidgetItem>
#include <QDebug>
#include <QFileDialog>
#include <QFileInfo>
// gstreamer includes
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>
// other includes
#include <stdio.h>
//local includes
#include "maemobarcodewindow.h"
#include "barcodedetector.h"
#include "videowidget.h"
#include "resultswindow.h"
#include "plugininterfaces.h"
#include "pluginaction.h"
#include "settingswindow.h"
#include "pluginsettings.h"
#include "recentdialog.h"
#include "pluginloaderthread.h"


const QDBusArgument & operator<<(QDBusArgument &arg, const Property &change)
{
        arg.beginStructure();
        arg << change.name << change.added << change.removed;
        arg.endStructure();
        return arg;
}
const QDBusArgument & operator>>(const QDBusArgument &arg, Property &change)
{
        arg.beginStructure();
        arg >> change.name >> change.added >> change.removed;
        arg.endStructure();
        return arg;
}

MaemoBarcodeWindow::MaemoBarcodeWindow(QWidget *parent)
    : QMainWindow(parent)
{
    qDebug() << "MainWindow::MainWindow(): Started MaemoBarcodeWindow";

    recentDialog = NULL;
    pluginsLoaded = false;

    // first of all, disable the camera app:
    qDebug() << "MainWindow::MainWindow(): Killing the camera app";
    QProcess process;
    process.execute("/usr/sbin/dsmetool -k /usr/bin/camera-ui");

    qDebug() << "MainWindow::MainWindow(): Setting window title";
    setWindowTitle("Maemo Barcode Reader");

    // set up dbus
    qDebug() << "MainWindow::MainWindow(): Setting up dbus";
    qDBusRegisterMetaType< Property >();
    qDBusRegisterMetaType< QList<Property> >();
    qDebug() << "MainWindow::MainWindow(): Setting up lensCoverPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_SHUTTER_STATE,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(lensCoverPropertyModified(int, QList<Property>)));
    qDebug() << "MainWindow::MainWindow(): Setting up focusButtonPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_FOCUS_BUTTON,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(focusButtonPropertyModified(int, QList<Property>)));
    qDebug() << "MainWindow::MainWindow(): Setting up cameraButtonPropertyModified";
    QDBusConnection::systemBus().connect(QString(),
            DBUS_RELEASE_BUTTON,
            "org.freedesktop.Hal.Device", "PropertyModified",
            this, SLOT(cameraButtonPropertyModified(int,QList<Property>)));

    // init the state of the lens cover
    qDebug() << "MainWindow::MainWindow(): Initing state of lens cover";
    initLensCoverState();

    // make sure the events are not called to pause the pipeline unecessary in the beginning
    firstChangeEvent = true;
    secondChangeEvent = true;

    // makes sure we don't try to do anything with a pointer to nowhere (using null-checks)
    videoWidget = NULL;

    qDebug() << "MainWindow::MainWindow(): Creating settings";
    settings = new QSettings("mbarcode", "mbarcode"); // load settings
    if(settings->value("settings_version", 1).toInt() < SETTINGS_VERSION) {
        settings->clear();
        settings->setValue("settings_version", SETTINGS_VERSION);
    }

    qDebug() << "MainWindow::MainWindow(): Initializing decoder";
    decoder = new BarcodeDetector(); // create a decoder

    qDebug() << "MainWindow::MainWindow(): Initializing results window";
    resultsWindow = new ResultsWindow(this); // create a resultswindow

    // set up the videoWidget
    qDebug() << "MainWindow::MainWindow(): Initializing video widget";
    videoWidget = new VideoWidget(this, decoder);
    qDebug() << "MainWindow::MainWindow(): Setting video widget attributes";
    videoWidget->setAttribute(Qt::WA_NativeWindow, true);
    videoWidget->setContinuousFocusTime((int)(settings->value("autofocus_delay",2.0).toFloat()*1000)); // pull from the settings
    qDebug() << "MainWindow::MainWindow(): Finished setting video widget attributes";
    //    videoWidget->setMinimumSize(300, 200);

    ////videoWidget->setMinimumSize(640,400);


    if(settings->value("continous_scanning").toBool()){
        videoWidget->setContinuousScanning(true);
        this->continuousScanningMode = true;
    }else{
        videoWidget->setContinuousScanning(false);
        this->continuousScanningMode = false;
    }

    // connect the signal from the barcode decoder
    qDebug() << "MainWindow::MainWindow(): Connecting signals";
    connect(decoder, SIGNAL(imageAnalysed(QString, QString, QImage*)),
            this, SLOT(imageAnalysed(QString, QString, QImage*)));
    // connect signals to disable/enable the refocus button
    connect(videoWidget, SIGNAL(focusingValueChanged(bool)), this, SLOT(setFocusing(bool)));
    connect(videoWidget, SIGNAL(doSaveBuffer()), this, SLOT(saveBufferImage()));

    qDebug() << "MainWindow::MainWindow(): Initializing buttons";
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true); // currently not working since libqt4-maemo5 is broken. Will work in PR1.2

    // if the lens cover is open, start scanning
    qDebug() << "MainWindow::MainWindow(): Checking lens cover state";
    if (isLensCoverOpen()) {
        videoWidget->start(true); // start and keep running
    } else {
        // We don't need to do anything here;
        // wait until the lens cover is opened.
    }

    qDebug() << "MainWindow::MainWindow(): Setting central widget";
    this->setCentralWidget(videoWidget);
    qDebug() << "MainWindow::MainWindow(): Finished";

    pluginloaderthread = new PluginLoaderThread(this);
    pluginloaderthread->run();
}

MaemoBarcodeWindow::~MaemoBarcodeWindow() {
    // revive the camera app
    QProcess process;
    process.execute("/usr/sbin/dsmetool -t /usr/bin/camera-ui");

    // clean up any temp files
    QDir d(QDir::homePath() + "/.mbarcode/temp/");
    if(!d.exists(QDir::homePath() + "/.mbarcode/temp/"))
        return;

    foreach ( QString file, d.entryList( QDir::Files))
        QFile::remove(QFileInfo( d, file).absoluteFilePath());
}

void MaemoBarcodeWindow::showRecentDialog() {
    int result = getRecentDialog()->exec();
    if(result == QDialog::Accepted) {
        ensurePluginsLoaded();
        emit barcodeAnalysedSignal(getRecentDialog()->selectedBarcodeType(),
                                   getRecentDialog()->selectedBarcodeData());
        showResultsWindow(getRecentDialog()->selectedBarcodeData());
    }
}

void MaemoBarcodeWindow::sendBarcodeAnalysedSignal(QString barcodeType, QString barcodeData) {
    ensurePluginsLoaded();
    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)
}

void MaemoBarcodeWindow::lensCoverPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_SHUTTER_STATE,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    lensCoverOpen = !propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::lensCoverPropertyModified(): is" << lensCoverOpen;
    if(lensCoverOpen && isActiveWindow()) {
        videoWidget->start(false);
    } else {
        videoWidget->stop(false);
    }
}

void MaemoBarcodeWindow::focusButtonPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_FOCUS_BUTTON,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    bool buttonPressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::focusButtonPropertyModified(): is" << buttonPressed;
    if(buttonPressed) {
        videoWidget->refocus();
    }
}

void MaemoBarcodeWindow::cameraButtonPropertyModified(int num_updates, QList<Property> updates) {
    Q_UNUSED(num_updates)
    Q_UNUSED(updates)
    QDBusInterface propertyInterface("org.freedesktop.Hal",
                DBUS_RELEASE_BUTTON,
                "org.freedesktop.Hal.Device",
                QDBusConnection::systemBus());
    bool buttonPressed = propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool();
    qDebug() << "MaemoBarcodeWindow::cameraButtonPropertyModified(: is" << buttonPressed;
    if(this->continuousScanningMode==true){
        if(buttonPressed) {
            saveBufferImage();
        }
    }else{
        if(buttonPressed) {
            this->videoWidget->startTimedScan(this->videoWidget->timedScanInterval);
        }
    }
}

void MaemoBarcodeWindow::changeEvent(QEvent *e) {
    if(firstChangeEvent) {
        firstChangeEvent = false;
        return;
    }
    if(secondChangeEvent) {
        secondChangeEvent = false;
        return;
    }
    qDebug() << "MaemoBarcodeWindow::changeEvent(): Event caught" << e->type();
    if(e->type() == QEvent::ActivationChange && isActiveWindow()) {
        qDebug() << "MaemoBarcodeWindow::changeEvent(): Is active";
        if(videoWidget != NULL && isLensCoverOpen()) {
            videoWidget->start(false);
        }
    } else {
        qDebug() << "MaemoBarcodeWindow::changeEvent(): Is not active window";
        if(videoWidget != NULL && videoWidget->isPipelineReady() && isLensCoverOpen()) {
            videoWidget->stop(false);
        }
    }
}

RecentDialog* MaemoBarcodeWindow::getRecentDialog()
{
    if (!recentDialog)
        recentDialog = new RecentDialog(this);
    return recentDialog;
}

void MaemoBarcodeWindow::ensurePluginsLoaded()
{
    if (!pluginsLoaded)
    {
        //loadPlugins();
        pluginloaderthread->wait();
        //pluginsLoaded = true;
    }
}

//void MaemoBarcodeWindow::loadPlugins() {
//    // TODO: Create a system which lets the user enable/disable plugins. These should not be loaded.
//
//    qDebug("Loading plugins...");
//
//    //    pluginsDir = QDir(qApp->applicationDirPath());.
//    QDir pluginsDir = QDir("/usr/share/mbarcode/");
//    QDir pluginsDirHome = QDir(QDir::homePath() + "/.config/mbarcode/plugins");
//
//    // some path stuff derived from Nokia's plugin tutorials - not really necessary in a Maemo app, but we'll just leave them here
//#if defined(Q_OS_WIN)
//    if (pluginsDir.dirName().toLower() == "debug" || pluginsDir.dirName().toLower() == "release")
//        pluginsDir.cdUp();
//#elif defined(Q_OS_MAC)
//    if (pluginsDir.dirName() == "MacOS") {
//        pluginsDir.cdUp();
//        pluginsDir.cdUp();
//        pluginsDir.cdUp();
//    }
//#endif
//    pluginsDir.cd("plugins");
//    QStringList allPluginFiles;
//
//    foreach (QString fileName, pluginsDir.entryList(QDir::Files)) { // go through all files in the plugin dir
//        allPluginFiles << pluginsDir.absoluteFilePath(fileName);
//    }
//    foreach (QString fileName, pluginsDirHome.entryList(QDir::Files)) { // go through all files in the plugin dir in home
//        allPluginFiles << pluginsDirHome.absoluteFilePath(fileName);
//    }
//    foreach (QString fileName, allPluginFiles) { // go through all files in the plugin dir
//        qDebug(qPrintable("Loading " + fileName));
//        QPluginLoader loader(fileName); // create a loader for the shared library
//        QObject *plugin = loader.instance();  // load it as a QObject
//        if (plugin) { // if it is a plugin (this fails if virtual functions are not implemented correct)
//            qDebug(qPrintable("Loaded successfully: " + fileName));
//            pluginFileNames += fileName; // list all our plugin file names
//            PluginInterface* interface = qobject_cast<PluginInterface*>(plugin); // cast it to our kind of interface
//            qDebug(qPrintable("Interface name: " + interface->getName()));
//            pluginInterfaces.append(interface); // add this interface to our list of interfaces
//            interface->initInterface(this); // initialize the plugin
//            foreach(PluginAction* pluginAction, interface->getPluginActions()) { // get the list of sink plugins in the interface
//                qDebug(qPrintable("Loading sink plugin " + pluginAction->getName()));
//                pluginActions.append(pluginAction); // add them to our list
//            }
//        } else {
//            qWarning(qPrintable("Could not load: " + fileName));
//        }
//    }
//    qDebug("Done loading plugins");
//    resultsWindow->setPluginActions(pluginActions); // let the resultwindow have all our plugins
//}

//void MaemoBarcodeWindow::on_btnScan_clicked()
//{
//    startScan();
//}

void MaemoBarcodeWindow::on_btnCancel_clicked()
{
    videoWidget->stop(true);
}

void MaemoBarcodeWindow::on_btnRefocus_clicked()
{
    videoWidget->refocus();
}

void MaemoBarcodeWindow::dataAnalysed(QString datatype, QString barcodeData, QHash<QString, QString> data) {
    // this is a slot for all data which may come from other plugins

    // let's just send this data back to the other plugins
    // later on we should store this in a database which other plugins may use as lookup later

    ensurePluginsLoaded();
    emit dataAnalysedSignal(datatype, barcodeData, data);
}

void MaemoBarcodeWindow::imageAnalysed(QString barcodeType, QString barcodeData, QImage *im) {
    videoWidget->stop(false); // cancel scanning first

    ensurePluginsLoaded();
    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)

    // display the output (this is done before anything is added to the tblTillRoll - avoids flickering)
    showResultsWindow(barcodeData);

    getRecentDialog()->imageAnalysed(barcodeType, barcodeData, im);
}

void MaemoBarcodeWindow::showResultsWindow(QString barcodeData) {
    ensurePluginsLoaded();
    resultsWindow->show();
    resultsWindow->processBarcode(barcodeData);
}
void MaemoBarcodeWindow::initLensCoverState() {
    QFile file("/sys/devices/platform/gpio-switch/cam_shutter/state");
    if (!file.open(QIODevice::ReadOnly))
        return;
    QTextStream stream(&file);
    QString state = stream.readLine();
    lensCoverOpen = (state == "open");
}

bool MaemoBarcodeWindow::isLensCoverOpen()
{
    return lensCoverOpen;
}

void MaemoBarcodeWindow::setFocusing(bool enabled) {
    focusing = enabled;
    //    ui->btnRefocus->setEnabled(!enabled); // if focusing, disable the Refocus button
}

void MaemoBarcodeWindow::showSettingsWindow()
{
    videoWidget->stop(false);
    qDebug() << "MainWindow::showSettingsWindow(): Creating settings window";
    SettingsWindow *settingsWindow = new SettingsWindow(this);
    settingsWindow->show();
}

void MaemoBarcodeWindow::on_btnOpenImage_clicked()
{
    QString fileName = QFileDialog::getOpenFileName(this, tr("Open Image"), "/home/user/MyDocs/.images/", tr("Image Files (*.png *.jpg *.bmp *.gif *.tif*)"));

    // check this exists
    QFileInfo qfi = QFileInfo(fileName);
    if(!qfi.exists())
        return;

    //videoWidget->useFilesrc(fileName);
    //videoWidget->start();

    // but how do we stop the pipeline? Or does it stop itself once it's processed the image, probably

}

void MaemoBarcodeWindow::on_actionPlugins_triggered()
{
    videoWidget->stop(false);
    PluginSettings* win = new PluginSettings(pluginInterfaces, this);
    win->show();
}

void MaemoBarcodeWindow::on_actionTest_portrait_mode_triggered()
{
    if(settings->value("portrait_mode", false).toBool()) {
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        settings->setValue("portrait_mode", false);
    } else {
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        settings->setValue("portrait_mode", true);
    }
}

void MaemoBarcodeWindow::saveBufferImage()
{
    videoWidget->stop(false);

    QImage *im = decoder->getLastImage();
    im->convertToFormat(QImage::Format_RGB32);

    // save this file
    // pop-up a save dialog
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save current image buffer to file"),
                                                    "/home/user/MyDocs/.images/barcode.jpg",
                                                    tr("JPEG Image (*.jpg)"));

    QFileInfo finfo(fileName);
    //if (finfo.exists())
    //    return; // should at least offer a new filename I suppose...

    // see if the extension survived....
    if(finfo.completeSuffix().isEmpty())
        fileName.append(".jpg");

    im->save(fileName, 0, 95);

}


//// Use text files instead, then an app can provide a text file with its DBus data when installed
//int MainWindow::register_external_plugin(QString appname, QString dbus_command_to_call, QString RequiredBarcodeTypes){
//    // we will be called by external programs over DBus to register them as being external plugins
//    // make a list somewhere
//
//}



