//Maemo Barcode Reader and Interpreter (mbarcode or maemo-barcode)
//Copyright (C) 2010 Simon Pickering
//Copyright (C) 2010 Svenn-Arne Dragly
//
//Some source code obtained from other individuals/companies (not affiliated with the project), such as
//Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
//
//Various parts of barcode recognition and GStreamer manipulation code written by:
//      Timothy Terriberry
//      Adam Harwell
//      Jonas Hurrelmann
//
//Original GStreamer code based on the maemo-examples package:
//Copyright (C) 2007-2008 Nokia Corporation. All rights reserved.
//Copyright (C) 2006 INdT.
//@author Talita Menezes <talita.menezes@indt.org.br>
//@author Cidorvan Leite <cidorvan.leite@indt.org.br>
//@author Jami Pekkanen <jami.pekkanen@nokia.com>
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

// qt includes
#include <QWidget>
#include <QtGui/QPushButton>
// gstreamer includes
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>
// other includes
#include <stdio.h>
//local includes
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "barcodedetector.h"
#include "videowidget.h"
#include "resultswindow.h"
#include "plugininterfaces.h"
#include "sinkplugin.h"
#include "settingswindow.h"
#include "pluginsettings.h"

#include <QTableView>
#include <QTableWidgetItem>
#include <QDebug>
#include <QFileDialog>
#include <QFileInfo>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindow)
{
    qDebug() << "MainWindow::MainWindow(): Setting up UI";
    ui->setupUi(this);
    qDebug() << "In MainWindow::MainWindow";
    settings = new QSettings("mbarcode", "mbarcode"); // load settings
    qDebug() << "MainWindow::MainWindow(): loading tillRollModel";
    tillRollModel = new QStandardItemModel(0,1);
    ui->tblTillRoll->setModel(tillRollModel);
    qDebug() << "MainWindow::MainWindow(): Initializing decoder";
    decoder = new BarcodeDetector(); // create a decoder
    qDebug() << "MainWindow::MainWindow(): Initializing results window";
    resultsWindow = new ResultsWindow(this); // create a resultswindow
    // set up the videoWidget
    qDebug() << "MainWindow::MainWindow(): Initializing video widget";
    videoWidget = new VideoWidget(this, decoder);
    videoWidget->setAttribute(Qt::WA_NativeWindow, true);
    ui->videoContainer->addWidget(videoWidget); // add the widget to the container
    // connect the signal from the barcode decoder
    connect(decoder, SIGNAL(imageAnalysed(QString, QString)),
            this, SLOT(imageAnalysed(QString, QString)));
    // connect signals to disable/enable the refocus button
    connect(videoWidget, SIGNAL(focusingValueChanged(bool)), this, SLOT(setFocusing(bool)));
    qDebug() << "MainWindow::MainWindow(): Setting tblTillRoll properties";
    ui->tblTillRoll->setProperty("FingerScrollable", true); // apparently this will change for 4.6
    //ui->tillRoll->setItemDelegate(new BarcodeTableItemDelegate);
    ui->tblTillRoll->setSelectionBehavior(QAbstractItemView::SelectRows);

    ui->tblTillRoll->verticalHeader()->setResizeMode(QHeaderView::ResizeToContents);
    ui->tblTillRoll->horizontalHeader()->setResizeMode(QHeaderView::ResizeToContents);
    qDebug() << "MainWindow::MainWindow(): Loading plugins";
    // load all plugins
    loadPlugins();
    qDebug() << "MainWindow::MainWindow(): Initializing buttons";
    // init our buttons
    ui->btnScan->setVisible(true);
    ui->btnCancel->setVisible(false);
    ui->btnRefocus->setVisible(false);
    ui->btnOpenImage->setVisible(true);
    bool fullscreenScan = settings->value("fullscreen_scan", false).toBool();
    if(fullscreenScan) {
        ui->tblTillRoll->setVisible(true);
        videoWidget->setVisible(false);
    } else {
        ui->tblTillRoll->setVisible(true);
        videoWidget->setVisible(true);
    }
    ui->menuMain->setTitle(""); // remove the title so we don't get options like "Menu / Settings" instead of "Settings"
    qDebug() << "MainWindow::MainWindow(): Finished";
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::loadPlugins() {
    // TODO: Create a system which lets the user enable/disable plugins. These should not be loaded.

    qDebug("Loading plugins...");

//    pluginsDir = QDir(qApp->applicationDirPath());.
    pluginsDir = QDir("/usr/share/mbarcode/");


    // some path stuff derived from Nokia's plugin tutorials - not really necessary in a Maemo app, but we'll just leave them here
#if defined(Q_OS_WIN)
     if (pluginsDir.dirName().toLower() == "debug" || pluginsDir.dirName().toLower() == "release")
         pluginsDir.cdUp();
#elif defined(Q_OS_MAC)
     if (pluginsDir.dirName() == "MacOS") {
         pluginsDir.cdUp();
         pluginsDir.cdUp();
         pluginsDir.cdUp();
     }
#endif
     pluginsDir.cd("plugins");

     foreach (QString fileName, pluginsDir.entryList(QDir::Files)) { // go through all files in the plugin dir
         qDebug(qPrintable("Loading " + fileName));
         QPluginLoader loader(pluginsDir.absoluteFilePath(fileName)); // create a loader for the shared library
         QObject *plugin = loader.instance();  // load it as a QObject
         if (plugin) { // if it is a plugin (this fails if virtual functions are not implemented correct)
             qDebug(qPrintable("Loaded successfully: " + fileName));
             pluginFileNames += fileName; // list all our plugin file names
             PluginInterface* interface = qobject_cast<PluginInterface*>(plugin); // cast it to our kind of interface
             qDebug(qPrintable("Interface name: " + interface->getName()));
             pluginInterfaces.insert(interface); // add this interface to our list of interfaces
             interface->initInterface(this); // initialize the plugin
             foreach(SinkPlugin* sinkPlugin, interface->getSinkPlugins()) { // get the list of sink plugins in the interface
                 qDebug(qPrintable("Loading sink plugin " + sinkPlugin->getName()));
                 sinkPlugins.insert(sinkPlugin); // add them to our list
             }
         } else {
             qWarning(qPrintable("Could not load: " + fileName));
         }
     }
     qDebug("Done loading plugins");
     resultsWindow->setSinkPlugins(sinkPlugins); // let the resultwindow have all our plugins
}

void MainWindow::on_btnScan_clicked()
{
    startScan();
}

void MainWindow::on_btnCancel_clicked()
{
    cancelScan();
}

void MainWindow::on_btnRefocus_clicked()
{
    videoWidget->refocus();
}

void MainWindow::cancelScan() {
    videoWidget->stop();
    ui->btnScan->setVisible(true);
    ui->btnCancel->setVisible(false);
    ui->btnRefocus->setVisible(false);
    ui->btnOpenImage->setVisible(true);
    bool fullscreenScan = settings->value("fullscreen_scan", false).toBool();
    if(fullscreenScan) {
        ui->tblTillRoll->setVisible(true);
        videoWidget->setVisible(false);
    } else { // must be set each time in case we've been in fullscreen mode in the meantime
        ui->tblTillRoll->setVisible(true);
        videoWidget->setVisible(true);
    }
}

void MainWindow::startScan() {
    //videoWidget->useVideosrc();
    videoWidget->start();
    ui->btnScan->setVisible(false);
    ui->btnRefocus->setVisible(true);
    ui->btnCancel->setVisible(true);
    ui->btnOpenImage->setVisible(false);
    bool fullscreenScan = settings->value("fullscreen_scan", false).toBool();
    if(fullscreenScan) {
        ui->tblTillRoll->setVisible(false);
        videoWidget->setVisible(true);
    } else { // must be set each time in case we've been in fullscreen mode in the meantime
        ui->tblTillRoll->setVisible(true);
        videoWidget->setVisible(true);
    }
}

void MainWindow::dataAnalysed(QString datatype, QString barcodeData, QHash<QString, QString> data) {
    // this is a slot for all data which may come from other plugins

    // let's just send this data back to the other plugins
    // later on we should store this in a database which other plugins may use as lookup later

    emit dataAnalysedSignal(datatype, barcodeData, data);
}

void MainWindow::imageAnalysed(QString barcodeType, QString barcodeData) {
    // this function is supposed to remove all previous results from the list (or at least all but the last three or something)
    // TODO: Fix the function so that is works...

    cancelScan(); // cancel scanning first

    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)

    // display the output (this is done before anything is added to the tblTillRoll - avoids flickering)
    showResultsWindow(barcodeData);

    qDebug() << "MainWindow::imageAnalysed(): tillRollModel->rowCount()=" << tillRollModel->columnCount();

    // see how many rows are in our table first
    if (tillRollModel->rowCount()>SCAN_HISTORY_LENGTH){
        // too many, so remove one
        qDebug("MainWindow::imageAnalysed(): Table too long, removing one item");
        tillRollModel->removeRow ( 0 );
    }


    // then we generate a TableItem and a TableItemWidget to render the data
    QStandardItem *barcodeItem = new QStandardItem(barcodeData);

    qDebug() << "MainWindow::imageAnalysed(): Created new item " << barcodeData;


    // now add our new data to the table
    //ui->tillRoll->insertColumn(ui->tillRoll->columnCount());
    //ui->tillRoll->setItem(0, ui->tillRoll->columnCount()-1, i);
    tillRollModel->insertRow(0); // insert at top
    tillRollModel->setItem(0, 0, barcodeItem);


    // create a widget to hold an image and text
    QWidget *w = new QWidget;
    w->setAutoFillBackground(true); // otherwise we'll see bits of the text contents

    QImage *im = decoder->getLastImage();
    QPixmap pm = QPixmap::fromImage(*im, Qt::AutoColor);

    pm = pm.scaledToWidth(300); // scale the image to 300 px wide
    // IMPORTANT: To change the size, you must also change the iconSize in mainwindow.ui for the table
    QIcon icon = QIcon(pm);

    tillRollModel->insertRow(0); // insert at top

    QStandardItem *imageItem = new QStandardItem(icon, "");
    imageItem->setSizeHint(QSize(300,200)); // make the image 300x200
    tillRollModel->setItem(0, 0, imageItem);

    qDebug() << "MainWindow::barcodeAnalysed(): Inserted new item. tillRollModel->rowCount()=" << tillRollModel->rowCount();

    //ui->tblTillRoll->resizeColumnsToContents();

    QVariantHash itemData;
    itemData.insert("barcodeType", barcodeType);
    itemData.insert("barcodeData", barcodeData);

    imageItem->setData(itemData);
    barcodeItem->setData(itemData);
}

void MainWindow::showResultsWindow(QString barcode) {
    resultsWindow->show();
    resultsWindow->processBarcode(barcode);
}
void MainWindow::setFocusing(bool enabled) {
    focusing = enabled;
    ui->btnRefocus->setEnabled(!enabled); // if focusing, disable the Refocus button
}

void MainWindow::on_actionSettings_triggered()
{
    cancelScan();
    qDebug() << "MainWindow::on_actionSettings_triggered(): Creating settings window";
    SettingsWindow *settingsWindow = new SettingsWindow(this);
    settingsWindow->setSettings(settings);
    settingsWindow->show();
}

void MainWindow::on_btnOpenImage_clicked()
{
    QString fileName = QFileDialog::getOpenFileName(this, tr("Open Image"), "/home/user/MyDocs/.images/", tr("Image Files (*.png *.jpg *.bmp *.gif *.tif*)"));

    // check this exists
    QFileInfo qfi = QFileInfo(fileName);
    if(!qfi.exists())
        return;

    //videoWidget->useFilesrc(fileName);
    //videoWidget->start();

    // but how do we stop the pipeline? Or does it stop itself once it's processed the image, probably

}

void MainWindow::on_tblTillRoll_clicked(QModelIndex index)
{
    QStandardItem *item = tillRollModel->itemFromIndex(index);
    QVariantHash itemData = item->data().toHash();
    QString barcodeType = itemData.value("barcodeType").toString();
    QString barcodeData = itemData.value("barcodeData").toString();
    qDebug() << "MainWindow::on_tblTillRoll_clicked(): " << barcodeType << barcodeData;
    emit barcodeAnalysedSignal(barcodeType, barcodeData); // emit a signal for whoever's listening. Especially plugins :)
    showResultsWindow(barcodeData);
}

void MainWindow::on_actionPlugins_triggered()
{
    cancelScan();
    PluginSettings* win = new PluginSettings(pluginInterfaces, this);
    win->show();
}
