//Maemo Barcode Reader and Interpreter (mbarcode or maemo-barcode)
//Copyright (C) 2010 Simon Pickering
//Copyright (C) 2010 Svenn-Arne Dragly
//
//Some source code obtained from other individuals/companies (not affiliated with the project), such as
//Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
//
//Various parts of barcode recognition and GStreamer manipulation code written by:
//      Timothy Terriberry
//      Adam Harwell
//      Jonas Hurrelmann
//
//Original GStreamer code based on the maemo-examples package:
//Copyright (C) 2007-2008 Nokia Corporation. All rights reserved.
//Copyright (C) 2006 INdT.
//@author Talita Menezes <talita.menezes@indt.org.br>
//@author Cidorvan Leite <cidorvan.leite@indt.org.br>
//@author Jami Pekkanen <jami.pekkanen@nokia.com>
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "videowidget.h"
// gstreamer includes
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/gstelement.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/photography.h>
// qt includes
#include <QPainter>
#include <QWidget>
#include <QApplication>
#include <QTime>
#include <QTimer>

#include <QFileDialog>
#include <QFileInfo>

// local includes
#include "common.h"
#include "barcodedetector.h"
#include "mainwindow.h"

VideoWidget *videoWidget_callback;
VideoWidget::VideoWidget(QWidget *parent) : QWidget(parent) {
    BarcodeDetector *newdecoder = new BarcodeDetector(); // create a new decoder object
    VideoWidget(parent, newdecoder);
}
VideoWidget::VideoWidget(QWidget *parent, BarcodeDetector *decoder) : QWidget(parent) {
    setAttribute(Qt::WA_OpaquePaintEvent);
    setAttribute(Qt::WA_PaintOnScreen);
    setAttribute(Qt::WA_NativeWindow);
    setAttribute(Qt::WA_DontCreateNativeAncestors);
    frame_width = 320;
    frame_height = 240;
    QApplication::syncX();
    // Set up the size of the widget
    QSizePolicy sizing(QSizePolicy::Preferred, QSizePolicy::Preferred);
    sizing.setHeightForWidth(true);
    setSizePolicy(sizing);

    pipelineReady = false;
    focusing = false;
    videoWidget_callback = this; // set the global pointer to this object
    getDeviceType(); // work out which device we are so we can specify what type of pipeline to setup
    initPipeline(); // create the GStreamer pipeline
    this->decoder = decoder; // create a new decoder object

    continuousFocusInterval = 1500;
    refocusTimer = new QTimer(this);
    connect(refocusTimer, SIGNAL(timeout()), this, SLOT(refocus()));
    refocusTimer->setInterval(continuousFocusInterval);

}

VideoWidget::~VideoWidget() {
    destroyPipeline();
}

void VideoWidget::setData(GstElement *data) {
    this->data = data;
}

void VideoWidget::paintEvent(QPaintEvent *) {
    // it appears to be unecessary to call this on every paintEvent...
//    if(pipelineReady) {
//        gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(data), this->winId());
//    }
}
// These callback functions are static wrappers making it possible to callback member functions of this class
// Remember to set the videoWidget_callback pointer correctly before calling these functions!
GstBusSyncReply VideoWidget::handleCreateWindow_callback (GstBus * bus, GstMessage * message, GstPipeline * pipeline) {
    return videoWidget_callback->handleCreateWindow(bus,message,pipeline);
}
gboolean VideoWidget::bufferProbeCallback_callback (GstElement *image_sink, GstBuffer *buffer, GstPad *pad) {
    return videoWidget_callback->bufferProbeCallback(image_sink, buffer, pad);
}
gboolean VideoWidget::busCallback_callback (GstBus * bus, GstMessage * message) {
    return videoWidget_callback->busCallback(bus,message);
}
// End listing callback functions
void VideoWidget::start() {
    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(data), this->winId());
//    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(video_crop, "src"), NULL);
    gst_element_set_state(pipeline, GST_STATE_PLAYING);
    gst_photography_set_scene_mode (GST_PHOTOGRAPHY (camera_src), GST_PHOTOGRAPHY_SCENE_MODE_CLOSEUP); // uncommented to fix focus although we are using crop
    pipelineReady = true;
    refocus();
}
void VideoWidget::stop() {
    gst_element_set_state(pipeline, GST_STATE_NULL);
    pipelineReady = false;
    refocusTimer->stop();
    focusing = false;
}
bool VideoWidget::initPipeline() {
    GstElement *screen_sink, *image_sink, *sink;
    GstElement *screen_queue, *image_queue;
    GstElement *csp_filter, *tee; // *image_filter,
    GstCaps *caps;
    GstBus *bus;


    /* Initialize Gstreamer */
    gst_init (0, 0);

    /* Create pipeline and attach a callback to it's
     * message bus */
    pipeline = gst_pipeline_new ("test-camera");
    if(!pipeline)
        qDebug("pipeline failed to init");

    bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
    gst_bus_set_sync_handler (bus, (GstBusSyncHandler) VideoWidget::handleCreateWindow_callback, pipeline);
    // for the xvimagesink stuff, from here: http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gst-plugins-base-libs/html/gst-plugins-base-libs-gstxoverlay.html#gst-x-overlay-set-xwindow-id

    gst_bus_add_watch (bus, (GstBusFunc) VideoWidget::busCallback_callback, NULL);
    gst_object_unref (GST_OBJECT (bus));

    /* Camera video stream comes from a Video4Linux driver */
    if (deviceType==DEVICE_N900){
        camera_src = gst_element_factory_make("v4l2camsrc", "camera_src"); // gst_photography_set_autofocus(GST_PHOTOGRAPHY(cameraSrc), TRUE)
        g_object_set(G_OBJECT(camera_src), "driver-name", "omap3cam", NULL); // thanks BBNS_ @ maemo irc aka Yun-Ta Tsai
    } else if (deviceType==DEVICE_N810)
        camera_src = gst_element_factory_make("gconfv4l2src", "camera_src");
    else
        camera_src = gst_element_factory_make("v4l2src", "camera_src");

    if(!camera_src)
        qDebug("camera_src failed to init");
    else
        qDebug("camera_src is now loaded");


    file_src = gst_element_factory_make("filesrc", "file_src");
    //useFilesrc("/usr/share/icons/hicolor/14x14/hildon/general_alarm_on.png"); // dummy file to do a quickfix
    input_selector = gst_element_factory_make("input-selector", "input_selector");
//    g_object_set(G_OBJECT(input_selector), "n-pads", 2, NULL); // set 2 input pads


    /* Colorspace filter is needed to make sure that sinks understands
     * the stream coming from the camera */
    csp_filter = gst_element_factory_make ("ffmpegcolorspace", "csp_filter");
    if(!csp_filter)
        qDebug("csp_filter failed to init");
    else
        qDebug("csp_filter is now loaded");

    video_crop = gst_element_factory_make ("videocrop", "videocrop");
    g_object_set (G_OBJECT (video_crop), "bottom", 120, NULL);
    g_object_set (G_OBJECT (video_crop), "left", 160, NULL);
    g_object_set (G_OBJECT (video_crop), "right", 160, NULL);
    g_object_set (G_OBJECT (video_crop), "top", 120, NULL);


    /* Tee that copies the stream to multiple outputs */
    tee = gst_element_factory_make ("tee", "tee");
    if(!tee)
        qDebug("tee failed to init");
    else
        qDebug("tee is now loaded");

    /* Queue creates new thread for the stream */
    screen_queue = gst_element_factory_make ("queue", "screen_queue");
    if(!screen_queue)
        qDebug("screen_queue failed to init");

    /* Sink that shows the image on screen. Xephyr doesn't support XVideo
     * extension, so it needs to use ximagesink, but the device uses
     * xvimagesink */
    screen_sink = gst_element_factory_make ("xvimagesink", "screen_sink");
    if(!screen_sink)
        qDebug("screen_sink failed to init");
    g_object_set (G_OBJECT (screen_sink), "sync", FALSE, NULL);

    /* Creates separate thread for the stream from which the image
     * is captured */
    image_queue = gst_element_factory_make ("queue", "image_queue");
    if(!pipeline)
        qDebug("pipeline failed to init");

    /* Filter to convert stream to use format that the gdkpixbuf library
     * can use */
    //image_filter = gst_element_factory_make ("ffmpegcolorspace", "image_filter");
    //if(!image_filter)
    //    qDebug("image_filter failed to init");

    /* A dummy sink for the image stream. Goes to bitheaven */
    image_sink = gst_element_factory_make ("fakesink", "image_sink");
    if(!image_sink)
        qDebug("image_sink failed to init");
    g_object_set (G_OBJECT (image_sink), "sync", FALSE, NULL);

    /* Check that elements are correctly initialized */
    if (!(pipeline && camera_src && screen_sink && csp_filter && screen_queue && image_queue && image_sink && video_crop /*&& file_src && input_selector*/)) {
        g_critical ("Couldn't create pipeline elements");
        return FALSE;
    }

    /* Set image sink to emit handoff-signal before throwing away
     * it's buffer */
    g_object_set (G_OBJECT (image_sink), "signal-handoffs", TRUE, "sync", FALSE, NULL);

    /* Add elements to the pipeline. This has to be done prior to
     * linking them */
    gst_bin_add_many (GST_BIN (pipeline), camera_src, /*file_src,*/ csp_filter, video_crop, /*input_selector,*/ tee, screen_queue, screen_sink, image_queue, image_sink, NULL);

    /* Specify what kind of video is wanted from the camera */
    if (deviceType==DEVICE_N900){
        qDebug("Using N900 caps");
        caps = gst_caps_from_string("video/x-raw-yuv,format=(fourcc)UYVY,width=640,height=480"); //,framerate=[1/10,10/1]"); //width=320,height=240
//        caps = gst_caps_new_simple("video/x-raw-yuv",
//            "width", G_TYPE_INT, 320,
//            "height", G_TYPE_INT, 240,
//            "framerate", GST_TYPE_FRACTION, 323, 25,
//            NULL);
    }else{
        qDebug("Using N8x0 caps");
        caps = gst_caps_new_simple ("video/x-raw-yuv",
                                "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
                                "width", G_TYPE_INT, 320, "height", G_TYPE_INT, 240, "framerate", GST_TYPE_FRACTION, 8, 1,
                                NULL); // change to 2 fps
    }
    /* Link the camera source and colorspace filter using capabilities
     * specified */
    if (!gst_element_link_filtered (camera_src, csp_filter, caps))
        return FALSE;

    gst_caps_unref (caps);

    /* Connect Colorspace Filter -> Tee -> Screen Queue -> Screen Sink
     * This finalizes the initialization of the screen-part of the pipeline */
//    if (!gst_element_link(csp_filter, video_crop))
//        return FALSE;


    // link up the input_selector here
//    gst_element_link_pads (video_crop, "src", input_selector, "sink");
//    gst_element_link_pads (file_src, "src", input_selector, "sink");

    //g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(video_crop, "src"), NULL);

//    if (!gst_element_link_many (input_selector, tee, screen_queue, screen_sink, NULL))
    if (!gst_element_link_many (csp_filter, video_crop, tee, screen_queue, screen_sink, NULL))
        return FALSE;


    //caps = gst_caps_new_simple ("video/x-raw-yuv", "width", G_TYPE_INT, WIDTH, "height", G_TYPE_INT, HEIGHT, NULL);

    /* Link the image-branch of the pipeline. The pipeline is
     * ready after this */
    if (!gst_element_link_many (tee, image_queue, image_sink, NULL))
        return FALSE;
    //if (!gst_element_link_filtered (image_filter, image_sink, caps))
    //    return FALSE;

    //gst_caps_unref (caps);

    /* As soon as screen is exposed, window ID will be advised to the sink */
    //g_signal_connect (screen, "expose-event", G_CALLBACK (expose_cb), screen_sink);

    buffer_cb_id = g_signal_connect (G_OBJECT (image_sink), "handoff", G_CALLBACK (bufferProbeCallback_callback), NULL);

//    sink = gst_element_factory_make("xvimagesink", "sink");
//    gst_element_set_state(sink, GST_STATE_READY);
//
//    QApplication::syncX();
//    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(sink), videoWidget->window()->winId());
    data = screen_sink;
}

void VideoWidget::destroyPipeline() {
    /* Free the pipeline. This automatically also unrefs all elements
     * added to the pipeline */
    gst_element_set_state (pipeline, GST_STATE_NULL);
    gst_object_unref (GST_OBJECT (pipeline));
}

GstBusSyncReply VideoWidget::handleCreateWindow (GstBus * bus, GstMessage * message, GstPipeline * pipeline) {
    // ignore anything but 'prepare-xwindow-id' element messages
    if (GST_MESSAGE_TYPE (message) != GST_MESSAGE_ELEMENT)
    return GST_BUS_PASS;

    if (!gst_structure_has_name (message->structure, "prepare-xwindow-id"))
    return GST_BUS_PASS;

    //win = XCreateSimpleWindow (disp, root, 0, 0, 320, 240, 0, 0, 0);

    //XSetWindowBackgroundPixmap (disp, win, None);

    //XMapRaised (disp, win);

    //XSync (disp, FALSE);

    // connect the signal from the sink to the QWidget

//    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY (GST_MESSAGE_SRC (message)), vw->window()->winId());

    gst_message_unref (message);

    return GST_BUS_DROP;
}
// This is a callback function which retrieves any message from the GstBus
gboolean VideoWidget::busCallback (GstBus * bus, GstMessage * message) {
    gchar *message_str;
    //const gchar *message_name;
    GError *error;
    /* Report errors to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_ERROR) {
        gst_message_parse_error (message, &error, &message_str);
        qDebug("GST error: %s", message_str);
        g_error ("GST error: %s", message_str);
        g_free (error);
        g_free (message_str);
    }

    /* Report warnings to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_WARNING) {
        gst_message_parse_warning (message, &error, &message_str);
        qDebug("GST warning: %s", message_str);
        g_warning ("GST warning: %s", message_str);
        g_free (error);
        g_free (message_str);
    }
    if(GST_MESSAGE_TYPE (message) == GST_MESSAGE_ELEMENT) { // look for GST_MESSAGE_ELEMENT
        const GstStructure *structure = gst_message_get_structure (message);
        const gchar *message_name = gst_structure_get_name (structure);
        gint status = GST_PHOTOGRAPHY_FOCUS_STATUS_NONE;

        if (g_strcmp0 (message_name, GST_PHOTOGRAPHY_AUTOFOCUS_DONE) == 0) { // autofocus messages
            focusing = false;
            emit focusingValueChanged(false);
            gst_structure_get_int (structure, "status", &status);
            switch (status) {
            case GST_PHOTOGRAPHY_FOCUS_STATUS_FAIL:
                qDebug("VideoWidget::busCallback(): Autofocus failed message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_SUCCESS:
                qDebug("VideoWidget::busCallback(): Autofocus success message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_NONE:
                qDebug("VideoWidget::busCallback(): Autofocus none message received.");
                break;
            case GST_PHOTOGRAPHY_FOCUS_STATUS_RUNNING:
                qDebug("VideoWidget::busCallback(): Autofocus running message received.");
                break;
            default:
                qDebug("VideoWidget::busCallback(): Autofocus unknown message recieved.");
                break;
            }
        }
    }

    /* See if the message type is GST_MESSAGE_APPLICATION which means
     * thet the message is sent by the client code (this program) and
     * not by gstreamer. */

    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_APPLICATION) {
        //char EAN[14], ISBN[11], title[255], author[255];
        //float price;
        //char place[255];
        //float number;
        //char temp[50];
        //unsigned int datetime;
        int message_type;
        char *message_output;
//        BarcodeData *bcdata = new BarcodeData;

        qDebug("Got a message from the GStreamer thread");
//        scanning_status=0;


        /* Get name of the message's structure */
        //message_name = gst_structure_get_name (gst_message_get_structure (message));
//        bcdata->message_type = g_value_get_int(gst_structure_get_value(gst_message_get_structure (message), "type"));
//        bcdata->message_output = g_strdup(g_value_get_string(gst_structure_get_value (gst_message_get_structure (message), "output")));

        //gst_structure_get_int(gst_message_get_structure(message), "type", &message_type);
        //message_output = gst_structure_get_string(gst_message_get_structure(message), "output");

        //qDebug("busCallback(): message_name=%s", message_name);


//        gtk_button_set_label(GTK_BUTTON(scan_button), "Scan!");

        //handle_barcode_data(message_output, message_type);
//        g_idle_add(handle_barcode_data, bcdata);

        qDebug("Just finished processing barcode");

        // unref/free this message
        // gst_message_unref(message); // perhaps not needed....?

//        processing=0;
//#ifndef USE_ZBAR
//        QR1D_timer=0;
//#endif
//        dmtx_timer=0;

        // do we need to free message_output now?
        //g_free (message_name);
        //g_free (message_output);


    }
    return TRUE;
}
/* This callback will be registered to the image sink
 * after user requests a photo */
gboolean VideoWidget::bufferProbeCallback (GstElement *image_sink, GstBuffer *buffer, GstPad *pad) {
    // this needs to be uncommented, but now it returns a segfault.
    GstMessage *message;
    //gchar *message_name;
    char *data = (char *) GST_BUFFER_DATA (buffer);
    int type = 0;
    //unsigned char output[14];
    char *src, *dst;
    char fourcc[5] = "Y800";
    int bpp = 8;
    guint z = frame_width * frame_height;

//    qDebug("In bufferProbeCallback");
//
//    if (processing==1) // make sure we don't get called again while we're already scanning
//        return TRUE;
//
//    // check the frame is large enough here
//
//    processing=1; // set the var, to avoid being called again while we're still running the last one


    // write over ourselves
    src = data;
    dst = data; //image_buffer;
    do {
        src++;
        *(dst++) = (*(src++));
    } while (--z);

    type = decoder->analyseImage(data, fourcc, bpp); // type is ignored anyway

    /* Returning TRUE means that the buffer can is OK to be
     * sent forward. When using fakesink this doesn't really
     * matter because the data is discarded anyway */
    return TRUE;
}

void VideoWidget::getDeviceType() {
    char d[255];
    FILE *f;
    f = fopen("/proc/component_version", "r");
    fgets(d,255,f);
    fclose(f);

    if (strstr(d, "SU-18")) {
        qDebug("We're a 770");
        deviceType = DEVICE_770;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-34")) {
        qDebug( "We're an N800");
        deviceType = DEVICE_N800;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-44")) {
        qDebug( "We're an N810");
        deviceType = DEVICE_N810;
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-48")) {
        qDebug( "We're an N810 WiMax");
        deviceType = DEVICE_N810; // wimax one, thanks Qwerty :)
        frame_height = HEIGHT;
        frame_width = WIDTH;
    } else if (strstr(d, "RX-51")) {
        qDebug( "We're an N900");
        deviceType = DEVICE_N900;
        frame_height = HEIGHT; //N900_HEIGHT;
        frame_width = WIDTH; //N900_WIDTH;
    } else
        qDebug( "Oh no... We're something else.");
}
void VideoWidget::refocus() {
    if(!focusing && pipelineReady) { // we are running and are not focusing.
        emit focusingValueChanged(true);
        gst_element_set_state (pipeline, GST_STATE_PAUSED);

        qDebug("VideoWidget::refocus(): Calling refocus");
        focusing = true;
        gst_photography_set_autofocus(GST_PHOTOGRAPHY(camera_src), TRUE);
        qDebug("VideoWidget::refocus(): Refocus called");
//        gst_element_set_state (pipeline, GST_STATE_READY); // causes some problems with flickering, should probably be uncommented
        gst_element_set_state (pipeline, GST_STATE_PLAYING);
        refocusTimer->start(); // try to refocus using the pre-set interval
    } else {
        qDebug("VideoWidget::refocus(): Already attempting autofocus or pipeline not ready... Please wait :)");
    }
}
void VideoWidget::setFocusing(bool enabled) {
    focusing = enabled;
    if(!enabled)
        refocusTimer->stop(); // stop the timer to save cycles
}
bool VideoWidget::isFocusing() {
    return focusing;
}

void VideoWidget::useVideosrc() {
//    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(video_crop, "src"), NULL); // make video_crop the source
}

void VideoWidget::useFilesrc(QString filename) {
//    g_object_set(G_OBJECT(input_selector), "active-pad", gst_element_get_pad(file_src, "src"), NULL); // make filesrc the source
//    g_object_set(G_OBJECT(file_src), "location", filename.toAscii(), NULL); // set the location
}

void VideoWidget::setContinuousFocusTime(int milliseconds)
{
    if (refocusTimer->isActive()){
        refocusTimer->stop();
        refocusTimer->setInterval(milliseconds);
        refocusTimer->start(); // try to refocus every one-and-a-half seconds
    }else
        refocusTimer->setInterval(milliseconds);

}

void VideoWidget::mousePressEvent(QMouseEvent *event){

    emit doSaveBuffer();
}
