/**
 * maemo-barcode - barcode detection/recognition application
 *
 * Copyright (c) 2008 Simon Pickering <S.G.Pickering@bath.ac.uk>
 *
 * Various parts of barcode recognition and GStreamer manipulation code written by:
 *       Timothy Terriberry
 *       Adam Harwell
 *       Jonas Hurrelmann
 *
 * Original GStreamer code based on the maemo-examples package
 * Copyright (c) 2007-2008 Nokia Corporation. All rights reserved.
 * Copyright (c) 2006 INdT.
 * @author Talita Menezes <talita.menezes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 * @author Jami Pekkanen <jami.pekkanen@nokia.com>
 */

#include <conic.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <libosso.h>
#include <gtk/gtk.h>
#include <curl/curl.h>
#include <curl/types.h> /* new for v7 */
#include <curl/easy.h> /* new for v7 */
#include <dbus/dbus-glib.h>

#ifdef FREMANTLE
#include <libosso-abook/osso-abook-contact.h>
#include <libosso-abook/osso-abook-util.h>
#endif /* FREMANTLE */

#include <libmodest-dbus-client/libmodest-dbus-client.h>

//#include <hildon/hildon-helper.h>
//#include <hildon/hildon-defines.h>

#include <hildon-mime.h>

#include "web.h"

#ifdef FREMANTLE
#include "gui_fremantle.h"
#else
#include "gui_diablo.h"
#endif /* N900 */

#include "common.h"

struct connection_object {
    int (*call_fn)(char*); // this points to the fn which we want to invoke after checking the connection has connected ok
    gpointer data; // this might point to the char *EAN
};


extern AppData *appdata;


struct curl_memorybuffer
{
    char *memory;
    size_t size;
};
 



// will add more and differnt ones, but this will do for the time being to test
struct lookup_results
{
    char *provider;
    char *ean;
    char *isbn;
    char *title;
    char *author;
    char *description;
    char *image;
    char *image_url;
    char *price;
    char *notes;
};

int signal_thread_to_quit=0;

// utility fns to wrap creation & destruction of this struct
// only need to make major here if struct changes
struct lookup_results* CreateLookupResultsStruct(char *provider_text)
{
    struct lookup_results* results = (struct lookup_results*)malloc(sizeof(struct lookup_results));
    // now zero each element so we will be able to work out which to destroy in the end
    results->provider=0;
    results->ean=0;
    results->isbn=0;
    results->title=0;
    results->author=0;
    results->description=0;
    results->image=0;
    results->image_url=0;
    results->price=0;
    results->notes=0;
    
    // now make a copy of provider text and copy that into the structure
    results->provider = malloc(strlen(provider_text));
    strcpy(results->provider, provider_text);
    
    return results;
}

void DestroyLookupResultsStruct(struct lookup_results* results)
{
    if(results->provider)
        free(results->provider);

    if(results->ean)
        free(results->ean);

    if(results->isbn)
        free(results->isbn);

    if(results->title)
        free(results->title);
        
    if(results->author)
        free(results->author);
        
    if(results->description)
        free(results->description);
        
    if(results->image)
        free(results->image);
        
    if(results->image_url)
        free(results->image_url);
        
    if(results->price)
        free(results->price);
        
    if(results->notes)
        free(results->notes);
    
    free(results);
}


// Following fn from liq*, thanks Gary/lcuk
static size_t curl_writebuffer(void *ptr, size_t size, size_t nmemb, void *data)
{
    struct curl_memorybuffer *mem = (struct curl_memorybuffer *) data;
    int realsize = size * nmemb;
    
    printf("In memory write callback, size=%d\n", realsize);
 
    mem->memory = (char *)realloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory){
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }
    return realsize;
}






// from modified libconic examples: http://maemo.org/maemo_release_documentation/maemo4.1.x/node10.html#sec:maemo_connectivity
int GetConnected(void (*call_fn)(char*), gpointer user_data) {
    gboolean success = FALSE;

    struct connection_object *con_obj;
    con_obj = g_malloc(sizeof(struct connection_object));
    con_obj->call_fn = call_fn;
    con_obj->data = user_data;

    /* Create connection object */
    ConIcConnection *connection = con_ic_connection_new();
    /* Connect signal to receive connection events */
    g_signal_connect(G_OBJECT(connection), "connection-event",
                     (GCallback)connection_handler, con_obj);
    /* Request connection and check for the result */
    success = con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
    if (!success)
        g_warning("Request for connection failed");
    return success;
}

// from modified libconic examples: http://maemo.org/maemo_release_documentation/maemo4.1.x/node10.html#sec:maemo_connectivity
void connection_handler(ConIcConnection *connection, ConIcConnectionEvent *event, gpointer user_data) {
    ConIcConnectionStatus status = con_ic_connection_event_get_status(event);
    ConIcConnectionError error;
    struct connection_object *con_obj = user_data;
    switch (status) {
    case CON_IC_STATUS_CONNECTED:
        printf("We are connected\n");

        // therefore let's call the connection in question
        (*(con_obj->call_fn))(con_obj->data);

        break;
    case CON_IC_STATUS_DISCONNECTING:
        printf("We are disconnecting...\n");
        break;
    case CON_IC_STATUS_DISCONNECTED:
        printf("And we are disconnected. Let's see what went wrong...\n");
        error = con_ic_connection_event_get_error(event);
        switch (error) {
        case CON_IC_CONNECTION_ERROR_NONE:
            printf("Libconic thinks there was nothing wrong.\n");
            break;
        case CON_IC_CONNECTION_ERROR_INVALID_IAP:
            printf("Invalid (non-existing?) IAP was requested.\n");
            break;
        case CON_IC_CONNECTION_ERROR_CONNECTION_FAILED:
            printf("Connection just failed.\n");
            break;
        case CON_IC_CONNECTION_ERROR_USER_CANCELED:
            printf("User canceled the connection attempt\n");
            break;
        }
        break;
    default:
        printf("Unknown connection status received\n");
    }

    // free the structure which we were passed
    g_free(con_obj);
}


// this callback is called by the threads if they have some data to add
// how we use that data is anyone's guess!
gboolean thread_data_callback(gpointer data)
{
    struct lookup_results *results = (struct lookup_results *)data;
    
    // at the moment we'll overwrite anything there, but perhaps we should just try filling in the gaps?
    
    printf("In thread_data_callback()\n");
    printf("results->ean: %s\n", results->ean);
    printf("appdata->EAN_text: %s\n", appdata->EAN_text);
    
    // check we're on the same barcode
    // need to also check and see if the relevant fields have been filled in!
    if(results->ean || results->isbn)
    {
        // ok, the codes match
        if(results->ean)
            strcpy(appdata->EAN_text, results->ean);
        else if(results->isbn)
            strcpy(appdata->EAN_text, results->isbn);
            
        if(results->title)
            strcpy(appdata->title_text, results->title);
            
        if(results->author)
            strcpy(appdata->author_text, results->author);
            
        if(results->price)
            strcpy(appdata->price_text, results->price);
            
        update_scan_window();
        
    }
    // otherwise the code has changed, e.g. may have been rescanned
        
    DestroyLookupResultsStruct(results); // we have to free the memory allocated by the thread, this includes the EAN passed in originally
    return FALSE; // we don't want to be scheduled to run again
}


/* ===================================
 *
 * AWS
 *
 * ===================================
 */



// this is the thread (lookup/download/parse)
void *amazon_aws_ean_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    char *temp_item_start, *temp_item_end;
    int item_len;
    char *image_url=0;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://www.amazon.co.uk/gp/aw/s/ref=nb_ss/276-8705072-1032160?k="; 
    // "http://www.amazon.co.uk/gp/aw/s/ref=is_box_dvd/276-8705072-1032160?k="; // http://www.amazon.co.uk/s/ref=nb_ss?url=search-alias%3Daps&field-keywords=
    strcat(url, ean);
    
    printf("amazon: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("About to get data\n");
        res = curl_easy_perform(curl);
        if(res){
            printf("amazon: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("amazon: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;
        
    results = CreateLookupResultsStruct("Amazon");
    
    // make a copy otherwise when this structure is freed, the EAN won't be available for any other thread
    results->ean = malloc(strlen(ean));
    strcpy(results->ean, ean);
    
    printf("amazon: About to parse the data\n");    
    
    // now try parsing the data
    
    
    // if no match, we get this:
    item_start = strstr(resultchunk.memory,"We're sorry. We cannot find any matches for your search term.");
    if(item_start){
        // didn't find anything using this service
        printf("amazon: Didn't find the code\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit
    }
    
    
    
    // find the name of the item
    
    item_start = strstr(resultchunk.memory,"<b>");
    if(item_start){
        printf("amazon: item_start = %d\n", item_start);
        item_start = (char *)(item_start + 3); //skip past the <b> business
        if(!item_start)
            goto pass1;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"</b>");
        if(!item_end)
            goto pass1;
        printf("amazon: item_end = %d\n", item_end);
            
        item_len  = item_end - item_start;
        printf("amazon: item_len = %d\n", item_len);
        results->title = (char *)malloc(item_len+1);
        memcpy(results->title, item_start, item_len);
        results->title[item_len] = 0; // null terminate the string
        printf("amazon: title = %s\n", results->title);
    }else
        printf("amazon: No title found\n");
    
pass1:    
    
    // we seem to not find the end of the buffer in one of the following functions
    // so let's just skip them and at least return the title
    goto pass3;
    
    // the wonders of gotos!
    
    
    
    /*
    // find the image of the item
    // this is a link to a small image with an arrow, we can chop this up though to get a large image with no arraw
    // e.g. this code will find: http://ecx.images-amazon.com/images/I/41Z1Z34FDGL._SL75_PIsitb-sticker-arrow-st,TopRight,8,-14_OU02_.jpg
    // we should chop everything off after the . in the last part to give: http://ecx.images-amazon.com/images/I/41Z1Z34FDGL.jpg
    
    item_start = strstr(item_end,"<img src=\""); // use item_end as this occurs after the name
    printf("amazon: item_start = %d\n", item_start);
    if (item_start!=0){
        item_start = (char *)(item_start + 10); //skip past the <???> business
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"\"");
        printf("amazon: item_end = %d\n", item_end);

        if(item_start<=0 || item_end<=0)
            printf("amazon: Error\n");
            
        // now we need to find the last "/", the work back until we find a "." and chop the string there, and add a "jpg" on the end
        // this needs to be constrained by the end of tag - sort it out!if(item_start)
        temp_item_end = item_end;
        temp_item_start = strstr(temp_item_end,"/");
        while(temp_item_start){
            if(temp_item_start)
                temp_item_end = temp_item_start;
            else
                break;
            temp_item_start = strstr(temp_item_end,"/");
        }
        // now temp_item_end should contain a pointer to the last "/"
        // now we want to look for the "." following it.
        temp_item_end = strstr(temp_item_end, ".");
        item_len  = temp_item_end - item_start;    
        printf("amazon: item_len = %d\n", item_len);       
        image_url = (char *)malloc(item_len+1);
        memcpy(image_url, item_start, item_len);
        image_url[item_len] = 0; // null terminate the string
        printf("amazon: image_url = %s\n", image_url);
    }else
        printf("amazon: No image url found\n");
    */


    
    // next look for Author
    // will find this: <a href="/gp/aw/s/ref=id_c_books/276-1557247-4290238?i=stripbooks&field-author=Wolfgang%20Steinchen">Wolfgang Steinchen</a>&#160;
    // extract the name at the end of the <a href> tag.

    item_start = strstr(item_end,"field-author="); // find part of the URL
    if(item_start){
        item_start = strstr(item_start,"\">"); // skip past
        if(!item_start)
            goto pass2;
        item_start = (char *)(item_start + 2); //skip past the <???> business
        if(!item_start)
            goto pass2;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"</a>");
        if(!item_end)
            goto pass2;
        item_len  = temp_item_end - item_start;

        results->author = (char *)malloc(item_len+1);
        memcpy(results->author, item_start, item_len);
        results->author[item_len] = 0; // null terminate the string
        printf("amazon: author = %s\n", results->author);
    }else
        printf("No author found\n");
    
pass2:    
    
    // editorial reviews, e.g.:
    //<a href="/gp/aw/d.html/276-1557247-4290238?rn=0&a=0819441104&er=1">Editorial Reviews</a><br />
    
    // product details page:
    // <a href="/gp/aw/d.html/276-1557247-4290238?pd=1&l=Product%20Details&a=0819441104">Product Details</a><br />

    
    
    
    // find the item price
    item_start = strstr(item_end,"<b>Our price</b>"); // use item_end as this occurs after the name
    if(item_start){
        printf("amazon: item_start = %d\n", item_start);
        item_start = (char *)(item_start + 16); //skip past the <???> business
        if(!item_start)
            goto pass3;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"<br />");
        if(!item_end)
            goto pass3;
        printf("amazon: item_end = %d\n", item_end);
        item_len  = item_end - item_start;
        printf("amazon: item_len = %d\n", item_len);
    
        results->price = (char *)malloc(item_len+1);
        memcpy(results->price, item_start, item_len);
        results->price[item_len] = 0; // null terminate the string
    }else
        printf("No Amazon price, change code to parse the market place results too\n");

pass3:

    /*
    results->image = (struct curl_memorybuffer*)malloc(sizeof(struct curl_memorybuffer));
    
    // now we might consider downloading the image to a buffer, so that we can display it later on.
    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, image_url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)(&results + offsetof(struct lookup_results, icon_url)));
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("About to get data\n");
        res = curl_easy_perform(curl);
        if(res)
            printf("curl error code = %d\n", res);

        // always cleanup
        curl_easy_cleanup(curl);
    }else
        printf("curl startup error code = %d\n", curl);    
    
    // now we have the image data in results->image
    */
    
    printf("1\n");
    if(results->title)
        printf("amazon: %s\n", results->title);

    printf("2\n");
    if(results->author)
        printf("amazon: %s\n", results->author);

    printf("3\n");
    if(results->image_url)
        printf("amazon: %s\n", results->image_url);

    printf("4\n");
    if(results->price)
        printf("amazon: %s\n", results->price);

    printf("5\n");

    // update the relevant table from here
    // or use a g_timeout_add_seconds to wait until we have produced some data
    // then allow that function to update the GUI...
    
    // free the main chunk of memory we allocated to store the downloaded HTML data
    free(resultchunk.memory);
    if(image_url)
        free(image_url);

    // add the callback so we can pass our data back to the main thread    
    g_idle_add(thread_data_callback, results);
    
    // add our callback, passing back the pointer to the data we have decoded
    //g_timeout_add(1, amazon_lookup_ean_callback, results);
    printf("amazon: returning\n");
    return NULL;
}


void amazon_aws_lookup_ean(char *ean)
{    
    // depending on what happens after this fn returns, we might need to make a copy of the ean pointer data and pass that on...
    
    if (!g_thread_create(&amazon_aws_ean_thread, ean, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");
}






/* 
 * ==================================
 * Amazon EAN lookup
 * ==================================
 */ 

// this is the thread (lookup/download/parse)
void *amazon_ean_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    char *temp_item_start, *temp_item_end;
    int item_len;
    char *image_url=0;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://www.amazon.co.uk/gp/aw/s/ref=nb_ss/276-8705072-1032160?k="; 
    // "http://www.amazon.co.uk/gp/aw/s/ref=is_box_dvd/276-8705072-1032160?k="; // http://www.amazon.co.uk/s/ref=nb_ss?url=search-alias%3Daps&field-keywords=
    strcat(url, ean);
    
    printf("amazon: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("About to get data\n");
        res = curl_easy_perform(curl);
        if(res){
            printf("amazon: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("amazon: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;
        
    results = CreateLookupResultsStruct("Amazon");
    
    // make a copy otherwise when this structure is freed, the EAN won't be available for any other thread
    results->ean = malloc(strlen(ean));
    strcpy(results->ean, ean);
    
    printf("amazon: About to parse the data\n");    
    
    // now try parsing the data
    
    
    // if no match, we get this:
    item_start = strstr(resultchunk.memory,"We're sorry. We cannot find any matches for your search term.");
    if(item_start){
        // didn't find anything using this service
        printf("amazon: Didn't find the code\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit
    }
    
    
    
    // find the name of the item
    
    item_start = strstr(resultchunk.memory,"<b>");
    if(item_start){
        printf("amazon: item_start = %d\n", item_start);
        item_start = (char *)(item_start + 3); //skip past the <b> business
        if(!item_start)
            goto pass1;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"</b>");
        if(!item_end)
            goto pass1;
        printf("amazon: item_end = %d\n", item_end);
            
        item_len  = item_end - item_start;
        printf("amazon: item_len = %d\n", item_len);
        results->title = (char *)malloc(item_len+1);
        memcpy(results->title, item_start, item_len);
        results->title[item_len] = 0; // null terminate the string
        printf("amazon: title = %s\n", results->title);
    }else
        printf("amazon: No title found\n");
    
pass1:    
    
    // we seem to not find the end of the buffer in one of the following functions
    // so let's just skip them and at least return the title
    goto pass3;
    
    // the wonders of gotos!
    
    
    
    /*
    // find the image of the item
    // this is a link to a small image with an arrow, we can chop this up though to get a large image with no arraw
    // e.g. this code will find: http://ecx.images-amazon.com/images/I/41Z1Z34FDGL._SL75_PIsitb-sticker-arrow-st,TopRight,8,-14_OU02_.jpg
    // we should chop everything off after the . in the last part to give: http://ecx.images-amazon.com/images/I/41Z1Z34FDGL.jpg
    
    item_start = strstr(item_end,"<img src=\""); // use item_end as this occurs after the name
    printf("amazon: item_start = %d\n", item_start);
    if (item_start!=0){
        item_start = (char *)(item_start + 10); //skip past the <???> business
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"\"");
        printf("amazon: item_end = %d\n", item_end);

        if(item_start<=0 || item_end<=0)
            printf("amazon: Error\n");
            
        // now we need to find the last "/", the work back until we find a "." and chop the string there, and add a "jpg" on the end
        // this needs to be constrained by the end of tag - sort it out!if(item_start)
        temp_item_end = item_end;
        temp_item_start = strstr(temp_item_end,"/");
        while(temp_item_start){
            if(temp_item_start)
                temp_item_end = temp_item_start;
            else
                break;
            temp_item_start = strstr(temp_item_end,"/");
        }
        // now temp_item_end should contain a pointer to the last "/"
        // now we want to look for the "." following it.
        temp_item_end = strstr(temp_item_end, ".");
        item_len  = temp_item_end - item_start;    
        printf("amazon: item_len = %d\n", item_len);       
        image_url = (char *)malloc(item_len+1);
        memcpy(image_url, item_start, item_len);
        image_url[item_len] = 0; // null terminate the string
        printf("amazon: image_url = %s\n", image_url);
    }else
        printf("amazon: No image url found\n");
    */


    
    // next look for Author
    // will find this: <a href="/gp/aw/s/ref=id_c_books/276-1557247-4290238?i=stripbooks&field-author=Wolfgang%20Steinchen">Wolfgang Steinchen</a>&#160;
    // extract the name at the end of the <a href> tag.

    item_start = strstr(item_end,"field-author="); // find part of the URL
    if(item_start){
        item_start = strstr(item_start,"\">"); // skip past
        if(!item_start)
            goto pass2;
        item_start = (char *)(item_start + 2); //skip past the <???> business
        if(!item_start)
            goto pass2;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"</a>");
        if(!item_end)
            goto pass2;
        item_len  = temp_item_end - item_start;

        results->author = (char *)malloc(item_len+1);
        memcpy(results->author, item_start, item_len);
        results->author[item_len] = 0; // null terminate the string
        printf("amazon: author = %s\n", results->author);
    }else
        printf("No author found\n");
    
pass2:    
    
    // editorial reviews, e.g.:
    //<a href="/gp/aw/d.html/276-1557247-4290238?rn=0&a=0819441104&er=1">Editorial Reviews</a><br />
    
    // product details page:
    // <a href="/gp/aw/d.html/276-1557247-4290238?pd=1&l=Product%20Details&a=0819441104">Product Details</a><br />

    
    
    
    // find the item price
    item_start = strstr(item_end,"<b>Our price</b>"); // use item_end as this occurs after the name
    if(item_start){
        printf("amazon: item_start = %d\n", item_start);
        item_start = (char *)(item_start + 16); //skip past the <???> business
        if(!item_start)
            goto pass3;
        printf("amazon: item_start = %d\n", item_start);
        item_end = strstr(item_start,"<br />");
        if(!item_end)
            goto pass3;
        printf("amazon: item_end = %d\n", item_end);
        item_len  = item_end - item_start;
        printf("amazon: item_len = %d\n", item_len);
    
        results->price = (char *)malloc(item_len+1);
        memcpy(results->price, item_start, item_len);
        results->price[item_len] = 0; // null terminate the string
    }else
        printf("No Amazon price, change code to parse the market place results too\n");

pass3:

    /*
    results->image = (struct curl_memorybuffer*)malloc(sizeof(struct curl_memorybuffer));
    
    // now we might consider downloading the image to a buffer, so that we can display it later on.
    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, image_url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)(&results + offsetof(struct lookup_results, icon_url)));
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("About to get data\n");
        res = curl_easy_perform(curl);
        if(res)
            printf("curl error code = %d\n", res);

        // always cleanup
        curl_easy_cleanup(curl);
    }else
        printf("curl startup error code = %d\n", curl);    
    
    // now we have the image data in results->image
    */
    
    printf("1\n");
    if(results->title)
        printf("amazon: %s\n", results->title);

    printf("2\n");
    if(results->author)
        printf("amazon: %s\n", results->author);

    printf("3\n");
    if(results->image_url)
        printf("amazon: %s\n", results->image_url);

    printf("4\n");
    if(results->price)
        printf("amazon: %s\n", results->price);

    printf("5\n");

    // update the relevant table from here
    // or use a g_timeout_add_seconds to wait until we have produced some data
    // then allow that function to update the GUI...
    
    // free the main chunk of memory we allocated to store the downloaded HTML data
    free(resultchunk.memory);
    if(image_url)
        free(image_url);

    // add the callback so we can pass our data back to the main thread    
    g_idle_add(thread_data_callback, results);
    
    // add our callback, passing back the pointer to the data we have decoded
    //g_timeout_add(1, amazon_lookup_ean_callback, results);
    printf("amazon: returning\n");
    return NULL;
}


void amazon_lookup_ean(char *ean)
{    
    // depending on what happens after this fn returns, we might need to make a copy of the ean pointer data and pass that on...
    
    if (!g_thread_create(&amazon_ean_thread, ean, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");
}



// ------------------------------------
//
// Amazon ISBN lookup stuff
//
// ------------------------------------

int amazon_lookup_isbn(char *ISBN) 
{
    // look it up at amazon (for example):
    // http://www.amazon.com/exec/obidos/ISBN=1565922867/
    
    // or we can use the cut-down version as above and use the same parser code.

    if (!g_thread_create(&amazon_ean_thread, ISBN, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;
}


// ------------------------------------
//
// Barcodepedia lookup stuff
//
// ------------------------------------



// bad code:
// <h1>Unknown Item</h1>
//

// good code:
//
//<h2>Description</h2>
//        <p>
//            Diet Coke 330ml Can
//        </p>


// this is the thread (lookup/download/parse)
void *barcodepedia_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    int item_len;
    char *image_url;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://en.barcodepedia.com/"; 
    strcat(url, ean);
    
    printf("barcodepedia: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("barcodepedia: About to get data\n");
        if(res){
            printf("barcodepedia: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("barcodepedia: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit){
        return NULL;
    }
        
    results = CreateLookupResultsStruct("barcodepedia");
    
    
    printf("barcodepedia: About to parse the data\n");    
    
    // now try parsing the data
    
    
    // if no match, we get this:
    item_start = strstr(resultchunk.memory,"<h1>Unknown Item</h1>");
    if(item_start){
        // didn't find anything using this service
        printf("barcodepedia: Didn't find the code\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit
    }
    
    
    // find the image of the item    
    item_start = strstr(resultchunk.memory,"<img alt=\"Product image\" src=\""); // use item_end as this occurs after the name
    printf("barcodepedia: item_start = %d\n", item_start);
    if(item_start){
        item_start = (char *)(item_start + 30); //skip past the <???> business - check this number!
        printf("barcodepedia: item_start = %d\n", item_start);
        item_end = strstr(item_start,"\"");
        printf("barcodepedia: item_end = %d\n", item_end);

        if(item_start<=0 || item_end<=0)
            printf("barcodepedia: Error\n");
            
        item_len  = item_end - item_start;    
        printf("barcodepedia: item_len = %d\n", item_len);       
        image_url = (char *)malloc(item_len+1);
        memcpy(image_url, item_start, item_len);
        image_url[item_len] = 0; // null terminate the string
        printf("barcodepedia: image_url = %s\n", image_url);    
    }else{
        printf("barcodepedia: no image\n");
        item_end = resultchunk.memory;
    }
    // actually looks like there are other ways for the image to be displayed...
    
    
    // find the name of the item
    
    item_start = strstr(item_end,"<h1>");
    printf("barcodepedia: item_start = %d\n", item_start);
    item_start = (char *)(item_start + 4); //skip past
    printf("barcodepedia: item_start = %d\n", item_start);
    item_end = strstr(item_start,"</h1>");
    printf("barcodepedia: item_end = %d\n", item_end);
    
    if(item_start<=0 || item_end<=0)
        printf("barcodepedia: Error\n");
        
    item_len  = item_end - item_start;
    printf("barcodepedia: item_len = %d\n", item_len);
    results->title = (char *)malloc(item_len+1);
    memcpy(results->title, item_start, item_len);
    results->title[item_len] = 0; // null terminate the string
    printf("barcodepedia: title = %s\n", results->title);
    
    
    
    



    /*
    results->image = (struct curl_memorybuffer*)malloc(sizeof(struct curl_memorybuffer));
    
    // now we might consider downloading the image to a buffer, so that we can display it later on.
    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, image_url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)(&results + offsetof(struct lookup_results, icon_url)));
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("About to get data\n");
        res = curl_easy_perform(curl);
        if(res)
            printf("curl error code = %d\n", res);

        // always cleanup
        curl_easy_cleanup(curl);
    }else
        printf("curl startup error code = %d\n", curl);    
    
    // now we have the image data in results->image
    */
    
    
    // make a copy otherwise when this structure is freed, the EAN won't be available for any other thread
    results->ean = malloc(strlen(ean));
    strcpy(results->ean, ean);
    
    printf("barcodepedia: %s\n", results->title);
    //printf("barcodepedia: %s\n", results->image_url);

    // update the relevant table from here
    // or use a g_timeout_add_seconds to wait until we have produced some data
    // then allow that function to update the GUI...
    
    // free the main chunk of memory we allocated to store the downloaded HTML data
    free(resultchunk.memory);
    free(image_url);
    
    // add the callback so we can pass our data back to the main thread    
    g_timeout_add(1, thread_data_callback, results);
    
    // add our callback, passing back the pointer to the data we have decoded
    //g_timeout_add(1, amazon_lookup_ean_callback, results);
    printf("barcodepedia: returning\n");
    return NULL;
}


void barcodepedia_lookup(char *ean)
{
    
    if (!g_thread_create(&barcodepedia_thread, ean, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;

}

// ------------------------------------
//
// UPCdatabase lookup stuff
//
// ------------------------------------


// bad code
// <h2 align="center">UPC Error</h2>



// good code
// <tr><td>Description</td><td></td><td>Diet Coke</td></tr>
// <tr><td>Size/Weight</td><td></td><td>330ml can</td></tr>


// this is the thread (lookup/download/parse)
void *upcdatabase_com_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    int item_len;
    char *image_url;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://www.upcdatabase.com/item/"; 
    strcat(url, ean);
    
    printf("upcdatabase_com: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("upcdatabase_com: About to get data\n");
        res = curl_easy_perform(curl);
        if(res){
            printf("upcdatabase_com: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("upcdatabase_com: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;
        
    results = CreateLookupResultsStruct("upcdatabase.com");
    
    
    printf("upcdatabase_com: About to parse the data\n");    
    
    // now try parsing the data
    
    
    // if no match, we get this:
    item_start = strstr(resultchunk.memory,"UPC Error");
    if(item_start){
        // didn't find anything using this service
        printf("upcdatabase_com: Didn't find the code\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit
    }else{
        // we need to prepare the pointer for the next line
        item_end = resultchunk.memory;
    }
    
    
    // see if this is a book or a different type of item
    
    
    item_start = strstr(item_end,"This is a Bookland ISBN code");
    if(item_start){
        // it's a book, so we should search for the book related text
        
        // the data is available on the page, but looks like it comes from another site, so just pass this for the time being
        
        printf("upcdatabase_com: is a book, skipping\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit 
        
    }else{
        // not a book
        // find the name of the item
        
        item_start = strstr(item_end,"<tr><td>Description</td><td></td><td>");
        printf("upcdatabase_com: item_start = %d\n", item_start);
        item_start = (char *)(item_start + 37); //skip past - check this number!
        printf("upcdatabase_com: item_start = %d\n", item_start);
        item_end = strstr(item_start,"</td></tr>");
        printf("upcdatabase_com: item_end = %d\n", item_end);
        
        if(item_start<=0 || item_end<=0)
            printf("Error\n");
            
        item_len  = item_end - item_start;
        printf("upcdatabase_com: item_len = %d\n", item_len);
        results->title = (char *)malloc(item_len+1);
        memcpy(results->title, item_start, item_len);
        results->title[item_len] = 0; // null terminate the string
        printf("upcdatabase_com: title = %s\n", results->title);


        printf("upcdatabase_com: %s\n", results->title);

    }
    
    // make a copy otherwise when this structure is freed, the EAN won't be available for any other thread
    results->ean = malloc(strlen(ean));
    strcpy(results->ean, ean);
    

    // update the relevant table from here
    // or use a g_timeout_add_seconds to wait until we have produced some data
    // then allow that function to update the GUI...
    
    // free the main chunk of memory we allocated to store the downloaded HTML data
    free(resultchunk.memory);

    // add the callback so we can pass our data back to the main thread    
    g_timeout_add(1, thread_data_callback, results);

    
    // add our callback, passing back the pointer to the data we have decoded
    //g_timeout_add(1, amazon_lookup_ean_callback, results);
    printf("upcdatabase_com: returning\n");
    return NULL;
}



int upcdatabase_com_lookup(char *EAN) 
{
    
    if (!g_thread_create(&upcdatabase_com_thread, EAN, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;
}




// ------------------------------------
//
// CheckUPC lookup stuff
//
// ------------------------------------


// bad: 
// <strong>Your search didn't match any products.</strong>

// good
// <img src="http://cf.checkupc.com/images/products/254315.jpg" alt="
// <strong>Linksys WRT160N Ultra RangePlus Wireless-N Broadband Router</strong>



// this is the thread (lookup/download/parse)
void *checkupc_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    int item_len;
    char *image_url;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://www.checkupc.com/search.php?keyword="; 
    strcat(url, ean);
    
    printf("checkupc: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("checkupc: About to get data\n");
        res = curl_easy_perform(curl);
        res = curl_easy_perform(curl);
        if(res){
            printf("checkupc: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("checkupc: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;
        
    results = CreateLookupResultsStruct("checkupc.com");
    
    
    printf("checkupc: About to parse the data\n");    
    
    // now try parsing the data
    
    
    // if no match, we get this:
    item_start = strstr(resultchunk.memory,"Your search didn't match any products.");
    if(item_start){
        // didn't find anything using this service
        printf("checkupc: Didn't find the code\n");
        free(resultchunk.memory);
        DestroyLookupResultsStruct(results);
        return NULL; // just quit
    }
    
    
    // find the image url of the item
    
    item_start = strstr(resultchunk.memory,"<div align=\"left\"");
    item_start = strstr(item_start,"<img src=\"");
    item_start = (char *)(item_start + 10); //skip past - check this number!
    printf("checkupc: item_start = %d\n", item_start);
    item_end = strstr(item_start,"\"");
    printf("checkupc: item_end = %d\n", item_end);
    
    if(item_start<=0 || item_end<=0)
        printf("Error\n");
        
    item_len  = item_end - item_start;
    printf("checkupc: item_len = %d\n", item_len);
    image_url = (char *)malloc(item_len+1);
    memcpy(image_url, item_start, item_len);
    image_url[item_len] = 0; // null terminate the string
    printf("checkupc: image_url = %s\n", image_url);
 
 

    // find item title
    item_start = strstr(item_end,"<strong>");
    item_start = (char *)(item_start + 8); //skip past - check this number!
    printf("checkupc: item_start = %d\n", item_start);
    item_end = strstr(item_start,"\"");
    printf("checkupc: item_end = %d\n", item_end);
    
    if(item_start<=0 || item_end<=0)
        printf("checkupc: Error\n");
        
    item_len  = item_end - item_start;
    printf("checkupc: item_len = %d\n", item_len);
    results->title = (char *)malloc(item_len+1);
    memcpy(results->title, item_start, item_len);
    results->title[item_len] = 0; // null terminate the string
    printf("checkupc: image_url = %s\n", results->title);
    
    
    // make a copy otherwise when this structure is freed, the EAN won't be available for any other thread
    results->ean = malloc(strlen(ean));
    strcpy(results->ean, ean);
    
    
    printf("checkupc: %s\n", results->title);

    // update the relevant table from here
    // or use a g_timeout_add_seconds to wait until we have produced some data
    // then allow that function to update the GUI...
    
    // free the main chunk of memory we allocated to store the downloaded HTML data
    free(resultchunk.memory);

    // add the callback so we can pass our data back to the main thread    
    g_timeout_add(1, thread_data_callback, results);

    
    // add our callback, passing back the pointer to the data we have decoded
    //g_timeout_add(1, amazon_lookup_ean_callback, results);
    printf("checkupc: returning\n");
    return NULL;
}


int checkupc_lookup(char *EAN) 
{
    
    if (!g_thread_create(&checkupc_thread, EAN, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;
}





// ------------------------------------
//
// MusicBrainz lookup stuff
//
// ------------------------------------



// this is the thread (lookup/download/parse)
void *musicbrainz_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    int item_len;
    char *image_url;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://musicbrainz.org/ws/1/release/?type=xml&query=barcode:"; 
    strcat(url, ean);
    
    printf("musicbrainz: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("musicbrainz: About to get data\n");
        res = curl_easy_perform(curl);
        if(res){
            printf("musicbrainz: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("musicbrainz: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;

    // add the callback so we can pass our data back to the main thread    
    //g_timeout_add(1, thread_data_callback, results);
    
    printf("musicbrainz: returning\n");
    return NULL;
}


int musicbrainz_lookup(char *EAN) 
{

    if (!g_thread_create(&musicbrainz_thread, EAN, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;
}

// ------------------------------------
//
// ISBN.nu lookup stuff
//
// ------------------------------------



// this is the thread (lookup/download/parse)
void *isbn_nu_thread(void *ptr)
{
    CURL *curl;
    CURLcode res;
    struct curl_memorybuffer resultchunk = {NULL,0};
    gchar *ean = ptr;
    char *big_buffer;
    char *item_start, *item_end;
    int item_len;
    char *image_url;
    struct lookup_results *results; // this we will pass back to the main thread via the timer callback
        
    gchar url[255] = "http://isbn.nu/"; 
    strcat(url, ean);
    
    printf("isbn_nu: Entered thread\n");
    

    curl = curl_easy_init();
    if(curl){

        curl_easy_setopt(curl, CURLOPT_URL, url);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&resultchunk);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_writebuffer);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        //curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
        //curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

        printf("isbn_nu: About to get data\n");
        res = curl_easy_perform(curl);
        res = curl_easy_perform(curl);
        if(res){
            printf("isbn_nu: curl error code = %d\n", res);
            curl_easy_cleanup(curl);
            return NULL;
        }
        printf("Got data! :)\n");

        /* always cleanup */
        curl_easy_cleanup(curl);
    }else
        printf("isbn_nu: curl startup error code = %d\n", curl);
    
    if (signal_thread_to_quit)
        return NULL;
    
    // add the callback so we can pass our data back to the main thread    
    //g_timeout_add(1, thread_data_callback, results);
    
    printf("isbn_nu: returning\n");
    return NULL;
}



int isbn_nu_lookup_isbn(char *ISBN) 
{

    if (!g_thread_create(&isbn_nu_thread, ISBN, FALSE, NULL) != 0) // the thread will pass the ean handle onward, do not free it
        g_warning("can't create the thread");

    return 1;
}










// Following fns from: http://www.geekhideout.com/urlcode.shtml

/* Converts a hex character to its integer value */
char from_hex(char ch) {
  return isdigit(ch) ? ch - '0' : tolower(ch) - 'a' + 10;
}

/* Converts an integer value to its hex character*/
char to_hex(char code) {
  static char hex[] = "0123456789abcdef";
  return hex[code & 15];
}

/* Returns a url-encoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char *url_encode(char *str) {
  char *pstr = str, *buf = malloc(strlen(str) * 3 + 1), *pbuf = buf;
  while (*pstr) {
    if (isalnum(*pstr) || *pstr == '-' || *pstr == '_' || *pstr == '.' || *pstr == '~') 
      *pbuf++ = *pstr;
    else if (*pstr == ' ') 
      *pbuf++ = '+';
    else 
      *pbuf++ = '%', *pbuf++ = to_hex(*pstr >> 4), *pbuf++ = to_hex(*pstr & 15);
    pstr++;
  }
  *pbuf = '\0';
  return buf;
}

/* Returns a url-decoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char *url_decode(char *str) {
  char *pstr = str, *buf = malloc(strlen(str) + 1), *pbuf = buf;
  while (*pstr) {
    if (*pstr == '%') {
      if (pstr[1] && pstr[2]) {
        *pbuf++ = from_hex(pstr[1]) << 4 | from_hex(pstr[2]);
        pstr += 2;
      }
    } else if (*pstr == '+') { 
      *pbuf++ = ' ';
    } else {
      *pbuf++ = *pstr;
    }
    pstr++;
  }
  *pbuf = '\0';
  return buf;
}














// Background reading:
// http://www.eblong.com/zarf/bookscan/
// http://isbn.nu/

/*
// work out what type of product this barcode codes for
int GetEANtype(char *EAN, char *ISBN) {
    // e.g. book, DVD, CD, other, etc.
    // http://www.av1611.org/666/barcode.html
    if (ean2isbn(EAN, ISBN)) {
        // it's a book
        return EAN_TYPE_BOOK;
    } else {
        // it's not a book!
        return EAN_TYPE_OTHER;
    }

}


// obtain basic data about this EAN
// description stuff
int LookupEAN(char *EAN) { // add more fields to pass data back
    // look at the HTML submission from this page and try hooking into it?: http://www.upcdatabase.com/itemform.asp
    // http://www.amazon.com/exec/obidos/ISBN=1565922867/
    // http://isbntools.com/
    // http://upcdata.info/ - limited lookups, registration, pay-for-lookups, etc. :(
    // http://gepir.gs1.org/V31/xx/ - this is company information

    // check and see what type it is:

    // if it's a book: http://www.amazon.com/exec/obidos/ISBN=1565922867/ or http://isbn.nu/
    return 1;
}
*/

// need to alter this function so it accepts the EAN as a char string rather than an array of ints.
int ean2isbn(char ean_str[14], char *isbn10_str) { // this uses ean[0] as the length isbn10 and isbn13 also need arry[0] to be their lengths
    int ean[13], isbn10[10];
    int cksum, i;

    // see: http://www.bisg.org/isbn-13/conversions.html
    // see: http://www.eblong.com/zarf/bookscan/eantoisbn-java.txt

    // create the arrays
    for (i=0;i<13;i++) {
        ean[i] = (int)ean_str[i]-(int)"0"; // subtract 48 to make our character number into a number number
    }

    if (ean[0]!=9 | ean[1]!=7 | ((ean[2]!=8) & (ean[2]!=9))) {
        // error
        fprintf(stderr, "Not an ISBN barcode\n");
        return 0;
    }

    // so we have a proper isbn code, let's allocate some memory for the string
    isbn10_str = g_malloc(11*sizeof(char)); // 10 + 1

    if (ean[2]==8)
        fprintf(stderr, "978 barcode\n");
    else if (ean[2]==9)
        fprintf(stderr, "979 barcode\n");

    //isbn = ean(4:end-1);
    memcpy(&isbn10[0], &ean[3], 9); // copy from 4th char to last minus 1

    // http://www.bisg.org/pi/appendix_1.html
    cksum = 0;
    for (i=0;i<9;i++) // for 9 digits
        cksum += (10-i)*isbn10[i];

    // modulus 11
    cksum %= 11;
    cksum = 11 - cksum;

    // spit the char array back out
    for (i=0;i<13;i++) {
        isbn10_str[i] = isbn10[i]+48; // add 48 to make our number number into a character number
    }

    if (cksum==10)
        isbn10_str[9] = "X";
    else if (cksum==11)
        isbn10_str[9] = "0";
    else
        isbn10_str[9] = (char)(cksum +48);

    return 1; //success
}



int handle_barcode_data(BarcodeData *bcdata) //(char *data, int barcode_type)
{
    char *data = bcdata->message_output;
    int barcode_type = bcdata->message_type;
    
    signal_barcode_payload(data, bcdata->message_type);
    
    // decide what sort of data this is, and decide what we want to do with it
    fprintf(stdout, "Entered handle_barcode_data()\n");
    if (barcode_type == ZBAR_QRCODE){ //BARCODE_TYPE_DMTX
        // this is a datamatrix barcode
        printf("handle_barcode_Data(): Datamatrix type barcode: %s\n", data);
        if (parse_mixed_barcode_data(data)) //# hand it off to be analysed
            free(data);
            free(bcdata);
            return FALSE;
    }
    else if (barcode_type <= ZBAR_CODE39 && barcode_type >= ZBAR_EAN8){ //(barcode_type == BARCODE_TYPE_1D){
        int type=1; // 0 = not all digits; 1 = all digits; 2 = all digits + X/x
        int i;
        printf("handle_barcode_Data(): 1D type barcode: %s\n", data);
        for(i=0; i<strlen(data); i++){
            if (!isdigit(data[i])){
                printf("%c\n", data[i]);
                if (data[i]!='x' && data[i]!='X'){
                    type=0;
                    break;
                }
                else if (data[i]=='x' && data[i]=='X'){
                    type=2;
                }
            }
        }
        if(type==0){ // not all digits
            printf("handle_barcode_Data(): not all digits barcode: %s\n", data);
            if (parse_mixed_barcode_data(data)){ //# hand it off to be analysed
                free(data);
                free(bcdata);
                return FALSE; // return TRUE;
            }
                        
        }else if(type==1){ // all digits
            printf("handle_barcode_Data(): all numeric type barcode: %s\n", data);
            if (parse_numeric_barcode_data(data)){
                free(data);
                free(bcdata);
                return FALSE; // return TRUE;
            }
            
        }else if(type==2){ // digits + X/x
            if (check_ISBN10_checksum(data)){
                printf("handle_barcode_Data(): isbn10 type barcode: %s\n", data);
                if (handle_ISBN10(data)){
                    free(data);
                    free(bcdata);
                    return FALSE; // return TRUE;
                }
            }else
                printf("ISBN 10 checksum fail");
        }
    /*
    }else if (barcode_type == BARCODE_TYPE_QRCODE){
        printf("handle_barcode_Data(): QR code type barcode: %s\n", data);
        if (parse_mixed_barcode_data(data)){ //# hand it off to be analysed
            free(data);
            free(bcdata);
            return FALSE; // return TRUE;
        }*/
    }
    printf("handle_barcode_Data(): fallback mixed type barcode: %s\n", data);
    parse_mixed_barcode_data(data); //# hand it off to be analysed
    free(data);
    free(bcdata);
    return FALSE; // return TRUE;
}

int parse_mixed_barcode_data(char *data)
{
    char *urlp;
    GtkWidget *banner;

    // the assumption here is that the barcode is definitely not an EAN

    // this link used for identifiers and ideas.
    // http://code.google.com/p/zxing/wiki/BarcodeContents

    printf("Mixed barcode: %s\n", data);

    strcpy(appdata->EAN_text, data);
    strcpy(appdata->title_text, "");
    strcpy(appdata->author_text, "");
    strcpy(appdata->price_text, "");
    update_scan_window();


    // check for a vCard
    //urlp = strstr(data, "BEGIN:VCARD");
    if (strncasecmp(data, "BEGIN:VCARD", 11)==0){
        urlp = data;
        // we assume there's nothing of interest on the end
        printf("vCard!\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "vCard");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
#ifdef N900        
        addVCard(urlp); // open the url
#endif /* N900 */        
        return TRUE;
    }




    // Let's try something Maemo specific - link to an app. E.g. and install file in a barcode
    // we should probably parse the file and let the user know what it's for
    // then we can also save the install file somewhere, then pass it onto app mngr
    // so, what do we name the file? Use something from the contents (so it's understandable)
    // or give it a unique (and not very useful) name?
    if (strncasecmp(data, "[install]", 9)==0){
        printf("Maemo install file\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Maemo install file");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        
        openInstallFile(data);        
        return TRUE;
    }


    // see if this contains an URL
    // we should check for this last, as there may be an URL in e.g. a VCard
    
    // check for the URL: header
    //urlp = strstr(data, "URL:");
    if (strncasecmp(data, "URL:", 4)==0){
        urlp = urlp+4; // skip the URL: business
        
        printf("URL!\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "URL");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        // we assume there's nothing of interest on the end
        //openURL(urlp); // open the url
        openGeneric(data, "Open URL: %s?");
        return TRUE;
    }
    
    // check for the URLTO: header
    //urlp = strstr(data, "URLTO:");
    if (strncasecmp(data, "URLTO:", 6)==0){
        urlp = data+6; // skip the URLTO: business
        
        printf("URL!\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "URL");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        // we assume there's nothing of interest on the end
        //openURL(urlp); // open the url
        openGeneric(data, "Open URL: %s?");
        return TRUE;
    }
    
    // check for a plain http://
    // might want to check for odd cases, though it *should* be lower case
    urlp = strstr(data, "http://");
    if (urlp){        
        printf("URL!\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "URL");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        // we assume there's nothing of interest on the end
        //openURL(urlp); // open the url
        openGeneric(data, "Open URL: %s?");
        return TRUE;
    }
    
    
    
    
    // if it's QR code, check the iMode stuff
    // http://www.nttdocomo.co.jp/english/service/imode/make/content/barcode/function/content/index.html
    
    if (strncasecmp(data, "CNTS:", 5)==0){
        printf("i-mode content marker\n");
        banner = hildon_banner_show_informationf (appdata->window, NULL, "i-mode content marker: %s", data);
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        //type = data[10:14]
        //lng = data[18:22]
        //body = data[27:27+lng+1]
        return TRUE;
    }else if (strncasecmp(data, "MECARD:", 7)==0){ //#http://www.nttdocomo.co.jp/english/service/imode/make/content/barcode/function/application/common/index.html
        printf("Phone book registration message\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Phone book registration message");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }else if (strncasecmp(data, "MATMSG:", 7)==0){
        printf("Email sync message\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Email sync message");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }else if (strncasecmp(data, "MEBKM:", 6)==0){
        printf("Bookmark registration message\n");
        // we could try parsing this out and asking the user if they want to save it as a bookmark, or just open it in the browser
        banner = hildon_banner_show_information (appdata->window, NULL, "Bookmark registration message");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }else if (strncasecmp(data,"LAPL:", 5)==0){
        printf("i-appli sync message\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "i-appli sync message");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }
    
    
    
    
    // check for an email address...
    if (strncasecmp(data, "mailto:", 7)==0){        
        printf("Email address\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Email address");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        //openEmail((char*)(data+7)); // just pass the remainder on
        openGeneric(data, "Open Email: %s?");
        return TRUE;
    }


    // check for a telephone number...
    if (strncasecmp(data, "tel:", 4)==0){      
        printf("Telephone number\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Telephone number");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        //openTel(data);
        openGeneric(data, "Open Telephone Number: %s?");
        return TRUE;
    }
    


    // check for a BIZCARD
    if (strncasecmp(data, "BIZCARD:", 8)==0){      
        printf("BIZCARD\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "BIZCARD");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }


    // check for an SMS
    if (strncasecmp(data, "sms:", 4)==0){      
        printf("SMS\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "SMS");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        openGeneric(data, "Open SMS: %s?");
        return TRUE;
    }
    if (strncasecmp(data, "smsto:", 6)==0){
        printf("SMS\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "SMS");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        openGeneric(data, "Open SMS: %s?");
        return TRUE;
    }


    // check for an MMS
    if (strncmp(data, "mms:", 4)==0){      
        printf("MMS\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "MMS");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        openGeneric(data, "Open MMS: %s?");
        return TRUE;
    }
    if (strncasecmp(data, "mmsto:", 6)==0){
        printf("MMS\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "MMS");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        openGeneric(data, "Open MMS: %s?");
        return TRUE;
    }
    

    // check for geographic information
    if (strncasecmp(data, "geo:", 4)==0){      
        printf("Geographical information\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Geographical information");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        openGeneric(data, "Show Geo Data: %s?");
        return TRUE;
    }



    // Android specific - link to app in Market
    if (strncasecmp(data, "market:", 7)==0){      
        printf("Android Market link\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Android Market link");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }



    // iPhone rumoured - link to YouTube
    if (strncasecmp(data, "youtube:", 8)==0){      
        printf("Link to YouTube\n");
        banner = hildon_banner_show_information (appdata->window, NULL, "Link to YouTube");
        hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
        return TRUE;
    }






    
    // iCal
    

    
    return TRUE;


}


int parse_numeric_barcode_data(char *data)
{
    // the assumption here is that the barcode is probably an EAN

    printf("Numeric: length=%d, %s\n", strlen(data), data);

    // check the length, and EAN can be 13 or 8 chars, a upc is 12 chars. ISBN can be 13/12 or 10. Wonderful, lots of choice! :p
    if (strlen(data)==12){
        //# UPC barcode
        //# turn it into an EAN by adding a leading 0
        printf("converting UPC to EAN\n");
        //sprintf(data, "0%s", data);
        memmove(data+1, data, 12);
        data[0] = '0';
    }

    if (strlen(data)==13){
        // this is an EAN or an ISBN
        // see what the first 3 digits are, was it manufactured in bookland? :)
        //if (strstr(data,"979")==data | strstr(data,"978")==data | strstr(data,"977")==data){
       
        if (data[0]=='9' && data[1]=='7' && (data[2]=='7' || data[2]=='8' || data[2]=='9')){ 
            // this is an ISBN13/EAN13 for a book
            printf("It's a book\n");
            if (check_ISBN13_checksum(data)){
                // valid checksum, good
                 //# handle the data
                return handle_ISBN13(data);
            }else
                printf("ISBN13 checksum fail");
        }else
            printf("Not a book\n");
            
        if (check_EAN13_checksum(data)){
            return handle_EAN13(data); //# handle the EAN number
        }else{
            printf("EAN13 checksum fail");
        }
    }else if (strlen(data)==8){
        // EAN8
        if (check_EAN8_checksum(data)){
            return handle_EAN8(data);
        }else
            printf("EAN8 checksum fail");
    }
    // otherwise must be something else
    // send it over to the other fn to be displayed/parsed/etc.
    return FALSE;
}


int check_ISBN13_checksum(char *data)
{
    printf("%s\n", data);
    return TRUE;
}

int check_ISBN10_checksum(char *data)
{
    printf("%s\n", data);
    return TRUE;
}

int check_EAN13_checksum(char *data)
{
    printf("%s\n", data);
    return TRUE;
}

int check_EAN8_checksum(char *data)
{
    printf("%s\n", data);
    return TRUE;
}





// will probably want to allow the user to chose which services to use
// will also need to allow them to choose the country they want, for e.g. the Amazon services

// start the threads running!
void do_ISBN13_lookup(char *data)
{
    amazon_lookup_isbn(data);
    //barcodepedia_lookup(data);
    //upcdatabase_com_lookup(data);
    //checkupc_lookup(data);
    //musicbrainz_lookup(data);
    //isbn_nu_lookup_isbn(data); - not implemented atm
    return;
}

void do_ISBN10_lookup(char *data)
{
    amazon_lookup_isbn(data);
    //barcodepedia_lookup(data);
    //upcdatabase_com_lookup(data);
    //checkupc_lookup(data);
    //musicbrainz_lookup(data);
    //isbn_nu_lookup_isbn(data); - not implemented atm
    return;
}

void do_EAN13_lookup(char *data)
{
    amazon_lookup_ean(data);
    //barcodepedia_lookup(data);
    //upcdatabase_com_lookup(data);
    //checkupc_lookup(data);
    //musicbrainz_lookup(data);  - not implemented atm
    return;
}

void do_EAN8_lookup(char *data)
{
    amazon_lookup_ean(data);
    //barcodepedia_lookup(data);
    //upcdatabase_com_lookup(data);
    //checkupc_lookup(data);
    //musicbrainz_lookup(data); - not implemented atm
    return;
}






// this should be a user setting
int use_web=0;

int handle_ISBN13(char *data)
{
    printf("ISBN13: %s\n", data);

    // use a web-scraper to lookup this ISBN13 code
    // might need to convert to an ISBN10
    //ean2isbn(data, char *isbn10_str)

    strcpy(appdata->EAN_text, data);
    strcpy(appdata->title_text, ""); // don't tell!
    strcpy(appdata->author_text, "");
    strcpy(appdata->price_text, "");
    update_scan_window();
    
    if(use_web) // should also check and make sure we're connected
        do_ISBN13_lookup(data);
    
    return TRUE;
}

int handle_ISBN10(char *data)
{
    printf("ISBN10: %s\n", data);

    // use a web-scraper to lookup this ISBN10 code
    
    strcpy(appdata->EAN_text, data);
    strcpy(appdata->title_text, ""); // don't tell!
    strcpy(appdata->author_text, "");
    strcpy(appdata->price_text, "");
    update_scan_window();

    if(use_web) // should also check and make sure we're connected
        do_ISBN10_lookup(data);

    return TRUE;
}

int handle_EAN13(char *data)
{
    printf("EAN13: %s\n", data);

    // use a web-scraper to lookup this EAN13 code
    
    strcpy(appdata->EAN_text, data);
    strcpy(appdata->title_text, "");
    strcpy(appdata->author_text, "");
    strcpy(appdata->price_text, "");
    update_scan_window();

    if(use_web) // should also check and make sure we're connected
        do_EAN13_lookup(data);

    return TRUE;
}

int handle_EAN8(char *data)
{
    printf("EAN8: %s\n", data);

    // use a web-scraper to lookup this EAN8 code

    strcpy(appdata->EAN_text, data);
    strcpy(appdata->title_text, "");
    strcpy(appdata->author_text, "");
    strcpy(appdata->price_text, "");
    update_scan_window();

    if(use_web) // should also check and make sure we're connected
        do_EAN8_lookup(data);

    return TRUE;
}




// ===============================================================================================
// libosso notifiers for our different payload types
// ===============================================================================================

void signal_barcode_payload(char* barcode, int type)
{
    // send out a generic "We have a barcode" message
    //gchar *payload = g_strdup(barcode);
    DBusMessage* msg;
    DBusMessageIter args; 
    dbus_uint32_t serial;

    // add check for benefit of Diablo users, need to track down the problem here...
    // I wonder if it has to do with the service file or more likely the .desktop which doesn't
    // work under Diablo....?
    if(!appdata->con){
        fprintf(stderr, "DBus connection failed, debug this!\n"); 
        return;
    }

    msg = dbus_message_new_signal("/org/maemo/mbarcode", // object name of the signal
         "org.maemo.mbarcode", // interface name of the signal
         "BarcodeDecoded"); // name of the signal

    if (msg == NULL) 
    { 
        fprintf(stderr, "Message Null\n"); 
        return; 
    }
       
    // append arguments onto signal
    dbus_message_iter_init_append(msg, &args);
    if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &barcode)) { 
        fprintf(stderr, "Out Of Memory!\n"); 
        return;
    }
    if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_INT32, &type)) { 
        fprintf(stderr, "Out Of Memory!\n"); 
        return;
    }

    // send the message and flush the connection
    if (!dbus_connection_send(appdata->con, msg, &serial)) { 
        fprintf(stderr, "Out Of Memory!\n"); 
        return;
    }
    dbus_connection_flush(appdata->con); // not sure if this is needed or a good idea
   
    // free the message 
    dbus_message_unref(msg);

    //g_free(payload);    
}



// ===============================================================================================
// Handlers for our different payload types
// ===============================================================================================


void openInstallFile(char *data1)
{
    FILE *install_file;
    gchar *data = g_strdup(data1);
    char *filename=NULL;
    char *p, *pp, *s, *e;
    
    FILE *fid;
    
    fid = fopen("/home/user/mbarcode.debug", "w+");
    fprintf(fid,"Start\n");
    fflush(fid);
    
    // decide on a file name for our install file
    
    // first see if this is an install file for a package or for a repo only
    p = strstr(data, "package"); // see if we find the keyword "package"
    if(p){
        fprintf(fid,"First loop found text\n");
        fflush(fid);
        p = strstr(p, "="); // if so, look for the "="
        if(p){
            p++; // skip past the =
            // find the end of the line, so we can work out the name (it might have spaces)
            s = strstr(p,"\r");
            if(!s)
                s = strstr(p,"\n");
            
            if(!s)
                pp = g_strdup(p); // no line breaks found apparently
            else
                pp = g_strndup(p, (int)(s-p+1)); // so we don't get the line breaks
            
        
            // assume we've now found the package name between p and the end of the string
            // get rid of white space on both ends
            fprintf(fid,"First loop about to strip\n");
            fflush(fid);
            pp = g_strstrip(pp);
            
            fprintf(fid,"First loop about to remove spaces\n");
            fflush(fid);
            // remove any spaces in the name
            e = strstr(pp," "); 
            while(e){
                e[0] = '_';
                e = strstr(pp," ");
            }
            
            fprintf(fid,"First loop generating filename: %s\n", pp);
            fflush(fid);
            //filename = g_strdup_printf("%s/%s.install", getenv ("MYDOCSDIR"), pp); // assuming the env var exists here, no checks made
            filename = g_strdup_printf("/home/user/MyDocs/%s.install", pp);
            g_free(pp);

        }else{ // something wrong here
            fprintf(fid,"First loop no =\n");
            fflush(fid);
        }
        
    }

    
    fprintf(fid,"Left first loop\n");
    fflush(fid);
    
    if(!filename){
        // it might just have been a repo, so try to extract this name now
        p = strstr(s, "repo_name"); // see if we find the keyword "package"
        if(p){
            fprintf(fid,"Second loop found text\n");
            fflush(fid);
            p = strstr(p, "="); // if so, look for the "="
            if(p){
                p++; // skip past the =
                // find the end of the line, so we can work out the name (it might have spaces)
                s = strstr(p,"\r");
                if(!s)
                    s = strstr(p,"\n");
                
                if(!s)
                    pp = g_strdup(p); // no line breaks found apparently
                else
                    pp = g_strndup(p, (int)(s-p+1)); // so we don't get the line breaks
            
                // assume we've now found the package name between p and the end of the string
                // get rid of white space on both ends
                fprintf(fid,"First loop about to strip\n");
                fflush(fid);
                pp = g_strstrip(pp);

                // remove any spaces in the name
                fprintf(fid,"Second loop about to remove spaces\n");
                fflush(fid);
                e = strstr(pp," "); 
                while(e){
                    e[0] = '_';
                    e = strstr(pp," ");
                }

                fprintf(fid,"Second loop generating filename: %s\n", pp);
                fflush(fid);
                //filename = g_strdup_printf("%s/%s.install", getenv ("MYDOCSDIR"), pp); // assuming the env var exists here, no checks made
                filename = g_strdup_printf("/home/user/MyDocs/%s.install", pp);
                g_free(pp);
            
            }else{ // something wrong here
                fprintf(fid,"Second loop no =\n");
                fflush(fid);
                
            }       
        }            
    }
    
    fprintf(fid,"Left second loop\n");
    fflush(fid);
    
    
    if(!filename){
        // we'll just have to generate a filename now it seems
        //filename = g_strdup_printf("%s//%s.install", getenv ("MYDOCSDIR"), tmpnam(NULL));
        filename = g_strdup_printf("/home/user/MyDocs/%s.install", tmpnam(NULL));
        fflush(fid);
        fprintf(fid,"Generating random filename\n");
    }
    
    /*
    // see if it already exists
    install_file = fopen(filename, "r");
    if(install_file){
        // it already exists!
        fclose(install_file);
    }
    */
    

    if((install_file = fopen(filename, "w+")) == NULL){
        fprintf(fid,"Failed to open filename\n");
        fflush(fid);
        g_free(filename);
        g_free(data);
        return; // and give up, for want of a better way of recovering...
    }
    
    fprintf(fid,"Writing data\n");
    fflush(fid);
    
    fwrite(data, 1, strlen(data), install_file);
    fclose(install_file);

    fprintf(fid,"about to openGeneric()\n");
    fflush(fid);
    
    openGeneric(filename, "Open install file: %s?");

    fclose(fid);
    g_free(filename);
    g_free(data);

}



void openGeneric(char *uri1, char *question)
{
    GtkWidget *dialog;
    gchar *uri = g_strdup(uri1);

    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            question, uri);
            
    g_signal_connect(dialog, "response", G_CALLBACK (openGeneric_response_cb), uri);
    gtk_window_set_transient_for  (GTK_WINDOW (dialog), GTK_WINDOW (appdata->window));
    gtk_widget_show_all (dialog);
    
}


void openGeneric_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *uri = (char*)user_data;
    GError* error;
    HildonURIAction* action;
    
    gtk_widget_destroy(dialog);
    
    switch (response) {
        case GTK_RESPONSE_YES:
            action = hildon_uri_get_default_action_by_uri(uri, &error);
            hildon_uri_open(uri, action, &error); //hildon_uri_open (uri, HILDON_URI_ACTION_NORMAL, error);
            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    g_free(uri);
}




/*
        dbus-send --dest=com.nokia.CallUI --type=method_call /com/nokia/CallUI com.nokia.CallUI.CreateCall string:"tel:0123456"
        
        msg = dbus_message_new_method_call("com.nokia.CallUI", // target for the method call
         "/com/nokia/CallUI", // object to call on
         "com.nokia.CallUI.CreateCall", // interface to call on
         "string:\"tel:0123456\""); // method name
*/

void openTel(char *uri)
{
    GtkWidget *dialog;
    gchar *tel = g_strdup(uri);
    
    printf("Opening Phone app to dial: %s\n", tel);

    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            "Dial number: %s", tel);
            
    g_signal_connect(dialog, "response", G_CALLBACK (openTel_response_cb), tel);
    gtk_window_set_transient_for  (GTK_WINDOW (dialog), GTK_WINDOW (appdata->window));
    gtk_widget_show_all (dialog);
    
}


// http://efforts.embedded.ufcg.edu.br/maemo/
void openTel_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *tel = (char*)user_data;
    GError* error;
    HildonURIAction* action;
    
    gtk_widget_destroy(dialog);
    
    switch (response) {
        case GTK_RESPONSE_YES:
            action = hildon_uri_get_default_action_by_uri(tel, &error);
            hildon_uri_open(tel, action, &error); //hildon_uri_open (uri, HILDON_URI_ACTION_NORMAL, error);
            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    g_free(tel);
}






void openEmail(char *address)
{
    GtkWidget *dialog;
    gchar *uri = g_strdup(address);
    
    printf("Opening Email client to send to: %s\n", uri);

    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            "Send email to: %s", uri);
            
    g_signal_connect(dialog, "response", G_CALLBACK (openEmail_response_cb), uri);
    gtk_window_set_transient_for  (GTK_WINDOW (dialog), GTK_WINDOW (appdata->window));
    gtk_widget_show_all (dialog);
    
}


// http://wiki.maemo.org/Documentation/Maemo_5_Developer_Guide/Using_Data_Sharing/SendVia_Functionality
void openEmail_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *uri = (char*)user_data;
    gboolean result = TRUE;
        
    gtk_widget_destroy(dialog);
    
    switch (response) {
        case GTK_RESPONSE_YES:
            result = libmodest_dbus_client_compose_mail(appdata->osso, /*osso_context_t*/
                uri, /*to*/
                NULL, /*cc*/
                NULL, /*bcc*/
                NULL, /*body*/
                NULL, /*subject*/
                NULL /*attachments*/);

            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    if(result)
        printf("Opened email... leaving.\n");
    else
        printf("Failed to open email... leaving.\n");

    g_free(uri);
}


// good lad qwerty12 (http://lists.maemo.org/pipermail/maemo-developers/2009-September/020791.html)

// check this!
// I've been naughty here and have hardcoded the method names etc. but
// these are all defined in /usr/include/tablet-browser-interface.h



#ifdef N900

void openURL(char *open)
{
    GtkWidget *dialog;
    gchar *uri = g_strdup(open);
    
    printf("Opening URL in browser: %s\n", uri);

    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            "Open URL: %s", uri);
            
    g_signal_connect(dialog, "response", G_CALLBACK (openURL_response_cb), uri);
    gtk_window_set_transient_for  (GTK_WINDOW (dialog), GTK_WINDOW (appdata->window));
    gtk_widget_show_all (dialog);
    
}


void openURL_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *uri = (char*)user_data;
    DBusGConnection *connection;
    GError *error = NULL;
    DBusGProxy *proxy;
    
    gtk_widget_destroy(dialog);
    
    switch (response) {
        case GTK_RESPONSE_YES:
            connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
            if (connection == NULL){
                printf("%s\n",error->message); // make this a notification eventually
                g_error_free (error);
                return;
            }

            proxy = dbus_g_proxy_new_for_name (connection,
				         "com.nokia.osso_browser",
				         "/com/nokia/osso_browser/request",
				         "com.nokia.osso_browser");

            error = NULL;
            
            printf("About to open %s\n", uri);
            
            if (!dbus_g_proxy_call (proxy, "load_url", &error, G_TYPE_STRING, uri, G_TYPE_INVALID, G_TYPE_INVALID)){
                printf("%s\n",error->message); // make this a notification eventually
                g_error_free (error);
            }
            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    printf("Opened URL... leaving.\n");
    g_free(uri);
}

#else

void openURL(char *open)
{
    GtkWidget *dialog;
    gchar *uri = g_strdup(open);
    
    
    
    printf("Opening URL in browser: %s\n", uri);

    

    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            "Open URL: %s", uri);
            
    g_signal_connect(dialog, "response", G_CALLBACK (openURL_response_cb), uri);
        
    gtk_widget_show_all (dialog);
    
}

void openURL_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *uri = (char*)user_data;
    osso_rpc_t* osso_retval = { 0 };

    gtk_widget_destroy(dialog);
    
    
    switch (response) {
        case GTK_RESPONSE_YES:
            /*Create the osso context for the browser application.*/
            /*Call the given function.*/
            
            printf("Open URL: %s, length=%d\n", uri, strlen(uri));
            
            osso_rpc_run(appdata->osso, "com.nokia.osso_browser","/com/nokia/osso_browser",
            "com.nokia.osso_browser","open_new_window", NULL, DBUS_TYPE_STRING, uri, DBUS_TYPE_INVALID);

/*
            osso_rpc_run(appdata->osso, "com.nokia.osso_browser","/com/nokia/osso_browser/request",
            "com.nokia.osso_browser","load_url", NULL, DBUS_TYPE_STRING, uri, DBUS_TYPE_INVALID);
*/
            /*Free all memory allocated*/

            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    printf("Opened URL... leaving.\n");
    g_free(uri);
}
#endif



#ifdef N900
void addVCard(char *vcard)
{    
    GtkWidget *dialog;
        
    printf("Adding vCard...\n");

   
    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_QUESTION,
            GTK_BUTTONS_YES_NO,
            "Add vCard: %s", "");
            
            // we should really parse the vCard first and let the user see what they are adding!
       
    g_signal_connect(dialog, "response", G_CALLBACK (addVCard_response_cb), (gpointer)vcard);
        
    gtk_widget_show_all (dialog);            
}




void addVCard_response_cb (GtkDialog *dialog, gint response, gpointer user_data)
{
    char *vcard = (char*)user_data;
    OssoABookContact *contact;
    GError *error = NULL;
    EBook *book = NULL;
    
    gtk_widget_destroy(dialog);
     
    
    
    switch (response) {
        case GTK_RESPONSE_YES:

            //
            // Thanks to Reto Zingg and his compatriotes at Nokia for this code
            //
            
            book = osso_abook_system_book_dup_singleton (TRUE, NULL);
            g_return_val_if_fail (E_IS_BOOK (book), FALSE);
            
            if (!vcard)
                return;

            contact = osso_abook_contact_new_from_vcard (NULL, vcard);
            g_assert (OSSO_ABOOK_IS_CONTACT (contact));

            if (!e_book_add_contact (book, E_CONTACT (contact), &error)) {
                    g_warning ("%s: %s", G_STRLOC, error->message);
                    g_assert_not_reached ();
            }
            break;
        default:
            break; // dialog was cancelled, so do nothing
    }
    printf("Added vCard... leaving.\n");
    return;   
}
#endif /* N900 */


