/**
 * maemo-barcode - barcode detection/recognition application
 *
 * Copyright (c) 2008 Simon Pickering <S.G.Pickering@bath.ac.uk>
 *
 * Various parts of barcode recognition and GStreamer manipulation code written by:
 *       Timothy Terriberry
 *       Adam Harwell
 *       Jonas Hurrelmann
 *
 * Original GStreamer code based on the maemo-examples package
 * Copyright (c) 2007-2008 Nokia Corporation. All rights reserved.
 * Copyright (c) 2006 INdT.
 * @author Talita Menezes <talita.menezes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 * @author Jami Pekkanen <jami.pekkanen@nokia.com>
 */

#include <stdio.h>
#include <sqlite3.h>
#include <glib.h>
#include <ctype.h>

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>

#include "common.h"
#include "lists.h"


/*
   TABLE: lists
   COL_KEY=0,
   COL_SQL_NAME=0,
   COL_PRETTY_NAME

   TABLE: (sql_name)
   COL_KEY=0,
   COL_EAN,
   COL_ISBN,
   COL_TITLE,
   COL_AUTHOR,
   COL_PRICE,
   COL_PLACE,
   COL_NUMBER,
   COL_DATE
   COL_TABLE_KEY

*/


extern AppData *appdata;


char pretty_column_names[9][6] = {"KEY", "EAN", "ISBN", "Title", "Author", "Price", "Place", "Number", "Date"};
char sql_column_names[9][10] = {"COL_KEY", "COL_EAN", "COL_ISBN", "COL_TITLE", "COL_AUTHOR", "COL_PRICE", "COL_PLACE", "COL_NUMBER", "COL_DATE"};


/* ======================= */
/* UI related stuff        */
/* ======================= */

// Lists views - these are the tables



// This is to create the 'table' which shows the tables in the database (e.g. CDs, DVDs, Wish-list)
GtkTreeModel *create_and_fill_lists_model () {
    GtkListStore  *store;
    GtkTreeIter    iter;

    char sqlite_name[255], pretty_name[255];
    int key;

    sqlite3_stmt *stmt;
    gchar command[] = "select * from lists;";
    int rc=1;

    // we need to read the number of columns and their types from the database
    // thought we could just set some standard ones for the time being
    // COL_SQL_NAME, COL_PRETTY_NAME
    store = gtk_list_store_new (3, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING);

    // now fill in this list store from the database in a loop

    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc)
        printf("create_and_fill_lists_model: select * from lists: %i, %s\n", rc, sqlite3_errmsg(appdata->db));
        // need to add a way to escape here
        
    if (rc==SQLITE_OK) {
        while (1) {
            rc = get_lists_row(stmt, &key, sqlite_name, pretty_name);
            if (rc==SQLITE_ROW) {
                /* Append a row and fill in some data */
                gtk_list_store_append (store, &iter); // add a new "row"
                gtk_list_store_set (store, &iter,
                                    COL_KEY, key,
                                    COL_SQL_NAME, sqlite_name,
                                    COL_PRETTY_NAME, pretty_name,
                                    -1);
            } else
                break;
        }
    } else {
        // some sort of error
    }
    rc = sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("create_and_fill_lists_model: finalise: %s\n", sqlite3_errmsg(appdata->db));

    return GTK_TREE_MODEL (store);
}

// This is to create the 'table' which shows the tables in the database (e.g. CDs, DVDs, Wish-list)
GtkWidget *create_lists_view_and_model () {
    GtkCellRenderer     *renderer;
    //GtkTreeModel        *model;
    GtkWidget           *view;
    GtkTreeSelection *select;
    GtkTreeIter    iter;

    view = gtk_tree_view_new ();

    /* --- Column #1 --- */
    /* We only want one colum, and we only want to display the pretty name, not the name that sqlite uses */

    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "List Name",
            renderer,
            "text",
            COL_PRETTY_NAME,
            NULL);
    appdata->lists_store = create_and_fill_lists_model(); //(GtkListStore *)
    gtk_tree_view_set_model(GTK_TREE_VIEW (view), appdata->lists_store); //(GtkTreeModel *)


    /* Setup the selection handler */
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
    gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

    g_signal_connect(select, "changed", G_CALLBACK(on_list_view_changed), NULL);


    // select the first item in the list
    // sod it, segfault-tastic
    /*
    if(gtk_tree_model_get_iter_first(appdata->lists_store, &iter)){
        printf("before\n");        
        gtk_tree_view_set_cursor(view, gtk_tree_model_get_path(appdata->lists_store,&iter), NULL, FALSE);
    }
    printf("after\n");
    */
    
    /*
    if(gtk_tree_model_get_iter_first(appdata->lists_store, &iter)){
        printf("before\n");        
        gtk_tree_selection_select_iter(select, &iter); // gtk_tree_view_get_selection(view)
    }
    printf("after\n");
    */

    /* The tree view has acquired its own reference to the
     *  model, so we can drop ours. That way the model will
     *  be freed automatically when the tree view is destroyed */
    //g_object_unref (model);

    return view;
}

char *GetCurrentListViewTable() // make sure you g_free() the output of this fn!
{
    GtkTreeIter    iter;    
    GtkTreeSelection *select;
    char *lists_sqlite_name;
    GtkTreeModel        *model;
    
    fprintf(stdout,"Entered GetCurrentListViewTable()\n");
    
    lists_sqlite_name = NULL; // by default we want to return null
    
    
    if (!appdata->lists_view)
        return lists_sqlite_name;
    
    fprintf(stdout,"Still in GetCurrentListViewTable()\n");
    
    // the list has been changed...
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->lists_view));
    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) // if there's a selection
        gtk_tree_model_get(model, &iter, COL_SQL_NAME, &lists_sqlite_name, -1); // get the SQL name for this list
    
    fprintf(stdout,"Leaving GetCurrentListViewTable()\n");
    return lists_sqlite_name;
}

char *GetCurrentDataViewItem() // make sure you g_free() the output of this fn!
{
    GtkTreeIter    iter;    
    GtkTreeSelection *select;
    char *EAN;
    GtkTreeModel        *model;

    fprintf(stdout,"Entered GetCurrentDataViewItem()\n");

    if (!appdata->data_view)
        return EAN;
    
    fprintf(stdout,"Still in GetCurrentDataViewItem()\n");
    
    // the list has been changed...
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->data_view));
    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) // if an item is selected
        gtk_tree_model_get(model, &iter, COL_EAN, &EAN, -1); // get the EAN for this item
    return EAN;
}

void  on_list_view_changed(GtkWidget *widget) 
{
    char *lists_sqlite_name =GetCurrentListViewTable();
    
    fprintf(stdout,"Entered on_list_view_changed()\n");
    
    // alter the data for the data_view
    refill_data_model(lists_sqlite_name);
    
    g_free(lists_sqlite_name);
    
    return;
}

void  on_data_view_changed(GtkWidget *widget) 
{
    char *EAN; // = GetCurrentDataViewItem();
    
    fprintf(stdout,"Entered on_data_view_changed()\n");
    
    return;
        
    // alter something, perhaps the detail view, to show the data for this record...
    fprintf(stdout, "Selected EAN = %s\n", EAN);
    g_free(EAN);
}

// Data views - these are the data in the tables

GtkTreeModel *create_and_fill_data_model (char *sql_table_name) {
    GtkListStore  *store;
    GtkTreeIter    iter;

    char EAN[14], ISBN[14], title[255], author[255], place[255];
    float price, number;
    unsigned int datetime;
    int key, table_key;

    //char *zErrMsg = 0;
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;

    // obtain the key (i.e. unique record identifier of this table) from the lists table so we can save it in the treeview object for later use
    if (sql_table_name!=NULL && strlen(sql_table_name)>0)
        table_key = get_key_for_table(sql_table_name);

    // we need to read the number of columns and their types from the database
    // thought we could just set some standard ones for the time being
    // EAN, ISBN, Title, Author, Price, Place, Number, Time/Date, table_key
    //store = gtk_list_store_new (8, G_TYPE_UINT64, G_TYPE_UINT64, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT);
    store = gtk_list_store_new (10, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT, G_TYPE_UINT);

    // now fill in this list store from the database in a loop
    if (sql_table_name==NULL || strlen(sql_table_name)==0){
        printf("Invalid data table name, return empty store\n");
        return GTK_TREE_MODEL (store); // or jump out if no table selected
    }
    
    // create the correct command string, with the right table name in it
    command = sqlite3_mprintf("select * from '%q';", sql_table_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    sqlite3_free(command);
    if (rc==SQLITE_OK) {
        while (1) {
            rc = get_next_data_row(stmt, &key, EAN, ISBN, title, author, &price, place, &number, &datetime);
            if (rc==SQLITE_ROW) {
                /* Append a row and fill in some data */
                gtk_list_store_append (store, &iter); // add a new "row"
                gtk_list_store_set (store, &iter,
                                    COL_KEY, key,
                                    COL_EAN, EAN,
                                    COL_ISBN, ISBN,
                                    COL_TITLE, title,
                                    COL_AUTHOR, author,
                                    COL_PRICE, price,
                                    COL_PLACE, place,
                                    COL_NUMBER, number,
                                    COL_DATE, datetime,
                                    COL_TABLE_KEY, table_key,
                                    -1);
            } else {
                break;
            }

        }
    } else {
        printf("create_and_fill_data_model: select * from %s: %s\n", sql_table_name, sqlite3_errmsg(appdata->db));
    }
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("create_and_fill_data_model: finalise: %s\n", sqlite3_errmsg(appdata->db));
    
    strcpy(appdata->current_data_table_name, sql_table_name);
    
    return GTK_TREE_MODEL (store);

}


void refill_data_model (char *sql_table_name) {
    GtkListStore  *store;
    GtkTreeIter    iter;

    char EAN[14], ISBN[14], title[255], author[255], place[255];
    float price, number;
    unsigned int datetime;
    int key, table_key;

    //char *zErrMsg = 0;
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;

    if (sql_table_name==NULL || strlen(sql_table_name)==0){ // don't refill is there's no table selected
        printf("refill_data_model(): No table selected, returning\n");
        return;
    }else if (strcmp(sql_table_name, appdata->current_data_table_name)==0){
        printf("refill_data_model(): Same table, returning\n");
        return;
    }else
        printf("refill_data_model(): Table changed, continuing\n");
    
    
    
    
    
    fprintf(stdout, "Refilling data view\n");
    
    // obtain the key (i.e. unique record identifier of this table) from the lists table so we can save it in the treeview object for later use   
    table_key = get_key_for_table(sql_table_name);
    
    // remove the data from the store
    if(appdata->data_store){ // if we exist
        printf("refill_data_model(): data_store exists, clear data and refill\n");
        gtk_list_store_clear(appdata->data_store); // clear existing data
    }else{
        printf("refill_data_model(): data_store does not exist, create and fill\n");
        appdata->data_store = gtk_list_store_new (10, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT, G_TYPE_UINT);
    }
    
    // create the correct command string, with the right table name in it
    command = sqlite3_mprintf("select * from '%q';", sql_table_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    sqlite3_free(command);
    if (rc==SQLITE_OK) {
        while (1) {
            rc = get_next_data_row(stmt, &key, EAN, ISBN, title, author, &price, place, &number, &datetime);
            if (rc==SQLITE_ROW) {
                /* Append a row and fill in some data */
                gtk_list_store_append (appdata->data_store, &iter); // add a new "row"
                gtk_list_store_set (appdata->data_store, &iter,
                                    COL_KEY, key,
                                    COL_EAN, EAN,
                                    COL_ISBN, ISBN,
                                    COL_TITLE, title,
                                    COL_AUTHOR, author,
                                    COL_PRICE, price,
                                    COL_PLACE, place,
                                    COL_NUMBER, number,
                                    COL_DATE, datetime,
                                    COL_TABLE_KEY, table_key,
                                    -1);
            } else {
                break;
            }

        }
    } else {
        printf("create_and_fill_data_model: select * from %s: %s\n", sql_table_name, sqlite3_errmsg(appdata->db));
    }
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    
    
    // update the current data table name
    strcpy(appdata->current_data_table_name, sql_table_name);
    
}


// Data views - these are the data in the tables
GtkWidget *create_data_view_and_model(char *sql_table_name) {
    GtkCellRenderer     *renderer;
    //GtkTreeModel        *model;
    GtkWidget           *view;
    GtkTreeSelection *select;

    view = gtk_tree_view_new ();

    /* --- Column #1 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "EAN",
            renderer,
            "text", COL_EAN,
            NULL);
    /* --- Column #2 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "ISBN",
            renderer,
            "text", COL_ISBN,
            NULL);
    /* --- Column #3 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Title",
            renderer,
            "text", COL_TITLE,
            NULL);
    /* --- Column #4 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Author",
            renderer,
            "text", COL_AUTHOR,
            NULL);
    /* --- Column #5 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Price",
            renderer,
            "text", COL_PRICE,
            NULL);
    /* --- Column #6 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Place",
            renderer,
            "text", COL_PLACE,
            NULL);
    /* --- Column #7 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Number",
            renderer,
            "text", COL_NUMBER,
            NULL);
    /* --- Column #8 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Time/Date",
            renderer,
            "text", COL_DATE,
            NULL);

    // see if there's a table for us to link to

    if (sql_table_name!=NULL && strlen(sql_table_name)!=0) {
        printf("create_data_view_and_model(): sql_table_name=%s\n", sql_table_name);
    } else {
        printf("create_data_view_and_model(): no table selected\n");
    }
    
    printf("DEBUG: %s\n", sql_table_name);
    appdata->data_store = create_and_fill_data_model (sql_table_name); // (GtkListStore *)
    gtk_tree_view_set_model(GTK_TREE_VIEW (view), appdata->data_store); // (GtkTreeModel *)


    /* Setup the selection handler */
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
    gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW (view), TRUE);


    g_signal_connect(select, "changed", G_CALLBACK(on_data_view_changed), NULL);

    /* The tree view has acquired its own reference to the
     *  model, so we can drop ours. That way the model will
     *  be freed automatically when the tree view is destroyed */

    //g_object_unref (model);

    return view;
}




// Data views - these are the data in the tables

GtkTreeModel *create_and_fill_search_model (char *sql_table_name) {
    GtkListStore  *store;
    GtkTreeIter    iter;

    char EAN[14], ISBN[14], title[255], author[255], place[255];
    float price, number;
    unsigned int datetime;
    int key, table_key;

    //char *zErrMsg = 0;
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;

    // obtain the key (i.e. unique record identifier of this table) from the lists table so we can save it in the treeview object for later use
    table_key = get_key_for_table(sql_table_name);

    // we need to read the number of columns and their types from the database
    // thought we could just set some standard ones for the time being
    // EAN, ISBN, Title, Author, Price, Place, Number, Time/Date, table_key
    //store = gtk_list_store_new (8, G_TYPE_UINT64, G_TYPE_UINT64, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT);
    store = gtk_list_store_new (10, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT, G_TYPE_UINT);

    // now fill in this list store from the database in a loop

    // create the correct command string, with the right table name in it
    command = sqlite3_mprintf("select * from '%q';", sql_table_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    sqlite3_free(command);
    if (rc==SQLITE_OK) {
        rc = SQLITE_ROW;
        while (rc==SQLITE_ROW) {
            rc = get_next_data_row(stmt, &key, EAN, ISBN, title, author, &price, place, &number, &datetime);
            /* Append a row and fill in some data */
            gtk_list_store_append (store, &iter); // add a new "row"
            gtk_list_store_set (store, &iter,
                                COL_KEY, key,
                                COL_EAN, EAN,
                                COL_ISBN, ISBN,
                                COL_TITLE, title,
                                COL_AUTHOR, author,
                                COL_PRICE, price,
                                COL_PLACE, place,
                                COL_NUMBER, number,
                                COL_DATE, datetime,
                                COL_TABLE_KEY, table_key,
                                -1);
        }
    } else {
        printf("create_and_fill_search_model: select * from %s: %s\n", sql_table_name, sqlite3_errmsg(appdata->db));
    }
    rc = sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("create_and_fill_search_model: finalise: %s\n", sqlite3_errmsg(appdata->db));
    return GTK_TREE_MODEL (store);

}

// Data views - these are the data in the tables
GtkWidget *create_search_view () {
    GtkCellRenderer     *renderer;
    //GtkTreeModel        *model;
    GtkWidget           *view;
    GtkTreeSelection *select;

    view = gtk_tree_view_new ();

    /* --- Column #1 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "EAN",
            renderer,
            "text", COL_EAN,
            NULL);
    /* --- Column #2 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "ISBN",
            renderer,
            "text", COL_ISBN,
            NULL);
    /* --- Column #3 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Title",
            renderer,
            "text", COL_TITLE,
            NULL);
    /* --- Column #4 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Author",
            renderer,
            "text", COL_AUTHOR,
            NULL);
    /* --- Column #5 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Price",
            renderer,
            "text", COL_PRICE,
            NULL);
    /* --- Column #6 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Place",
            renderer,
            "text", COL_PLACE,
            NULL);
    /* --- Column #7 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Number",
            renderer,
            "text", COL_NUMBER,
            NULL);
    /* --- Column #8 --- */
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
            -1,
            "Time/Date",
            renderer,
            "text", COL_DATE,
            NULL);

    

    //gtk_tree_view_set_model(GTK_TREE_VIEW (view), appdata->data_store);


    /* Setup the selection handler */
    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
    gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);


    /* The tree view has acquired its own reference to the
     *  model, so we can drop ours. That way the model will
     *  be freed automatically when the tree view is destroyed */

    //g_object_unref (model);

    return view;
}




/* ================================== */
/* Overall database stuff             */
/* ================================== */

/* Do any init, open the database */
sqlite3* init_db() {
    sqlite3 *db;
    int rc;
    char db_name[255] = "\n";
    char *db_dirname = NULL;
    char **pszResult;
    int nRow, nColumn;

    // find homedir
    strcpy(db_name, strcat(getenv("HOME"), "/.mbarcode/mbarcode.db"));

    printf("Opening database... %s\n", db_name);


    // borrowed this error checking from maemo-mapper, poi.c

    if(!g_file_test(db_name, G_FILE_TEST_EXISTS)){
        fprintf(stdout, "Directory does not exist, creating.\n");
        g_mkdir_with_parents(g_path_get_dirname(db_name), 0755);
    }
    
    if(SQLITE_OK != (sqlite3_open(db_name, &db))){
        printf("Error opening database: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        db = NULL;
        // create a popup warning
        exit(1); // quit
    } 
        
    // check the table exists
    if (SQLITE_OK != sqlite3_get_table(db,"select COL_KEY from lists limit 1;",&pszResult, &nRow, &nColumn, NULL)) {
        
        gchar *create_sql = sqlite3_mprintf("create table lists (COL_KEY integer PRIMARY KEY, COL_SQL_NAME text, COL_PRETTY_NAME text);");

        if (SQLITE_OK != sqlite3_exec(db, create_sql, NULL, NULL, NULL)){
            printf("Error with database: %s\n", sqlite3_errmsg(db));
            sqlite3_close(db);
            db = NULL;
            exit(1);
        }
    } else {
        sqlite3_free_table(pszResult);
    }

    g_free(db_dirname);




    /*
        // I must create the directory if it doesn't alreay exist!

        rc = sqlite3_open(db_name, &db);
        if( rc ){
            fprintf(stderr, "Can't open database: %s\n", sqlite3_errmsg(db));
            sqlite3_close(db);
            return 0;
        }

        // now we try to create the lists table, this should fail if it already exists, that's fine, just as long as it exists!
        create_lists_table(db);
    */
    return db;


}

/* Do any cleanup, close the database */
void close_db() {
    int rc;
    
    /*
    // finalise any outstanding statements
    sqlite3_stmt *pStmt;
    while( (pStmt = sqlite3_next_stmt(appdata->db, 0))!=0 ){
        printf("Finalising...\n");
        sqlite3_finalize(pStmt);
    }
    */
    
    // close the database
    rc = sqlite3_close(appdata->db);
    if (rc==SQLITE_BUSY)
        printf("Error closing the database\n");
}

/* Get a row from the lists table - this is the table which contains the names of the tables in the db (CDs, DVDs, etc.) */
// the unsigned int *key, char *sqlite_name, char *pretty_name pointers are allocated outside this function (as char[255]) and data is returned in them
int get_lists_row(sqlite3_stmt *stmt, int *key, char *sqlite_name, char *pretty_name) {
    int rc;
    char *zErrMsg = 0;
    char *temp;
    rc = sqlite3_step(stmt);
    if ( rc==SQLITE_DONE ) { // no more rows left
        sqlite_name[0]=0; //"";
        pretty_name[0]=0; //"";
        return rc;
    } else if ( rc==SQLITE_ROW ) {
        *key = (int)sqlite3_column_int(stmt, COL_KEY);
        temp = (char *)sqlite3_column_text(stmt, COL_SQL_NAME);
        strcpy(sqlite_name, temp);
        temp = (char *)sqlite3_column_text(stmt, COL_PRETTY_NAME);
        strcpy(pretty_name, temp);
    } else {
        fprintf(stderr, "get_lists_row(): SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
    }
    return rc;
}

/*
int test_db_is_ok() {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;
    command = sqlite3_mprintf("select * from lists;");
    //rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(appdata->db, command, stmt);
    rc = sqlite3_exec(appdata->db, command, NULL, NULL, NULL); //rc = submit_query(appdata->db, command, stmt);
    if (rc==SQLITE_OK) {
        printf("test_db_is_ok: Prepare statement was ok\n");
    } else {
        printf("test_db_is_ok: Prepare statement error: %s\n", sqlite3_errmsg(appdata->db));
        //rc = sqlite3_finalize(stmt);
        return rc;
    }
    rc = sqlite3_step(stmt);
    if ( rc==SQLITE_DONE ) { // no more rows left
        printf("test_db_is_ok: step statement was ok, no rows\n");
    } else if ( rc==SQLITE_ROW ) {
        printf("test_db_is_ok: step statement was ok, found rows\n");
    } else {
        printf("test_db_is_ok: step statement error: %s\n", sqlite3_errmsg(appdata->db));
        rc = sqlite3_finalize(stmt);
        return rc;
    }
    rc = sqlite3_finalize(stmt);
    return rc;
}
*/

// search the lists table for a given string return its key (unique id)
int get_key_for_table(char *sqlite_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;
    int key;

    command = sqlite3_mprintf("select * from lists where COL_SQL_NAME='%q';", sqlite_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(appdata->db, command, stmt);
    if (rc==SQLITE_OK) {
        char *zErrMsg = 0;
        rc = sqlite3_step(stmt);
        if ( rc==SQLITE_DONE ) { // no more rows left
            key=-1;
        } else if ( rc==SQLITE_ROW ) {
            key = (int)sqlite3_column_int(stmt, COL_KEY);
        } else {
            fprintf(stderr, "get_lists_row(): SQL error: %s\n", zErrMsg);
            sqlite3_free(zErrMsg);
            key=-1;
        }
    } else {
        printf("get_key_for_table(): submit_query failed searching for table '%s', returning 0: %s\n", sqlite_name, sqlite3_errmsg(appdata->db));
        key=-1;
    }
    sqlite3_free(command);
    rc = sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_key_for_table: finalise: %s\n", sqlite3_errmsg(appdata->db));
    return key;
}

// search the lists table for a given key and return its name (string)
int get_table_from_key(int key, char *sqlite_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    gchar *temp;
    int rc=1;

    command = sqlite3_mprintf("select * from lists where COL_KEY=%i;", key);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(appdata->db, command, stmt);
    if (rc==SQLITE_OK) {
        char *zErrMsg = 0;
        rc = sqlite3_step(stmt);
        if ( rc==SQLITE_DONE ) { // no more rows left
            rc=0;
        } else if ( rc==SQLITE_ROW ) {
            rc = 1;
            temp = (char *)sqlite3_column_text(stmt, COL_PRETTY_NAME);
            strcpy(sqlite_name, temp); // make a copy as this will be freed automatically
        } else {
            fprintf(stderr, "get_lists_row(): SQL error: %s\n", zErrMsg);
            sqlite3_free(zErrMsg);
            rc=0;
        }
    } else {
        printf("get_table_from_key(): submit_query failed, returning NULL\n");
        key=0;
    }
    sqlite3_free(command);
    rc = sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_table_from_key: finalise: %s\n", sqlite3_errmsg(appdata->db));
    return key;
}


/* Get a row from a data table - currently we fix the data in the table, this could be made user-alterable eventually */
int get_next_data_row(sqlite3_stmt *stmt, int *key, char *EAN, char *ISBN, char *title, char *author, float *price, char *place, float *number, unsigned int *datetime) {
    int rc;
    char *zErrMsg = 0;
    char *temp;
    rc = sqlite3_step(stmt);
    if ( rc==SQLITE_ROW ) {
        // EAN, ISBN, Title, Author, Price, Place, Number
        // G_TYPE_UINT64, G_TYPE_UINT64, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT

        *key = (int)sqlite3_column_int(stmt,COL_KEY);
        temp = (char *)sqlite3_column_text(stmt,COL_EAN);
        strcpy(EAN, temp);
        temp = (char *)sqlite3_column_text(stmt, COL_ISBN);
        strcpy(ISBN, temp);
        temp = (char *)sqlite3_column_text(stmt, COL_TITLE);
        strcpy(title, temp);
        temp = (char *)sqlite3_column_text(stmt, COL_AUTHOR);
        strcpy(author, temp);
        *price = (float)sqlite3_column_double(stmt, COL_PRICE);
        temp = (char *)sqlite3_column_text(stmt, COL_PLACE);
        strcpy(place, temp);
        *number = (float)sqlite3_column_double(stmt, COL_NUMBER);
        *datetime = (unsigned int)sqlite3_column_int(stmt,COL_DATE);
    } else if (rc!=SQLITE_DONE) {
        fprintf(stderr, "SQL error: %s\n", zErrMsg);
        sqlite3_free(zErrMsg);
    }
    // else, just no rows left, which is reflected in the value of rc
    return rc;
}


/* Get a row from a data table - currently we fix the data in the table, this could be made user-alterable eventually */
int get_data_row(char *sql_name, int *key, char *EAN, char *ISBN, char *title, char *author, float *price, char *place, float *number, unsigned int *datetime) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc;
    char *temp;

    command = sqlite3_mprintf("select * from '%q' where COL_KEY=%d;", sql_name, *key);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    printf("get_data_row: select * from %s where COL_KEY=%i: %s\n", sql_name, *key, sqlite3_errmsg(appdata->db));
    rc = sqlite3_step(stmt);
    sqlite3_free(command);
    if ( rc!=SQLITE_ROW ) {
        rc=sqlite3_finalize(stmt); //finish_query(stmt);
        printf("get_data_row: error: %s\n", sqlite3_errmsg(appdata->db));
        return rc; // not found
    }

    //*key = (int)sqlite3_column_int(stmt,COL_KEY);
    temp = (char *)sqlite3_column_text(stmt,COL_EAN);
    strcpy(EAN, temp);
    temp = (char *)sqlite3_column_text(stmt, COL_ISBN);
    strcpy(ISBN, temp);
    temp = (char *)sqlite3_column_text(stmt, COL_TITLE);
    strcpy(title, temp);
    temp = (char *)sqlite3_column_text(stmt, COL_AUTHOR);
    strcpy(author, temp);
    *price = (float)sqlite3_column_double(stmt, COL_PRICE);
    temp = (char *)sqlite3_column_text(stmt, COL_PLACE);
    strcpy(place, temp);
    *number = (float)sqlite3_column_double(stmt, COL_NUMBER);
    *datetime = (unsigned int)sqlite3_column_int(stmt,COL_DATE);
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_data_row: finalise: %s\n", sqlite3_errmsg(appdata->db));
    return rc; // returning row
}



/* ================================== */
/* Table (list) stuff                 */
/* ================================== */

/* Return a list of the current lists */
// should we allocate the memory for the strings here, no they are allocated outside this function
void get_first_data_table(int *key, char *sqlite_name, char *pretty_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;

    command = sqlite3_mprintf("select * from lists;");
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc==SQLITE_OK) {
        rc = get_lists_row(stmt, key, sqlite_name, pretty_name);
        if (rc!=SQLITE_ROW) {
            printf("get_first_data_table(): get_lists_row failed, returning empty strings: %s\n", sqlite3_errmsg(appdata->db));
            sqlite_name[0]=0; //"";
            pretty_name[0]=0; //"";

        }
        // else, it did find them, so we're ok
    } else {
        printf("get_first_data_table(): submit_query failed, returning empty strings\n");
        sqlite_name[0]=0;//"";
        pretty_name[0]=0;//"";
    }
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_first_data_table: finalise: %s\n", sqlite3_errmsg(db));
    sqlite3_free(command);
    return;
}


// get pretty name from sql name
void get_pretty_from_sql_name(char *sqlite_name, char *pretty_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;
    char *temp;

    printf("get_pretty_from_sql_name(): SQL name=%s\n", sqlite_name);

    command = sqlite3_mprintf("select * from lists where COL_SQL_NAME='%q';", sqlite_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc==SQLITE_OK) {
        char *zErrMsg = 0;
        rc = sqlite3_step(stmt);
        if ( rc==SQLITE_DONE ) { // no more rows left
            pretty_name[0]=0; //"";
        } else if ( rc==SQLITE_ROW ) {
            temp = (char *)sqlite3_column_text(stmt, COL_PRETTY_NAME);
            strcpy(pretty_name, temp);
        } else {
            fprintf(stderr, "get_pretty_from_sql_name(): SQL error: %s\n", zErrMsg);
            sqlite3_free(zErrMsg);
            pretty_name[0]=0; //"";
        }
    } else {
        printf("get_pretty_from_sql_name(): submit_query failed, returning empty strings, rc=%i\n", rc);
        pretty_name[0]=0;//"";
    }
    sqlite3_free(command);
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_pretty_from_sql_name: finalise: %s\n", sqlite3_errmsg(db));
    return;
}


// get pretty name from sql name
void get_sql_from_pretty_name(char *pretty_name, char *sqlite_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;
    char *temp;

    printf("get_sql_from_pretty_name(): Pretty name=%s\n", pretty_name);

    command = sqlite3_mprintf("select * from lists where COL_PRETTY_NAME='%q';", pretty_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc==SQLITE_OK) {
        char *zErrMsg = 0;
        rc = sqlite3_step(stmt);
        if ( rc==SQLITE_DONE ) { // no more rows left
            sqlite_name[0]=0; //"";
        } else if ( rc==SQLITE_ROW ) {
            temp = (char *)sqlite3_column_text(stmt, COL_SQL_NAME);
            strcpy(sqlite_name, temp);
        } else {
            fprintf(stderr, "get_sql_from_pretty_name(): SQL error: %s\n", zErrMsg);
            sqlite3_free(zErrMsg);
            sqlite_name[0]=0; //"";
        }
    } else {
        printf("get_sql_from_pretty_name(): submit_query failed, returning empty strings, rc=%i\n", rc);
        sqlite_name[0]=0;//"";
    }
    sqlite3_free(command);
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("get_sql_from_pretty_name: finalise: %s\n", sqlite3_errmsg(db));
    return;
}


/* Add a list to the db */
int add_list(char *pretty_list_name, char *sql_name) {
    sqlite3_stmt *stmt;
    gchar *command;
    int rc=1;
    int key;

    // add the entry to the tables table
    // construct the command:
    command = sqlite3_mprintf("insert into lists (COL_SQL_NAME, COL_PRETTY_NAME) values ('%q', '%q');", sql_name, pretty_list_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc) {
        printf("add_list: insert into lists failed: %s\n", sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return -1;
    }
    rc = sqlite3_step(stmt);
    /*if(rc)
       printf("add_list: step: %s\n", sqlite3_errmsg(appdata->db));*/
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("add_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);

    // create the table to hold the data
    // G_TYPE_UINT64, G_TYPE_UINT64, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_STRING, G_TYPE_FLOAT, G_TYPE_UINT
    // construct the command:
    command = sqlite3_mprintf("create table %q (COL_KEY integer PRIMARY KEY, COL_EAN integer, COL_ISBN integer,"
                              " COL_TITLE text, COL_AUTHOR text, COL_PRICE real, COL_PLACE text, COL_NUMBER real, COL_DATE integer);", sql_name);
    // Looks like unsigned long longs can be stored in an INTEGER field. Good :)
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    if (rc) {
        printf("add_list: create table %s error: %s\n", sql_name, sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return -1;
    }
    rc = sqlite3_step(stmt);
    /*if(rc)
       printf("add_list: step: %s\n", sqlite3_errmsg(appdata->db));*/
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("add_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);


    command = sqlite3_mprintf("select * from lists where COL_SQL_NAME='%q';", sql_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc==SQLITE_OK) {
        char *zErrMsg = 0;
        rc = sqlite3_step(stmt);
        if ( rc==SQLITE_DONE ) { // no more rows left
            key=-1;
        } else if ( rc==SQLITE_ROW ) {
            key = (int)sqlite3_column_int(stmt, COL_KEY);
        } else {
            fprintf(stderr, "get_lists_row(): SQL error: %s\n", zErrMsg);
            sqlite3_free(zErrMsg);
            key=-1;
        }
    } else {
        printf("add list: select * from lists failed: %s\n", sqlite3_errmsg(appdata->db));
        key=-1;
    }
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("add_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);
    return key;
}

/* Remove a list from the db */
void remove_list(char *sql_list_name) {
    sqlite3_stmt *stmt;
    int rc;
    // delete the table in question
    // construct the command:
    gchar *command = sqlite3_mprintf("drop table %q;", sql_list_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc=submit_query(db, command, stmt);
    if (rc) {
        printf("remove_list: drop table %s error: %s\n", sql_list_name, sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return;
    }
    rc = sqlite3_step(stmt);
    //if(rc)
    //   printf("remove_list: step: %s\n", sqlite3_errmsg(appdata->db));
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("remove_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);

    // remove the entry from the tables table
    // construct the command:
    command = sqlite3_mprintf("delete from lists where COL_SQL_NAME='%q';", sql_list_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc=submit_query(db, command, stmt);
    if (rc) {
        printf("remove_list: delete from lists where COL_SQL_NAME='%s' error: %s\n", sql_list_name, sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return;
    }
    rc = sqlite3_step(stmt);
    //if(rc)
    //   printf("remove_list: step: %s\n", sqlite3_errmsg(appdata->db));
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("remove_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);
}

/* rename a list from the db */
void rename_list(char *sql_list_name, char *pretty_list_name) {
    // rename the table in question, just the pretty name, the sql one can stay the same (is hidden)
    // construct the command:
    sqlite3_stmt *stmt;
    int rc;
    gchar *command = sqlite3_mprintf("update table set COL_PRETTY_NAME='%q' where COL_SQL_NAME='%q';",pretty_list_name, sql_list_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(db, command, stmt);
    if (rc) {
        printf("rename_list: SQL name=%s to Pretty name=%s error, rc=%i\n", sql_list_name, pretty_list_name, rc);
        rc=sqlite3_finalize(stmt);
        return;
    }
    rc = sqlite3_step(stmt);
    //if(rc)
    //   printf("rename_list: step: %s\n", sqlite3_errmsg(appdata->db));
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("rename_list: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);
}


/* ================================== */
/* Entries in table (list) stuff      */
/* ================================== */

/* Add an entry to the list */
void add_entry(char *sql_list_name, char *EAN, char *ISBN, char *title, char *author, float *price, char *place, float *number, unsigned int *datetime) {
    gchar *command;
    //sqlite3_stmt *stmt;
    int rc;
    char *zErrMsg = 0;

    command = sqlite3_mprintf("insert into %s (COL_EAN, COL_ISBN, COL_TITLE, COL_AUTHOR, COL_PRICE, COL_PLACE, COL_NUMBER, COL_DATE) values ('%q', '%q', '%q', '%q', %f, '%q', %f, %u);", \
                              sql_list_name, EAN, ISBN, title, author, *price, place, *number, *datetime);
    printf("add_entry(): about to run command: %s\n", command);
    //rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    rc = sqlite3_exec(appdata->db, command, NULL, NULL, &zErrMsg); //submit_query(db, command, stmt);
    if (rc) {
        //printf("add_entry(): insert into %s error: %s\n", sql_list_name, sqlite3_errmsg(appdata->db));
        fprintf(stderr, "SQL error: %s\n", zErrMsg);
        sqlite3_free(command);
        sqlite3_free(zErrMsg);
        return;
    }
    //rc = sqlite3_step(stmt);
    //if(rc)
    //   printf("add_entry(): step: %s\n", sqlite3_errmsg(appdata->db));
    //rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("add_entry(): finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);
    printf("add_entry(): successful\n");
}

/* Remove an entry from the list */
void remove_entry(char *sql_list_name, int key) {
    // remove the entry from the table
    // construct the command:
    gchar *command = NULL;
    sqlite3_stmt *stmt;
    int rc;
    command = sqlite3_mprintf("delete from %q where COL_KEY=%u;", sql_list_name, key);
    printf("remove_entry: command: %s\n", command);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    if (rc) {
        printf("remove_entry: delete from %s error: %s\n", sql_list_name, sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return;
    }
    rc = sqlite3_step(stmt);
    //if(rc)
    //   printf("remove_entry: step: %s\n", sqlite3_errmsg(appdata->db));
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
    //if(rc)
    //   printf("remove_entry: finalise: %s\n", sqlite3_errmsg(appdata->db));
    sqlite3_free(command);
}

/* change an entry in a list */
void change_entry(char *sql_list_name, int key, char *EAN, char *ISBN, char *title, char *author, float *price, char *place, float *number, unsigned int *datetime) {
    // probably easiest to remove the old row and add in a new one
    // or is this very slow?
    // the alternative is to see what needs to be changed in this row and handle it like that
    // need to check this out
    remove_entry(sql_list_name, key);
    add_entry(sql_list_name, EAN, ISBN, title, author, price, place, number, datetime);
}


/* ================================== */
/* Utility stuff                      */
/* ================================== */

// turn a pretty name into something acceptable as an actual SQL lite table name
int deprettify_name(char* pretty_name, char* sql_name) {
    // I have no idea what can't be in a table name
    // can't seem to find anything in the sqlite docs
    // anyway, we'll change to uppercase, remove spaces and other punctuation and add a number on the end if not unique and leave it at that
    gchar *command = NULL;
    int i, rc, string_length=0;
    sqlite3_stmt *stmt;

    // scan the string
    for (i=0; i<strlen(pretty_name); i++) {
        if (pretty_name[i]<'0') // this means all non-printing chars, punctuation and spaces
            continue;
        else if (pretty_name[i]>='0' && pretty_name[i]<='Z') // is upper case or a number
            sql_name[string_length++] = pretty_name[i];
        else if (pretty_name[i]>='a' && pretty_name[i]<='z') // is lower case
            sql_name[string_length++] = pretty_name[i]-32; // turn to upper case
    }
    if (string_length==0) // check to see that the entire name wasn't made of illegal chars
        sql_name[string_length++]='a'; // if so, give it an arbitrary name

    sql_name[string_length++]=0; // add trailing null


    // see if this name is already in the db
    command = sqlite3_mprintf("select * from lists where COL_SQL_NAME='%s';", sql_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    if (rc) {
        printf("deprettify_name: %s error: %s\n", sql_name, sqlite3_errmsg(appdata->db));
        rc=sqlite3_finalize(stmt);
        return -1;
    }
    rc = sqlite3_step(stmt);
    sqlite3_free(command);
    if ( rc!=SQLITE_ROW ) {
        printf("deprettify_name: name not found\n");
        rc=sqlite3_finalize(stmt); //finish_query(stmt); // clean up
        return 0; // not found, so this one is ok
    } else {
        char sql_name_orig[255]; // declare it here, no point wasting space if this name wasn't found
        int appendix;

        rc=sqlite3_finalize(stmt); //finish_query(stmt); // clean up

        // it was found, let's start appending something to the name
        // initialise this variable
        strcpy(sql_name_orig, sql_name);

        appendix=0;
        while (1) {

            sprintf(sql_name, "%s%d", sql_name_orig, appendix);

            command = sqlite3_mprintf("select * from lists where COL_SQL_NAME='%s';", sql_name);
            rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
            if (rc) {
                printf("deprettify_name: %s error: %s\n", sql_name, sqlite3_errmsg(appdata->db));
                return -1;
            }
            rc = sqlite3_step(stmt);
            if ( rc!=SQLITE_ROW ) {
                rc=sqlite3_finalize(stmt); //finish_query(stmt);
                sqlite3_free(command);
                return 0; // not found, so this one is ok
            }
            rc=sqlite3_finalize(stmt); //finish_query(stmt);
            sqlite3_free(command);
            appendix++;

            if (appendix>99999) {
                // problems here, no more space in our string
                return -1;
            }
        }
    }


    return 0;
}



/* ================================== */
/* Search stuff                       */
/* ================================== */

/* search for entries in a (or multiple) list(s) */
void submit_search(sqlite3_stmt *stmt, char* sql_name, int key, char *EAN, char *ISBN,
                   char *title, char *author, float price, char *place, float number, unsigned int datetime) {
    //char* column_name = sql_column_names[column];
    GString *command = g_string_new("select * from ");
    char *command_string;
    int rc;

    g_string_append_printf(command, "%s where", sql_name);
    if (key) {
        command_string = sqlite3_mprintf(" COL_KEY=%d", key);
        command = g_string_append(command, command_string);
        sqlite3_free(command_string);
    }
    if (EAN) {
        command_string = sqlite3_mprintf(" COL_EAN='%q'", EAN);
        command = g_string_append(command, command_string);
        sqlite3_free(command_string);
    }
    if (ISBN) {
        command_string = sqlite3_mprintf(" COL_ISBN='%q'", ISBN);
        command = g_string_append(command, command_string);
        sqlite3_free(command_string);
    }
    if (title) {
        command_string = sqlite3_mprintf(" COL_TITLE='%q'", title);
        command = g_string_append(command, command_string);
        sqlite3_free(command_string);
    }
    if (author) {
        command_string = sqlite3_mprintf(" COL_AUTHOR='%q'", author);
        command = g_string_append(command, command_string);
        sqlite3_free(command_string);
    }
    if (price) {
        g_string_append_printf(command, " COL_PRICE=%f", price);
    }
    if (place) {
        g_string_append_printf(command, " COL_PLACE='%s'", place);
    }
    if (number) {
        g_string_append_printf(command, " COL_NUMBER=%f", number);
    }
    if (datetime) {
        g_string_append_printf(command, " COL_DATE=%u;", datetime);
    }
    rc = sqlite3_prepare_v2(appdata->db, command->str, command->len, &stmt, NULL); //submit_query(db, command, stmt);
    if (rc)
        printf("submit_search: %s\n", sqlite3_errmsg(appdata->db));

    rc=sqlite3_finalize(stmt);

    g_string_free(command, TRUE);
}


