#include "searchuserswindow.h"
#include "ui_searchuserswindow.h"
#include "plugin.h"
#include <QtXml>
#include <QtGui>
#include "trustdialog.h"
#include "htmldelegate.h"

SearchUsersWindow::SearchUsersWindow(QWidget *parent, Plugin *plugin) :
    QMainWindow(parent),
    ui(new Ui::SearchUsersWindow)
{
    this->plugin = plugin;
    ui->setupUi(this);
    manager = new QNetworkAccessManager(this);

    ui->tblResults->setProperty("FingerScrollable", true);
    ui->tblResults->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    HtmlDelegate *delegate = new HtmlDelegate();
    ui->tblResults->setItemDelegateForColumn(0,delegate);

    model = new QStandardItemModel(this);
    model->setColumnCount(3);
    connect(manager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(replyFinished(QNetworkReply*)));
    ui->tblResults->setModel(model);
}

SearchUsersWindow::~SearchUsersWindow()
{
    delete ui;
}

void SearchUsersWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
void SearchUsersWindow::replyFinished(QNetworkReply *reply) {
    QString data = QString::fromUtf8(reply->readAll()); // use UTF-8 encoding (why doesn't Qt detect this by itself?)
    qDebug() << "\n\n----Returned data---\n\n" << data << "\n\n\n";
    if(reply->error() == QNetworkReply::NoError) {
        if(requestType==SearchUsers) {
            QDomDocument doc;
            doc.setContent(data);
            QDomElement response = doc.documentElement();
            if(response.isNull()) {
                qDebug("No response found!");
            }
            QDomElement nodes = response.firstChildElement("users");
            if(nodes.isNull()) {
                qDebug("No users element found!");
            } else {
                QDomElement resource = nodes.firstChildElement("resource");
                // let's empty the results!
                model->clear();
                int row = 0;
                while(!resource.isNull()) {
                    QDomElement username = resource.firstChildElement("username");
                    QDomElement id = resource.firstChildElement("id");
                    QDomElement inNetwork = resource.firstChildElement("in_network");
                    QDomElement score = resource.firstChildElement("score");
                    QString convertedScore = QString::number(score.text().toDouble() * 100.0);
                    QStandardItem *userItem = new QStandardItem(username.text() +
                                                                "<br/><span style='font-size: small; color: grey'>Trust: " +
                                                                convertedScore +
                                                                "</span>");
                    QStandardItem *toggleItem;

                    if(inNetwork.text().toLower() == "true") {
                        toggleItem = new QStandardItem("Remove");
                        userItem->setIcon(QIcon(":/icons/agt_action_success.png"));
                    } else {
                        toggleItem = new QStandardItem("Add");
                    }
                    QVariantHash itemData;
                    itemData.insert("id", id.text());
                    itemData.insert("inNetwork", inNetwork.text());
                    userItem->setData(itemData);
                    toggleItem->setData(itemData);
                    toggleItem->setTextAlignment(Qt::AlignRight);
                    model->setItem(row,0,userItem);
                    model->setItem(row,1,toggleItem);
                    resource = resource.nextSiblingElement("resource");
                    row++;
                }

                ui->tblResults->resizeRowsToContents();
            }
        } else if(requestType == AddUser || requestType == RemoveUser){
            on_btnSearch_clicked();
        }
    } else if(reply->error() == QNetworkReply::ContentNotFoundError) {
        QMessageBox msgBox(QMessageBox::Information, "Not found", "We know it is sad, but we can't help that no users matched your query.");
        msgBox.exec();
    } else {
        QMessageBox msgBox(QMessageBox::Critical, "Error", "An error occured, but we have no idea why. Please check the application logs and report the mighty developers.");
        msgBox.exec();
    }
}

void SearchUsersWindow::on_btnSearch_clicked()
{
    //Getting data
    QString dataUrl = plugin->baseUrl + "users/search.xml?q=" + ui->txtSearch->text(); //
    QNetworkRequest request = QNetworkRequest(QUrl(dataUrl + "&" + plugin->generateOAuthData()));
    requestType = SearchUsers;
    manager->get(request);
}

void SearchUsersWindow::on_txtSearch_returnPressed()
{
    on_btnSearch_clicked(); // just invoke it as if the user clicked the button
}

void SearchUsersWindow::on_tblResults_clicked(QModelIndex index)
{
    QStandardItem *item = model->itemFromIndex(index);
    QVariantHash itemData = item->data().toHash();
    int userId = itemData.value("id").toString().toInt();
    QString inNetwork = itemData.value("inNetwork").toString();
    if(item->column() == 1) { // add/remove user
        qDebug() << QString::number(userId);
        if(inNetwork.toLower() == "false") { // add this user as trusted
            TrustDialog *dialog = new TrustDialog(this);
            if(dialog->exec() == QDialog::Accepted) {
                QString dataUrl = plugin->baseUrl + "trustnode/create.xml"; //
                QNetworkRequest request = QNetworkRequest(QUrl(dataUrl + "?" + plugin->generateOAuthData()));
                QString postdata = "user_id=" + QString::number(userId);
                double scoreAsPercent = ((double)dialog->score) / 100.0;
                qDebug() << QString::number(scoreAsPercent) << QString::number(dialog->score);
                postdata += "&score=" + QString::number(scoreAsPercent);
                requestType = AddUser;
                manager->post(request,postdata.toLatin1());
            }
        } else {
            QMessageBox *prompt = new QMessageBox(this);
            prompt->setWindowTitle("Remove user?");
            prompt->setText("Are you sure you want to remove this user from your trust network?");
            prompt->setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
            int ret = prompt->exec();
            if(ret == QMessageBox::Ok) {
                QString dataUrl = plugin->baseUrl + "trustnode/remove/" + QString::number(userId) + ".xml"; //
                QNetworkRequest request = QNetworkRequest(QUrl(dataUrl + "?" + plugin->generateOAuthData()));
                requestType = RemoveUser;
                request.setRawHeader("X-HTTP-Method-Override","DELETE");
                manager->post(request,"");
            }
        }
    }
    ui->tblResults->clearSelection(); // mimic the behaviour of Hildon tables (like in the e-mail app)
}
