#define _GNU_SOURCE

#include <cairo.h>
#include <gtk/gtk.h>
#include <microfeed/microfeedconfiguration.h>
#include <microfeed/microfeedsubscriber.h>
#include <microfeed/microfeedmisc.h>
#include <microfeed/microfeedprotocol.h>
#include <string.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>


typedef struct _Badge Badge;

struct _Badge {
	Badge* next;
	Badge* previous;
	
	char* publisher;
	time_t timestamp;
	char* uid;
	char* uri;
	char* text;
	char* info;
	char* sender_image;

	int shown : 1;
	int placeholder : 1;
};

MicrofeedSubscriber* subscriber;
Badge* current_badge = NULL;
GtkWidget* window;
GtkWidget* label = NULL;
guint timeout = 0;
GdkPixmap* pixmap = NULL;
cairo_surface_t* cairo_surface = NULL;
int hide_show_counter = 0;
guint hide_show_timeout = 0;
double alpha;
int buttons_visible = 0;
int button_previous_activated = 0;
int button_next_activated = 0;
int button_open_activated = 0;

GHashTable* image_cache;

#define CYCLES 30
#define PUBLISHER_ID "testtwitter:org.microfeed.Provider.Twitter"
#define PADDING 8

static Badge* badge_new(const char* publisher, time_t timestamp, const char* uid, const char* uri, const char* text, const char* info, const char* sender_image) {
	Badge* badge;
	
	badge = microfeed_memory_allocate(Badge);
	badge->publisher = strdup(publisher);
	badge->uid = strdup(uid);
	badge->uri = strdup(uri);
	badge->text = strdup(text);
	badge->info = strdup(info);
	if (sender_image) {
		badge->sender_image = strdup(sender_image);
	}
	
	return badge;
}

static void badge_free(Badge* badge) {
	free(badge->uid);
	free(badge->uri);
	free(badge->text);
	microfeed_memory_free(badge);
}

static void badge_add_into_list(Badge* badge) {
	Badge* i;
	
	if (!current_badge) {
		current_badge = badge;
	} else {
		for (i = current_badge; i->previous; i = i->previous) {
		}
		badge->next = i;
		i->previous = badge;
	}
}

static badge_remove_from_list(Badge* badge) {
	if (badge->previous) {
		badge->previous->next = badge->next;
	}
	if (badge->next) {
		badge->next->previous = badge->previous;
	}
	badge->previous = badge->next = NULL;
}

GdkPixbuf* image_cache_get_image(const char* uid) {
	GdkPixbuf* pixbuf = NULL;
	char* reply;
	size_t len;
	GdkPixbufLoader* loader;
	
	return g_hash_table_lookup(image_cache, uid);
}

void image_cache_set_image(const char* uid, const void* data, size_t data_size) {
	GdkPixbufLoader* loader;
	GdkPixbuf* pixbuf;
	
	loader = gdk_pixbuf_loader_new();
	if (gdk_pixbuf_loader_write(loader, data, data_size, NULL) &&
	    gdk_pixbuf_loader_close(loader, NULL) &&
	    (pixbuf = gdk_pixbuf_loader_get_pixbuf(loader))) {
		g_object_ref(pixbuf);
		g_hash_table_insert(image_cache, g_strdup(uid), pixbuf);
	}
	g_object_unref(loader);
}

static gboolean view_hide_show_timeout(void* user_data) {
	gboolean continue_animation = TRUE;
	int width;
	int height;

	if (pixmap) {
		gdk_drawable_get_size(pixmap, &width, &height);

		if (hide_show_counter < CYCLES) {
			alpha = 1.0 - (hide_show_counter / (double)CYCLES);
		} else if (hide_show_counter == CYCLES) {
			alpha = 0.0;
			if (cairo_surface) {
				cairo_surface_destroy(cairo_surface);
				cairo_surface = NULL;
			}
		} else if (hide_show_counter == CYCLES * 2) {
			alpha = 1.0;
			continue_animation = FALSE;
			hide_show_timeout = 0;
		} else {
			alpha = (hide_show_counter - CYCLES) / (double)CYCLES;
		}
		hide_show_counter++;

		gtk_widget_queue_draw_area(window, 0, 0, width, height);
	}
	
	return continue_animation;
}

static void view_hide_show(void) {
	if (!hide_show_timeout) {
		hide_show_counter = 0;
		hide_show_timeout = g_timeout_add(15, view_hide_show_timeout, NULL);
	} else if (hide_show_counter > CYCLES) {
		hide_show_counter = (CYCLES * 2) - hide_show_counter;
	}
}

static gboolean view_show_next_unshown_timeout(void* user_data) {
	gboolean more_unshown = FALSE;
	Badge* b;
	
	for (b = current_badge; b->shown && b->previous; b = b->previous) {
	}

	if (b != current_badge || !b->shown) {	
		current_badge = b;
		view_hide_show();
		if (!timeout) {
			timeout = g_timeout_add(5000, view_show_next_unshown_timeout, NULL);
		} else if (current_badge->previous) {
			more_unshown = TRUE;
		}
	}

	return more_unshown;
}

static void view_show_next_unshown(void) {
	if (!timeout) {
		view_show_next_unshown_timeout(NULL);	
	}
}

static void view_show_next_unshown_delayed(void) {
	if (!timeout) {
		timeout = g_timeout_add(5000, view_show_next_unshown_timeout, NULL);
	}
}

static void feed_subscribed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	if (error_name) {
		printf("Subscribing failed: %s %s: %s (%s)\n", publisher, uri, error_name, error_message);
	} else {

		printf("Subscribed: %s %s\n", publisher, uri);
	
		microfeed_subscriber_republish_items(subscriber, publisher, uri, NULL, NULL, 15, NULL, NULL);
		microfeed_subscriber_update_feed(subscriber, publisher, uri, NULL, NULL);
	}
}

static void error_occured(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	printf("Failed: %s %s (%s): %s (%s)\n", publisher, uri, uid, error_name, error_message);
}

static void item_added(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	const char* text;
	const char* thread_uri;
	const char* sender_image;
	const char* sender_nick;
	char* s;
	const char* publisher_part;
	char* info;
	Badge* badge;
	
	printf("Added: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
	/*if (microfeed_item_get_status(item) & MICROFEED_ITEM_STATUS_UNREAD || !current_badge) {*/

	if (strcmp(microfeed_item_get_uid(item), MICROFEED_ITEM_UID_FEED_METADATA)) {
		if (!(text = microfeed_item_get_property(item, "content.text"))) {
			text = "<no content>";
		}
		if (!(thread_uri = microfeed_item_get_property(item, "thread.uri")) &&
		    !(thread_uri = microfeed_item_get_property(item, "feed.uri")) &&
		    !(thread_uri = microfeed_item_get_property(item, "sender.uri"))) {
			thread_uri = uri;
		}
		if ((sender_image = microfeed_item_get_property(item, "sender.image")) &&
		    !image_cache_get_image(sender_image)) {
			microfeed_subscriber_send_item_data(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, sender_image, NULL, NULL);
		}
		if (!(sender_nick = microfeed_item_get_property(item, "sender.nick"))) {
			sender_nick = "<anonymous>";
		}
		if ((s = strchr(publisher, ':'))) {
			publisher_part = strndup(publisher, s - publisher);
		} else {
			publisher_part = strdup("<not specified>");
		}
		info = microfeed_util_string_concatenate("By ", sender_nick, " in ", publisher_part, " ", NULL);		
		badge = badge_new(publisher, microfeed_item_get_timestamp(item), microfeed_item_get_uid(item), uri, text, info, sender_image);
		badge_add_into_list(badge);
		view_show_next_unshown();
	}
/*	}*/
}

static void item_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	printf("Changed: %s %s %s %ld %u\n", publisher, uri, microfeed_item_get_uid(item), (unsigned long)microfeed_item_get_timestamp(item), (unsigned int)microfeed_item_get_status(item));
}

static void item_republished(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data) {
	/*if (!current_badge) {*/
		item_added(subscriber, publisher, uri, item, user_data);
/*	} */
}

static void item_removed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, void* user_data) {
	printf("Removed: %s %s %s\n", publisher, uri, uid);
}

static void item_status_changed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const MicrofeedItemStatus status, void* user_data) {
	printf("Status changed: %s %s %s %u\n", publisher, uri, uid, (unsigned int)status);
}

static void item_data(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t length, void* user_data) {
	int width;
	int height;

	printf("Data: %s %s %s (length: %u)\n", publisher, uri, uid, length);
	image_cache_set_image(uid, data, length);
	if (!current_badge->sender_image || !strcmp(current_badge->sender_image, uid)) {
		if (cairo_surface) {
			cairo_surface_destroy(cairo_surface);
			cairo_surface = NULL;
		}
		
		if (pixmap) {
			gdk_drawable_get_size(pixmap, &width, &height);
			gtk_widget_queue_draw_area(window, 0, 0, width, height);
		}
	}
}

static MicrofeedSubscriberCallbacks callbacks = {
	error_occured,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	item_added,
	item_changed,
	item_republished,
	item_removed,
	item_status_changed,
	item_data
};

gboolean on_configure_event(GtkWidget* widget, GdkEventConfigure* event, gpointer user_data) {
	static int width = 0;
	static int height = 0;
	GdkPixmap* new_pixmap;

	if (width != event->width || height != event->height){
		new_pixmap = gdk_pixmap_new(widget->window, event->width,  event->height, -1);
		if (pixmap) {
			gdk_draw_drawable(new_pixmap, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], pixmap, 0, 0, 0, 0, (width < event->width ? width : event->width), (height < event->height ? height : event->height));
			g_object_unref(pixmap); 
		}
		pixmap = new_pixmap;
		width = event->width;
		height = event->height;
		if (cairo_surface) {
			cairo_surface_destroy(cairo_surface);
			cairo_surface = NULL;
		}
		gtk_widget_queue_draw_area(window, 0, 0, width, height);
	}

	return TRUE;
}

gboolean on_expose_event(GtkWidget* widget, GdkEventExpose* event, gpointer user_data) {
	int x = 0;
	int y;
	int width;
	int height;
	cairo_t* cairo;
	PangoLayout* layout;
	GdkPixbuf* sender_pixbuf = NULL;
	GdkPixbuf* content_pixbuf;
	GdkPixbuf* pixbuf1;
	GdkPixbuf* pixbuf2;
	

	if (pixmap) {
		gdk_drawable_get_size(pixmap, &width, &height);
		width -= PADDING * 2;
		height -= PADDING * 2;

		if (!cairo_surface) {
			cairo_surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, width, height);
			cairo = cairo_create(cairo_surface);
			cairo_translate(cairo, PADDING, PADDING);
			cairo_set_source_rgba(cairo, 1.0, 1.0, 1.0, 0.0);
			cairo_set_operator(cairo, CAIRO_OPERATOR_SOURCE);
			cairo_paint(cairo);
			if (current_badge) {
				current_badge->shown = 1;

				cairo_set_operator(cairo, CAIRO_OPERATOR_OVER);
				cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 1.0);
				if (current_badge->sender_image && (sender_pixbuf = image_cache_get_image(current_badge->sender_image))) {
					x += gdk_pixbuf_get_width(sender_pixbuf) + 8;
				}

				cairo_move_to(cairo, x, 0);
				layout = pango_cairo_create_layout(cairo);
				pango_layout_set_width(layout, (width - x) * PANGO_SCALE);
				pango_layout_set_height(layout, (height - 24) * PANGO_SCALE);
				pango_layout_set_wrap(layout, PANGO_WRAP_WORD_CHAR);
				pango_layout_set_ellipsize(layout, PANGO_ELLIPSIZE_END);
				pango_layout_set_text(layout, current_badge->text, -1);
				pango_cairo_show_layout(cairo, layout);
				pango_layout_get_size(layout, NULL, &y);
				y /= PANGO_SCALE;
				g_object_unref(layout);

				cairo_move_to(cairo, x, y + 4);
				layout = pango_cairo_create_layout(cairo);
				pango_layout_set_width(layout, (width - x) * PANGO_SCALE);
				pango_layout_set_height(layout, (height - y - 4) * PANGO_SCALE);
				pango_layout_set_wrap(layout, PANGO_WRAP_WORD_CHAR);
				pango_layout_set_ellipsize(layout, PANGO_ELLIPSIZE_END);
				pango_layout_set_text(layout, current_badge->info, -1);
				pango_cairo_show_layout(cairo, layout);
				pango_layout_get_size(layout, NULL, &y);
				g_object_unref(layout);

				if (sender_pixbuf) {
					gdk_cairo_set_source_pixbuf(cairo, sender_pixbuf, 1, 1);
					cairo_paint(cairo);
					x = gdk_pixbuf_get_width(sender_pixbuf);
					y = gdk_pixbuf_get_height(sender_pixbuf);
					cairo_set_source_rgba(cairo, 0.5, 0.5, 0.5, 0.9);
					cairo_set_line_width(cairo, 0.5);
					cairo_move_to(cairo, x + 2, 0);
					cairo_line_to(cairo, 0, 0);
					cairo_line_to(cairo, 0, y + 2);
					cairo_stroke(cairo); 
					cairo_set_source_rgba(cairo, 0.8, 0.8, 0.8, 0.9);
					cairo_set_line_width(cairo, 0.5);
					cairo_move_to(cairo, x + 2, 0);
					cairo_line_to(cairo, x + 2, y + 2);
					cairo_line_to(cairo, 0, y + 2);
					cairo_stroke(cairo); 
					if ((pixbuf1 = gdk_pixbuf_scale_simple(sender_pixbuf, x, y / 3, GDK_INTERP_BILINEAR))) {
						if ((pixbuf2 = gdk_pixbuf_flip(pixbuf1, FALSE))) {
							gdk_cairo_set_source_pixbuf(cairo, pixbuf2, 1, y + 2);
							cairo_paint_with_alpha(cairo, 0.2);					
							gdk_pixbuf_unref(pixbuf2);
						}
						gdk_pixbuf_unref(pixbuf1);
					}
				}
			}
			cairo_destroy(cairo);
		}

		cairo = gdk_cairo_create(pixmap);
		cairo_set_source_rgba(cairo, 1.0, 1.0, 1.0, 0.75);
		cairo_set_operator(cairo, CAIRO_OPERATOR_SOURCE);
		cairo_paint(cairo);
		cairo_set_source_surface(cairo, cairo_surface, 0, 0);
		cairo_set_operator(cairo, CAIRO_OPERATOR_OVER);
		cairo_paint_with_alpha(cairo, alpha);
		if (buttons_visible && current_badge) {
			cairo_translate(cairo, PADDING, PADDING);
			if (current_badge->next) {
				if (button_next_activated) {
					cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
				} else {
					cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
				}
				cairo_move_to(cairo, width / 4, 0);
				cairo_line_to(cairo, 0, height / 2);
				cairo_line_to(cairo, width / 4, height);
				cairo_line_to(cairo, width / 4, 0);
				cairo_fill(cairo);
			}
			if (current_badge->previous) {
				if (button_previous_activated) {
					cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
				} else {
					cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
				}
				cairo_move_to(cairo, 3 * width / 4, 0);
				cairo_line_to(cairo, width, height / 2);
				cairo_line_to(cairo, 3 * width / 4, height);
				cairo_line_to(cairo, 3 * width / 4, 0);
				cairo_fill(cairo);			
			}
			if (button_open_activated) {
				cairo_set_source_rgba(cairo, 1.0, 0.0, 0.0, 0.6);
			} else {
				cairo_set_source_rgba(cairo, 0.0, 0.0, 0.0, 0.6);
			}
			cairo_move_to(cairo, width / 2, 0);
			cairo_line_to(cairo, 5 * width / 8, height);
			cairo_line_to(cairo, 3 * width / 8, height);
			cairo_line_to(cairo, width / 2, 0);
			cairo_fill(cairo);			
			
		}
		cairo_destroy(cairo);

		gdk_draw_drawable(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], pixmap, event->area.x, event->area.y, event->area.x, event->area.y, event->area.width, event->area.height);
	}	

	return TRUE;
}

gboolean on_button_press_event(GtkWidget* widget, GdkEventButton* event, gpointer user_data) {
	int width;
	int height;

	buttons_visible = 1;

	button_next_activated = button_previous_activated = button_open_activated = 0;

	if (pixmap) {
		gdk_drawable_get_size(pixmap, &width, &height);
		if (event->x > 0 && event->y > 0 && event->x < width && event->y < height) {
			if (event->x < width / 4) {
				button_next_activated = 1;
			} else if (event->x > 3 * width / 4) {
				button_previous_activated = 1;
			} else if (event->x > 3 * width / 8 && event->x < 5 * width / 8) {
				button_open_activated = 1;
			}
		}
		gtk_widget_queue_draw_area(window, 0, 0, width, height);
	}

	return FALSE;
}

gboolean on_button_release_event(GtkWidget* widget, GdkEventButton* event, gpointer user_data) {
	int width;
	int height;

	buttons_visible = 0;

	if (pixmap) {
		gdk_drawable_get_size(pixmap, &width, &height);
		if (event->x > 0 && event->y > 0 && event->x < width && event->y < height) {
			if (event->x < width / 4 && current_badge && current_badge->next) {
				if (timeout) {
					g_source_remove(timeout);
					timeout = 0;
				}
				current_badge = current_badge->next;
				view_hide_show();
				view_show_next_unshown_delayed();
			} else if (event->x > 3 * width / 4 && current_badge && current_badge->previous) {
				if (timeout) {
					g_source_remove(timeout);
					timeout = 0;
				}
				current_badge = current_badge->previous;
				view_hide_show();
				view_show_next_unshown_delayed();
			} else if (event->x > 3 * width / 8 && event->x < 5 * width / 8 && current_badge) {
/* TODO */
			}
		}

		gtk_widget_queue_draw_area(window, 0, 0, width, height);
	}

	return FALSE;
}

int main(int argc, char** argv) {
	GdkColormap* colormap;
	DBusError error;
	DBusConnection* connection;
	MicrofeedConfiguration* configuration;

	configuration = microfeed_configuration_new();

	microfeed_thread_init();
	gtk_init(&argc, &argv);
	
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	
	if ((colormap = gdk_screen_get_rgba_colormap(gdk_screen_get_default()))) {
		gtk_widget_set_colormap(window, colormap);
	}

	gtk_widget_set_double_buffered(window, FALSE);
	gtk_window_set_default_size(GTK_WINDOW(window), 600, 100);
	g_signal_connect(window, "destroy", G_CALLBACK(gtk_main_quit), NULL);
	g_signal_connect(window, "expose_event", G_CALLBACK(on_expose_event), NULL);
	g_signal_connect(window, "configure_event", G_CALLBACK(on_configure_event), NULL);
	g_signal_connect(window, "button_press_event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(window, "motion_notify_event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(window, "button_release_event", G_CALLBACK(on_button_release_event), NULL);

	gtk_widget_add_events(window, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_MOTION_MASK);
	
	gtk_widget_show_all(window);

	image_cache = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);

	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SESSION, &error);
	dbus_connection_setup_with_g_main(connection, NULL);
	configuration = microfeed_configuration_new();
	subscriber = microfeed_subscriber_new("/org/microfeed/Subscriber/mauku_home_plugin", configuration, connection);

	microfeed_subscriber_subscribe_feed(subscriber, PUBLISHER_ID, MICROFEED_FEED_URI_IMAGES, &callbacks, NULL, NULL);
	microfeed_subscriber_subscribe_feed(subscriber, PUBLISHER_ID, MICROFEED_FEED_URI_OVERVIEW, &callbacks, feed_subscribed, NULL);
	
	gtk_main();
	
	return 0;
}
