/*
 * connection-watcher.c - Source for ConnectionWatcher
 * Copyright (C) 2010 Guillaume Desmottes
 * Copyright (C) 2010 Collabora
 * @author Guillaume Desmottes <gdesmott@gnome.org>
 * @author Alban Crequy <alban.crequy@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <stdlib.h>

#include <telepathy-glib/account-manager.h>
#include <telepathy-glib/account.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/gtypes.h>
#include <telepathy-glib/util.h>
#include <telepathy-glib/channel.h>
#include <telepathy-glib/contact.h>

#include "contact-watcher.h"
#include "connection-watcher.h"
#include "mapbuddy-marshallers.h"

G_DEFINE_TYPE(ContactWatcher, contact_watcher, G_TYPE_OBJECT)

/* signal enum */
enum
{
    CONTACT_ADDED,
    CONTACT_REMOVED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* private structure */
typedef struct _ContactWatcherPrivate ContactWatcherPrivate;

struct _ContactWatcherPrivate
{
  ConnectionWatcher *connection_watcher;
  TpDBusDaemon *bus_daemon;

  /* key: connection
   * value: ConnectionData */
  GHashTable *connections;

  gboolean dispose_has_run;
};

typedef struct
{
  TpContact *contact;
  GHashTable *location;
} ContactLocation;

typedef struct
{
  TpAccount *account;
  /* key: TpHandle
   * value: ContactLocation */
  GHashTable *contacts_locations;
} ConnectionData;

#define CONTACT_WATCHER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), \
      CONTACT_WATCHER_TYPE, ContactWatcherPrivate))

static void conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    ContactWatcher *self);

static void
contact_location_free (gpointer data)
{
  ContactLocation *contact_location = data;

  g_object_unref (contact_location->contact);
  contact_location->contact = NULL;

  g_hash_table_destroy (contact_location->location);
  contact_location->location = NULL;

  g_slice_free (ContactLocation, contact_location);
}

static void
connection_data_free (gpointer data)
{
  ConnectionData *connection_data = data;

  g_hash_table_destroy (connection_data->contacts_locations);
}

static void
remove_connection (ContactWatcher *self,
    TpConnection *conn)
{
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);

  g_signal_handlers_disconnect_by_func (conn, G_CALLBACK (conn_invalidated_cb),
      self);
  g_hash_table_remove (priv->connections, conn);
  g_object_unref (conn);
}

static void
conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    ContactWatcher *self)
{
  g_debug ("connection %s invalidated; removing\n", tp_proxy_get_object_path (
        conn));

  remove_connection (self, TP_CONNECTION (conn));
}

static void
contacts_location_updated (TpConnection *connection,
    guint handle,
    GHashTable *location,
    gpointer user_data,
    GObject *weak_object G_GNUC_UNUSED)
{
  ContactWatcher *self = CONTACT_WATCHER (user_data);
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);
  ConnectionData *connection_data = g_hash_table_lookup (priv->connections,
      connection);
  TpAccount *account = connection_data->account;
  GHashTable *contacts_locations = connection_data->contacts_locations;
  ContactLocation *contact_location = g_hash_table_lookup (contacts_locations,
      GUINT_TO_POINTER (handle));
  GValue *value;
  gdouble flon, flat;

  g_debug ("contacts_location_updated: Called for handle %u\n", handle);

  if (contact_location == NULL)
    {
      g_printerr ("Got location for an unknown contact\n");
      return;
    }
  if (contact_location->contact == NULL)
    {
      g_printerr ("Got location for an released contact\n");
      return;
    }
  if (contact_location->location != NULL)
    {
      g_hash_table_unref (contact_location->location);
    }
  contact_location->location = g_hash_table_ref (location);

  value = g_hash_table_lookup (location, "lat");
  if (value == NULL)
    {
      g_printerr ("Contact '%s' has no location 'lat'\n",
        tp_contact_get_identifier (contact_location->contact));
      g_signal_emit (self, signals[CONTACT_REMOVED], 0,
          contact_location->contact);
      return;
    }
  flat = g_value_get_double (value);

  value = g_hash_table_lookup (location, "lon");
  if (value == NULL)
    {
      g_printerr ("Contact '%s' has no location 'lon'\n",
        tp_contact_get_identifier (contact_location->contact));
      g_signal_emit (self, signals[CONTACT_REMOVED], 0,
          contact_location->contact);
      return;
    }
  flon = g_value_get_double (value);

  g_debug ("Emit contact-added for '%s': lon=%f lat=%f\n",
      tp_contact_get_identifier (contact_location->contact),
      flon, flat);
  g_signal_emit (self, signals[CONTACT_ADDED], 0, account, connection,
      contact_location->contact, flon, flat);
}

static void
contacts_got_locations_cb (TpConnection *connection,
    GHashTable *locations,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  ContactWatcher *self = CONTACT_WATCHER (weak_object);
  GHashTableIter iter;
  gpointer key, value;

  g_debug ("Got %u locations\n", g_hash_table_size (locations));

  if (error != NULL)
    {
      g_printerr ("GetLocations failed with %s %u: %s",
          g_quark_to_string (error->domain), error->code, error->message);
      return;
    }

  g_hash_table_iter_init (&iter, locations);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      contacts_location_updated (connection, GPOINTER_TO_UINT (key),
          value, self, NULL);
    }
}

static void
got_contacts_cb (TpConnection *connection,
                 guint n_contacts,
                 TpContact * const *contacts,
                 guint n_failed,
                 const TpHandle *failed,
                 const GError *error,
                 ContactWatcher *self,
                 GObject *weak_object)
{
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);
  ConnectionData *connection_data;
  GHashTable *contacts_locations;
  GArray *handles;
  int i;

  g_debug ("Got %d TpContact (%d failed)\n",
      n_contacts, n_failed);

  connection_data = g_hash_table_lookup (priv->connections, connection);
  contacts_locations = connection_data->contacts_locations;

  handles = g_array_sized_new (TRUE, TRUE, sizeof (TpHandle), n_contacts);

  for (i = 0 ; i < n_contacts ; i++)
    {
      TpHandle handle = tp_contact_get_handle (contacts[i]);
      ContactLocation *contact_location = g_slice_new0 (ContactLocation);
      contact_location->contact = g_object_ref (contacts[i]);

      g_hash_table_insert (contacts_locations, GUINT_TO_POINTER (handle),
          contact_location);
      g_array_append_val (handles, handle);

      g_debug ("Ask location for handle %u '%s'\n",
        handle, tp_contact_get_identifier (contact_location->contact));
    }

  tp_cli_connection_interface_location_call_get_locations (connection,
      -1, handles, contacts_got_locations_cb, NULL, NULL, self);

  g_array_free (handles, TRUE);
}

static void
got_members_cb (TpChannel *channel,
                const GArray *handles,
                const GError *error,
                ContactWatcher *self,
                GObject *weak_object)
{
  TpContactFeature features[3] = {
      TP_CONTACT_FEATURE_ALIAS,
      TP_CONTACT_FEATURE_AVATAR_TOKEN,
      TP_CONTACT_FEATURE_PRESENCE
  };

  g_debug ("Got members handles\n");

  tp_connection_get_contacts_by_handle (
      tp_channel_borrow_connection (channel), handles->len, handles->data,
      3, features, got_contacts_cb, self, NULL, G_OBJECT (self));
}

static void
got_publish_channel_cb (TpConnection *conn,
                        gboolean yours,
                        const gchar *path,
                        GHashTable *properties,
                        const GError *error,
                        ContactWatcher *self,
                        GObject *weak_object)
{
  TpChannel *channel;
  if (error != NULL)
    {
      g_printerr ("got_publish_channel_cb failed: %s\n", error->message);
      remove_connection (self, conn);
      return;
    }

  g_debug ("Got publish channel\n");
  channel = tp_channel_new_from_properties (conn, path, properties, NULL);
  tp_cli_channel_interface_group_call_get_members (channel, -1,
      got_members_cb, self, NULL, G_OBJECT (self));
}

static void
connection_added_cb (ConnectionWatcher *connection_watcher,
    TpAccount *account,
    TpConnection *conn,
    ContactWatcher *self)
{
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);
  GHashTable *request;
  ConnectionData *connection_data;

  if (g_hash_table_lookup (priv->connections, conn) != NULL)
    return;

  if (!tp_proxy_has_interface_by_id (conn,
        TP_IFACE_QUARK_CONNECTION_INTERFACE_LOCATION))
    return;

  g_debug ("Add new connection\n");

  connection_data = g_slice_new0 (ConnectionData);
  connection_data->account = account;
  connection_data->contacts_locations = g_hash_table_new_full (g_direct_hash,
      g_direct_equal, NULL, (GDestroyNotify) contact_location_free);
  g_hash_table_insert (priv->connections, g_object_ref (conn),
      connection_data);

  g_signal_connect (conn, "invalidated",
      G_CALLBACK (conn_invalidated_cb), self);

  tp_cli_connection_interface_location_connect_to_location_updated
    (conn, contacts_location_updated, self, NULL, self, NULL);

  request = tp_asv_new (
      TP_IFACE_CHANNEL ".ChannelType", G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_CONTACT_LIST,
      TP_IFACE_CHANNEL ".TargetHandleType", G_TYPE_UINT, TP_HANDLE_TYPE_LIST,
      TP_IFACE_CHANNEL ".TargetID", G_TYPE_STRING, "publish",
      NULL);
  tp_cli_connection_interface_requests_call_ensure_channel (conn, -1, request,
      got_publish_channel_cb, self, NULL, G_OBJECT (self));
  g_hash_table_unref (request);
}

static void
contact_watcher_init (ContactWatcher *obj)
{
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (obj);

  priv->connection_watcher = connection_watcher_new ();
  priv->connections = g_hash_table_new_full (g_direct_hash,  g_direct_equal,
      (GDestroyNotify) g_object_unref, (GDestroyNotify) connection_data_free);

  g_signal_connect (priv->connection_watcher, "connection-added",
      G_CALLBACK (connection_added_cb), obj);
}

static void contact_watcher_dispose (GObject *object);
static void contact_watcher_finalize (GObject *object);

static void
contact_watcher_class_init (ContactWatcherClass *contact_watcher_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (contact_watcher_class);

  g_type_class_add_private (contact_watcher_class, sizeof (ContactWatcherPrivate));

  object_class->dispose = contact_watcher_dispose;
  object_class->finalize = contact_watcher_finalize;

  signals[CONTACT_ADDED] = g_signal_new ("contact-added",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    mapbuddy_marshal_VOID__OBJECT_OBJECT_OBJECT_DOUBLE_DOUBLE,
    G_TYPE_NONE, 5, TP_TYPE_ACCOUNT, TP_TYPE_CONNECTION, TP_TYPE_CONTACT,
    G_TYPE_DOUBLE, G_TYPE_DOUBLE);

  signals[CONTACT_REMOVED] = g_signal_new ("contact-removed",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    g_cclosure_marshal_VOID__OBJECT,
    G_TYPE_NONE, 1, TP_TYPE_CONTACT);
}

void
contact_watcher_dispose (GObject *object)
{
  ContactWatcher *self = CONTACT_WATCHER (object);
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;

  if (priv->connection_watcher)
    {
      g_object_unref (priv->connection_watcher);
      priv->connection_watcher = NULL;
    }

  if (priv->connections)
    {
      g_hash_table_destroy (priv->connections);
      priv->connections = NULL;
    }

  if (G_OBJECT_CLASS (contact_watcher_parent_class)->dispose)
    G_OBJECT_CLASS (contact_watcher_parent_class)->dispose (object);
}

void
contact_watcher_finalize (GObject *object)
{
  //ContactWatcher *self = CONTACT_WATCHER (object);
  //ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);

  G_OBJECT_CLASS (contact_watcher_parent_class)->finalize (object);
}

ContactWatcher *
contact_watcher_new (void)
{
  return g_object_new (CONTACT_WATCHER_TYPE,
      NULL);
}

void
contact_watcher_start (ContactWatcher *self)
{
  ContactWatcherPrivate *priv = CONTACT_WATCHER_GET_PRIVATE (self);

  connection_watcher_start (priv->connection_watcher);
}
