/*
 *  Subtitles control panel applet.
 *  Copyright (C) 2010 Roman Moravcik
 *
 *  encodings structure imported from totem-subtitle-encoding.c
 *  Copyright (C) 2001-2006 Bastien Nocera <hadess@hadess.net>
 *
 *  font family detection imported from hildon-font-selection-dialog.c
 *  Copyright (C) 2005, 2006 Nokia Corporation, all rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

#include <gconf/gconf-client.h>

#define GCONF_MAFW_GST_SUBTITLE_RENDERER "/system/mafw/mafw-gst-subtitles-renderer"
#define _HL(str) dgettext("hildon-libs",str)

typedef enum
{
	FONT_STYLE_REGULAR,
	FONT_STYLE_ITALIC,
	FONT_STYLE_BOLD,
	FONT_STYLE_ITALIC_BOLD,
	FONT_STYLE_LAST,
} FontStyleIndex;

typedef struct {
	int index;
	const char *name;
} FontStyle;

static FontStyle font_styles[] = {
	{FONT_STYLE_REGULAR,		N_("Regular")},
	{FONT_STYLE_ITALIC,		N_("Italic")},
	{FONT_STYLE_BOLD,		N_("Bold")},
	{FONT_STYLE_ITALIC_BOLD,	N_("Italic Bold")}
};

static const gint font_sizes[] =
{
  10, 12, 14, 16, 18, 20, 22, 24, 26, 28, -1
};

typedef enum
{
	SUBTITLE_ENCODING_ISO_8859_6,
	SUBTITLE_ENCODING_IBM_864,
	SUBTITLE_ENCODING_MAC_ARABIC,
	SUBTITLE_ENCODING_WINDOWS_1256,

	SUBTITLE_ENCODING_ARMSCII_8,

	SUBTITLE_ENCODING_ISO_8859_4,
	SUBTITLE_ENCODING_ISO_8859_13,
	SUBTITLE_ENCODING_WINDOWS_1257,

	SUBTITLE_ENCODING_ISO_8859_14,

	SUBTITLE_ENCODING_ISO_8859_2,
	SUBTITLE_ENCODING_IBM_852,
	SUBTITLE_ENCODING_MAC_CE,
	SUBTITLE_ENCODING_WINDOWS_1250,

	SUBTITLE_ENCODING_GB18030,
	SUBTITLE_ENCODING_GB2312,
	SUBTITLE_ENCODING_GBK,
	SUBTITLE_ENCODING_HZ,

	SUBTITLE_ENCODING_BIG5,
	SUBTITLE_ENCODING_BIG5_HKSCS,
	SUBTITLE_ENCODING_EUC_TW,

	SUBTITLE_ENCODING_MAC_CROATIAN,

	SUBTITLE_ENCODING_ISO_8859_5,
	SUBTITLE_ENCODING_IBM_855,
	SUBTITLE_ENCODING_ISO_IR_111,
	SUBTITLE_ENCODING_KOI8_R,
	SUBTITLE_ENCODING_MAC_CYRILLIC,
	SUBTITLE_ENCODING_WINDOWS_1251,

	SUBTITLE_ENCODING_CP_866,

	SUBTITLE_ENCODING_MAC_UKRAINIAN,
	SUBTITLE_ENCODING_KOI8_U,

	SUBTITLE_ENCODING_GEOSTD8,

	SUBTITLE_ENCODING_ISO_8859_7,
	SUBTITLE_ENCODING_MAC_GREEK,
	SUBTITLE_ENCODING_WINDOWS_1253,

	SUBTITLE_ENCODING_MAC_GUJARATI,

	SUBTITLE_ENCODING_MAC_GURMUKHI,

	SUBTITLE_ENCODING_ISO_8859_8_I,
	SUBTITLE_ENCODING_IBM_862,
	SUBTITLE_ENCODING_MAC_HEBREW,
	SUBTITLE_ENCODING_WINDOWS_1255,

	SUBTITLE_ENCODING_ISO_8859_8,

	SUBTITLE_ENCODING_MAC_DEVANAGARI,

	SUBTITLE_ENCODING_MAC_ICELANDIC,

	SUBTITLE_ENCODING_EUC_JP,
	SUBTITLE_ENCODING_ISO_2022_JP,
	SUBTITLE_ENCODING_SHIFT_JIS,

	SUBTITLE_ENCODING_EUC_KR,
	SUBTITLE_ENCODING_ISO_2022_KR,
	SUBTITLE_ENCODING_JOHAB,
	SUBTITLE_ENCODING_UHC,

	SUBTITLE_ENCODING_ISO_8859_10,

	SUBTITLE_ENCODING_MAC_FARSI,

	SUBTITLE_ENCODING_ISO_8859_16,
	SUBTITLE_ENCODING_MAC_ROMANIAN,

	SUBTITLE_ENCODING_ISO_8859_3,

	SUBTITLE_ENCODING_TIS_620,

	SUBTITLE_ENCODING_ISO_8859_9,
	SUBTITLE_ENCODING_IBM_857,
	SUBTITLE_ENCODING_MAC_TURKISH,
	SUBTITLE_ENCODING_WINDOWS_1254,

	SUBTITLE_ENCODING_UTF_7,
	SUBTITLE_ENCODING_UTF_8,
	SUBTITLE_ENCODING_UTF_16,
	SUBTITLE_ENCODING_UCS_2,
	SUBTITLE_ENCODING_UCS_4,

	SUBTITLE_ENCODING_ISO_8859_1,
	SUBTITLE_ENCODING_ISO_8859_15,
	SUBTITLE_ENCODING_IBM_850,
	SUBTITLE_ENCODING_MAC_ROMAN,
	SUBTITLE_ENCODING_WINDOWS_1252,

	SUBTITLE_ENCODING_TCVN,
	SUBTITLE_ENCODING_VISCII,
	SUBTITLE_ENCODING_WINDOWS_1258,

	SUBTITLE_ENCODING_CURRENT_LOCALE,

	SUBTITLE_ENCODING_LAST
} SubtitleEncodingIndex;

typedef struct {
	int index;
	const char *charset;
	const char *name;
} SubtitleEncoding;

static SubtitleEncoding encodings[] = {
	{SUBTITLE_ENCODING_ISO_8859_6,		"ISO-8859-6",		N_("Arabic")},
	{SUBTITLE_ENCODING_IBM_864,		"IBM864",		N_("Arabic")},
	{SUBTITLE_ENCODING_MAC_ARABIC,		"MAC_ARABIC",		N_("Arabic")},
	{SUBTITLE_ENCODING_WINDOWS_1256,	"WINDOWS-1256",		N_("Arabic")},

	{SUBTITLE_ENCODING_ARMSCII_8,		"ARMSCII-8",		N_("Armenian")},

	{SUBTITLE_ENCODING_ISO_8859_4,		"ISO-8859-4",		N_("Baltic")},
	{SUBTITLE_ENCODING_ISO_8859_13,		"ISO-8859-13",		N_("Baltic")},
	{SUBTITLE_ENCODING_WINDOWS_1257,	"WINDOWS-1257",		N_("Baltic")},

	{SUBTITLE_ENCODING_ISO_8859_14,		"ISO-8859-14",		N_("Celtic")},

	{SUBTITLE_ENCODING_ISO_8859_2,		"ISO-8859-2",		N_("Central European")},
	{SUBTITLE_ENCODING_IBM_852,		"IBM852",		N_("Central European")},
	{SUBTITLE_ENCODING_MAC_CE,		"MAC_CE",		N_("Central European")},
	{SUBTITLE_ENCODING_WINDOWS_1250,	"WINDOWS-1250",		N_("Central European")},

	{SUBTITLE_ENCODING_GB18030,		"GB18030",		N_("Chinese Simplified")},
	{SUBTITLE_ENCODING_GB2312,		"GB2312",		N_("Chinese Simplified")},
	{SUBTITLE_ENCODING_GBK,			"GBK",			N_("Chinese Simplified")},
	{SUBTITLE_ENCODING_HZ,			"HZ",			N_("Chinese Simplified")},

	{SUBTITLE_ENCODING_BIG5,		"BIG5",			N_("Chinese Traditional")},
	{SUBTITLE_ENCODING_BIG5_HKSCS,		"BIG5-HKSCS",		N_("Chinese Traditional")},
	{SUBTITLE_ENCODING_EUC_TW,		"EUC-TW",		N_("Chinese Traditional")},

	{SUBTITLE_ENCODING_MAC_CROATIAN,	"MAC_CROATIAN",		N_("Croatian")},

	{SUBTITLE_ENCODING_ISO_8859_5,		"ISO-8859-5",		N_("Cyrillic")},
	{SUBTITLE_ENCODING_IBM_855,		"IBM855",		N_("Cyrillic")},
	{SUBTITLE_ENCODING_ISO_IR_111,		"ISO-IR-111",		N_("Cyrillic")},
	{SUBTITLE_ENCODING_KOI8_R,		"KOI8-R",		N_("Cyrillic")},
	{SUBTITLE_ENCODING_MAC_CYRILLIC,	"MAC-CYRILLIC",		N_("Cyrillic")},
	{SUBTITLE_ENCODING_WINDOWS_1251,	"WINDOWS-1251",		N_("Cyrillic")},

	{SUBTITLE_ENCODING_CP_866,		"CP866",		N_("Cyrillic/Russian")},

	{SUBTITLE_ENCODING_MAC_UKRAINIAN,	"MAC_UKRAINIAN",	N_("Cyrillic/Ukrainian")},
	{SUBTITLE_ENCODING_KOI8_U,		"KOI8-U",		N_("Cyrillic/Ukrainian")},

	{SUBTITLE_ENCODING_GEOSTD8,		"GEORGIAN-PS",		N_("Georgian")},

	{SUBTITLE_ENCODING_ISO_8859_7,		"ISO-8859-7",		N_("Greek")},
	{SUBTITLE_ENCODING_MAC_GREEK,		"MAC_GREEK",		N_("Greek")},
	{SUBTITLE_ENCODING_WINDOWS_1253,	"WINDOWS-1253",		N_("Greek")},

	{SUBTITLE_ENCODING_MAC_GUJARATI,	"MAC_GUJARATI",		N_("Gujarati")},

	{SUBTITLE_ENCODING_MAC_GURMUKHI,	"MAC_GURMUKHI",		N_("Gurmukhi")},

	{SUBTITLE_ENCODING_ISO_8859_8_I,	"ISO-8859-8-I",		N_("Hebrew")},
	{SUBTITLE_ENCODING_IBM_862,		"IBM862",		N_("Hebrew")},
	{SUBTITLE_ENCODING_MAC_HEBREW,		"MAC_HEBREW",		N_("Hebrew")},
	{SUBTITLE_ENCODING_WINDOWS_1255,	"WINDOWS-1255",		N_("Hebrew")},

	{SUBTITLE_ENCODING_ISO_8859_8,		"ISO-8859-8",		N_("Hebrew Visual")},

	{SUBTITLE_ENCODING_MAC_DEVANAGARI,	"MAC_DEVANAGARI",	N_("Hindi")},

	{SUBTITLE_ENCODING_MAC_ICELANDIC,	"MAC_ICELANDIC",	N_("Icelandic")},

	{SUBTITLE_ENCODING_EUC_JP,		"EUC-JP",		N_("Japanese")},
	{SUBTITLE_ENCODING_ISO_2022_JP,		"ISO2022JP",		N_("Japanese")},
	{SUBTITLE_ENCODING_SHIFT_JIS,		"SHIFT-JIS",		N_("Japanese")},

	{SUBTITLE_ENCODING_EUC_KR,		"EUC-KR",		N_("Korean")},
	{SUBTITLE_ENCODING_ISO_2022_KR,		"ISO2022KR",		N_("Korean")},
	{SUBTITLE_ENCODING_JOHAB,		"JOHAB",		N_("Korean")},
	{SUBTITLE_ENCODING_UHC,			"UHC",			N_("Korean")},

	{SUBTITLE_ENCODING_ISO_8859_10,		"ISO-8859-10",		N_("Nordic")},

	{SUBTITLE_ENCODING_MAC_FARSI,		"MAC_FARSI",		N_("Persian")},

	{SUBTITLE_ENCODING_ISO_8859_16,		"ISO-8859-16",		N_("Romanian")},
	{SUBTITLE_ENCODING_MAC_ROMANIAN,	"MAC_ROMANIAN",		N_("Romanian")},

	{SUBTITLE_ENCODING_ISO_8859_3,		"ISO-8859-3",		N_("South European")},

	{SUBTITLE_ENCODING_TIS_620,		"TIS-620",		N_("Thai")},

	{SUBTITLE_ENCODING_ISO_8859_9,		"ISO-8859-9",		N_("Turkish")},
	{SUBTITLE_ENCODING_IBM_857,		"IBM857",		N_("Turkish")},
	{SUBTITLE_ENCODING_MAC_TURKISH,		"MAC_TURKISH",		N_("Turkish")},
	{SUBTITLE_ENCODING_WINDOWS_1254,	"WINDOWS-1254",		N_("Turkish")},

	{SUBTITLE_ENCODING_UTF_7,		"UTF-7",		N_("Unicode")},
	{SUBTITLE_ENCODING_UTF_8,		"UTF-8",		N_("Unicode")},
	{SUBTITLE_ENCODING_UTF_16,		"UTF-16",		N_("Unicode")},
	{SUBTITLE_ENCODING_UCS_2,		"UCS-2",		N_("Unicode")},
	{SUBTITLE_ENCODING_UCS_4,		"UCS-4",		N_("Unicode")},

	{SUBTITLE_ENCODING_ISO_8859_1,		"ISO-8859-1",		N_("Western")},
	{SUBTITLE_ENCODING_ISO_8859_15,		"ISO-8859-15",		N_("Western")},
	{SUBTITLE_ENCODING_IBM_850,		"IBM850",		N_("Western")},
	{SUBTITLE_ENCODING_MAC_ROMAN,		"MAC_ROMAN",		N_("Western")},
	{SUBTITLE_ENCODING_WINDOWS_1252,	"WINDOWS-1252",		N_("Western")},

	{SUBTITLE_ENCODING_TCVN,		"TCVN",			N_("Vietnamese")},
	{SUBTITLE_ENCODING_VISCII,		"VISCII",		N_("Vietnamese")},
	{SUBTITLE_ENCODING_WINDOWS_1258,	"WINDOWS-1258",		N_("Vietnamese")},

	{SUBTITLE_ENCODING_CURRENT_LOCALE,	NULL,			N_("Current Locale")}
};

static gboolean
gconf_get_bool (GConfClient *client,
		const gchar *key)
{
	gboolean value = FALSE;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_MAFW_GST_SUBTITLE_RENDERER, key);

	value = gconf_client_get_bool (client, tmp, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_bool (GConfClient *client,
		const gchar *key,
		gboolean value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_MAFW_GST_SUBTITLE_RENDERER, key);

	gconf_client_set_bool (client, tmp, value, NULL);

	if (tmp)
		g_free (tmp);
}

static gchar *
gconf_get_string (GConfClient *client,
		  gchar *key)
{
	gchar *value = FALSE;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_MAFW_GST_SUBTITLE_RENDERER, key);

	value = gconf_client_get_string (client, tmp, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_string (GConfClient *client,
		gchar *key,
		const gchar *value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_MAFW_GST_SUBTITLE_RENDERER, key);

	if (value)
		gconf_client_set_string (client, tmp, value, NULL);
	else
		gconf_client_unset (client, tmp, NULL);

	if (tmp)
		g_free (tmp);
}

static gboolean
is_internal_font (const gchar * name)
{
	/* FIXME Extremally BAD BAD BAD way of doing things */

	return strcmp (name, "DeviceSymbols") == 0
		       || strcmp(name, "Nokia Smiley") == 0;
}

static void
filter_out_internal_fonts (PangoFontFamily **families,
			   int *n_families)
{
	int i;
	int n; /* counts valid fonts */
	const gchar * name = NULL;

	for (i = 0, n = 0; i < * n_families; i++) {
		name = pango_font_family_get_name (families[i]);

		if(!is_internal_font(name)) {
			if (i != n) { /* there are filtered out families */
				families[n] = families[i]; /* shift the current family */
			}
			n++; /* count one more valid */
		}
	} /* foreach font family */

	*n_families = n;
}

static int
cmp_families (const void *a,
	      const void *b)
{
	const char *a_name = pango_font_family_get_name (* (PangoFontFamily **) a);
	const char *b_name = pango_font_family_get_name (* (PangoFontFamily **) b);

	return g_utf8_collate (a_name, b_name);
}

static int
cmp_encodings (const void *a,
	       const void *b)
{
	const SubtitleEncoding *a_encoding = (SubtitleEncoding *) a;
	const SubtitleEncoding *b_encoding = (SubtitleEncoding *) b;

	return g_utf8_collate (_(a_encoding->name), _(b_encoding->name));
}

static void
font_selector_dialog (HildonButton *button,
		      gpointer user_data)
{
	GtkWidget *dialog, *hbox, *family_selector, *style_selector, *size_selector;
	gint index = 0;
	const gchar *font = NULL;
	PangoFontDescription *font_desc = NULL;
	PangoFontFamily **families;
	gint n_families = 0;
	PangoWeight pango_weight;
	PangoStyle pango_style;

	font = hildon_button_get_value (HILDON_BUTTON (button));
	if (font == NULL)
		return;

	font_desc = pango_font_description_from_string (font);

	dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_title (GTK_WINDOW (dialog), _("Font"));
	gtk_dialog_add_button(GTK_DIALOG (dialog), "OK", GTK_RESPONSE_ACCEPT);
	gtk_window_set_default_size (GTK_WINDOW (dialog), -1, 400);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox);

	/* font family selector */
	family_selector = hildon_touch_selector_new_text ();
	gtk_box_pack_start (GTK_BOX (hbox), family_selector, TRUE, TRUE, 0);

	pango_context_list_families (gtk_widget_get_pango_context (GTK_WIDGET (dialog)),
				     &families, &n_families);

	filter_out_internal_fonts (families, &n_families);

	qsort (families, n_families, sizeof(PangoFontFamily *), cmp_families);

	for (index = 0; index < n_families; index++) {
		const gchar *family = pango_font_family_get_name (families[index]);
		hildon_touch_selector_insert_text (HILDON_TOUCH_SELECTOR (family_selector),
						   index, family);

		if (strcmp (family, pango_font_description_get_family (font_desc)) == 0) {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (family_selector), 0,
							  index);
			hildon_touch_selector_center_on_selected (HILDON_TOUCH_SELECTOR (family_selector));
		}
	}
	g_free (families);

	/* font style selector */
	style_selector = hildon_touch_selector_new_text ();
	gtk_widget_set_size_request (style_selector, 200, -1);
	gtk_box_pack_start (GTK_BOX (hbox), style_selector, FALSE, TRUE, 0);

	index = 0;
	while (index < FONT_STYLE_LAST) {
		const gchar *style = g_strdup_printf ("%s", _(font_styles[index].name));
		hildon_touch_selector_insert_text (HILDON_TOUCH_SELECTOR (style_selector),
						   font_styles[index].index, style);
		index++;
	}
	pango_weight = pango_font_description_get_weight (font_desc);
	pango_style = pango_font_description_get_style (font_desc);

	if (pango_weight == PANGO_WEIGHT_NORMAL) {
		if (pango_style == PANGO_STYLE_NORMAL) {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (style_selector), 0,
							  FONT_STYLE_REGULAR);
		} else {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (style_selector), 0,
							  FONT_STYLE_ITALIC);
		}
	} else {
		if (pango_style == PANGO_STYLE_NORMAL) {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (style_selector), 0,
							  FONT_STYLE_BOLD);
		} else {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (style_selector), 0,
							  FONT_STYLE_ITALIC_BOLD);
		}
	}
	hildon_touch_selector_center_on_selected (HILDON_TOUCH_SELECTOR (style_selector));

	/* font size selector */
	size_selector = hildon_touch_selector_new_text ();
	gtk_widget_set_size_request (size_selector, 100, -1);
	gtk_box_pack_start (GTK_BOX (hbox), size_selector, FALSE, TRUE, 0);

	index = 0;
	while (font_sizes[index] != -1) {
		const gchar *size = g_strdup_printf ("%d", font_sizes[index]);
		hildon_touch_selector_insert_text (HILDON_TOUCH_SELECTOR (size_selector),
						   index, size);

		if (font_sizes[index] == (pango_font_description_get_size (font_desc) / PANGO_SCALE)) {
			hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (size_selector), 0,
							  index);
			hildon_touch_selector_center_on_selected (HILDON_TOUCH_SELECTOR (size_selector));
		}

		index++;
	}

	/* Run the dialog */
	gtk_widget_show_all (GTK_WIDGET (dialog));
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		if (font_desc)
			pango_font_description_free (font_desc);

		font_desc = pango_font_description_new ();

		/* set font family */
		pango_font_description_set_family (font_desc,
						   hildon_touch_selector_get_current_text (HILDON_TOUCH_SELECTOR (family_selector)));

		/* set font style */
		switch (hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (style_selector), 0)) {
			case FONT_STYLE_REGULAR:
				pango_font_description_set_style (font_desc, PANGO_STYLE_NORMAL);
				pango_font_description_set_weight (font_desc, PANGO_WEIGHT_NORMAL);
				break;

			case FONT_STYLE_ITALIC:
				pango_font_description_set_style (font_desc, PANGO_STYLE_ITALIC);
				pango_font_description_set_weight (font_desc, PANGO_WEIGHT_NORMAL);
				break;

			case FONT_STYLE_BOLD:
				pango_font_description_set_style (font_desc, PANGO_STYLE_NORMAL);
				pango_font_description_set_weight (font_desc, PANGO_WEIGHT_BOLD);
				break;

			case FONT_STYLE_ITALIC_BOLD:
				pango_font_description_set_style (font_desc, PANGO_STYLE_ITALIC);
				pango_font_description_set_weight (font_desc, PANGO_WEIGHT_BOLD);
				break;
		}

		/* set font size */
		pango_font_description_set_size (font_desc,
						 font_sizes[hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (size_selector), 0)] * PANGO_SCALE);

		hildon_button_set_value (HILDON_BUTTON (button), pango_font_description_to_string (font_desc));
	}

	if (font_desc)
		pango_font_description_free (font_desc);

	gtk_widget_destroy(GTK_WIDGET(dialog));
}


static GtkWidget *
create_encoding_selector (void)
{
	GtkWidget *selector;
	gint index = 0;

	selector = hildon_touch_selector_new_text ();

	qsort (encodings, SUBTITLE_ENCODING_LAST - 1, sizeof (SubtitleEncoding), cmp_encodings);

	while (index < SUBTITLE_ENCODING_LAST) {
		const gchar *encoding = NULL;

		if (encodings[index].charset) {
			encoding = g_strdup_printf ("%s (%s)", _(encodings[index].name),
						    encodings[index].charset);
		} else {
			encoding = g_strdup_printf ("%s", _(encodings[index].name));
		}

		hildon_touch_selector_insert_text (HILDON_TOUCH_SELECTOR (selector),
						   index,
						   encoding);
		index++;
	}

	return selector;
}

static GtkWidget *
create_autoload_subtitles_button (GConfClient *gconf_client)
{
	GtkWidget *button;
	gboolean autoload_subtitles = FALSE;

	button = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (button), _("Automatically load subtitle files"));

	autoload_subtitles = gconf_get_bool (gconf_client, "autoload_subtitles");
	if (autoload_subtitles)
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (button), TRUE);
	else
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (button), FALSE);

	return button;
}

static void
save_autoload_subtitles (GConfClient *gconf_client,
			 GtkWidget *widget)
{
	if (hildon_check_button_get_active (HILDON_CHECK_BUTTON (widget)))
		gconf_set_bool (gconf_client, "autoload_subtitles", TRUE);
	else
		gconf_set_bool (gconf_client, "autoload_subtitles", FALSE);
}

static GtkWidget *
create_subtitles_font_button (GConfClient *gconf_client)
{
	GtkWidget *button;
	const gchar *font = NULL;

	button = hildon_button_new (HILDON_SIZE_FINGER_HEIGHT,
				    HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Font"));
	hildon_button_set_alignment (HILDON_BUTTON (button), 0.0, 0.5, 1.0, 0.0);
	hildon_button_set_title_alignment (HILDON_BUTTON(button), 0.0, 0.5);
	hildon_button_set_value_alignment (HILDON_BUTTON (button), 0.0, 0.5);
	hildon_button_set_style (HILDON_BUTTON (button), HILDON_BUTTON_STYLE_PICKER);

	font = gconf_get_string (gconf_client, "subtitle_font");
	if (font) {
		hildon_button_set_value (HILDON_BUTTON (button), font);
	} else {
		hildon_button_set_value (HILDON_BUTTON (button), "Sans Bold 18");
	}

	g_signal_connect (button, "clicked", G_CALLBACK (font_selector_dialog),
			  NULL);

	return button;
}

static void
save_subtitles_font (GConfClient *gconf_client,
		     GtkWidget *widget)
{
	const gchar *font = NULL;

	font = hildon_button_get_value (HILDON_BUTTON (widget));
	gconf_set_string (gconf_client, "subtitle_font", font);
}

static GtkWidget *
create_subtitles_encoding_button (GConfClient *gconf_client)
{
	GtkWidget *button, *selector;
	const gchar *encoding = NULL;

	button = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Encoding"));
	hildon_button_set_alignment (HILDON_BUTTON (button), 0.0, 0.5, 1.0, 0.0);
	hildon_button_set_title_alignment (HILDON_BUTTON(button), 0.0, 0.5);
	hildon_button_set_value_alignment (HILDON_BUTTON (button), 0.0, 0.5);

	selector = create_encoding_selector ();
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
					   HILDON_TOUCH_SELECTOR (selector));

	encoding = gconf_get_string (gconf_client, "subtitle_encoding");
	if (encoding) {
		gint index = 0;

		while (index < SUBTITLE_ENCODING_LAST) {
			if (encodings[index].charset) {
				if (strcmp (encodings[index].charset, encoding) == 0) {
					hildon_picker_button_set_active (HILDON_PICKER_BUTTON (button),
									 index);
					break;
				}
			}
			index++;
		}
	} else {
		hildon_picker_button_set_active (HILDON_PICKER_BUTTON (button),
						 SUBTITLE_ENCODING_CURRENT_LOCALE);
	}

	return button;
}

static void
save_subtitles_encoding (GConfClient *gconf_client,
			 GtkWidget *widget)
{
	gint encoding = 0, index = 0;

	encoding = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (widget));

	while (index < SUBTITLE_ENCODING_LAST) {
		if (encoding == index) {
			gconf_set_string (gconf_client, "subtitle_encoding",
					  encodings[index].charset);
			break;
		}
		index++;
	}
}

osso_return_t
execute (osso_context_t *osso,
	 gpointer data,
	 gboolean user_activated)
{
	GConfClient *gconf_client = NULL;
	GtkWidget *dialog, *vbox, *autoload_subtitles_button;
	GtkWidget *subtitles_font_button, *subtitles_encoding_button;

	gconf_client = gconf_client_get_default ();
	if (gconf_client == NULL) {
		return OSSO_ERROR;
	}

	dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data));
	gtk_window_set_title (GTK_WINDOW (dialog), _("Subtitles"));
	gtk_dialog_add_button(GTK_DIALOG (dialog), _HL("wdgt_bd_save"), GTK_RESPONSE_ACCEPT);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);

	/* autoload subtitles button */
	autoload_subtitles_button = create_autoload_subtitles_button (gconf_client);
	gtk_box_pack_start (GTK_BOX (vbox), autoload_subtitles_button, TRUE, TRUE, 0);

	/* font selector */
	subtitles_font_button = create_subtitles_font_button (gconf_client);
	gtk_box_pack_start (GTK_BOX (vbox), subtitles_font_button, TRUE, TRUE, 0);

	/* font encoding */
	subtitles_encoding_button = create_subtitles_encoding_button (gconf_client);
	gtk_box_pack_start (GTK_BOX (vbox), subtitles_encoding_button, TRUE, TRUE, 0);

	/* Run the dialog */
	gtk_widget_show_all (GTK_WIDGET (dialog));
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		/* save autoload subtitles option */
		save_autoload_subtitles (gconf_client, autoload_subtitles_button);

		/* save subtitle font option */
		save_subtitles_font (gconf_client, subtitles_font_button);

		/* save subtitle encoding option */
		save_subtitles_encoding (gconf_client, subtitles_encoding_button);
	}

	gtk_widget_destroy(GTK_WIDGET(dialog));
	return OSSO_OK;
}

osso_return_t
save_state (osso_context_t *osso,
	    gpointer data)
{
	return OSSO_OK;
}

