/*
 * tangle-clamp.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-clamp.h"

/**
 * SECTION:tangle-clamp
 * @Short_description: An object holding an original value and possible clamped value
 * @Title: TangleClamp
 *
 * This object is typically used when scrolling actors with #TangleScrollingActor.
 */

G_DEFINE_TYPE(TangleClamp, tangle_clamp, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_DIRECTION,
	PROP_ORIGINAL_VALUE,
	PROP_CLAMPED_VALUE,
	PROP_CLAMPED_VALUE_SET
};

struct _TangleClampPrivate {
	TangleClampDirection direction;
	gfloat original_value;
	gfloat clamped_value;
	
	guint clamped_value_set : 1;
};

static const GEnumValue clamp_direction_values[] = {
	{ TANGLE_CLAMP_UNKNOWN, "TANGLE_CLAMP_UNKNOWN", "unknown" },
	{ TANGLE_CLAMP_HORIZONTAL, "TANGLE_CLAMP_HORIZONTAL", "horizontal" },
	{ TANGLE_CLAMP_VERTICAL, "TANGLE_CLAMP_VERTICAL", "vertical" },
	{ TANGLE_CLAMP_BOTH, "TANGLE_CLAMP_BOTH", "both" },
	{ 0, NULL, NULL }
};

GType tangle_clamp_direction_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("TangleClampDirection", clamp_direction_values);
	}
	
	return type;
}
TangleClamp* tangle_clamp_new(TangleClampDirection direction, gfloat original_value) {

	return TANGLE_CLAMP(g_object_new(TANGLE_TYPE_CLAMP, "direction", direction, "original-value", original_value, NULL));
}

TangleClampDirection tangle_clamp_get_direction(TangleClamp* clamp) {
	g_return_val_if_fail(TANGLE_IS_CLAMP(clamp), TANGLE_CLAMP_UNKNOWN);

	return clamp->priv->direction;
}

gfloat tangle_clamp_get_original_value(TangleClamp* clamp) {
	g_return_val_if_fail(TANGLE_IS_CLAMP(clamp), 0.0);

	return clamp->priv->original_value;
}

gboolean tangle_clamp_get_clamped_value_set(TangleClamp* clamp) {
	g_return_val_if_fail(TANGLE_IS_CLAMP(clamp), FALSE);

	return clamp->priv->clamped_value_set;
}

gfloat tangle_clamp_get_clamped_value(TangleClamp* clamp) {
	g_return_val_if_fail(TANGLE_IS_CLAMP(clamp), 0.0);

	return clamp->priv->clamped_value;
}

void tangle_clamp_set_clamped_value(TangleClamp* clamp, gfloat clamped_value) {
	gboolean was_clamped_value_set;

	g_return_if_fail(TANGLE_IS_CLAMP(clamp));
	
	if (clamp->priv->clamped_value != clamped_value || !clamp->priv->clamped_value_set) {
		was_clamped_value_set = clamp->priv->clamped_value_set;
		
		clamp->priv->clamped_value = clamped_value;
		clamp->priv->clamped_value_set = TRUE;
		
		g_object_notify(G_OBJECT(clamp), "clamped-value");
		if (!was_clamped_value_set) {
			g_object_notify(G_OBJECT(clamp), "clamped-value-set");
		}
	}
}

static void tangle_clamp_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleClamp* clamp;
	
	clamp = TANGLE_CLAMP(object);

	switch (prop_id) {
		case PROP_DIRECTION:
			clamp->priv->direction = g_value_get_enum(value);
			break;
		case PROP_ORIGINAL_VALUE:
			clamp->priv->original_value = g_value_get_float(value);
			break;
		case PROP_CLAMPED_VALUE:
			tangle_clamp_set_clamped_value(clamp, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_clamp_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleClamp* clamp;

	clamp = TANGLE_CLAMP(object);

        switch (prop_id) {
		case PROP_DIRECTION:
			g_value_set_enum(value, clamp->priv->direction);
			break;
		case PROP_ORIGINAL_VALUE:
			g_value_set_float(value, clamp->priv->original_value);
			break;
		case PROP_CLAMPED_VALUE:
			g_value_set_float(value, clamp->priv->clamped_value);
			break;
		case PROP_CLAMPED_VALUE_SET:
			g_value_set_boolean(value, clamp->priv->clamped_value_set);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_clamp_class_init(TangleClampClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->set_property = tangle_clamp_set_property;
	gobject_class->get_property = tangle_clamp_get_property;

	/**
	 * TangleClamp:direction:
	 */
	g_object_class_install_property(gobject_class, PROP_DIRECTION,
	                                g_param_spec_enum("direction",
	                                "Direction",
	                                "The direction of the value in the context of the signal emitter",
	                                TANGLE_TYPE_CLAMP_DIRECTION,
					TANGLE_CLAMP_UNKNOWN,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleClamp:original-value:
	 */
	g_object_class_install_property(gobject_class, PROP_ORIGINAL_VALUE,
	                                g_param_spec_float("original-value",
	                                "Original value",
	                                "The original value of the clamped value",
	                                -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleClamp:clamped-value:
	 */
	g_object_class_install_property(gobject_class, PROP_CLAMPED_VALUE,
	                                g_param_spec_float("clamped-value",
	                                "Clamped value",
	                                "The clamped value, if set",
	                                -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleClamp:clamped-value-set:
	 */
	g_object_class_install_property(gobject_class, PROP_CLAMPED_VALUE_SET,
	                                g_param_spec_boolean("clamped-value-set",
	                                "Clamped value set",
	                                "Whether the clamped value is set",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof (TangleClampPrivate));
}

static void tangle_clamp_init(TangleClamp* clamp) {
	clamp->priv = G_TYPE_INSTANCE_GET_PRIVATE(clamp, TANGLE_TYPE_CLAMP, TangleClampPrivate);
}

