/*
 * tangle-bufferer.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-bufferer.h"

G_DEFINE_TYPE(TangleBufferer, tangle_bufferer, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
};

typedef struct {
	ClutterActor* actor;
	CoglMaterial* material;
	CoglMatrix matrix;
	guint has_matrix : 1;
} Buffer;

typedef struct {
	guint cycle;
	glong prepaint_signal_handler;
	glong postpaint_signal_handler;
} Redrawable;

struct _TangleBuffererPrivate {
	guint cycle;
	GHashTable* queued_actors;
	CoglHandle offscreen;
	GList* buffers;
	guint nested_paints;
	gfloat absolute_x;
	gfloat absolute_y;
	gfloat absolute_width;
	gfloat absolute_height;
	
	guint buffering : 1;
	guint redraw_self : 1;
};

static void begin_paint_actor(ClutterActor* actor, gpointer user_data);
static void end_paint_actor(ClutterActor* actor, gpointer user_data);
static void queue_actor(TangleBufferer* bufferer, ClutterActor* actor);
static void destroy_buffers(TangleBufferer* bufferer);
static void start_buffering(TangleBufferer* bufferer);
static void switch_buffer(TangleBufferer* bufferer, ClutterActor* actor);
static void end_buffering(TangleBufferer* bufferer);
static void draw_buffers(TangleBufferer* bufferer);
static gboolean check_queued_actor(gpointer key, gpointer value, gpointer user_data);
static void remove_queued_actor(gpointer user_data, GObject* object);
static void get_absolute_allocation_box(ClutterActor* actor, ClutterActorBox* actor_box);

static TangleBufferer* active_bufferer;

ClutterActor* tangle_bufferer_new(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUFFERER, NULL));
}

ClutterActor* tangle_bufferer_new_with_layout(TangleLayout* layout) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUFFERER, "layout", layout, NULL));
}

void tangle_bufferer_mark_allocation_changed(ClutterActor* actor) {
	if (active_bufferer && actor != CLUTTER_ACTOR(active_bufferer) &&
	    (actor = clutter_actor_get_parent(actor)) && actor != CLUTTER_ACTOR(active_bufferer)) {
		queue_actor(active_bufferer, actor);
	}
}

static void tangle_bufferer_queue_redraw(ClutterActor* actor, ClutterActor* queued_actor) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(actor);
	
	if (actor != queued_actor) {
		queue_actor(bufferer, queued_actor);
	}
	
	CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->queue_redraw(actor, queued_actor);
}

static void tangle_bufferer_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	g_return_if_fail(active_bufferer == NULL);
	
	active_bufferer = TANGLE_BUFFERER(actor);
	CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->allocate(actor, box, flags);
	active_bufferer = NULL;
}

static void tangle_bufferer_paint(ClutterActor* actor) {
	TangleBufferer* bufferer;
	ClutterActorBox box;
	guint n_queued_actors;
	gfloat width, height;
	GList* list;
	
	bufferer = TANGLE_BUFFERER(actor);
	
	if (g_hash_table_foreach_remove(bufferer->priv->queued_actors, check_queued_actor, bufferer) > 0) {
		bufferer->priv->buffering = TRUE;
	}

	get_absolute_allocation_box(CLUTTER_ACTOR(bufferer), &box);
	if (bufferer->priv->absolute_width != box.x2 - box.x1 ||
	    bufferer->priv->absolute_height != box.y2 - box.y1) {
		bufferer->priv->absolute_width = box.x2 - box.x1;
		bufferer->priv->absolute_height = box.y2 - box.y1;
	
		bufferer->priv->buffering = TRUE;
	}
	bufferer->priv->absolute_x = box.x1;
	bufferer->priv->absolute_y = box.y1;

	if (bufferer->priv->redraw_self || g_hash_table_size(bufferer->priv->queued_actors) * 2 + 1 > 100) {
		bufferer->priv->buffering = FALSE;		
		destroy_buffers(bufferer);
		
		CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->paint(actor);
	} else {
		if (bufferer->priv->buffering) {
			destroy_buffers(bufferer);

			start_buffering(bufferer);
			CLUTTER_ACTOR_CLASS(tangle_bufferer_parent_class)->paint(actor);
			end_buffering(bufferer);
		}
		
		draw_buffers(bufferer);
	}
	
	bufferer->priv->cycle++;
	if (bufferer->priv->redraw_self) {
		bufferer->priv->redraw_self = FALSE;
		bufferer->priv->buffering = TRUE;
	} else {
		bufferer->priv->buffering = FALSE;
	}
}

static void tangle_bufferer_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_bufferer_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleBufferer* bufferer;

	bufferer = TANGLE_BUFFERER(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_bufferer_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_bufferer_parent_class)->finalize(object);
}

static void tangle_bufferer_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_bufferer_parent_class)->dispose(object);
}

static void tangle_bufferer_class_init(TangleBuffererClass* bufferer_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(bufferer_class);
	ClutterActorClass* actor_class = CLUTTER_ACTOR_CLASS(bufferer_class);

	gobject_class->finalize = tangle_bufferer_finalize;
	gobject_class->dispose = tangle_bufferer_dispose;
	gobject_class->set_property = tangle_bufferer_set_property;
	gobject_class->get_property = tangle_bufferer_get_property;

	actor_class->queue_redraw = tangle_bufferer_queue_redraw;
	actor_class->allocate = tangle_bufferer_allocate;
	actor_class->paint = tangle_bufferer_paint;

	g_type_class_add_private(gobject_class, sizeof(TangleBuffererPrivate));
}

static void tangle_bufferer_init(TangleBufferer* bufferer) {
	bufferer->priv = G_TYPE_INSTANCE_GET_PRIVATE(bufferer, TANGLE_TYPE_BUFFERER, TangleBuffererPrivate);

	bufferer->priv->queued_actors = g_hash_table_new(g_direct_hash, g_direct_equal);
}

static void begin_paint_actor(ClutterActor* actor, gpointer user_data) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(user_data);
	
	if (bufferer->priv->buffering && bufferer->priv->nested_paints++ == 0) {
		switch_buffer(bufferer, actor);
	}
}

static void end_paint_actor(ClutterActor* actor, gpointer user_data) {
	TangleBufferer* bufferer;
	
	bufferer = TANGLE_BUFFERER(user_data);
	
	if (bufferer->priv->buffering && --bufferer->priv->nested_paints == 0) {
		switch_buffer(bufferer, NULL);
	}
}

static void queue_actor(TangleBufferer* bufferer, ClutterActor* actor) {
	Redrawable* redrawable;

	if (!(redrawable = (Redrawable*)g_hash_table_lookup(bufferer->priv->queued_actors, actor))) {
		redrawable = g_new(Redrawable, 1);

		redrawable->prepaint_signal_handler = g_signal_connect(actor, "paint", G_CALLBACK(begin_paint_actor), bufferer);
		redrawable->postpaint_signal_handler = g_signal_connect_after(actor, "paint", G_CALLBACK(end_paint_actor), bufferer);
		g_object_weak_ref(G_OBJECT(actor), remove_queued_actor, bufferer);

		g_hash_table_insert(bufferer->priv->queued_actors, actor, redrawable);

		bufferer->priv->buffering = TRUE;
	}
	redrawable->cycle = bufferer->priv->cycle;
}

static void create_buffer(TangleBufferer* bufferer, CoglHandle texture, ClutterActor* actor) {
	Buffer* buffer;
	ClutterActor* parent;
	
	buffer = g_new0(Buffer, 1);
	buffer->actor = actor;
	buffer->material = cogl_material_new();
	cogl_material_set_layer(buffer->material, 0, texture);

	bufferer->priv->buffers = g_list_prepend(bufferer->priv->buffers, buffer);
}

static void destroy_buffers(TangleBufferer* bufferer) {
	GList* list;
	Buffer* buffer;
	
	for (list = bufferer->priv->buffers; list; list = list->next) {
		buffer = (Buffer*)list->data;
		if (buffer->material) {
			cogl_handle_unref(buffer->material);
		}
		g_free(buffer);
	}
	g_list_free(bufferer->priv->buffers);
	bufferer->priv->buffers = NULL;		
}

static void start_buffering(TangleBufferer* bufferer) {
	CoglHandle texture;
	Buffer* buffer;
	ClutterPerspective perspective;
	CoglMatrix projection_matrix;
	CoglMatrix modelview_matrix;
	gfloat width, height;
	CoglColor color;
	gfloat viewport[4];

	if ((texture = cogl_texture_new_with_size(bufferer->priv->absolute_width, bufferer->priv->absolute_height, COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE)) != COGL_INVALID_HANDLE) {
		if ((bufferer->priv->offscreen = cogl_offscreen_new_to_texture(texture)) != COGL_INVALID_HANDLE) {
			create_buffer(bufferer, texture, NULL);

			clutter_stage_get_perspective(CLUTTER_STAGE(clutter_actor_get_stage(CLUTTER_ACTOR(bufferer))), &perspective);
			cogl_matrix_init_identity(&projection_matrix);
			cogl_matrix_perspective(&projection_matrix, perspective.fovy, perspective.aspect, perspective.z_near, perspective.z_far);
			cogl_get_modelview_matrix(&modelview_matrix);

			cogl_push_framebuffer(bufferer->priv->offscreen);

			clutter_actor_get_size(clutter_actor_get_stage(CLUTTER_ACTOR(bufferer)), &width, &height);
			cogl_set_viewport(-bufferer->priv->absolute_x, -bufferer->priv->absolute_y, width, height);
			cogl_set_projection_matrix(&projection_matrix);
			cogl_set_modelview_matrix(&modelview_matrix);

			cogl_color_init_from_4ub(&color, 0, 0, 0, 0);
			cogl_clear(&color, COGL_BUFFER_BIT_COLOR | COGL_BUFFER_BIT_DEPTH);

			cogl_push_matrix();
		}
		
		cogl_handle_unref(texture);
	}

}

static void switch_buffer(TangleBufferer* bufferer, ClutterActor* actor) {
	CoglHandle texture;
	CoglColor color;
	gboolean success = FALSE;

	if ((texture = cogl_texture_new_with_size(bufferer->priv->absolute_width, bufferer->priv->absolute_height, COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE)) != COGL_INVALID_HANDLE) {
		if (cogl_offscreen_set_to_texture(bufferer->priv->offscreen, texture)) {
			create_buffer(bufferer, texture, actor);

			cogl_color_init_from_4ub(&color, 0, 0, 0, 0);
			cogl_clear(&color, COGL_BUFFER_BIT_COLOR | COGL_BUFFER_BIT_DEPTH);

			success = TRUE;
		}
	
		cogl_handle_unref(texture);
	}
	
	if (!success) {
		end_buffering(bufferer);
		draw_buffers(bufferer);
		destroy_buffers(bufferer);
	}
}

static void end_buffering(TangleBufferer* bufferer) {
	if (bufferer->priv->offscreen) {
		cogl_pop_matrix();
		cogl_pop_framebuffer();

		cogl_handle_unref(bufferer->priv->offscreen);
		bufferer->priv->offscreen = NULL;

		bufferer->priv->buffers = g_list_reverse(bufferer->priv->buffers);
	}
}

static void draw_buffers(TangleBufferer* bufferer) {
	GList* list;
	Buffer* buffer;
	guint8 paint_opacity;
	ClutterActor* actor;
	CoglMatrix matrix;
	CoglMatrix result_matrix;

	for (list = bufferer->priv->buffers; list; list = list->next) {
		buffer = (Buffer*)list->data;

		if (buffer->material) {
			/* TODO: opacity should not be used every time; maybe we should triple buffer? */
			paint_opacity = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(bufferer));
			cogl_material_set_color4ub(buffer->material, paint_opacity, paint_opacity, paint_opacity, paint_opacity);

			cogl_set_source(buffer->material);
			cogl_rectangle_with_texture_coords(0, 0, bufferer->priv->absolute_width, bufferer->priv->absolute_height, 0.0, 0.0, 1.0, 1.0);

			if (buffer->actor) {
				cogl_handle_unref(buffer->material);
				buffer->material = NULL;
			}
		} else {	
			if (!buffer->has_matrix) {
				cogl_matrix_init_identity(&buffer->matrix);
				actor = buffer->actor;
				while ((actor = clutter_actor_get_parent(actor)) && actor != CLUTTER_ACTOR(bufferer)) {
					clutter_actor_get_transformation_matrix(actor, &matrix);
					cogl_matrix_multiply(&result_matrix, &matrix, &buffer->matrix);
					buffer->matrix = result_matrix;
				}
			}

			cogl_push_matrix();
			cogl_transform(&buffer->matrix);
			clutter_actor_paint(buffer->actor);
			cogl_pop_matrix();
		}
	}
}

static gboolean check_queued_actor(gpointer key, gpointer value, gpointer user_data) {
	gboolean remove = FALSE;
	TangleBufferer* bufferer;
	Redrawable* redrawable;
	
	bufferer = TANGLE_BUFFERER(user_data);
	redrawable = (Redrawable*)value;
	
	if (redrawable->cycle != bufferer->priv->cycle) {
		g_signal_handler_disconnect(key, redrawable->prepaint_signal_handler);
		g_signal_handler_disconnect(key, redrawable->postpaint_signal_handler);
		g_object_weak_unref(G_OBJECT(key), remove_queued_actor, bufferer);

		g_free(redrawable);
		
		remove = TRUE;
	}
	
	return remove;
}

static void remove_queued_actor(gpointer user_data, GObject* object) {
	TangleBufferer* bufferer;
	Redrawable* redrawable;
	
	bufferer = TANGLE_BUFFERER(user_data);
	
	if ((redrawable = (Redrawable*)g_hash_table_lookup(bufferer->priv->queued_actors, object))) {
		g_free(redrawable);
		g_hash_table_remove(bufferer->priv->queued_actors, object);
	}
}

static void get_absolute_allocation_box(ClutterActor* actor, ClutterActorBox* actor_box) {
	ClutterVertex vertices[4];
	gfloat min_x = G_MAXFLOAT, min_y = G_MAXFLOAT;
	gfloat max_x = 0, max_y = 0;
	gint i;
	gfloat viewport[4];

	clutter_actor_get_abs_allocation_vertices(actor, vertices);
	for (i = 0; i < G_N_ELEMENTS(vertices); i++) {
		if (vertices[i].x < min_x) {
			min_x = vertices[i].x;
		}
		if (vertices[i].y < min_y) {
			min_y = vertices[i].y;
		}
		if (vertices[i].x > max_x) {
			max_x = vertices[i].x;
		}
		if (vertices[i].y > max_y) {
			max_y = vertices[i].y;
		}
	}

	cogl_get_viewport(viewport);

#define ROUND(x) ((x) >= 0 ? (long)((x) + 0.5) : (long)((x) - 0.5))

	actor_box->x1 = ROUND(min_x) - viewport[0];
	actor_box->x2 = ROUND(max_x) - viewport[1];
	actor_box->y1 = ROUND(min_y) - viewport[0];
	actor_box->y2 = ROUND(max_y) - viewport[1];

	if (actor_box->x2 - actor_box->x1 < 1) {
		actor_box->x2 = actor_box->x1 + 1;
	}
	if (actor_box->y2 - actor_box->y1 < 1) {
		actor_box->y2 = actor_box->y1 + 1;
	}

#undef ROUND
}
