/*
 * tangle-widget.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-widget.h"
#include "tangle-stylable.h"
#include "tangle-stylesheet.h"
#include "tangle-misc.h"
#include "tangle-scroll-action.h"
#include "tangle-scroll-trick.h"

/**
 * SECTION:tangle-widget
 * @Short_description: A base class for user interface elements that consists of other actors
 * @Title: TangleWidget
 */

static void clutter_container_iface_init(ClutterContainerIface* iface);
static void tangle_stylable_iface_init(TangleStylableIface* iface);

G_DEFINE_TYPE_WITH_CODE(TangleWidget, tangle_widget, TANGLE_TYPE_ACTOR,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_CONTAINER, clutter_container_iface_init);
			G_IMPLEMENT_INTERFACE(TANGLE_TYPE_STYLABLE, tangle_stylable_iface_init));

enum {
	PROP_0,
	PROP_N_CHILDREN,
	PROP_CHILREN_READONLY,
	PROP_LAYOUT,
	PROP_PADDING,
	PROP_PADDING_SET,
	PROP_CHILDREN_ALLOCATION,
	PROP_BACKGROUND_COLOR,
	PROP_BACKGROUND_COLOR_SET,
	PROP_BACKGROUND_ACTOR,
	PROP_FOREGROUND_ACTOR,
	PROP_PREFER_BACKGROUND_SIZE,
	PROP_PREFER_BACKGROUND_SIZE_SET,
	PROP_PICKABLE,
	PROP_DESCENDANT_INTERACTING,
	PROP_DESCENDANT_DRAGGING
};

enum {
	PAINT_CHILDREN,
	LAST_SIGNAL
};
struct _TangleWidgetPrivate {
	GList* children;
	TangleLayout* layout;
	TangleSpacing padding;
	ClutterActorBox children_allocation;
	TangleStylesheet* stylesheet;

	ClutterColor background_color;
	ClutterActor* background_actor;
	ClutterActor* foreground_actor;
	
	TangleActor* interacting_descendant;
	TangleActor* dragging_descendant;
	
	guint padding_set : 1;
	guint background_color_set : 1;
	guint prefer_background_size : 1;
	guint prefer_background_size_set : 1;
	guint pickable : 1;
	
	guint background_parented : 1;
	guint foreground_parented : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

static void tangle_widget_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p);
static void tangle_widget_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p);
static void tangle_widget_add_impl(TangleWidget* container, ClutterActor* actor, ClutterActor* existing, gboolean before);
static gboolean tangle_widget_unapply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet);
static void change_actor_order(TangleWidget* widget, ClutterActor* actor, ClutterActor* sibling, gboolean before);
static gint compare_actor_depths(gconstpointer a, gconstpointer b);
static gint compare_actor_if_higher_depth_position(gconstpointer a, gconstpointer b);
static void force_actor_depth(ClutterActor* actor, ClutterActor* sibling, gboolean before);
static void on_notify_interacting(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_notify_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data);

ClutterActor* tangle_widget_new() {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, NULL));
}

ClutterActor* tangle_widget_new_with_layout(TangleLayout* layout) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, "layout", layout, NULL));
}

ClutterActor* tangle_widget_new_scrollable(TangleLayout* layout) {
	ClutterActor* actor;
	ClutterAction* action;
	TangleTrick* trick;

	actor = CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, "layout", layout, NULL));

	clutter_actor_set_clip_to_allocation(actor, TRUE);

	action = tangle_scroll_action_new();
	clutter_actor_add_action(actor, action);

	trick = tangle_scroll_trick_new();
	tangle_layout_add_trick(layout, trick);
	
	tangle_scroll_action_bind_to_scroll_trick(TANGLE_SCROLL_ACTION(action), TANGLE_SCROLL_TRICK(trick));
	
	return actor;
}


guint tangle_widget_get_n_children(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), 0);

	return g_list_length(widget->priv->children);
}

ClutterActor* tangle_widget_get_nth_child(TangleWidget* widget, guint n) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), NULL);

	return CLUTTER_ACTOR(g_list_nth_data(widget->priv->children, n));
}

gboolean tangle_widget_has_child(TangleWidget* widget, ClutterActor* actor) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return g_list_find(widget->priv->children, actor) != NULL;
}

GList* tangle_widget_get_children_readonly(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), NULL);

	return widget->priv->children;
}

void tangle_widget_foreach_swapped(TangleWidget* widget, TangleSwappedCallback swapped_callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		swapped_callback(user_data, CLUTTER_ACTOR(child_in_list->data));
	}
}

void tangle_widget_foreach_type(TangleWidget* widget, GType type, ClutterCallback callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		if (g_type_is_a(G_OBJECT_TYPE(child_in_list->data), type)) {
			callback(CLUTTER_ACTOR(child_in_list->data), user_data);
		}
	}
}

void tangle_widget_foreach_type_swapped(TangleWidget* widget, GType type, TangleSwappedCallback swapped_callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		if (g_type_is_a(G_OBJECT_TYPE(child_in_list->data), type)) {
			swapped_callback(user_data, CLUTTER_ACTOR(child_in_list->data));
		}
	}
}

void tangle_widget_add(TangleWidget* widget, ClutterActor* actor, ...) {
	GList* existing_in_children;
	
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	existing_in_children = g_list_find_custom(widget->priv->children, actor, compare_actor_if_higher_depth_position);
	if (existing_in_children) {
		tangle_widget_add_impl(TANGLE_WIDGET(widget), actor, CLUTTER_ACTOR(existing_in_children->data), TRUE);
	} else {
		tangle_widget_add_impl(TANGLE_WIDGET(widget), actor, NULL, FALSE);
	}
	
	/* TODO: properties */
}

void tangle_widget_add_before(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, ...) {
	tangle_widget_add_impl(widget, actor, existing, TRUE);

	/* TODO: properties */
}

void tangle_widget_add_after(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, ...) {
	tangle_widget_add_impl(widget, actor, existing, FALSE);

	/* TODO: properties */
}

void tangle_widget_set_layout(TangleWidget* widget, TangleLayout* layout) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->layout != layout) {
		if (widget->priv->layout) {
			g_object_unref(widget->priv->layout);
			/* TODO: g_signal_remove */
			widget->priv->layout = NULL;
		}
		if (layout) {
			widget->priv->layout = layout;
			g_object_ref_sink(layout);
			g_signal_connect_swapped(layout, "changed", G_CALLBACK(clutter_actor_queue_relayout), widget);
			tangle_layout_set_widget(layout, widget);
		}
		
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		g_object_notify(G_OBJECT(widget), "layout");
	}
}

TangleLayout* tangle_widget_get_layout(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), NULL);

	return widget->priv->layout;
}

void tangle_widget_get_padding(TangleWidget* widget, TangleSpacing* spacing) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	*spacing = widget->priv->padding;
}
	
void tangle_widget_set_padding(TangleWidget* widget, const TangleSpacing* spacing) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (!widget->priv->padding_set || !tangle_spacing_equal(&widget->priv->padding, spacing)) {
		g_object_freeze_notify(G_OBJECT(widget));
		
		if (widget->priv->padding_set) {
			widget->priv->children_allocation.x1 -= widget->priv->padding.left;
			widget->priv->children_allocation.y1 -= widget->priv->padding.top;
			widget->priv->children_allocation.x2 += widget->priv->padding.right;
			widget->priv->children_allocation.y2 += widget->priv->padding.bottom;
		}
		widget->priv->padding = *spacing;
		widget->priv->children_allocation.x1 += widget->priv->padding.left;
		widget->priv->children_allocation.y1 += widget->priv->padding.top;
		widget->priv->children_allocation.x2 -= widget->priv->padding.right;
		widget->priv->children_allocation.y2 -= widget->priv->padding.bottom;
		g_object_notify(G_OBJECT(widget), "padding");
		g_object_notify(G_OBJECT(widget), "children-allocation");
		tangle_widget_set_padding_set(widget, TRUE);

		g_object_thaw_notify(G_OBJECT(widget));
		
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

gboolean tangle_widget_get_padding_set(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->padding_set;
}

void tangle_widget_set_padding_set(TangleWidget* widget, gboolean is_set) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->padding_set != is_set) {
		g_object_freeze_notify(G_OBJECT(widget));

		widget->priv->padding_set = is_set;
		g_object_notify(G_OBJECT(widget), "padding-set");

		/* TODO: Calculate aligned allocation */

		g_object_thaw_notify(G_OBJECT(widget));
	}
}

void tangle_widget_get_background_color(TangleWidget* widget, ClutterColor* color) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));
	g_return_if_fail(color != NULL);

	*color = widget->priv->background_color;
}

gboolean tangle_widget_get_background_color_set(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->background_color_set;
}

void tangle_widget_set_background_color(TangleWidget* widget, const ClutterColor* color) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (!clutter_color_equal(&widget->priv->background_color, color)) {
		if (color) {
			widget->priv->background_color = *color;
			widget->priv->background_color_set = TRUE;
		} else {
			widget->priv->background_color_set = FALSE;
		}
		g_object_notify(G_OBJECT(widget), "background-color");
		g_object_notify(G_OBJECT(widget), "background-color-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(widget));
	}
}

ClutterActor* tangle_widget_get_background_actor(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), NULL);

	return widget->priv->background_actor;
}

void tangle_widget_set_background_actor(TangleWidget* widget, ClutterActor* actor) {	
	ClutterActorBox actor_box;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->background_actor != actor) {
		if (widget->priv->background_actor && widget->priv->background_parented) {
			clutter_actor_unparent(widget->priv->background_actor);
		}
		widget->priv->background_actor = actor;
		
		if (actor) {
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(widget), &actor_box);
			clutter_actor_allocate(actor, &actor_box, CLUTTER_ABSOLUTE_ORIGIN_CHANGED);
			clutter_actor_push_internal(CLUTTER_ACTOR(widget));
			clutter_actor_set_parent(actor, CLUTTER_ACTOR(widget));	
			clutter_actor_pop_internal(CLUTTER_ACTOR(widget));
			widget->priv->background_parented = TRUE;
		}
		
		g_object_notify(G_OBJECT(widget), "background-actor");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

/**
 * tangle_widget_change_background_actor:
 * @widget: a #TangleWidget
 * @actor: a #TangleActor that is already parented and referenced by the widget
 *
 * This function changes the active background actor. Note that the given actor must be
 * already parented and referenced by the widget itself! This function is meant to be used
 * by subclasses of #TangleWidget (such as #TangleButton). Usually do not use this function.
 * See tangle_widget_set_background_actor() instead.
 */
void tangle_widget_change_background_actor(TangleWidget* widget, ClutterActor* actor) {
	ClutterActorBox actor_box;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->background_actor != actor) {
		if (widget->priv->background_actor && widget->priv->background_parented) {
			clutter_actor_unparent(widget->priv->background_actor);
		}
		widget->priv->background_actor = actor;
		
		if (actor) {
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(widget), &actor_box);
			clutter_actor_allocate(actor, &actor_box, CLUTTER_ABSOLUTE_ORIGIN_CHANGED);
			widget->priv->background_parented = FALSE;
		}
		
		g_object_notify(G_OBJECT(widget), "background-actor");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

ClutterActor* tangle_widget_get_foreground_actor(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), NULL);

	return widget->priv->foreground_actor;
}

void tangle_widget_set_foreground_actor(TangleWidget* widget, ClutterActor* actor) {
	ClutterActorBox actor_box;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->foreground_actor != actor) {
		if (widget->priv->foreground_actor && widget->priv->foreground_parented) {
			clutter_actor_unparent(widget->priv->foreground_actor);
			widget->priv->foreground_parented = FALSE;
		}
		widget->priv->foreground_actor = actor;
		
		if (actor) {
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(widget), &actor_box);
			clutter_actor_allocate(actor, &actor_box, CLUTTER_ABSOLUTE_ORIGIN_CHANGED);
			clutter_actor_push_internal(CLUTTER_ACTOR(widget));
			clutter_actor_set_parent(actor, CLUTTER_ACTOR(widget));	
			clutter_actor_pop_internal(CLUTTER_ACTOR(widget));
			widget->priv->foreground_parented = TRUE;
		}
		
		g_object_notify(G_OBJECT(widget), "foreground-actor");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

gboolean tangle_widget_get_prefer_background_size(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->prefer_background_size;
}

void tangle_widget_set_prefer_background_size(TangleWidget* widget, gboolean prefer_background_size) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (!widget->priv->prefer_background_size_set ||
	    widget->priv->prefer_background_size != prefer_background_size) {
		widget->priv->prefer_background_size = prefer_background_size;
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		tangle_widget_set_prefer_background_size_set(widget, TRUE);
		g_object_notify(G_OBJECT(widget), "prefer-background-size");
	}
}

gboolean tangle_widget_get_prefer_background_size_set(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->prefer_background_size_set;
}

void tangle_widget_set_prefer_background_size_set(TangleWidget* widget, gboolean is_set) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	if (widget->priv->prefer_background_size_set != is_set) {
		widget->priv->prefer_background_size_set = is_set;
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		g_object_notify(G_OBJECT(widget), "prefer-background-size-set");
	}
}

void tangle_widget_get_real_preferred_size(TangleWidget* widget, gboolean interacting, gfloat* min_width_p, gfloat* min_height_p, gfloat* natural_width_p, gfloat* natural_height_p, gfloat* max_width_p, gfloat* max_height_p) {
	ClutterRequestMode request_mode;
	gfloat natural_size;
	
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	g_object_get(widget, "request-mode", &request_mode, NULL);
	if (request_mode == CLUTTER_REQUEST_HEIGHT_FOR_WIDTH) {
		tangle_widget_get_real_preferred_width(widget, -1, interacting, min_width_p, &natural_size, max_width_p);
		tangle_widget_get_real_preferred_height(widget, natural_size, interacting, min_height_p, natural_height_p, max_height_p);
		if (natural_width_p) {
			*natural_width_p = natural_size;
		}
	} else {
		tangle_widget_get_real_preferred_height(widget, -1, interacting, min_height_p, &natural_size, max_height_p);
		tangle_widget_get_real_preferred_width(widget, natural_size, interacting, min_width_p, natural_width_p, max_width_p);
		if (natural_height_p) {
			*natural_height_p = natural_size;
		}
	}
}

void tangle_widget_get_real_preferred_width(TangleWidget* widget, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	tangle_widget_get_preferred_width(TANGLE_ACTOR(widget), for_height, interacting, min_width_p, natural_width_p, max_width_p);
}

void tangle_widget_get_real_preferred_height(TangleWidget* widget, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	tangle_widget_get_preferred_height(TANGLE_ACTOR(widget), for_width, interacting, min_height_p, natural_height_p, max_height_p);
}

gboolean tangle_widget_get_descendant_interacting(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->interacting_descendant != NULL;
}

gboolean tangle_widget_get_descendant_dragging(TangleWidget* widget) {
	g_return_val_if_fail(TANGLE_IS_WIDGET(widget), FALSE);

	return widget->priv->dragging_descendant != NULL;
}


#define ABS_F(x) ((x) < 0 ? -(x) : (x))

void tangle_widget_clamp_child_boundaries(TangleWidget* widget, TangleClamp* clamp) {
	gfloat original_value;
	TangleClampDirection direction;
	GList* child_in_list;
        ClutterActor* actor;
        ClutterActorBox actor_box;
	gfloat child_value;
	gfloat child_distance;
	gfloat clamp_distance = G_MAXFLOAT;
	
	g_return_if_fail(TANGLE_IS_WIDGET(widget));
	
	original_value = tangle_clamp_get_original_value(clamp);
	direction = tangle_clamp_get_direction(clamp);
	g_return_if_fail(direction == TANGLE_CLAMP_HORIZONTAL || direction == TANGLE_CLAMP_VERTICAL);
	
	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		actor = CLUTTER_ACTOR(child_in_list->data);
		if (TANGLE_IS_ACTOR(actor)) {
			tangle_actor_get_layout_allocation(TANGLE_ACTOR(actor), &actor_box);
		} else {
			clutter_actor_get_allocation_box(actor, &actor_box);
		}

		if (direction == TANGLE_CLAMP_HORIZONTAL) {
			child_value = actor_box.x1;
		} else {
			child_value = actor_box.y1;
		}
		if ((child_distance = ABS_F(child_value - original_value)) < clamp_distance) {
			clamp_distance = child_distance;
			tangle_clamp_set_clamped_value(clamp, child_value);
		}
	}
}

static void paint_actor_if_mapped(ClutterActor* actor) {
	if (CLUTTER_ACTOR_IS_MAPPED(actor)) {
		clutter_actor_paint(actor);
	}
}

static void tangle_widget_paint_children(TangleWidget* widget) {
	g_list_foreach(widget->priv->children, (GFunc)paint_actor_if_mapped, NULL);
}

static void tangle_widget_paint(ClutterActor* actor) {
	TangleWidget* widget;
	gdouble scale_x;
	gdouble scale_y;
	gfloat scale_center_x;
	gfloat scale_center_y;
	guint8 alpha;
	ClutterActorBox box;
	gboolean transition_scale_children;
	
	widget = TANGLE_WIDGET(actor);
	transition_scale_children = tangle_actor_should_transition_scale_children(TANGLE_ACTOR(widget));

	cogl_push_matrix();
	tangle_actor_apply_transition_scale(TANGLE_ACTOR(widget));

	if (widget->priv->background_color_set) {
		alpha = clutter_actor_get_paint_opacity(actor) * widget->priv->background_color.alpha / 255;
		cogl_set_source_color4ub(widget->priv->background_color.red,
			 		 widget->priv->background_color.green,
			 		 widget->priv->background_color.blue,
					 alpha);
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
		cogl_rectangle(box.x1, box.y1, box.x2, box.y2);
	}
	if (widget->priv->background_actor) {
		clutter_actor_paint(widget->priv->background_actor);
	}

	if (!transition_scale_children) {
		cogl_pop_matrix();
	}

	g_signal_emit(widget, signals[PAINT_CHILDREN], 0);

	if (widget->priv->foreground_actor) {
		if (!transition_scale_children) {
			cogl_push_matrix();
			tangle_actor_apply_transition_scale(TANGLE_ACTOR(widget));
		}
		
		clutter_actor_paint(widget->priv->foreground_actor);
		
		cogl_pop_matrix();
	} else if (transition_scale_children) {
		cogl_pop_matrix();
	}
}

static void paint_actor_if_mapped_and_not_hiding_or_showing(ClutterActor* actor) {
	if (CLUTTER_ACTOR_IS_MAPPED(actor) &&
	    (!TANGLE_IS_ACTOR(actor) || (!TANGLE_ACTOR_IS_HIDING(TANGLE_ACTOR(actor)) && !TANGLE_ACTOR_IS_SHOWING(TANGLE_ACTOR(actor))))) {
		clutter_actor_paint(actor);
	}
}

static void tangle_widget_pick(ClutterActor* actor, const ClutterColor *color) {
	TangleWidget* widget;
	GList* child_in_list;

	widget = TANGLE_WIDGET(actor);

	if (widget->priv->pickable) {
		CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->pick(actor, color);
	}

	g_list_foreach(widget->priv->children,(GFunc)paint_actor_if_mapped_and_not_hiding_or_showing, NULL);
}

static void tangle_widget_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	TangleWidget* widget;
	TangleSpacing spacing;
	gfloat min_width;
	gfloat natural_width;
	GList* child;
	gfloat child_x, child_min_width, child_natural_width;

	widget = TANGLE_WIDGET(actor);

	if (widget->priv->background_actor &&
	    ((widget->priv->prefer_background_size_set && widget->priv->prefer_background_size) ||
	     (!widget->priv->prefer_background_size_set && !widget->priv->children))) {
		if (TANGLE_IS_ACTOR(widget->priv->background_actor)) {
			tangle_actor_get_preferred_width(TANGLE_ACTOR(widget->priv->background_actor), for_height, interacting, min_width_p, natural_width_p, max_width_p);
		} else {
			clutter_actor_get_preferred_width(widget->priv->background_actor, for_height, min_width_p, natural_width_p);
			if (max_width_p) {
				*max_width_p = 0.0;
			}
		}
	} else {
		if (widget->priv->layout) {
			tangle_layout_get_preferred_width(widget->priv->layout, for_height, min_width_p, natural_width_p);
		} else {
			min_width = natural_width = 0;
			for (child = widget->priv->children; child; child = child->next) {
				child_x = clutter_actor_get_x(CLUTTER_ACTOR(child->data));
				if (TANGLE_IS_ACTOR(child->data)) {
					tangle_actor_get_preferred_size(TANGLE_ACTOR(child->data), interacting, &child_min_width, NULL, &child_natural_width, NULL, NULL, NULL);
				} else {
					clutter_actor_get_preferred_size(CLUTTER_ACTOR(child->data), &child_min_width, NULL, &child_natural_width, NULL);
				}
				if (child_x + child_min_width > min_width) {
					min_width = child_x + child_min_width;
				}
				if (child_x + child_natural_width > natural_width) {
					natural_width = child_x + child_natural_width;
				}
			}
			
			if (min_width_p) {
				*min_width_p = min_width;
			}
			if (natural_width_p) {
				*natural_width_p = natural_width;
			}
			if (max_width_p) {
				*max_width_p = 0.0;
			}
		}

		if (widget->priv->padding_set) {
			if (*min_width_p) {
				*min_width_p += widget->priv->padding.left + widget->priv->padding.right;
			}
			if (*natural_width_p) {
				*natural_width_p += widget->priv->padding.left + widget->priv->padding.right;
			}
		}
	}
}

static void tangle_widget_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	TangleWidget* widget;
	TangleSpacing spacing;
	gfloat min_height;
	gfloat natural_height;
	GList* child;
	gfloat child_x, child_min_height, child_natural_height;
	
	widget = TANGLE_WIDGET(actor);

	if (widget->priv->background_actor &&
	    ((widget->priv->prefer_background_size_set && widget->priv->prefer_background_size) ||
	     (!widget->priv->prefer_background_size_set && !widget->priv->children))) {
		if (TANGLE_IS_ACTOR(widget->priv->background_actor)) {
			tangle_actor_get_preferred_height(TANGLE_ACTOR(widget->priv->background_actor), for_width, interacting, min_height_p, natural_height_p, max_height_p);
		} else {
			clutter_actor_get_preferred_height(widget->priv->background_actor, for_width, min_height_p, natural_height_p);
			if (max_height_p) {
				*max_height_p = 0.0;
			}
		}
	} else {
		if (widget->priv->layout) {
			tangle_layout_get_preferred_height(widget->priv->layout, for_width, min_height_p, natural_height_p);
			if (max_height_p) {
				*max_height_p = 0;
			}
		} else {
			min_height = natural_height = 0;
			for (child = widget->priv->children; child; child = child->next) {
				child_x = clutter_actor_get_x(CLUTTER_ACTOR(child->data));
				if (TANGLE_IS_ACTOR(child->data)) {
					tangle_actor_get_preferred_size(TANGLE_ACTOR(child->data), interacting, NULL, &child_min_height, NULL, &child_natural_height, NULL, NULL);
				} else {
					clutter_actor_get_preferred_size(CLUTTER_ACTOR(child->data), NULL, &child_min_height, NULL, &child_natural_height);
				}
				if (child_x + child_min_height > min_height) {
					min_height = child_x + child_min_height;
				}
				if (child_x + child_natural_height > natural_height) {
					natural_height = child_x + child_natural_height;
				}
			}
			
			if (min_height_p) {
				*min_height_p = min_height;
			}
			if (natural_height_p) {
				*natural_height_p = natural_height;
			}
			if (max_height_p) {
				*max_height_p = 0.0;
			}
		}
			
		if (widget->priv->padding_set) {
			if (*min_height_p) {
				*min_height_p += widget->priv->padding.top + widget->priv->padding.bottom;
			}
			if (*natural_height_p) {
				*natural_height_p += widget->priv->padding.top + widget->priv->padding.bottom;
			}
		}
	}
}

static void tangle_widget_allocate_children(TangleWidget* widget, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	GList* child_in_children;
	ClutterActor* child;
	gfloat width, height;
	ClutterActorBox actor_box;

	if (widget->priv->layout) {
		tangle_layout_allocate(widget->priv->layout, box, flags);
	} else {
		for (child_in_children = widget->priv->children; child_in_children; child_in_children = child_in_children->next) {
			child = CLUTTER_ACTOR(child_in_children->data);
			if (CLUTTER_ACTOR_IS_VISIBLE(child)) {
				clutter_actor_get_preferred_size(child, NULL, NULL, &width, &height);
				actor_box.x1 = box->x1 + clutter_actor_get_x(child);
				actor_box.y1 = box->y1 + clutter_actor_get_y(child);
				actor_box.x2 = actor_box.x1 + width;
				actor_box.y2 = actor_box.y1 + height;
				clutter_actor_allocate(child, &actor_box, flags);
			}
		}
	}
}

static void tangle_widget_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleWidget* widget;
	ClutterActorBox actor_box;
	ClutterActorBox old_children_allocation;

	widget = TANGLE_WIDGET(actor);

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->allocate(actor, box, flags);

	if (widget->priv->background_actor) {
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &actor_box);
		clutter_actor_allocate(widget->priv->background_actor, &actor_box, flags);
	}
	if (widget->priv->foreground_actor) {
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &actor_box);
		clutter_actor_allocate(widget->priv->foreground_actor, &actor_box, flags);
	}

	old_children_allocation = widget->priv->children_allocation;
	tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &widget->priv->children_allocation);
	if (widget->priv->padding_set) {
		widget->priv->children_allocation.x1 += widget->priv->padding.left;
		widget->priv->children_allocation.y1 += widget->priv->padding.top;
		widget->priv->children_allocation.x2 -= widget->priv->padding.right;
		widget->priv->children_allocation.y2 -= widget->priv->padding.bottom;
	}
	if (!clutter_actor_box_equal(&old_children_allocation, &widget->priv->children_allocation)) {
		g_object_notify(G_OBJECT(actor), "children-allocation");
	}

	TANGLE_WIDGET_GET_CLASS(actor)->allocate_children(widget, &widget->priv->children_allocation, flags);
}

static void tangle_widget_map(ClutterActor* actor) {
	TangleWidget* widget;
	TangleStylesheet* stylesheet;

	widget = TANGLE_WIDGET(actor);

	if (!widget->priv->stylesheet &&
	    (stylesheet = tangle_stylesheet_get_default())) {
		tangle_stylesheet_apply(stylesheet, G_OBJECT(actor));
	}

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->map(actor);

	if (widget->priv->background_actor && widget->priv->background_parented) {
		clutter_actor_map(widget->priv->background_actor);
	}
	if (widget->priv->foreground_actor && widget->priv->background_parented) {
		clutter_actor_map(widget->priv->foreground_actor);
	}
}

static void tangle_widget_unmap(ClutterActor* actor) {
	TangleWidget* widget;
	TangleStylesheet* stylesheet;

	widget = TANGLE_WIDGET(actor);

	if (widget->priv->background_actor && widget->priv->background_parented) {
		clutter_actor_unmap(widget->priv->background_actor);
	}
	if (widget->priv->foreground_actor && widget->priv->background_parented) {
		clutter_actor_unmap(widget->priv->foreground_actor);
	}

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->unmap(actor);
}

static void tangle_widget_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleWidget* widget;
	
	widget = TANGLE_WIDGET(object);

	switch (prop_id) {
		case PROP_LAYOUT:
			tangle_widget_set_layout(widget, g_value_get_object(value));
			break;
		case PROP_PADDING:
			tangle_widget_set_padding(widget, TANGLE_SPACING(g_value_get_boxed(value)));
			break;
		case PROP_PADDING_SET:
			tangle_widget_set_padding_set(widget, g_value_get_boolean(value));
			break;
		case PROP_BACKGROUND_COLOR:
			tangle_widget_set_background_color(widget, clutter_value_get_color(value));
			break;
		case PROP_BACKGROUND_ACTOR:
			tangle_widget_set_background_actor(widget, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_FOREGROUND_ACTOR:
			tangle_widget_set_foreground_actor(widget, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_PREFER_BACKGROUND_SIZE:
			tangle_widget_set_prefer_background_size(widget, g_value_get_boolean(value));
			break;
		case PROP_PREFER_BACKGROUND_SIZE_SET:
			tangle_widget_set_prefer_background_size_set(widget, g_value_get_boolean(value));
			break;
		case PROP_PICKABLE:
			widget->priv->pickable = g_value_get_boolean(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_widget_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleWidget* widget;

	widget = TANGLE_WIDGET(object);

        switch (prop_id) {
		case PROP_LAYOUT:
			g_value_set_object(value, tangle_widget_get_layout(widget));
			break;
		case PROP_PADDING:
			g_value_set_boxed(value, &widget->priv->padding);
			break;
		case PROP_PADDING_SET:
			g_value_set_boolean(value, widget->priv->padding_set);
			break;
		case PROP_CHILDREN_ALLOCATION:
			g_value_set_boxed(value, &widget->priv->children_allocation);
			break;
		case PROP_BACKGROUND_COLOR:
			clutter_value_set_color(value, &widget->priv->background_color);
			break;
		case PROP_BACKGROUND_ACTOR:
			g_value_set_object(value, tangle_widget_get_background_actor(widget));
			break;
		case PROP_FOREGROUND_ACTOR:
			g_value_set_object(value, tangle_widget_get_foreground_actor(widget));
			break;
		case PROP_PREFER_BACKGROUND_SIZE:
			g_value_set_boolean(value, widget->priv->prefer_background_size);
			break;
		case PROP_PREFER_BACKGROUND_SIZE_SET:
			g_value_set_boolean(value, widget->priv->prefer_background_size_set);
			break;
		case PROP_PICKABLE:
			g_value_set_boolean(value, widget->priv->pickable);
			break;
		case PROP_DESCENDANT_INTERACTING:
			g_value_set_boolean(value, widget->priv->interacting_descendant != NULL);
			break;
		case PROP_DESCENDANT_DRAGGING:
			g_value_set_boolean(value, widget->priv->dragging_descendant != NULL);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void unparent_child(ClutterActor* actor, TangleWidget* widget) {
	if (TANGLE_IS_ACTOR(actor)) {
		g_signal_handlers_disconnect_by_func(actor, G_CALLBACK(on_notify_interacting), widget);
		g_signal_handlers_disconnect_by_func(actor, G_CALLBACK(on_notify_dragging), widget);
	}
	clutter_actor_destroy(actor);
}

static void tangle_widget_dispose(GObject* object) {
	TangleWidget* widget;
	
	widget = TANGLE_WIDGET(object);
	
	if (widget->priv->children) {
		g_list_foreach(widget->priv->children, (GFunc)unparent_child, widget);
		g_list_free(widget->priv->children);
		widget->priv->children = NULL;
	}
	
	if (widget->priv->layout) {
		tangle_layout_set_widget(widget->priv->layout, NULL);
		g_object_unref(widget->priv->layout);
		widget->priv->layout = NULL;
	}
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT(widget->priv->stylesheet);
	if (widget->priv->background_parented) {
		TANGLE_UNPARENT_AND_NULLIFY_ACTOR(widget->priv->background_actor);
	}
	if (widget->priv->foreground_parented) {
		TANGLE_UNPARENT_AND_NULLIFY_ACTOR(widget->priv->foreground_actor);
	}

	G_OBJECT_CLASS(tangle_widget_parent_class)->dispose(object);
}

static void tangle_widget_class_init(TangleWidgetClass* widget_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(widget_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(widget_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(widget_class);
	
	gobject_class->dispose = tangle_widget_dispose;
	gobject_class->set_property = tangle_widget_set_property;
	gobject_class->get_property = tangle_widget_get_property;

	clutter_actor_class->paint = tangle_widget_paint;
	clutter_actor_class->pick = tangle_widget_pick;
	clutter_actor_class->allocate = tangle_widget_allocate;
	clutter_actor_class->map = tangle_widget_map;
	clutter_actor_class->unmap = tangle_widget_unmap;

	actor_class->get_preferred_width = tangle_widget_get_preferred_width;
	actor_class->get_preferred_height = tangle_widget_get_preferred_height;

	widget_class->paint_children = tangle_widget_paint_children;
	widget_class->allocate_children = tangle_widget_allocate_children;

	/**
	 * TangleWidget:layout:
	 */
	g_object_class_install_property(gobject_class, PROP_LAYOUT,
	                                g_param_spec_object("layout",
	                                "Layout",
	                                "The layout that is used to position children",
	                                TANGLE_TYPE_LAYOUT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:padding:
	 *
	 * The padding around the child actors. This does not affect to background or foreground.
	 */
	g_object_class_install_property(gobject_class, PROP_PADDING,
	                                g_param_spec_boxed("padding",
	                                                    "Padding",
	                                                    "The padding around the child actors",
	                                                    TANGLE_TYPE_SPACING,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:padding-set:
	 *
	 * Whether the :padding is set.
	 */
	g_object_class_install_property(gobject_class, PROP_PADDING_SET,
	                                g_param_spec_boolean("padding-set",
	                                                    "Padding set",
	                                                    "Whether the padding is set",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:children-allocation:
	 */
	g_object_class_install_property(gobject_class, PROP_CHILDREN_ALLOCATION,
	                                g_param_spec_boxed("children-allocation",
	                                                   "Children allocation",
	                                                   "The allocated area for the children of the widget",
	                                                   CLUTTER_TYPE_ACTOR_BOX,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-color:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_COLOR,
	                                clutter_param_spec_color("background-color",
	                                                	 "Background color",
	                                                	 "The background color of the widget",
	                                                	 NULL,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-color_set:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_COLOR_SET,
	                                g_param_spec_boolean("background-color-set",
	                                "Background color set",
	                                "Whether the background color is set or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_ACTOR,
	                                g_param_spec_object("background-actor",
	                                "Background actor",
	                                "The actor that is used as a background of the widget",
	                                CLUTTER_TYPE_ACTOR,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:foreground-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_FOREGROUND_ACTOR,
	                                g_param_spec_object("foreground-actor",
	                                "Foreground actor",
	                                "The actor that is used as a foreground of the widget",
	                                CLUTTER_TYPE_ACTOR,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:prefer-background-size:
	 *
	 * Overrides the usual size request that is based on the layout and the children of the widget.
	 * When TRUE (and #TangleWidget:prefer-background-size-set is TRUE), the size of the background actor
	 * is returned as a size request of the widget.
	 * 
	 * Writing this property sets the #TangleWidget:prefer-background-size-set property as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_PREFER_BACKGROUND_SIZE,
	                                g_param_spec_boolean("prefer-background-size",
	                                "Prefer background size",
	                                "Whether return the background size or the preferred size of children as a size request",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:prefer-background-size-set:
	 *
	 * Forces the widget to obey the value of the #TangleWidget:prefer-background-size property
	 * when returning a size request. When TRUE, the size request policy depends solely on the
	 * #TangleWidget:prefer-background-size property. When FALSE, the size request is
	 * normally based on the layout and the size requests of the children (like #TangleWidget:prefer-background-size
	 * was FALSE), but if the widget does not have any child, the size request of the background actor is used
	 * instead (like #TangleWidget:prefer-background-size was TRUE).
	 */
	g_object_class_install_property(gobject_class, PROP_PREFER_BACKGROUND_SIZE_SET,
	                                g_param_spec_boolean("prefer-background-size-set",
	                                "Prefer background size",
	                                "Whether to use the prefer-background-size property",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:pickable:
	 *
	 * Whether the widget is pickable or not. The picking is always forwarded to children,
	 * but this flags controls whether the whole widget area is used when picking.
	 *
	 * Especially, when a widget is on top of other actors (layered with #TangleStackLayout, for example),
	 * it should not be pickable.
	 */
	g_object_class_install_property(gobject_class, PROP_PICKABLE,
	                                g_param_spec_boolean("pickable",
	                                "Pickable",
	                                "Whether the widget is pickable or not",
	                                TRUE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:descendant-interacting:
	 *
	 * Whether an descendant #TangleActor is interacting.
	 */
	g_object_class_install_property(gobject_class, PROP_DESCENDANT_INTERACTING,
	                                g_param_spec_boolean("descendant-interacting",
	                                "Descendant interacting",
	                                "Whether an descendant actor is interacting",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:descendant-dragging:
	 *
	 * Whether an descendant #TangleActor is under dragging.
	 */
	g_object_class_install_property(gobject_class, PROP_DESCENDANT_DRAGGING,
	                                g_param_spec_boolean("descendant-dragging",
	                                "Descendant dragging",
	                                "Whether an descendant actor is under dragging",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	signals[PAINT_CHILDREN] = g_signal_new("paint-children",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleWidgetClass, paint_children),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__VOID,
					       G_TYPE_NONE, 0);

	g_type_class_add_private(gobject_class, sizeof(TangleWidgetPrivate));
}

static void tangle_widget_init(TangleWidget* widget) {
	widget->priv = G_TYPE_INSTANCE_GET_PRIVATE(widget, TANGLE_TYPE_WIDGET, TangleWidgetPrivate);
	
	widget->priv->pickable = TRUE;
}

static void tangle_widget_add_impl(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, gboolean before) {
	GList* existing_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(clutter_actor_get_parent(actor) == NULL);
	g_return_if_fail(existing == NULL || CLUTTER_IS_ACTOR(existing));
	g_return_if_fail(existing == NULL || clutter_actor_get_parent(existing) == CLUTTER_ACTOR(widget));

	g_object_ref(actor);
	
	if (existing) {
		existing_in_list = g_list_find(widget->priv->children, existing);
		g_return_if_fail(existing_in_list != NULL);
	
		if (before) {
			widget->priv->children = g_list_insert_before(widget->priv->children, existing_in_list, actor);
		} else {
			widget->priv->children = g_list_insert_before(widget->priv->children, g_list_next(existing_in_list), actor);
		}
		force_actor_depth(actor, existing, before);
	} else {
		if (before) {
			existing_in_list = g_list_first(widget->priv->children);
			widget->priv->children = g_list_prepend(widget->priv->children, actor);
		} else {
			/* Unfortunately this traverses the list twice. */
			existing_in_list = g_list_last(widget->priv->children);
			widget->priv->children = g_list_append(widget->priv->children, actor);
		}		
		if (existing_in_list) {
			force_actor_depth(actor, CLUTTER_ACTOR(existing_in_list->data), before);
		}
	}

	clutter_actor_set_parent(actor, CLUTTER_ACTOR(widget));
	if (widget->priv->stylesheet) {
		tangle_stylesheet_apply(widget->priv->stylesheet, G_OBJECT(actor));
	}
	
	if (TANGLE_IS_ACTOR(actor)) {
		g_signal_connect(actor, "notify::interacting", G_CALLBACK(on_notify_interacting), widget);
		g_signal_connect(actor, "notify::descendant-interacting", G_CALLBACK(on_notify_interacting), widget);
		g_signal_connect(actor, "notify::dragging", G_CALLBACK(on_notify_dragging), widget);
		g_signal_connect(actor, "notify::descendant-dragging", G_CALLBACK(on_notify_dragging), widget);
	}

	clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));

	g_signal_emit_by_name(widget, "actor-added", actor);

	g_object_unref(actor);
}

static void tangle_widget_real_add(ClutterContainer* widget, ClutterActor* actor) {	
	tangle_widget_add(TANGLE_WIDGET(widget), actor, NULL);
}

static void tangle_widget_remove(ClutterContainer* container, ClutterActor* actor) {
	TangleWidget* widget;
	gboolean was_visible;

	g_return_if_fail(TANGLE_IS_WIDGET(container));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(clutter_actor_get_parent(actor) == CLUTTER_ACTOR(container));

	widget = TANGLE_WIDGET(container);

	g_object_ref(actor);
	
	if (CLUTTER_ACTOR(widget->priv->interacting_descendant) == actor) {
		widget->priv->interacting_descendant = NULL;
		g_object_notify(G_OBJECT(widget), "descendant-interacting");
	}
	if (CLUTTER_ACTOR(widget->priv->dragging_descendant) == actor) {
		widget->priv->dragging_descendant = NULL;
		g_object_notify(G_OBJECT(widget), "descendant-dragging");
	}

	was_visible = CLUTTER_ACTOR_IS_VISIBLE(actor);

	widget->priv->children = g_list_remove(widget->priv->children, actor);
	if (widget->priv->stylesheet) {
		tangle_stylesheet_unapply(widget->priv->stylesheet, G_OBJECT(actor));
	}
	clutter_actor_unparent(actor);
	
	g_signal_handlers_disconnect_by_func(actor, G_CALLBACK(on_notify_interacting), widget);
	g_signal_handlers_disconnect_by_func(actor, G_CALLBACK(on_notify_dragging), widget);
	
	if (was_visible) {
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
	
	g_signal_emit_by_name(widget, "actor-removed", actor);

	g_object_unref(actor);
}

static void tangle_widget_foreach(ClutterContainer* widget, ClutterCallback callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		callback(CLUTTER_ACTOR(child_in_list->data), user_data);
	}
}

static void tangle_widget_raise(ClutterContainer* container, ClutterActor* actor, ClutterActor* sibling) {
	change_actor_order(TANGLE_WIDGET(container), actor, sibling, FALSE);
}

static void tangle_widget_lower(ClutterContainer* container, ClutterActor* actor, ClutterActor* sibling) {
	change_actor_order(TANGLE_WIDGET(container), actor, sibling, TRUE);
}

static void tangle_widget_sort_depth_order(ClutterContainer* container) {
	TangleWidget* widget;
	
	widget = TANGLE_WIDGET(container);
	
	widget->priv->children = g_list_sort(widget->priv->children, compare_actor_depths);
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(widget))) {
		clutter_actor_queue_redraw(CLUTTER_ACTOR(widget));
	}
}

static void clutter_container_iface_init(ClutterContainerIface* iface) {
	iface->add = tangle_widget_real_add;
	iface->remove = tangle_widget_remove;
	iface->foreach = tangle_widget_foreach;
	iface->raise = tangle_widget_raise;
	iface->lower = tangle_widget_lower;
	iface->sort_depth_order = tangle_widget_sort_depth_order;
}

static gboolean tangle_widget_apply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	TangleWidget* widget;
	GList* link;
	
	widget = TANGLE_WIDGET(stylable);
	if (!widget->priv->stylesheet) {
		tangle_widget_foreach_swapped(widget, TANGLE_SWAPPED_CALLBACK(tangle_stylesheet_apply), stylesheet);
		widget->priv->stylesheet = stylesheet;
		g_object_ref(stylesheet);
	}

	return FALSE;
}

static gboolean tangle_widget_unapply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	TangleWidget* widget;
	GList* link;
	
	widget = TANGLE_WIDGET(stylable);
	if (widget->priv->stylesheet == stylesheet) {
		tangle_widget_foreach_swapped(widget, TANGLE_SWAPPED_CALLBACK(tangle_stylesheet_unapply), stylesheet);
		g_object_unref(widget->priv->stylesheet);
		widget->priv->stylesheet = NULL;
	}

	return FALSE;
}

static void tangle_stylable_iface_init(TangleStylableIface* iface) {
	iface->apply_stylesheet = tangle_widget_apply_stylesheet;
	iface->unapply_stylesheet = tangle_widget_unapply_stylesheet;
}

static void change_actor_order(TangleWidget* widget, ClutterActor* actor, ClutterActor* sibling, gboolean before) {
	GList* actor_in_children;
	GList* sibling_in_children = NULL;
	
	actor_in_children = g_list_find(widget->priv->children, actor);
	g_return_if_fail(actor_in_children != NULL);

	if (sibling) {
		sibling_in_children = g_list_find(widget->priv->children, sibling);
		g_return_if_fail(sibling_in_children != NULL);
	}

	widget->priv->children = g_list_delete_link(widget->priv->children, actor_in_children);
	
	if (sibling) {
		if (before) {
			widget->priv->children = g_list_insert_before(widget->priv->children, sibling_in_children, actor);
		} else {
			widget->priv->children = g_list_insert_before(widget->priv->children, g_list_next(sibling_in_children), actor);
		}
		force_actor_depth(actor, sibling, before);
	} else {
		if (before) {
			sibling_in_children = g_list_first(widget->priv->children);
			widget->priv->children = g_list_prepend(widget->priv->children, actor);
		} else {
			/* Unfortunately this traverses the list twice. */
			sibling_in_children = g_list_last(widget->priv->children);
			widget->priv->children = g_list_append(widget->priv->children, actor);
		}
		if (sibling_in_children) {
			force_actor_depth(actor, CLUTTER_ACTOR(sibling_in_children->data), before);
		}
	}
}

static gint compare_actor_depths(gconstpointer a, gconstpointer b) {
	gint retvalue;
	gfloat depth_a;
	gfloat depth_b;
	gint depth_position_a = 0;
	gint depth_position_b = 0;
	
	depth_a = clutter_actor_get_depth(CLUTTER_ACTOR(a));
	depth_b = clutter_actor_get_depth(CLUTTER_ACTOR(b));
	if (depth_a == 0.0 && depth_b == 0.0) {
		if (TANGLE_IS_ACTOR(a)) {
			depth_position_a = tangle_actor_get_depth_position(TANGLE_ACTOR(a));
		}
		if (TANGLE_IS_ACTOR(b)) {
			depth_position_b = tangle_actor_get_depth_position(TANGLE_ACTOR(b));
		}
		retvalue = (depth_position_a < depth_position_b ? -1 : (depth_position_b < depth_position_a ? 1 : 0));
	} else {
		retvalue = (depth_a < depth_b ? -1 : (depth_b < depth_a ? 1 : 0));
	}
	
	return retvalue;
}

static gint compare_actor_if_higher_depth_position(gconstpointer a, gconstpointer b) {
	gint retvalue;
	gfloat depth_a;
	gfloat depth_b;
	gint depth_position_a = 0;
	gint depth_position_b = 0;
	
	depth_a = clutter_actor_get_depth(CLUTTER_ACTOR(a));
	depth_b = clutter_actor_get_depth(CLUTTER_ACTOR(b));
	if (depth_a == 0.0 && depth_b == 0.0) {
		if (TANGLE_IS_ACTOR(a)) {
			depth_position_a = tangle_actor_get_depth_position(TANGLE_ACTOR(a));
		}
		if (TANGLE_IS_ACTOR(b)) {
			depth_position_b = tangle_actor_get_depth_position(TANGLE_ACTOR(b));
		}
		retvalue = (depth_position_a > depth_position_b ? 0 : -1);
	} else {
		retvalue = (depth_a > depth_b ? 0 : -1);
	}

	return retvalue;
}

static void force_actor_depth(ClutterActor* actor, ClutterActor* sibling, gboolean before) {
	gfloat depth_actor;
	gfloat depth_sibling;
	gint depth_position_actor = 0;
	gint depth_position_sibling = 0;

	depth_actor = clutter_actor_get_depth(actor);
	depth_sibling = clutter_actor_get_depth(sibling);
	if (depth_actor == 0.0 && depth_sibling == 0.0) {
		if (TANGLE_IS_ACTOR(actor)) {
			depth_position_actor = tangle_actor_get_depth_position(TANGLE_ACTOR(actor));

			if (TANGLE_IS_ACTOR(sibling)) {
				depth_position_sibling = tangle_actor_get_depth_position(TANGLE_ACTOR(sibling));
			}

			if ((depth_position_actor > depth_position_sibling && before) ||
			    (depth_position_actor < depth_position_sibling && !before)) {
				/* This causes unnecessary clutter_container_sort_depth_order(), but no can do. */
				tangle_actor_set_depth_position(TANGLE_ACTOR(actor), depth_position_sibling);
			}
		}
	} else if ((depth_actor > depth_sibling && before) ||
	           (depth_actor < depth_sibling && !before)) {
		/* This causes unnecessary clutter_container_sort_depth_order(), but no can do. */
		clutter_actor_set_depth(actor, depth_sibling);
	}
}

static void stop_interacting(TangleActor* actor) {
	TangleWidget* widget;

	if (actor) {
		if (TANGLE_IS_WIDGET(actor)) {
			widget = TANGLE_WIDGET(actor);
			stop_interacting(widget->priv->interacting_descendant);
			widget->priv->interacting_descendant = NULL;
			g_object_notify(G_OBJECT(widget), "descendant-interacting");
		}
		tangle_actor_set_interacting(actor, FALSE);
	}
}

static void on_notify_interacting(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleWidget* widget;
	TangleActor* actor;
	
	widget = TANGLE_WIDGET(user_data);
	actor = TANGLE_ACTOR(object);
	
	if (tangle_actor_get_interacting(actor) || (TANGLE_IS_WIDGET(actor) && tangle_widget_get_descendant_interacting(TANGLE_WIDGET(actor)))) {
		if (widget->priv->interacting_descendant != actor) {
			stop_interacting(widget->priv->interacting_descendant);
			widget->priv->interacting_descendant = actor;
			g_object_notify(G_OBJECT(widget), "descendant-interacting");
		}
	} else {
		if (widget->priv->interacting_descendant == actor) {
			widget->priv->interacting_descendant = NULL;
			g_object_notify(G_OBJECT(widget), "descendant-interacting");
		}
	}
}

static void stop_dragging(TangleActor* actor) {
	TangleWidget* widget;

	if (actor) {
		if (TANGLE_IS_WIDGET(actor)) {
			widget = TANGLE_WIDGET(actor);
			stop_dragging(widget->priv->dragging_descendant);
			widget->priv->dragging_descendant = NULL;
			g_object_notify(G_OBJECT(widget), "descendant-dragging");
		}
		tangle_actor_set_dragging(actor, FALSE);
	}
}

static void on_notify_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleWidget* widget;
	TangleActor* actor;
	
	widget = TANGLE_WIDGET(user_data);

	actor = TANGLE_ACTOR(object);
	
	if (tangle_actor_get_dragging(actor) || (TANGLE_IS_WIDGET(actor) && tangle_widget_get_descendant_dragging(TANGLE_WIDGET(actor)))) {
		if (widget->priv->dragging_descendant != actor) {
			stop_dragging(widget->priv->dragging_descendant);
			widget->priv->dragging_descendant = actor;
			g_object_notify(G_OBJECT(widget), "descendant-dragging");
		}
	} else {
		if (widget->priv->dragging_descendant == actor) {
			widget->priv->dragging_descendant = NULL;
			g_object_notify(G_OBJECT(widget), "descendant-dragging");
		}
	}
}
