/*
 * tangle-actor.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-template.h"
#include "tangle-actor.h"
#include "tangle-properties.h"
#include "tangle-transition.h"
#include "tangle-vault.h"
#include "tangle-misc.h"
#include "tangle-private.h"
#include "marshalers.h"
#include <gobject/gvaluecollector.h>
#include <string.h>

/* We know that this is available in the Clutter library (from clutter-script-private.h). */
gboolean clutter_script_parse_node (ClutterScript *script, GValue *value, const gchar *name, JsonNode *node, GParamSpec *pspec);

/**
 * SECTION:tangle-actor
 * @Short_description: A base class for actors that display content
 * @Title: TangleActor
 */

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

G_DEFINE_ABSTRACT_TYPE_WITH_CODE(TangleActor, tangle_actor, CLUTTER_TYPE_ACTOR,
                                 G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

struct _TangleActorPrivate {
	gfloat transition_move_x;
	gfloat transition_move_y;
	gdouble transition_scale_x;
	gdouble transition_scale_y;
	gfloat transition_scale_center_x;
	gfloat transition_scale_center_y;
	ClutterGravity transition_scale_gravity;
	TangleTransition* transition;

	guint depth_position;

	gdouble picking_scale_x;
	gdouble picking_scale_y;
	gfloat picking_scale_center_x;
	gfloat picking_scale_center_y;
	ClutterGravity picking_scale_gravity;
	
	gfloat squint_x;
	gfloat squint_y;

	gulong paint_event_handler_id;
	gulong pick_event_handler_id;

	TangleSpacing margin;
	gfloat max_width;
	gfloat max_height;
	gdouble aspect_ratio;
	gdouble alignment_x;
	gdouble alignment_y;

	ClutterActorBox aligned_allocation;
	gdouble interactive_size_multiplier;
	
	ClutterActorBox layout_allocation;
	
	ClutterTimeline* hiding_timeline;
	
	GHashTable* template_parameters;
	gboolean template_found;

	guint transition_scale_children : 1;
	guint margin_set : 1;
	guint max_width_set : 1;
	guint max_height_set : 1;
	guint aspect_ratio_set : 1;
	guint interacting : 1;
	guint dragging : 1;
	guint ancestor_interacting : 1;
	guint ancestor_dragging : 1;
	guint interactive_size_multiplier_set : 1;
	guint hide_all : 1;
	guint destroy : 1;
};

enum {
	PROP_0,
	PROP_TEMPLATE,
	PROP_TRANSITION_MOVE_X,
	PROP_TRANSITION_MOVE_Y,
	PROP_TRANSITION_SCALE_X,
	PROP_TRANSITION_SCALE_Y,
	PROP_TRANSITION_SCALE_CENTER_X,
	PROP_TRANSITION_SCALE_CENTER_Y,
	PROP_TRANSITION_SCALE_GRAVITY,
	PROP_TRANSITION,
	PROP_DEPTH_POSITION,
	PROP_PICKING_SCALE_X,
	PROP_PICKING_SCALE_Y,
	PROP_PICKING_SCALE_CENTER_X,
	PROP_PICKING_SCALE_CENTER_Y,
	PROP_PICKING_SCALE_GRAVITY,
	PROP_SQUINT_X,
	PROP_SQUINT_Y,
	PROP_MARGIN,
	PROP_MARGIN_SET,
	PROP_MAX_WIDTH,
	PROP_MAX_WIDTH_SET,
	PROP_MAX_HEIGHT,
	PROP_MAX_HEIGHT_SET,
	PROP_ASPECT_RATIO,
	PROP_ASPECT_RATIO_SET,
	PROP_ALIGNMENT_X,
	PROP_ALIGNMENT_Y,
	PROP_ALIGNMENT_MODE,
	PROP_ALIGNED_ALLOCATION,
	PROP_INTERACTING,
	PROP_DRAGGING,
	PROP_ANCESTOR_INTERACTING,
	PROP_ANCESTOR_DRAGGING,
	PROP_INTERACTIVE_SIZE_MULTIPLIER,
	PROP_INTERACTIVE_SIZE_MULTIPLIER_SET,
	PROP_STYLE,
	PROP_ACTION
};

enum {
	ANIMATE_TRANSITION,
	SHOW_COMPLETED,
	TRANSITION_COMPLETED,
	LAST_SIGNAL
};

static ClutterTimeline* animate_transition(TangleActor* actor, ClutterActorBox* from_allocation, ClutterActorBox* to_allocation);
static void set_max_width(TangleActor* actor, gfloat max_width);
static void set_max_height(TangleActor* actor, gfloat max_height);
static void set_scale_centers_by_gravity(TangleActor* actor);
static void set_picking_scale_gravity(TangleActor* actor, ClutterGravity gravity);
static void set_transition_scale_gravity(TangleActor* actor, ClutterGravity gravity);
static void unset_transition_scale_children(TangleActor* actor);
static void on_hiding_timeline_completed(TangleActor* actor);
static void complete_timeline(ClutterTimeline* timeline);
static gboolean parse_spacing_from_json_array(TangleSpacing* spacing, JsonArray* array);
static gboolean parse_spacing_from_json_object(TangleSpacing* spacing, JsonObject* object);
static void update_ancestor_interacting_and_dragging(TangleActor* actor, TangleActor* parent);
static void on_notify_interacting_or_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };
static ClutterScriptableIface* parent_scriptable_iface = NULL;

ClutterAnimation* tangle_actor_animate(TangleActor* actor, gulong mode, guint duration, ...) {
	ClutterAnimation* animation;
	va_list args;

	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), NULL);
	
	va_start(args, duration);
	animation = tangle_object_animate_valist(G_OBJECT(actor), mode, duration, args);
	va_end(args);
	
	return animation;
}

ClutterAnimation* tangle_actor_animate_with_timeline(TangleActor* actor, gulong mode, ClutterTimeline* timeline, ...) {
	ClutterAnimation* animation;
	va_list args;
	
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), NULL);

	va_start(args, timeline);
	animation = tangle_object_animate_valist_with_timeline(G_OBJECT(actor), mode, timeline, args);
	va_end(args);

	return animation;
}

ClutterAnimation* tangle_actor_animatev(TangleActor* actor, gulong mode, guint duration, gint n_properties, const gchar* const property_names[], const GValue* property_values) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), NULL);

	return tangle_object_animatev(G_OBJECT(actor), mode, duration, n_properties, property_names, property_values);
}

gboolean tangle_actor_stop_animation(TangleActor* actor, ClutterAnimation* animation) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);
	
	return tangle_object_stop_animation(G_OBJECT(actor), animation);
}

TangleTransition* tangle_actor_get_transition(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), NULL);

	return actor->priv->transition;
}

void tangle_actor_set_transition(TangleActor* actor, TangleTransition* transition) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->transition != transition) {
		if (transition) {
			g_object_ref(transition);
		}
		if (actor->priv->transition) {
			g_object_unref(actor->priv->transition);
		}
		actor->priv->transition = transition;
		g_object_notify(G_OBJECT(actor), "transition");	
	}
}

gboolean tangle_actor_should_transition_scale_children(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->transition_scale_children;
}

void tangle_actor_apply_transition_scale(TangleActor* actor) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->transition_scale_x != 1.0 || actor->priv->transition_scale_y != 1.0) {
		cogl_translate(actor->priv->transition_scale_center_x, actor->priv->transition_scale_center_y, 0.0);
		cogl_scale(actor->priv->transition_scale_x, actor->priv->transition_scale_y, 1.0);
		cogl_translate(-actor->priv->transition_scale_center_x, -actor->priv->transition_scale_center_y, 0.0);
	}
}

void tangle_actor_get_transition_move(TangleActor* actor, gfloat* x_return, gfloat* y_return) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (x_return) {
		*x_return = actor->priv->transition_move_x;
	}
	if (y_return) {
		*y_return = actor->priv->transition_move_y;
	}
}

void tangle_actor_set_transition_move(TangleActor* actor, gfloat x, gfloat y) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	g_object_freeze_notify(G_OBJECT(actor));
	
	if (actor->priv->transition_move_x != x) {
		actor->priv->transition_move_x = x;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		g_object_notify(G_OBJECT(actor), "transition-move-x");
	}
	if (actor->priv->transition_move_y != y) {
		actor->priv->transition_move_y = y;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		g_object_notify(G_OBJECT(actor), "transition-move-y");
	}
	
	g_object_thaw_notify(G_OBJECT(actor));
}

void tangle_actor_get_transition_scale(TangleActor* actor, gdouble* x_return, gdouble* y_return) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (x_return) {
		*x_return = actor->priv->transition_scale_x;
	}
	if (y_return) {
		*y_return = actor->priv->transition_scale_y;
	}
}

void tangle_actor_set_transition_scale(TangleActor* actor, gdouble x, gdouble y) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	g_object_freeze_notify(G_OBJECT(actor));
	
	if (actor->priv->transition_scale_x != x) {
		actor->priv->transition_scale_x = x;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		g_object_notify(G_OBJECT(actor), "transition-scale-x");
	}
	if (actor->priv->transition_scale_y != y) {
		actor->priv->transition_scale_y = y;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		g_object_notify(G_OBJECT(actor), "transition-scale-y");
	}
	
	g_object_thaw_notify(G_OBJECT(actor));
}

void tangle_actor_show(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterTimeline* timeline = NULL;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (TANGLE_ACTOR_IS_HIDING(actor) && !actor->priv->destroy) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);
		TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_SHOWING);

		complete_timeline(actor->priv->hiding_timeline);
		g_object_unref(actor->priv->hiding_timeline);
		actor->priv->hiding_timeline = NULL;

		g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
		actor->priv->transition_scale_children = TRUE;
		if ((timeline = animate_transition(actor, NULL, current_allocation))) {
			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			actor->priv->transition_scale_children = FALSE;
		}
		clutter_actor_box_free(current_allocation);
		
		/* TODO: if hiding_animation should have done hide_all... */
	}

	if (!actor->priv->destroy && !CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_show(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_hide_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterTimeline* timeline = NULL;
	
	g_return_if_fail(TANGLE_IS_ACTOR(actor));
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_ALLOCATED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			timeline = animate_transition(actor, current_allocation, NULL);
			clutter_actor_box_free(current_allocation);
		}
		if (timeline) {
			actor->priv->hiding_timeline = timeline;
			g_object_ref(timeline);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(on_hiding_timeline_completed), actor);
			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_hide(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	}
}

void tangle_actor_hide_all_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterTimeline* timeline = NULL;
	
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	actor->priv->hide_all = TRUE;
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_ALLOCATED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			timeline = animate_transition(actor, current_allocation, NULL);
			clutter_actor_box_free(current_allocation);
		}
		if (timeline) {
			actor->priv->hiding_timeline = timeline;
			g_object_ref(timeline);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(on_hiding_timeline_completed), actor);
			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_hide_all(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	} else if (!CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_hide_all(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_destroy_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterTimeline* timeline = NULL;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));
	
	actor->priv->destroy = TRUE;
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_ALLOCATED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			timeline = animate_transition(actor, current_allocation, NULL);
			clutter_actor_box_free(current_allocation);
		}
		if (timeline) {
			actor->priv->hiding_timeline = timeline;
			g_object_ref(timeline);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(on_hiding_timeline_completed), actor);
			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_destroy(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	} else if (!CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_destroy(CLUTTER_ACTOR(actor));
	}
}

/**
 * tangle_actor_get_depth_position:
 * @actor: a #TangleActor
 * 
 * See :depth-position.
 *
 * Return value: the depth position of the actor
 */
gint tangle_actor_get_depth_position(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), 0);

	return actor->priv->depth_position;
}

/**
 * tangle_actor_set_depth_position:
 * @actor: a #TangleActor
 * @depth_position: a relative stacking position in depth order
 *
 * See :depth-position.
 *
 * This function sets ClutterActor:depth to zero as a side effect.
 */
void tangle_actor_set_depth_position(TangleActor* actor, gint depth_position) {
	ClutterActor* parent;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));
	
	if (actor->priv->depth_position != depth_position) {
		actor->priv->depth_position = depth_position;
		if (clutter_actor_get_depth(CLUTTER_ACTOR(actor)) != 0.0) {
			clutter_actor_set_depth(CLUTTER_ACTOR(actor), 0.0);
		} else if ((parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor))) && CLUTTER_IS_CONTAINER(parent)) {
			clutter_container_sort_depth_order(CLUTTER_CONTAINER(parent));
		}
		g_object_notify(G_OBJECT(actor), "depth-position");
	}
}

void tangle_actor_get_margin(TangleActor* actor, TangleSpacing* spacing) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	*spacing = actor->priv->margin;
}
	
void tangle_actor_set_margin(TangleActor* actor, const TangleSpacing* spacing) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (!actor->priv->margin_set || !tangle_spacing_equal(&actor->priv->margin, spacing)) {
		g_object_freeze_notify(G_OBJECT(actor));
		
		if (actor->priv->margin_set) {
			actor->priv->aligned_allocation.x1 -= actor->priv->margin.left;
			actor->priv->aligned_allocation.y1 -= actor->priv->margin.top;
			actor->priv->aligned_allocation.x2 += actor->priv->margin.right;
			actor->priv->aligned_allocation.y2 += actor->priv->margin.bottom;
		}
		actor->priv->margin = *spacing;
		actor->priv->aligned_allocation.x1 += actor->priv->margin.left;
		actor->priv->aligned_allocation.y1 += actor->priv->margin.top;
		actor->priv->aligned_allocation.x2 -= actor->priv->margin.right;
		actor->priv->aligned_allocation.y2 -= actor->priv->margin.bottom;
		g_object_notify(G_OBJECT(actor), "margin");
/* TODO		g_object_notify(G_OBJECT(actor), "aligned-allocation"); */
		tangle_actor_set_margin_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

gboolean tangle_actor_get_margin_set(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->margin_set;
}

void tangle_actor_set_margin_set(TangleActor* actor, gboolean is_set) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->margin_set != is_set) {
		g_object_freeze_notify(G_OBJECT(actor));

		actor->priv->margin_set = is_set;
		g_object_notify(G_OBJECT(actor), "margin-set");

		/* TODO: Calculate aligned allocation */

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

void tangle_actor_get_max_size(TangleActor* actor, gfloat* max_width_p, gfloat* max_height_p) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->max_width_set) {
		*max_width_p = actor->priv->max_width_set;
	} else {
		*max_width_p = 0.0;
	}
	if (actor->priv->max_height_set) {
		*max_height_p = actor->priv->max_height_set;
	} else {
		*max_height_p = 0.0;
	}
}

void tangle_actor_set_max_size(TangleActor* actor, gfloat max_width, gfloat max_height) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	g_object_freeze_notify(G_OBJECT(actor));
	
	set_max_width(actor, max_width);
	set_max_height(actor, max_height);
	
	g_object_thaw_notify(G_OBJECT(actor));
}

gdouble tangle_actor_get_aspect_ratio(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), 0.0);

	return actor->priv->aspect_ratio;
}

void tangle_actor_set_aspect_ratio(TangleActor* actor, gdouble aspect_ratio) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->aspect_ratio != aspect_ratio) {
		actor->priv->aspect_ratio = aspect_ratio;
		tangle_actor_set_aspect_ratio_set(actor, TRUE);
		g_object_notify(G_OBJECT(actor), "aspect-ratio");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
	}
}

gboolean tangle_actor_get_aspect_ratio_set(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->aspect_ratio_set;
}

void tangle_actor_set_aspect_ratio_set(TangleActor* actor, gboolean is_set) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->aspect_ratio_set != is_set) {
		actor->priv->aspect_ratio_set = is_set;
		g_object_notify(G_OBJECT(actor), "aspect-ratio-set");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
	}
}

gdouble tangle_actor_get_alignment_x(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), 0.0);

	return actor->priv->alignment_x;
}

void tangle_actor_set_alignment_x(TangleActor* actor, gdouble alignment_x) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->alignment_x != alignment_x) {
		actor->priv->alignment_x = alignment_x;
		g_object_notify(G_OBJECT(actor), "alignment-x");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

gdouble tangle_actor_get_alignment_y(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), 0.0);

	return actor->priv->alignment_y;
}

void tangle_actor_set_alignment_y(TangleActor* actor, gdouble alignment_y) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->alignment_y != alignment_y) {
		actor->priv->alignment_y = alignment_y;
		g_object_notify(G_OBJECT(actor), "alignment-y");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_get_aligned_allocation(TangleActor* actor, ClutterActorBox* aligned_allocation) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	*aligned_allocation = actor->priv->aligned_allocation;
}


gboolean tangle_actor_get_interacting(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->interacting;
}

void tangle_actor_set_interacting(TangleActor* actor, gboolean interacting) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->interacting != interacting) {
		actor->priv->interacting = interacting;
		g_object_notify(G_OBJECT(actor), "interacting");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
	}
}

gboolean tangle_actor_get_dragging(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->dragging;
}

void tangle_actor_set_dragging(TangleActor* actor, gboolean dragging) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->dragging != dragging) {
		actor->priv->dragging = dragging;
		g_object_notify(G_OBJECT(actor), "dragging");
	}
}

gboolean tangle_actor_get_ancestor_interacting(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->ancestor_interacting;
}

gboolean tangle_actor_get_ancestor_dragging(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->ancestor_dragging;
}

void tangle_actor_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	gfloat min_width;
	gboolean min_width_set;
	gfloat natural_width;
	gboolean natural_width_set;
	TangleActorClass* actor_class;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (for_height >= 0 && actor->priv->aspect_ratio_set) {
		if (min_width_p) {
			*min_width_p = actor->priv->aspect_ratio * for_height;
		}
		if (natural_width_p) {
			*natural_width_p = actor->priv->aspect_ratio * for_height;
		}
		if (max_width_p) {
			*max_width_p = actor->priv->aspect_ratio * for_height;
		}
	} else {
		g_object_get(actor, "min-width", &min_width, "min-width-set", &min_width_set, "natural-width", &natural_width, "natural-width-set", &natural_width_set, NULL);
		if (min_width_set && min_width_p) {
			*min_width_p = min_width;
			min_width_p = NULL;
		}
		if (natural_width_set && natural_width_p) {
			*natural_width_p = natural_width;
			natural_width_p = NULL;	
		}

		if (min_width_p || natural_width_p || max_width_p) {
			actor_class = TANGLE_ACTOR_GET_CLASS(actor);
			if (actor_class->get_preferred_width) {
				actor_class->get_preferred_width(actor, for_height, interacting, min_width_p, natural_width_p, max_width_p);

				if (natural_width_p && min_width_p && *natural_width_p < *min_width_p) {
					*natural_width_p = *min_width_p;
				}
				if (max_width_p && natural_width_p && *max_width_p != 0 && *max_width_p < *natural_width_p) {
					*max_width_p = *natural_width_p;
				}
				if (actor->priv->interacting && actor->priv->interactive_size_multiplier_set) {
					if (min_width_p) {
						*min_width_p *= actor->priv->interactive_size_multiplier;
					}
					if (natural_width_p) {
						*natural_width_p *= actor->priv->interactive_size_multiplier;
					}
					if (max_width_p) {
						*max_width_p *= actor->priv->interactive_size_multiplier;
					}
				}
			} else {
				if (min_width_p) {
					*min_width_p = 0.0;
				}
				if (natural_width_p) {
					*natural_width_p = 0.0;
				}
				if (max_width_p) {
					*max_width_p = 0.0;
				}
			}

			if (actor->priv->margin_set) {
				if (*min_width_p) {
					*min_width_p += actor->priv->margin.left + actor->priv->margin.right;
				}
				if (*natural_width_p) {
					*natural_width_p += actor->priv->margin.left + actor->priv->margin.right;
				}
			}
		}
	}
}

void tangle_actor_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	gfloat min_height;
	gboolean min_height_set;
	gfloat natural_height;
	gboolean natural_height_set;
	TangleActorClass* actor_class;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (for_width >= 0 && actor->priv->aspect_ratio_set) {
		if (min_height_p) {
			*min_height_p = for_width / actor->priv->aspect_ratio;
		}
		if (natural_height_p) {
			*natural_height_p = for_width / actor->priv->aspect_ratio;
		}
		if (max_height_p) {
			*max_height_p = for_width / actor->priv->aspect_ratio;
		}
	} else {
		g_object_get(actor, "min-height", &min_height, "min-height-set", &min_height_set, "natural-height", &natural_height, "natural-height-set", &natural_height_set, NULL);
		if (min_height_set && min_height_p) {
			*min_height_p = min_height;
			min_height_p = NULL;
		}
		if (natural_height_set && natural_height_p) {
			*natural_height_p = natural_height;
			natural_height_p = NULL;	
		}

		if (min_height_p || natural_height_p || max_height_p) {
			actor_class = TANGLE_ACTOR_GET_CLASS(actor);
			if (actor_class->get_preferred_height) {
				actor_class->get_preferred_height(actor, for_width, interacting, min_height_p, natural_height_p, max_height_p);

				if (natural_height_p && min_height_p && *natural_height_p < *min_height_p) {
					*natural_height_p = *min_height_p;
				}
				if (max_height_p && natural_height_p && *max_height_p != 0 && *max_height_p < *natural_height_p) {
					*max_height_p = *natural_height_p;
				}
				if (actor->priv->interacting && actor->priv->interactive_size_multiplier_set) {
					if (min_height_p) {
						*min_height_p *= actor->priv->interactive_size_multiplier;
					}
					if (natural_height_p) {
						*natural_height_p *= actor->priv->interactive_size_multiplier;
					}
					if (max_height_p) {
						*max_height_p *= actor->priv->interactive_size_multiplier;
					}
				}
			} else {
				if (min_height_p) {
					*min_height_p = 0.0;
				}
				if (natural_height_p) {
					*natural_height_p = 0.0;
				}
				if (max_height_p) {
					*max_height_p = 0.0;
				}
			}

			if (actor->priv->margin_set) {
				if (*min_height_p) {
					*min_height_p += actor->priv->margin.top + actor->priv->margin.bottom;
				}
				if (*natural_height_p) {
					*natural_height_p += actor->priv->margin.top + actor->priv->margin.bottom;
				}		
			}
		}
	}
}

void tangle_actor_get_preferred_size(TangleActor* actor, gboolean interacting, gfloat* min_width_p, gfloat* min_height_p, gfloat* natural_width_p, gfloat* natural_height_p, gfloat* max_width_p, gfloat* max_height_p) {
	ClutterRequestMode request_mode;
	gfloat natural_size;

	g_return_if_fail(TANGLE_IS_ACTOR(actor));
	
	g_object_get(actor, "request-mode", &request_mode, NULL);
	if (request_mode == CLUTTER_REQUEST_HEIGHT_FOR_WIDTH) {
		tangle_actor_get_preferred_width(actor, -1, interacting, min_width_p, &natural_size, max_width_p);
		tangle_actor_get_preferred_height(actor, natural_size, interacting, min_height_p, natural_height_p, max_height_p);
		if (natural_width_p) {
			*natural_width_p = natural_size;
		}
	} else {
		tangle_actor_get_preferred_height(actor, -1, interacting, min_height_p, &natural_size, max_height_p);
		tangle_actor_get_preferred_width(actor, natural_size, interacting, min_width_p, natural_width_p, max_width_p);
		if (natural_height_p) {
			*natural_height_p = natural_size;
		}
	}

}

gdouble tangle_actor_get_interactive_size_multiplier(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), 0.0);

	return actor->priv->interactive_size_multiplier;
}

void tangle_actor_set_interactive_size_multiplier(TangleActor* actor, gdouble multiplier) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->interactive_size_multiplier != multiplier) {
		g_object_freeze_notify(G_OBJECT(actor));
		
		actor->priv->interactive_size_multiplier = multiplier;
		g_object_notify(G_OBJECT(actor), "interactive-size-multiplier");
		tangle_actor_set_interactive_size_multiplier_set(actor, TRUE);
		if (actor->priv->interacting) {
			clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
		}
		
		g_object_thaw_notify(G_OBJECT(actor));
	}
}

gboolean tangle_actor_get_interactive_size_multiplier_set(TangleActor* actor) {
	g_return_val_if_fail(TANGLE_IS_ACTOR(actor), FALSE);

	return actor->priv->interactive_size_multiplier_set;
}

void tangle_actor_set_interactive_size_multiplier_set(TangleActor* actor, gboolean is_set) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	if (actor->priv->interactive_size_multiplier_set != is_set) {
		actor->priv->interactive_size_multiplier_set = is_set;
		g_object_notify(G_OBJECT(actor), "interactive-size-multiplier-set");
		if (actor->priv->interacting) {
			clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
		}
	}
}

void tangle_actor_get_layout_allocation(TangleActor* actor, ClutterActorBox* actor_box) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	*actor_box = actor->priv->layout_allocation;
}

void tangle_actor_set_layout_allocation(TangleActor* actor, const ClutterActorBox* actor_box) {
	g_return_if_fail(TANGLE_IS_ACTOR(actor));

	actor->priv->layout_allocation = *actor_box;
}

GType tangle_spacing_get_type(void) {
	static GType type = 0;

	if (!type)
		type = g_boxed_type_register_static("TangleSpacing",
		                                    (GBoxedCopyFunc)tangle_spacing_copy,
		                                    (GBoxedFreeFunc)tangle_spacing_free);

	return type;
}

TangleSpacing* tangle_spacing_new(gfloat top, gfloat right, gfloat bottom, gfloat left) {
	TangleSpacing* spacing;
	
	spacing = (TangleSpacing*)g_slice_alloc(sizeof(TangleSpacing));
	spacing->top = top;
	spacing->right = right;
	spacing->bottom = bottom;
	spacing->left = left;

	return spacing;
}

TangleSpacing* tangle_spacing_copy(const TangleSpacing* spacing) {
	TangleSpacing* copy_of_spacing;
	
	copy_of_spacing = (TangleSpacing*)g_slice_alloc(sizeof(TangleSpacing));
	*copy_of_spacing = *spacing;
	
	return copy_of_spacing;
}

void tangle_spacing_free(TangleSpacing* spacing) {
	g_slice_free1(sizeof(TangleSpacing), spacing);
}

gboolean tangle_spacing_equal(const TangleSpacing* spacing_a, const TangleSpacing* spacing_b) {
	
	return spacing_a->top == spacing_b->top &&
	       spacing_a->right == spacing_b->right &&
	       spacing_a->bottom == spacing_b->bottom &&
	       spacing_a->left == spacing_b->left;
}

gboolean tangle_spacing_parse_from_json(TangleSpacing* spacing, JsonNode* node) {
	gboolean retvalue = FALSE;
	
	switch (JSON_NODE_TYPE(node)) {
		case JSON_NODE_ARRAY:
			retvalue = parse_spacing_from_json_array(spacing, json_node_get_array(node));
			break;
		case JSON_NODE_OBJECT:
			retvalue = parse_spacing_from_json_object(spacing, json_node_get_object(node));
			break;
		default:
			g_warning("Failed to parse TangleSpacing from JSON node that was not an array or an object, skipping.\n");
			break;
	}
	
	return retvalue;
}

ClutterAction* tangle_actor_get_action_by_type(ClutterActor* actor, GType type) {
	ClutterAction* action = NULL;
	GList* list;
	GList* list_item;
	
	g_return_val_if_fail(CLUTTER_IS_ACTOR(actor), NULL);
	g_return_val_if_fail(g_type_is_a(type, CLUTTER_TYPE_ACTION), NULL);
	
	list = clutter_actor_get_actions(actor);
	for (list_item = list; !action && list_item; list_item = list_item->next) {
		if (g_type_is_a(G_OBJECT_TYPE(list_item->data), type)) {
			action = CLUTTER_ACTION(list_item->data);
		}
	}
	g_list_free(list);

	return action;
}

static void set_max_width_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->max_width_set != is_set) {
		actor->priv->max_width_set = is_set;
		g_object_notify(G_OBJECT(actor), "max-width-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

static void set_max_width(TangleActor* actor, gfloat max_width) {
	if (actor->priv->max_width != max_width) {
		g_object_freeze_notify(G_OBJECT(actor));
	
		actor->priv->max_width = max_width;
		g_object_notify(G_OBJECT(actor), "max-width");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		
		set_max_width_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

static void set_max_height_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->max_height_set != is_set) {
		actor->priv->max_height_set = is_set;
		g_object_notify(G_OBJECT(actor), "max-height-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

static void set_max_height(TangleActor* actor, gfloat max_height) {
	if (actor->priv->max_height != max_height) {
		g_object_freeze_notify(G_OBJECT(actor));
	
		actor->priv->max_height = max_height;
		g_object_notify(G_OBJECT(actor), "max-height");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		
		set_max_height_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

static void tangle_actor_override_get_preferred_width(ClutterActor* actor, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	tangle_actor_get_preferred_width(TANGLE_ACTOR(actor), for_height, TANGLE_ACTOR(actor)->priv->interacting, min_width_p, natural_width_p, NULL);
}

static void tangle_actor_override_get_preferred_height(ClutterActor* actor, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	tangle_actor_get_preferred_height(TANGLE_ACTOR(actor), for_width, TANGLE_ACTOR(actor)->priv->interacting, min_height_p, natural_height_p, NULL);
}

static void tangle_actor_allocate(ClutterActor* self, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleActor* actor;
	ClutterActorBox* old_allocation;
	ClutterActorBox* new_allocation;
	gfloat length;
	gfloat difference;
	ClutterTimeline* timeline;
	gdouble aspect_ratio;

	actor = TANGLE_ACTOR(self);
	if (TANGLE_ACTOR_IS_ALLOCATED(actor)) {
		g_object_get(G_OBJECT(self), "allocation", &old_allocation, NULL);
	} else {
		old_allocation = NULL;
		TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_ALLOCATED);
	}
	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->allocate(self, box, flags);
	g_object_get(G_OBJECT(self), "allocation", &new_allocation, NULL);

	actor->priv->aligned_allocation.x1 = 0.0;
	actor->priv->aligned_allocation.y1 = 0.0;
	actor->priv->aligned_allocation.x2 = new_allocation->x2 - new_allocation->x1;
	actor->priv->aligned_allocation.y2 = new_allocation->y2 - new_allocation->y1;

	if (actor->priv->margin_set) {
		actor->priv->aligned_allocation.x1 += actor->priv->margin.left;
		actor->priv->aligned_allocation.y1 += actor->priv->margin.top;
		actor->priv->aligned_allocation.x2 -= actor->priv->margin.right;
		actor->priv->aligned_allocation.y2 -= actor->priv->margin.bottom;
		if (actor->priv->aligned_allocation.x2 < 0.0) {
			actor->priv->aligned_allocation.x2 = 0.0;
		}
		if (actor->priv->aligned_allocation.y2 < 0.0) {
			actor->priv->aligned_allocation.y2 = 0.0;
		}
		if (actor->priv->aligned_allocation.x1 > actor->priv->aligned_allocation.x2) {
			actor->priv->aligned_allocation.x1 = actor->priv->aligned_allocation.x2;
		}
		if (actor->priv->aligned_allocation.y1 > actor->priv->aligned_allocation.y2) {
			actor->priv->aligned_allocation.y1 = actor->priv->aligned_allocation.y2;
		}
	}

	if (actor->priv->max_width_set) {
		length = actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1;
		difference = length - actor->priv->max_width;
		if (difference > 0) {
			actor->priv->aligned_allocation.x1 += difference * actor->priv->alignment_x;
			actor->priv->aligned_allocation.x2 = actor->priv->aligned_allocation.x1 + actor->priv->max_width;
		}
	}
	if (actor->priv->max_height_set) {
		length = actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1;
		difference = length - actor->priv->max_height;
		if (difference > 0) {
			actor->priv->aligned_allocation.y1 += difference * actor->priv->alignment_y;
			actor->priv->aligned_allocation.y2 = actor->priv->aligned_allocation.y1 + actor->priv->max_height;
		}
	}
	
	if (actor->priv->aspect_ratio_set &&
	    (aspect_ratio = (actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1) / (actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1)) != actor->priv->aspect_ratio) {
		if (aspect_ratio > actor->priv->aspect_ratio) {
			length = actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1;
			difference = length - actor->priv->aspect_ratio * (actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1);
			actor->priv->aligned_allocation.x1 += difference * actor->priv->alignment_x;
			actor->priv->aligned_allocation.x2 -= difference * (1 - actor->priv->alignment_x);
		} else {
			length = actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1;
			difference = length - actor->priv->aspect_ratio * (actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1);
			actor->priv->aligned_allocation.y1 += difference * actor->priv->alignment_y;
			actor->priv->aligned_allocation.y2 -= difference * (1 - actor->priv->alignment_y);
		}
	}

	set_scale_centers_by_gravity(actor);

	if (!old_allocation ||
	    old_allocation->x1 != new_allocation->x1 || old_allocation->x2 != new_allocation->x2 ||
	    old_allocation->y1 != new_allocation->y1 || old_allocation->y2 != new_allocation->y2) {
		if (!old_allocation) {
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_SHOWING);
			actor->priv->transition_scale_children = TRUE;
		}
		timeline = animate_transition(actor, old_allocation, new_allocation);
		if (timeline && !old_allocation) {
			g_signal_connect_swapped(timeline, "completed", G_CALLBACK(unset_transition_scale_children), self);
		} else {
			actor->priv->transition_scale_children = FALSE;
		}
	}
	
	if (old_allocation) {
		clutter_actor_box_free(old_allocation);
	}
	clutter_actor_box_free(new_allocation);
}

static void tangle_actor_paint(ClutterActor* clutter_actor) {
	TangleActor* actor;
	gfloat width, height;
	TangleActorClass* actor_class;
	
	actor = TANGLE_ACTOR(clutter_actor);
	width = actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1;
	height = actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1;
	
	cogl_translate(actor->priv->aligned_allocation.x1, actor->priv->aligned_allocation.y1, 0.0);
	
	actor_class = TANGLE_ACTOR_GET_CLASS(actor);
	actor_class->paint_aligned(actor, width, height);
}

static void tangle_actor_pick(ClutterActor* clutter_actor, const ClutterColor* color) {
	TangleActor* actor;
	
	actor = TANGLE_ACTOR(clutter_actor);

	if (!TANGLE_ACTOR_IS_HIDING(actor) && !TANGLE_ACTOR_IS_SHOWING(actor)) {
		CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->pick(clutter_actor, color);
	}
}

static void tangle_actor_real_show(ClutterActor* clutter_actor) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);

	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->show(clutter_actor);
}

static void tangle_actor_hide(ClutterActor* clutter_actor) {
	TangleActor* actor;
	ClutterActorBox box = { 0.0, 0.0, 0.0, 0.0 };
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);

	TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_ALLOCATED);
	/* TODO: dunno, if this is causing mysterious crashes...
	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->allocate(clutter_actor, &box, CLUTTER_ALLOCATION_NONE);
	*/
	
	if (TANGLE_ACTOR_IS_HIDING(actor)) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);

		complete_timeline(actor->priv->hiding_timeline);
		g_object_unref(actor->priv->hiding_timeline);
		actor->priv->hiding_timeline = NULL;
	}

	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->hide(clutter_actor);
}

static void tangle_actor_parent_set(ClutterActor* actor, ClutterActor* old_parent) {
	ClutterActor* parent;
	
	if (old_parent && TANGLE_IS_ACTOR(old_parent)) {
		g_signal_handlers_disconnect_by_func(old_parent, G_CALLBACK(on_notify_interacting_or_dragging), actor);
	}
	
	parent = clutter_actor_get_parent(actor);
	if (parent && TANGLE_IS_ACTOR(parent)) {
		update_ancestor_interacting_and_dragging(TANGLE_ACTOR(actor), TANGLE_ACTOR(parent));

		g_signal_connect(parent, "notify::interacting", G_CALLBACK(on_notify_interacting_or_dragging), actor);
		g_signal_connect(parent, "notify::ancestor-interacting", G_CALLBACK(on_notify_interacting_or_dragging), actor);
		g_signal_connect(parent, "notify::dragging", G_CALLBACK(on_notify_interacting_or_dragging), actor);
		g_signal_connect(parent, "notify::ancestor-dragging", G_CALLBACK(on_notify_interacting_or_dragging), actor);
	} else {
		update_ancestor_interacting_and_dragging(TANGLE_ACTOR(actor), NULL);
	}	
}

static ClutterTimeline* tangle_actor_animate_transition(TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterTimeline* timeline = NULL;
	
	if (actor->priv->transition) {
		timeline = tangle_transition_animate_actor(actor->priv->transition, actor, current_box, new_box);
	}
	
	return timeline;
}

static void tangle_actor_transition_completed(TangleActor* actor, ClutterActorBox* old_box, ClutterActorBox* current_box) {
}

static void tangle_actor_show_completed(TangleActor* actor, ClutterActorBox* current_box) {
}

static void tangle_actor_paint_event_handler(ClutterActor* self, gpointer user_data) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(self);

	cogl_translate(actor->priv->transition_move_x + actor->priv->squint_x,
	               actor->priv->transition_move_y + actor->priv->squint_y,
		       0);
}

static void tangle_actor_pick_event_handler(ClutterActor* self, gpointer user_data) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(self);
	cogl_translate(actor->priv->transition_move_x + actor->priv->squint_x,
	               actor->priv->transition_move_y + actor->priv->squint_y,
		       0);
	if (actor->priv->picking_scale_x != 1.0 || actor->priv->picking_scale_y != 1.0) {
		cogl_translate(actor->priv->picking_scale_center_x, actor->priv->picking_scale_center_y, 0.0);
		cogl_scale(actor->priv->picking_scale_x, actor->priv->picking_scale_y, 1.0);
		cogl_translate(-actor->priv->picking_scale_center_x, -actor->priv->picking_scale_center_y, 0.0);
	}
}

static void tangle_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleActor* actor;
	
	actor = TANGLE_ACTOR(object);
	switch (prop_id) {
		case PROP_TEMPLATE:
			if (g_value_get_object(value) != NULL) {
				tangle_template_apply_properties(TANGLE_TEMPLATE(g_value_get_object(value)), object);
			}
			break;
		case PROP_TRANSITION_MOVE_X:
			actor->priv->transition_move_x = g_value_get_float(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_MOVE_Y:
			actor->priv->transition_move_y = g_value_get_float(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_SCALE_X:
			if ((actor->priv->transition_scale_x = g_value_get_double(value)) < 0) {
				actor->priv->transition_scale_x = 0.0;
			}
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_SCALE_Y:
			if ((actor->priv->transition_scale_y = g_value_get_double(value)) < 0) {
				actor->priv->transition_scale_y = 0.0;
			}
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_SCALE_CENTER_X:
			actor->priv->transition_scale_center_x = g_value_get_float(value);
			set_transition_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_TRANSITION_SCALE_CENTER_Y:
			actor->priv->transition_scale_center_y = g_value_get_float(value);
			set_transition_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_TRANSITION_SCALE_GRAVITY:
			set_transition_scale_gravity(actor, g_value_get_enum(value));
			break;
		case PROP_TRANSITION:
			tangle_actor_set_transition(actor, TANGLE_TRANSITION(g_value_get_object(value)));
			break;
		case PROP_DEPTH_POSITION:
			tangle_actor_set_depth_position(actor, g_value_get_int(value));
			break;
		case PROP_PICKING_SCALE_X:
			actor->priv->picking_scale_x = g_value_get_double(value);
			break;
		case PROP_PICKING_SCALE_Y:
			actor->priv->picking_scale_y = g_value_get_double(value);
			break;
		case PROP_PICKING_SCALE_CENTER_X:
			actor->priv->picking_scale_center_x = g_value_get_float(value);
			set_picking_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_PICKING_SCALE_CENTER_Y:
			actor->priv->picking_scale_center_y = g_value_get_float(value);
			set_picking_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_PICKING_SCALE_GRAVITY:
			set_picking_scale_gravity(actor, g_value_get_enum(value));
			break;
		case PROP_SQUINT_X:
			actor->priv->squint_x = g_value_get_float(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_SQUINT_Y:
			actor->priv->squint_y = g_value_get_float(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_MARGIN:
			tangle_actor_set_margin(actor, TANGLE_SPACING(g_value_get_boxed(value)));
			break;
		case PROP_MARGIN_SET:
			tangle_actor_set_margin_set(actor, g_value_get_boolean(value));
			break;
		case PROP_ASPECT_RATIO:
			tangle_actor_set_aspect_ratio(actor, g_value_get_double(value));
			break;
		case PROP_ASPECT_RATIO_SET:
			tangle_actor_set_aspect_ratio_set(actor, g_value_get_boolean(value));
			break;
		case PROP_INTERACTING:
			tangle_actor_set_interacting(actor, g_value_get_boolean(value));
			break;
		case PROP_DRAGGING:
			tangle_actor_set_dragging(actor, g_value_get_boolean(value));
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER:
			tangle_actor_set_interactive_size_multiplier(actor, g_value_get_double(value));
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER_SET:
			tangle_actor_set_interactive_size_multiplier_set(actor, g_value_get_boolean(value));
			break;
		case PROP_ACTION:
			clutter_actor_add_action(CLUTTER_ACTOR(actor), CLUTTER_ACTION(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void tangle_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleActor* actor;

	actor = TANGLE_ACTOR(object);
        switch (prop_id) {
		case PROP_TRANSITION_MOVE_X:
			g_value_set_float(value, actor->priv->transition_move_x);
			break;
		case PROP_TRANSITION_MOVE_Y:
			g_value_set_float(value, actor->priv->transition_move_y);
			break;
		case PROP_TRANSITION_SCALE_X:
			g_value_set_double(value, actor->priv->transition_scale_x);
			break;
		case PROP_TRANSITION_SCALE_Y:
			g_value_set_double(value, actor->priv->transition_scale_y);
			break;
		case PROP_TRANSITION_SCALE_CENTER_X:
			g_value_set_float(value, actor->priv->transition_scale_center_x);
			break;
		case PROP_TRANSITION_SCALE_CENTER_Y:
			g_value_set_float(value, actor->priv->transition_scale_center_y);
			break;
		case PROP_TRANSITION_SCALE_GRAVITY:
			g_value_set_enum(value, actor->priv->transition_scale_gravity);
			break;
		case PROP_TRANSITION:
			g_value_set_object(value, actor->priv->transition);
			break;
		case PROP_DEPTH_POSITION:
			g_value_set_int(value, actor->priv->depth_position);
			break;
		case PROP_PICKING_SCALE_X:
			g_value_set_double(value, actor->priv->picking_scale_x);
			break;
		case PROP_PICKING_SCALE_Y:
			g_value_set_double(value, actor->priv->picking_scale_y);
			break;
		case PROP_PICKING_SCALE_CENTER_X:
			g_value_set_float(value, actor->priv->picking_scale_center_x);
			break;
		case PROP_PICKING_SCALE_CENTER_Y:
			g_value_set_float(value, actor->priv->picking_scale_center_y);
			break;
		case PROP_PICKING_SCALE_GRAVITY:
			g_value_set_enum(value, actor->priv->picking_scale_gravity);
			break;
		case PROP_SQUINT_X:
			g_value_set_float(value, actor->priv->squint_x);
			break;
		case PROP_SQUINT_Y:
			g_value_set_float(value, actor->priv->squint_y);
			break;
		case PROP_MARGIN:
			g_value_set_boxed(value, &actor->priv->margin);
			break;
		case PROP_MARGIN_SET:
			g_value_set_boolean(value, actor->priv->margin_set);
			break;
		case PROP_ASPECT_RATIO:
			g_value_set_double(value, actor->priv->aspect_ratio);
			break;
		case PROP_ASPECT_RATIO_SET:
			g_value_set_boolean(value, actor->priv->aspect_ratio_set);
			break;
		case PROP_INTERACTING:
			g_value_set_boolean(value, actor->priv->interacting);
			break;
		case PROP_DRAGGING:
			g_value_set_boolean(value, actor->priv->dragging);
			break;
		case PROP_ANCESTOR_INTERACTING:
			g_value_set_boolean(value, actor->priv->ancestor_interacting);
			break;
		case PROP_ANCESTOR_DRAGGING:
			g_value_set_boolean(value, actor->priv->ancestor_dragging);
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER:
			g_value_set_double(value, actor->priv->interactive_size_multiplier);
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER_SET:
			g_value_set_boolean(value, actor->priv->interactive_size_multiplier_set);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
        }
}

static void tangle_actor_dispose(GObject* object) {
	TangleActor* actor;
	ClutterActor* parent;
	
	actor = TANGLE_ACTOR(object);

	if (actor->priv->hiding_timeline) {
		complete_timeline(actor->priv->hiding_timeline);
		g_object_unref(actor->priv->hiding_timeline);
	}
	TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);
	
	if (actor->priv->template_parameters) {
		g_hash_table_destroy(actor->priv->template_parameters);
		actor->priv->template_parameters = NULL;
	}
	
	parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor));
	if (parent && TANGLE_IS_ACTOR(parent)) {
		g_signal_handlers_disconnect_by_func(parent, G_CALLBACK(on_notify_interacting_or_dragging), actor);
	}
	
	G_OBJECT_CLASS (tangle_actor_parent_class)->dispose (object);
}

static void tangle_actor_class_init(TangleActorClass* actor_class) {
	GObjectClass* gobject_class;
	ClutterActorClass* clutter_actor_class;
	
	gobject_class = G_OBJECT_CLASS(actor_class);
	gobject_class->dispose = tangle_actor_dispose;
	gobject_class->set_property = tangle_actor_set_property;
	gobject_class->get_property = tangle_actor_get_property;

	clutter_actor_class = CLUTTER_ACTOR_CLASS(actor_class);
	clutter_actor_class->get_preferred_width = tangle_actor_override_get_preferred_width;
	clutter_actor_class->get_preferred_height = tangle_actor_override_get_preferred_height;
	clutter_actor_class->allocate = tangle_actor_allocate;
	clutter_actor_class->paint = tangle_actor_paint;
	clutter_actor_class->pick = tangle_actor_pick;
	clutter_actor_class->show = tangle_actor_real_show;
	clutter_actor_class->hide = tangle_actor_hide;
	clutter_actor_class->parent_set = tangle_actor_parent_set;
	
	actor_class->animate_transition = tangle_actor_animate_transition;
	actor_class->transition_completed = tangle_actor_transition_completed;
	actor_class->show_completed = tangle_actor_show_completed;

	/**
	 * TangleActor:template:
	 *
	 * Applies the given #TangleTemplate to this actor.
	 */
	g_object_class_install_property(gobject_class, PROP_TEMPLATE,
	                                g_param_spec_object("template",
	                                "Template",
	                                "Applies the given template to this actor",
	                                TANGLE_TYPE_TEMPLATE,
	                                G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-move-x:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_MOVE_X,
	                                g_param_spec_float("transition-move-x",
	                                                   "Transition move x",
	                                                   "Moves actor along x axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-move-y:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_MOVE_Y,
	                                g_param_spec_float("transition-move-y",
	                                                   "Transition move y",
	                                                   "Moves actor along y axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-x:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_X,
	                                g_param_spec_double("transition-scale-x",
	                                                   "Transition scale x",
	                                                   "Scales actor along x axis when transiting from an allocation to another",
	                                                   -G_MAXDOUBLE, G_MAXDOUBLE, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-y:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_Y,
	                                g_param_spec_double("transition-scale-y",
	                                                   "Transition scale y",
	                                                   "Scales actor along y axis when transiting from an allocation to another",
	                                                   -G_MAXDOUBLE, G_MAXDOUBLE, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-center-x:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_CENTER_X,
	                                g_param_spec_float("transition-scale-center-x",
	                                                   "Transition scale center x",
	                                                   "The horizontal center point for transition scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-center-y:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_CENTER_Y,
	                                g_param_spec_float("transition-scale-center-y",
	                                                   "Transition scale center y",
	                                                   "The vertical center point for transition scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-gravity
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_GRAVITY,
	                                g_param_spec_enum("transition-scale-gravity",
	                                                   "Transition scale gravity",
	                                                   "The center point for transition as gravity",
	                                                   CLUTTER_TYPE_GRAVITY, CLUTTER_GRAVITY_CENTER,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition:
	 *
	 * The transition used when animating between allocations.
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION,
	                                g_param_spec_object("transition",
	                                                   "Transition",
	                                                    "The transition used when animating between allocations",
	                                                    TANGLE_TYPE_TRANSITION,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:depth-position:
	 *
	 * When an actor is in #TangleWidget container, :depth-position
	 * defines the relative position of the actor in stacking order.
	 * An actor that has higher :depth-position is drawn in front of
	 * an actor that has lower :depth-position.
	 *
	 * :depth-position is related to ClutterActor:depth, but it does not
	 * translate the Z coordinate of the actor. It is only used for
	 * actors that have ClutterActor:depth set to zero (default),
	 * otherwise it is ignored.
	 *
	 * This can interpreted as follows: First, ClutterActor:depth determines
	 * the depth of (and the Z coordinate of) the actor. If ClutterActor:depth
	 * is zero, then :depth-position determines the depth of the actor
	 * related to other actors that has ClutterActor:depth set to zero.
	 *
	 * If an actor is not #TangleActor, but some other #ClutterActor
	 * the :depth-position is considered zero when stacked in #TangleWidget.
	 *
	 * Note! This property does not work in other #ClutterContainer implementations
	 * than #TangleWidget and derivates.
	 *
	 * Setting this property resets ClutterActor:depth to zero, as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_DEPTH_POSITION,
	                                g_param_spec_int("depth-position",
	                                                 "Depth position",
	                                                 "The relative depth position of the actor in stacking order",
	                                                 G_MININT, G_MAXINT, 0,
	                                                 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-x:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_X,
	                                g_param_spec_double("picking-scale-x",
	                                                   "Picking scale x",
	                                                   "Scales the picking area of the actor along x axis",
	                                                   0, G_MAXDOUBLE, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-y:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_Y,
	                                g_param_spec_double("picking-scale-y",
	                                                   "Picking scale y",
	                                                   "Scales the picking area of the actor along y axis",
	                                                   0, G_MAXDOUBLE, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-center-x:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_CENTER_X,
	                                g_param_spec_float("picking-scale-center-x",
	                                                   "Picking scale center x",
	                                                   "The horizontal center point for picking scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-center-y:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_CENTER_Y,
	                                g_param_spec_float("picking-scale-center-y",
	                                                   "Picking scale center y",
	                                                   "The vertical center point for picking scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-gravity
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_GRAVITY,
	                                g_param_spec_enum("picking-scale-gravity",
	                                                   "Picking scale gravity",
	                                                   "The center point for picking as gravity",
	                                                   CLUTTER_TYPE_GRAVITY, CLUTTER_GRAVITY_NONE,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:squint-x:
	 */
	g_object_class_install_property(gobject_class, PROP_SQUINT_X,
	                                g_param_spec_float("squint-x",
	                                                   "Squint X",
	                                                   "The horizontal squint (drawing offset)",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:squint-y:
	 */
	g_object_class_install_property(gobject_class, PROP_SQUINT_Y,
	                                g_param_spec_float("squint-y",
	                                                   "Squint Y",
	                                                   "The vertical squint (drawing offset)",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:margin:
	 *
	 * The marginals around the actor.
	 */
	g_object_class_install_property(gobject_class, PROP_MARGIN,
	                                g_param_spec_boxed("margin",
	                                                    "Margin",
	                                                    "The marginals around the actor",
	                                                    TANGLE_TYPE_SPACING,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:margin-set:
	 *
	 * Whether the :margin is set.
	 */
	g_object_class_install_property(gobject_class, PROP_MARGIN_SET,
	                                g_param_spec_boolean("margin-set",
	                                                    "Margin set",
	                                                    "Whether the margin is set",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:aspect-ratio:
	 */
	g_object_class_install_property(gobject_class, PROP_ASPECT_RATIO,
	                                g_param_spec_double("aspect-ratio",
	                                                    "Aspect ratio",
	                                                    "The fixed ratio of width / height",
	                                                    0.0, G_MAXFLOAT, 1.0,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:aspect-ratio-set:
	 */
	g_object_class_install_property(gobject_class, PROP_ASPECT_RATIO_SET,
	                                g_param_spec_boolean("aspect-ratio-set",
	                                                     "Aspect ratio set",
	                                                     "Whether the aspect ratio property is set or not",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interacting:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTING,
	                                g_param_spec_boolean("interacting",
	                                "Interacting",
	                                "Whether the actor is interacting with an user or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:dragging:
	 */
	g_object_class_install_property(gobject_class, PROP_DRAGGING,
	                                g_param_spec_boolean("dragging",
	                                "Dragging",
	                                "Whether the actor is under dragging",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:ancestor-interacting:
	 */
	g_object_class_install_property(gobject_class, PROP_ANCESTOR_INTERACTING,
	                                g_param_spec_boolean("ancestor-interacting",
	                                "Ancestor interacting",
	                                "Whether an ancestor of this actor is interacting with an user or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:ancestor-dragging:
	 */
	g_object_class_install_property(gobject_class, PROP_ANCESTOR_DRAGGING,
	                                g_param_spec_boolean("ancestor-dragging",
	                                "Ancestor Dragging",
	                                "Whether an ancestor of this actor is under dragging",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interactive-multiplier:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_SIZE_MULTIPLIER,
	                                g_param_spec_double("interactive-size-multiplier",
	                                "Interactive size multiplier",
	                                "The multiplier that is used to multiply the preferred size of the widget when it is interactive",
	                                0.0, G_MAXDOUBLE, 1.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interactive-multiplier-set:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_SIZE_MULTIPLIER_SET,
	                                g_param_spec_boolean("interactive-size-multiplier-set",
	                                "Interactive size multiplier set",
	                                "Whether the interactive size multiplier is set or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:action:
	 *
	 * Adds a new #ClutterAction into the actor.
	 */
	g_object_class_install_property(gobject_class, PROP_ACTION,
	                                g_param_spec_object("action",
	                                "Action",
	                                "Adds a new action into the actor",
	                                CLUTTER_TYPE_ACTION,
	                                G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));


	/**
	 * TangleActor::animate-transition:
	 * @actor: the object which received the signal
	 *
	 * The ::animate-transition signal is emitted when the allocation box of an actor has been
	 * changed.
	 */
	signals[ANIMATE_TRANSITION] = g_signal_new("animate-transition", G_TYPE_FROM_CLASS(gobject_class),
	                                           G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleActorClass, animate_transition),
						   tangle_signal_accumulator_non_null_handled, NULL,
						   tangle_marshal_OBJECT__BOXED_BOXED,
						   CLUTTER_TYPE_TIMELINE, 2,
						   CLUTTER_TYPE_ACTOR_BOX, CLUTTER_TYPE_ACTOR_BOX);
	/**
	 * TangleActor::show-completed:
	 * @actor: the object which received the signal
	 *
	 * The ::show-completed signal is emitted when the transition animation (see ::animate-transition), that is related
	 * to showing, is completed, or immediately if there is no animation.
	 */
	signals[SHOW_COMPLETED] = g_signal_new("show-completed", G_TYPE_FROM_CLASS(gobject_class),
	                                             G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleActorClass, show_completed),
						     NULL, NULL,
						     g_cclosure_marshal_VOID__BOXED,
						     G_TYPE_NONE, 1,
						     CLUTTER_TYPE_ACTOR_BOX);
	/**
	 * TangleActor::transition-completed:
	 * @actor: the object which received the signal
	 *
	 * The ::transition-completed signal is emitted when the transition animation (see ::animate-transition)
	 * is completed, or immediately if there is no animation.
	 */
	signals[TRANSITION_COMPLETED] = g_signal_new("transition-completed", G_TYPE_FROM_CLASS(gobject_class),
	                                             G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleActorClass, transition_completed),
						     NULL, NULL,
						     tangle_marshal_VOID__BOXED_BOXED,
						     G_TYPE_NONE, 2,
						     CLUTTER_TYPE_ACTOR_BOX, CLUTTER_TYPE_ACTOR_BOX);

	g_type_class_add_private(gobject_class, sizeof(TangleActorPrivate));
}

static void tangle_actor_init(TangleActor* actor) {
	actor->priv = G_TYPE_INSTANCE_GET_PRIVATE (actor, TANGLE_TYPE_ACTOR, TangleActorPrivate);

	actor->priv->transition_scale_x = 1.0;
	actor->priv->transition_scale_y = 1.0;
	actor->priv->transition_scale_gravity = CLUTTER_GRAVITY_CENTER;
	actor->priv->picking_scale_x = 1.0;
	actor->priv->picking_scale_y = 1.0;
	actor->priv->aspect_ratio = 1.0;
	actor->priv->alignment_x = 0.5;
	actor->priv->alignment_y = 0.5;

	actor->priv->paint_event_handler_id = g_signal_connect(actor, "paint", G_CALLBACK(tangle_actor_paint_event_handler), NULL);
	actor->priv->pick_event_handler_id = g_signal_connect(actor, "pick", G_CALLBACK(tangle_actor_pick_event_handler), NULL);
}

static gboolean tangle_actor_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node) {
	gboolean retvalue;
	TangleActor* actor;

	actor = TANGLE_ACTOR(scriptable);

	if (!(retvalue = _tangle_scriptable_parse_custom_node(scriptable, script, value, name, node, &actor->priv->template_parameters, &actor->priv->template_found)) &&
	    parent_scriptable_iface->parse_custom_node) {
		retvalue = parent_scriptable_iface->parse_custom_node(scriptable, script, value, name, node);
	}
	
	return retvalue;
}

static void tangle_actor_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(scriptable);
	
	_tangle_scriptable_set_custom_property(scriptable, script, name, value, &actor->priv->template_parameters, actor->priv->template_found);
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->parse_custom_node = tangle_actor_parse_custom_node;
	iface->set_custom_property = tangle_actor_set_custom_property;
}

static void on_transition_timeline_completed(ClutterTimeline* timeline, gpointer user_data) {
	TangleVault* vault;
	TangleActor* actor;
	ClutterActorBox* from_allocation;
	ClutterActorBox* to_allocation;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 3, TANGLE_TYPE_ACTOR, &actor, CLUTTER_TYPE_ACTOR_BOX, &from_allocation, CLUTTER_TYPE_ACTOR_BOX, &to_allocation);
	g_signal_emit(actor, signals[TRANSITION_COMPLETED], 0, from_allocation, to_allocation);
	if (TANGLE_ACTOR_IS_SHOWING(actor)) {
		g_signal_emit(actor, signals[SHOW_COMPLETED], 0, to_allocation);
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_SHOWING);
	}
	clutter_actor_box_free(from_allocation);
	clutter_actor_box_free(to_allocation);
}

static ClutterTimeline* animate_transition(TangleActor* actor, ClutterActorBox* from_allocation, ClutterActorBox* to_allocation) {
	ClutterTimeline* timeline = NULL;
	TangleVault* vault;
	
	g_signal_emit(actor, signals[ANIMATE_TRANSITION], 0, from_allocation, to_allocation, &timeline);
	if (timeline) {
		vault = tangle_vault_new(3, TANGLE_TYPE_ACTOR, actor, CLUTTER_TYPE_ACTOR_BOX, from_allocation, CLUTTER_TYPE_ACTOR_BOX, to_allocation);
		tangle_signal_connect_vault(timeline, "completed", G_CALLBACK(on_transition_timeline_completed), vault);
	} else {
		g_signal_emit(actor, signals[TRANSITION_COMPLETED], 0, from_allocation, to_allocation);
		if (TANGLE_ACTOR_IS_SHOWING(actor)) {
			g_signal_emit(actor, signals[SHOW_COMPLETED], 0, to_allocation);
			TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_SHOWING);
		}
	}

	return timeline;
}

static void set_coordinates_by_gravity(ClutterGravity gravity, gfloat width, gfloat height, gfloat* x_p, gfloat* y_p) {
	switch (gravity) {
		case CLUTTER_GRAVITY_NORTH_WEST:
		case CLUTTER_GRAVITY_WEST:
		case CLUTTER_GRAVITY_SOUTH_WEST:
			*x_p = 0.0;
			break;
		case CLUTTER_GRAVITY_NORTH_EAST:
		case CLUTTER_GRAVITY_EAST:
		case CLUTTER_GRAVITY_SOUTH_EAST:
			*x_p = width;
			break;
		case CLUTTER_GRAVITY_CENTER:
			*x_p = width / 2.0;
			break;
		default:
			break;
	}
	switch (gravity) {
		case CLUTTER_GRAVITY_NORTH_WEST:
		case CLUTTER_GRAVITY_NORTH:
		case CLUTTER_GRAVITY_NORTH_EAST:
			*y_p = 0.0;
			break;
		case CLUTTER_GRAVITY_SOUTH_WEST:
		case CLUTTER_GRAVITY_SOUTH:
		case CLUTTER_GRAVITY_SOUTH_EAST:
			*y_p = height;
			break;
		case CLUTTER_GRAVITY_CENTER:
			*y_p = height / 2.0;
			break;
		default:
			break;
	}
}

static void set_scale_centers_by_gravity(TangleActor* actor) {
	gfloat x, y;
	
	if (actor->priv->picking_scale_gravity != CLUTTER_GRAVITY_NONE) {
		set_coordinates_by_gravity(actor->priv->picking_scale_gravity,
		                           clutter_actor_get_width(CLUTTER_ACTOR(actor)), clutter_actor_get_height(CLUTTER_ACTOR(actor)),
					   &x, &y);
		if (actor->priv->picking_scale_center_x != x) {
			actor->priv->picking_scale_center_x = x;
			g_object_notify(G_OBJECT(actor), "picking-scale-center-x");
		}
		if (actor->priv->picking_scale_center_y != y) {
			actor->priv->picking_scale_center_y = y;
			g_object_notify(G_OBJECT(actor), "picking-scale-center-y");
		}
	}
	if (actor->priv->transition_scale_gravity != CLUTTER_GRAVITY_NONE) {
		set_coordinates_by_gravity(actor->priv->transition_scale_gravity,
		                           clutter_actor_get_width(CLUTTER_ACTOR(actor)), clutter_actor_get_height(CLUTTER_ACTOR(actor)),
					   &x, &y);
		if (actor->priv->transition_scale_center_x != x) {
			actor->priv->transition_scale_center_x = x;
			g_object_notify(G_OBJECT(actor), "transition-scale-center-x");
		}
		if (actor->priv->transition_scale_center_y != y) {
			actor->priv->transition_scale_center_y = y;
			g_object_notify(G_OBJECT(actor), "transition-scale-center-y");
		}
	}
}

static void set_picking_scale_gravity(TangleActor* actor, ClutterGravity gravity) {
	if (actor->priv->picking_scale_gravity != gravity) {
		actor->priv->picking_scale_gravity = gravity;
		set_scale_centers_by_gravity(actor);
		g_object_notify(G_OBJECT(actor), "picking-scale-gravity");
	}
}

static void set_transition_scale_gravity(TangleActor* actor, ClutterGravity gravity) {
	if (actor->priv->transition_scale_gravity != gravity) {
		actor->priv->transition_scale_gravity = gravity;
		set_scale_centers_by_gravity(actor);
		g_object_notify(G_OBJECT(actor), "transition-scale-gravity");
	}
}

static void unset_transition_scale_children(TangleActor* actor) {
	actor->priv->transition_scale_children = FALSE;
}

static void on_hiding_timeline_completed(TangleActor* actor) {
	if (TANGLE_ACTOR_IS_HIDING(actor)) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);

		g_object_unref(actor->priv->hiding_timeline);
		actor->priv->hiding_timeline = NULL;

		if (actor->priv->destroy) {
			clutter_actor_destroy(CLUTTER_ACTOR(actor));
		} else if (actor->priv->hide_all) {
			clutter_actor_hide_all(CLUTTER_ACTOR(actor));
		} else {
			clutter_actor_hide(CLUTTER_ACTOR(actor));
		}
	}
	actor->priv->hide_all = FALSE;
}

static void complete_timeline(ClutterTimeline* timeline) {
	clutter_timeline_skip(timeline, clutter_timeline_get_duration(timeline) - clutter_timeline_get_elapsed_time(timeline));
}

static gboolean parse_spacing_from_json_array(TangleSpacing* spacing, JsonArray* array) {
	gboolean retvalue = FALSE;
	
	if (json_array_get_length(array) != 4) {
		g_warning("Expected four elements in JSON array containing TangleSpacing, skipping.");
	} else {
		spacing->top = json_array_get_double_element(array, 0);
		spacing->right = json_array_get_double_element(array, 1);
		spacing->bottom = json_array_get_double_element(array, 2);
		spacing->left = json_array_get_double_element(array, 3);
		retvalue = TRUE;
	}
	
	return retvalue;
}

static gboolean parse_spacing_from_json_object(TangleSpacing* spacing, JsonObject* object) {
	if (json_object_has_member(object, "top")) {
		spacing->top = json_object_get_double_member(object, "top");
	} else {
		spacing->top = 0.0;
	}
	if (json_object_has_member(object, "right")) {
		spacing->right = json_object_get_double_member(object, "right");
	} else {
		spacing->right = 0.0;
	}
	if (json_object_has_member(object, "bottom")) {
		spacing->bottom = json_object_get_double_member(object, "bottom");
	} else {
		spacing->bottom = 0.0;
	}
	if (json_object_has_member(object, "left")) {
		spacing->left = json_object_get_double_member(object, "left");
	} else {
		spacing->left = 0.0;
	}
	
	return TRUE;
}

static void update_ancestor_interacting_and_dragging(TangleActor* actor, TangleActor* parent) {
	gboolean ancestor_interacting;
	gboolean ancestor_dragging;
	
	if (parent && (tangle_actor_get_interacting(parent) || tangle_actor_get_ancestor_interacting(parent))) {
		ancestor_interacting = TRUE;
	} else {
		ancestor_interacting = FALSE;
	}
	if (actor->priv->ancestor_interacting != ancestor_interacting) {
		actor->priv->ancestor_interacting = ancestor_interacting;
		g_object_notify(G_OBJECT(actor), "ancestor-interacting");
	}

	if (parent && (tangle_actor_get_dragging(parent) || tangle_actor_get_ancestor_dragging(parent))) {
		ancestor_dragging = TRUE;
	} else {
		ancestor_dragging = FALSE;
	}
	if (actor->priv->ancestor_dragging != ancestor_dragging) {
		actor->priv->ancestor_dragging = ancestor_dragging;
		g_object_notify(G_OBJECT(actor), "ancestor-dragging");
	}
}

static void on_notify_interacting_or_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	update_ancestor_interacting_and_dragging(TANGLE_ACTOR(user_data), TANGLE_ACTOR(object));
}
