/*
 * tangle-layout.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-layout.h"
#include "tangle-widget.h"
#include "marshalers.h"

/**
 * SECTION:tangle-layout
 * @Short_description: A base class for layout managers
 * @Title: TangleLayout
 */

G_DEFINE_ABSTRACT_TYPE(TangleLayout, tangle_layout, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_WIDGET,
	PROP_DIRECTION
};

enum {
	CHANGED,
	LAST_SIGNAL
};

struct _TangleLayoutPrivate {
	TangleWidget* widget;
	TangleLayoutDirection direction;
	ClutterActorBox allocation;
};

static GQuark quark_layout_meta = 0;
static guint signals[LAST_SIGNAL];

static const GEnumValue layout_direction_values[] = {
	{ TANGLE_LAYOUT_DIRECTION_LTR_TTB, "TANGLE_LAYOUT_DIRECTION_LTR_TTB", "ltr-ttb" },
	{ TANGLE_LAYOUT_DIRECTION_RTL_TTB, "TANGLE_LAYOUT_DIRECTION_RTL_TTB", "rtl-ttb" },
	{ TANGLE_LAYOUT_DIRECTION_LTR_BTT, "TANGLE_LAYOUT_DIRECTION_LTR_BTT", "ltr-btt" },
	{ TANGLE_LAYOUT_DIRECTION_RTL_BTT, "TANGLE_LAYOUT_DIRECTION_RTL_BTT", "rtl-btt" },
	{ TANGLE_LAYOUT_DIRECTION_TTB_LTR, "TANGLE_LAYOUT_DIRECTION_TTB_LTR", "ttb-ltr" },
	{ TANGLE_LAYOUT_DIRECTION_TTB_RTL, "TANGLE_LAYOUT_DIRECTION_TTB_RTL", "ttb-rtl" },
	{ TANGLE_LAYOUT_DIRECTION_BTT_LTR, "TANGLE_LAYOUT_DIRECTION_BTT_LTR", "btt-ltr" },
	{ TANGLE_LAYOUT_DIRECTION_BTT_RTL, "TANGLE_LAYOUT_DIRECTION_BTT_RTL", "btt-rtl" },
	{ 0, NULL, NULL }
};

GType tangle_layout_direction_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("TangleLayoutDirection", layout_direction_values);
	}
	
	return type;
}

TangleWidget* tangle_layout_get_widget(TangleLayout* layout) {

	return layout->priv->widget;
}

void tangle_layout_set_widget(TangleLayout* layout, TangleWidget* widget) {
	g_return_if_fail(TANGLE_IS_LAYOUT(layout));
	
	if (layout->priv->widget != widget) {
		if (widget) {
			g_return_if_fail(layout->priv->widget == NULL);
			
			layout->priv->widget = widget;
			g_object_add_weak_pointer(G_OBJECT(layout->priv->widget), (gpointer*)&layout->priv->widget);
		} else {
			g_return_if_fail(layout->priv->widget != NULL);
			
			g_object_remove_weak_pointer(G_OBJECT(layout->priv->widget), (gpointer*)&layout->priv->widget);
			layout->priv->widget = NULL;
		}
		
		g_object_notify(G_OBJECT(layout), "widget");
		/* TODO: signal */
	}
}

void tangle_layout_get_preferred_width(TangleLayout* layout, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	TangleLayoutClass* layout_class;
	
	layout_class = TANGLE_LAYOUT_GET_CLASS(layout);
	if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
		if (layout_class->get_preferred_height) {
			layout_class->get_preferred_height(layout, layout->priv->widget, for_height, min_width_p, natural_width_p);
		}
	} else {
		if (layout_class->get_preferred_width) {
			layout_class->get_preferred_width(layout, layout->priv->widget, for_height, min_width_p, natural_width_p);
		}
	}
}

void tangle_layout_get_preferred_actor_width(TangleLayout* layout, ClutterActor* actor, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
		clutter_actor_get_preferred_height(actor, for_height, min_width_p, natural_width_p);
	} else {
		clutter_actor_get_preferred_width(actor, for_height, min_width_p, natural_width_p);
	}
}

void tangle_layout_get_preferred_interacting_width(TangleLayout* layout, ClutterActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	if (TANGLE_IS_ACTOR(actor)) {
		if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
			tangle_actor_get_preferred_height(TANGLE_ACTOR(actor), for_height, interacting, min_width_p, natural_width_p, max_width_p);
		} else {
			tangle_actor_get_preferred_width(TANGLE_ACTOR(actor), for_height, interacting, min_width_p, natural_width_p, max_width_p);
		}
	} else {
		if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
			clutter_actor_get_preferred_height(actor, for_height, min_width_p, natural_width_p);
		} else {
			clutter_actor_get_preferred_width(actor, for_height, min_width_p, natural_width_p);
		}
		if (max_width_p) {
			*max_width_p = 0;
		}
	}
}

void tangle_layout_get_preferred_height(TangleLayout* layout, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	TangleLayoutClass* layout_class;
	
	layout_class = TANGLE_LAYOUT_GET_CLASS(layout);
	if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
		if (layout_class->get_preferred_width) {
			layout_class->get_preferred_width(layout, layout->priv->widget, for_width, min_height_p, natural_height_p);
		}	
	} else {
		if (layout_class->get_preferred_height) {
			layout_class->get_preferred_height(layout, layout->priv->widget, for_width, min_height_p, natural_height_p);
		}
	}
}

void tangle_layout_get_preferred_actor_height(TangleLayout* layout, ClutterActor* actor, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
		clutter_actor_get_preferred_height(actor, for_width, min_height_p, natural_height_p);
	} else {
		clutter_actor_get_preferred_width(actor, for_width, min_height_p, natural_height_p);
	}
}

void tangle_layout_get_preferred_interactive_height(TangleLayout* layout, ClutterActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	if (TANGLE_IS_ACTOR(actor)) {
		if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
			tangle_actor_get_preferred_width(TANGLE_ACTOR(actor), for_width, interacting, min_height_p, natural_height_p, max_height_p);
		} else {
			tangle_actor_get_preferred_height(TANGLE_ACTOR(actor), for_width, interacting, min_height_p, natural_height_p, max_height_p);
		}
	} else {
		if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
			clutter_actor_get_preferred_width(actor, for_width, min_height_p, natural_height_p);
		} else {
			clutter_actor_get_preferred_height(actor, for_width, min_height_p, natural_height_p);
		}
		if (max_height_p) {
			*max_height_p = 0;
		}
	}
}

void tangle_layout_allocate(TangleLayout* layout, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleLayoutClass* layout_class;
	
	layout->priv->allocation = *box;
	layout_class = TANGLE_LAYOUT_GET_CLASS(layout);
	if (layout_class->allocate) {
		if (layout->priv->direction & TANGLE_LAYOUT_DIRECTION_VERTICAL_FIRST) {
			layout_class->allocate(layout, layout->priv->widget, box->y2 - box->y1, box->x2 - box->x1, flags);
		} else {
			layout_class->allocate(layout, layout->priv->widget, box->x2 - box->x1, box->y2 - box->y1, flags);
		}
	}
}

void tangle_layout_allocate_actor(TangleLayout* layout, ClutterActor* actor, ClutterActorBox* box, ClutterAllocationFlags flags) {
	ClutterActorBox translated_box;
	
	switch (layout->priv->direction) {
		case TANGLE_LAYOUT_DIRECTION_LTR_TTB:
			translated_box.x1 = layout->priv->allocation.x1 + box->x1;
			translated_box.x2 = layout->priv->allocation.x1 + box->x2;
			translated_box.y1 = layout->priv->allocation.y1 + box->y1;
			translated_box.y2 = layout->priv->allocation.y1 + box->y2;
			break;
		case TANGLE_LAYOUT_DIRECTION_RTL_TTB:
			translated_box.x1 = layout->priv->allocation.x2 - box->x2;
			translated_box.x2 = layout->priv->allocation.x2 - box->x1;
			translated_box.y1 = layout->priv->allocation.y1 + box->y1;
			translated_box.y2 = layout->priv->allocation.y1 + box->y2;
			break;
		case TANGLE_LAYOUT_DIRECTION_LTR_BTT:
			translated_box.x1 = layout->priv->allocation.x1 + box->x1;
			translated_box.x2 = layout->priv->allocation.x1 + box->x2;
			translated_box.y1 = layout->priv->allocation.y2 - box->y2;
			translated_box.y2 = layout->priv->allocation.y2 - box->y1;
			break;
		case TANGLE_LAYOUT_DIRECTION_RTL_BTT:
			translated_box.x1 = layout->priv->allocation.x2 - box->x2;
			translated_box.x2 = layout->priv->allocation.x2 - box->x1;
			translated_box.y1 = layout->priv->allocation.y2 - box->y2;
			translated_box.y2 = layout->priv->allocation.y2 - box->y1;
			break;
		case TANGLE_LAYOUT_DIRECTION_TTB_LTR:
			translated_box.x1 = layout->priv->allocation.x1 + box->y1;
			translated_box.x2 = layout->priv->allocation.x1 + box->y2;
			translated_box.y1 = layout->priv->allocation.y1 + box->x1;
			translated_box.y2 = layout->priv->allocation.y1 + box->x2;
			break;
		case TANGLE_LAYOUT_DIRECTION_TTB_RTL:
			translated_box.x1 = layout->priv->allocation.x2 - box->y2;
			translated_box.x2 = layout->priv->allocation.x2 - box->y1;
			translated_box.y1 = layout->priv->allocation.y1 + box->x1;
			translated_box.y2 = layout->priv->allocation.y1 + box->x2;
			break;
		case TANGLE_LAYOUT_DIRECTION_BTT_LTR:
			translated_box.x1 = layout->priv->allocation.x1 + box->y1;
			translated_box.x2 = layout->priv->allocation.x1 + box->y2;
			translated_box.y1 = layout->priv->allocation.y2 - box->x2;
			translated_box.y2 = layout->priv->allocation.y2 - box->x1;
			break;
		case TANGLE_LAYOUT_DIRECTION_BTT_RTL:
			translated_box.x1 = layout->priv->allocation.x2 - box->y2;
			translated_box.x2 = layout->priv->allocation.x2 - box->y1;
			translated_box.y1 = layout->priv->allocation.y2 - box->x2;
			translated_box.y2 = layout->priv->allocation.y2 - box->x1;
			break;
	}
	clutter_actor_allocate(actor, &translated_box, flags);
}

TangleLayoutDirection tangle_layout_get_direction(TangleLayout* layout) {

	return layout->priv->direction;
}

void tangle_layout_set_direction(TangleLayout* layout, TangleLayoutDirection direction) {
	if (layout->priv->direction != direction) {
		layout->priv->direction = direction;
		g_object_notify(G_OBJECT(layout), "direction");
		g_signal_emit(layout, signals[CHANGED], 0);
	}
}

TangleLayoutMeta* tangle_layout_get_meta(TangleLayout* layout, ClutterActor* actor) {
	TangleLayoutMeta* layout_meta;
	TangleLayoutClass* layout_class;
	
	layout_class = TANGLE_LAYOUT_GET_CLASS(layout);
	
	if (!(layout_meta = g_object_get_qdata(G_OBJECT(actor), quark_layout_meta)) ||
	    tangle_layout_meta_get_layout(layout_meta) != layout) {
		if (layout_class->create_layout_meta &&
		    (layout_meta = layout_class->create_layout_meta(layout, actor))) {
			g_object_set_qdata_full(G_OBJECT(actor), quark_layout_meta, layout_meta, (GDestroyNotify)g_object_unref);
		} else {
			g_object_set_qdata(G_OBJECT(actor), quark_layout_meta, NULL);
		}
	}
	
	return layout_meta;
}

static void tangle_layout_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleLayout* layout;
	
	layout = TANGLE_LAYOUT(object);

	switch (prop_id) {
		case PROP_WIDGET:
			tangle_layout_set_widget(layout, TANGLE_WIDGET(g_value_get_object(value)));
			break;
		case PROP_DIRECTION:
			tangle_layout_set_direction(layout, g_value_get_enum(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_layout_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleLayout* layout;

	layout = TANGLE_LAYOUT(object);

        switch (prop_id) {
		case PROP_WIDGET:
			g_value_set_object(value, tangle_layout_get_widget(layout));
			break;
		case PROP_DIRECTION:
			g_value_set_enum(value, layout->priv->direction);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_layout_dispose(GObject* object) {
        TangleLayout* layout;

	layout = TANGLE_LAYOUT(object);

	if (layout->priv->widget) {
		g_object_remove_weak_pointer(G_OBJECT(layout->priv->widget), (gpointer*)&layout->priv->widget);
		layout->priv->widget = NULL;
	}

	G_OBJECT_CLASS(tangle_layout_parent_class)->dispose(object);
}

static void tangle_layout_class_init(TangleLayoutClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->dispose = tangle_layout_dispose;
	gobject_class->set_property = tangle_layout_set_property;
	gobject_class->get_property = tangle_layout_get_property;

	/**
	 * TangleLayout:widget:
	 *
	 * The widget of which children is allocated with this layout.
	 */
	g_object_class_install_property(gobject_class, PROP_WIDGET,
	                                g_param_spec_object ("widget",
	                                "Widget",
	                                "The widget of which children is allocated with this layout",
	                                TANGLE_TYPE_WIDGET,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleLayout:direction:
	 */
	g_object_class_install_property(gobject_class, PROP_DIRECTION,
	                                g_param_spec_enum("direction",
	                                "Direction",
	                                "The direction of which children are laid out",
	                                TANGLE_TYPE_LAYOUT_DIRECTION,
					TANGLE_LAYOUT_DIRECTION_LTR_TTB,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	signals[CHANGED] = g_signal_new("changed",
					G_TYPE_FROM_CLASS(gobject_class),
					G_SIGNAL_RUN_LAST,
					0,
					NULL, NULL,
					tangle_marshal_VOID__VOID,
					G_TYPE_NONE, 0);

	quark_layout_meta = g_quark_from_static_string("tangle-layout-meta");

	g_type_class_add_private (gobject_class, sizeof (TangleLayoutPrivate));
}

static void tangle_layout_init(TangleLayout* layout) {
	layout->priv = G_TYPE_INSTANCE_GET_PRIVATE(layout, TANGLE_TYPE_LAYOUT, TangleLayoutPrivate);
}

