/*
 * tangle-widget.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-widget.h"
#include "tangle-stylable.h"
#include "tangle-stylesheet.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-widget
 * @Short_description: A base class for user interface elements that consists of other actors
 * @Title: TangleWidget
 */

static void clutter_container_iface_init(ClutterContainerIface* iface);
static void tangle_stylable_iface_init(TangleStylableIface* iface);

G_DEFINE_TYPE_WITH_CODE(TangleWidget, tangle_widget, TANGLE_TYPE_ACTOR,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_CONTAINER, clutter_container_iface_init);
			G_IMPLEMENT_INTERFACE(TANGLE_TYPE_STYLABLE, tangle_stylable_iface_init));

enum {
	PROP_0,
	PROP_PARENT,
	PROP_N_CHILDREN,
	PROP_CHILREN_READONLY,
	PROP_LAYOUT,
	PROP_PADDING,
	PROP_CHILDREN_ALLOCATION,
	PROP_BACKGROUND_COLOR,
	PROP_BACKGROUND_COLOR_SET,
	PROP_BACKGROUND_ACTOR,
	PROP_PREFER_BACKGROUND_SIZE,
	PROP_PREFER_BACKGROUND_SIZE_SET
};

struct _TangleWidgetPrivate {
	GList* children;
	TangleLayout* layout;
	TangleSpacing padding;
	ClutterActorBox children_allocation;
	TangleStylesheet* stylesheet;

	ClutterColor background_color;
	ClutterActor* background_actor;
	
	guint padding_set : 1;
	guint background_color_set : 1;
	guint prefer_background_size : 1;
	guint prefer_background_size_set : 1;
};

static void tangle_widget_add_impl(TangleWidget* container, ClutterActor* actor, ClutterActor* existing, gboolean before);
static gboolean tangle_widget_unapply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet);

ClutterActor* tangle_widget_new() {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, NULL));
}

ClutterActor* tangle_widget_new_with_layout(TangleLayout* layout) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, "layout", layout, NULL));
}

ClutterActor* tangle_widget_new_with_parent(TangleWidget* parent) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_WIDGET, "parent", parent, NULL));
}


guint tangle_widget_get_n_children(TangleWidget* widget) {

	return g_list_length(widget->priv->children);
}

ClutterActor* tangle_widget_get_nth_child(TangleWidget* widget, guint n) {

	return CLUTTER_ACTOR(g_list_nth_data(widget->priv->children, n));
}

gboolean tangle_widget_has_child(TangleWidget* widget, ClutterActor* actor) {

	return g_list_find(widget->priv->children, actor) != NULL;
}

GList* tangle_widget_get_children_readonly(TangleWidget* widget) {

	return widget->priv->children;
}

void tangle_widget_foreach_swapped(TangleWidget* widget, TangleSwappedCallback swapped_callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		swapped_callback(user_data, CLUTTER_ACTOR(child_in_list->data));
	}
}

void tangle_widget_foreach_type(TangleWidget* widget, GType type, ClutterCallback callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		if (g_type_is_a(G_OBJECT_TYPE(child_in_list->data), type)) {
			callback(CLUTTER_ACTOR(child_in_list->data), user_data);
		}
	}
}

void tangle_widget_foreach_type_swapped(TangleWidget* widget, GType type, TangleSwappedCallback swapped_callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		if (g_type_is_a(G_OBJECT_TYPE(child_in_list->data), type)) {
			swapped_callback(user_data, CLUTTER_ACTOR(child_in_list->data));
		}
	}
}

void tangle_widget_add_before(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, const gchar* first_property, ...) {
	tangle_widget_add_impl(widget, actor, existing, TRUE);
	/* TODO: properties */
}

void tangle_widget_add_after(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, const gchar* first_property, ...) {
	tangle_widget_add_impl(widget, actor, existing, FALSE);
	/* TODO: properties */
}

void tangle_widget_set_layout(TangleWidget* widget, TangleLayout* layout) {
	if (widget->priv->layout != layout) {
		if (widget->priv->layout) {
			g_object_unref(widget->priv->layout);
			/* TODO: g_signal_remove */
			widget->priv->layout = NULL;
		}
		if (layout) {
			widget->priv->layout = layout;
			g_object_ref_sink(layout);
			g_signal_connect_swapped(layout, "changed", G_CALLBACK(clutter_actor_queue_relayout), widget);
			tangle_layout_set_widget(layout, widget);
		}
		
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		g_object_notify(G_OBJECT(widget), "layout");
	}
}

TangleLayout* tangle_widget_get_layout(TangleWidget* widget) {

	return widget->priv->layout;
}

void tangle_widget_get_padding(TangleWidget* widget, TangleSpacing* spacing) {
	*spacing = widget->priv->padding;
}
	
void tangle_widget_set_padding(TangleWidget* widget, const TangleSpacing* spacing) {
	if (!widget->priv->padding_set || !tangle_spacing_equal(&widget->priv->padding, spacing)) {
		g_object_freeze_notify(G_OBJECT(widget));
		
		if (widget->priv->padding_set) {
			widget->priv->children_allocation.x1 -= widget->priv->padding.left;
			widget->priv->children_allocation.y1 -= widget->priv->padding.top;
			widget->priv->children_allocation.x2 += widget->priv->padding.right;
			widget->priv->children_allocation.y2 += widget->priv->padding.bottom;
		}
		widget->priv->padding = *spacing;
		widget->priv->children_allocation.x1 += widget->priv->padding.left;
		widget->priv->children_allocation.y1 += widget->priv->padding.top;
		widget->priv->children_allocation.x2 -= widget->priv->padding.right;
		widget->priv->children_allocation.y2 -= widget->priv->padding.bottom;
		g_object_notify(G_OBJECT(widget), "padding");
		g_object_notify(G_OBJECT(widget), "children-allocation");
		tangle_widget_set_padding_set(widget, TRUE);

		g_object_thaw_notify(G_OBJECT(widget));
		
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

gboolean tangle_widget_get_padding_set(TangleWidget* widget) {

	return widget->priv->padding_set;
}

void tangle_widget_set_padding_set(TangleWidget* widget, gboolean is_set) {
	if (widget->priv->padding_set != is_set) {
		g_object_freeze_notify(G_OBJECT(widget));

		widget->priv->padding_set = is_set;
		g_object_notify(G_OBJECT(widget), "padding-set");

		/* TODO: Calculate aligned allocation */

		g_object_thaw_notify(G_OBJECT(widget));
	}
}

void tangle_widget_get_background_color(TangleWidget* widget, ClutterColor* color) {
	g_return_if_fail(color != NULL);

	*color = widget->priv->background_color;
}

gboolean tangle_widget_get_background_color_set(TangleWidget* widget) {

	return widget->priv->background_color_set;
}

void tangle_widget_set_background_color(TangleWidget* widget, const ClutterColor* color) {
	if (!clutter_color_equal(&widget->priv->background_color, color)) {
		if (color) {
			widget->priv->background_color = *color;
			widget->priv->background_color_set = TRUE;
		} else {
			widget->priv->background_color_set = FALSE;
		}
		g_object_notify(G_OBJECT(widget), "background-color");
		g_object_notify(G_OBJECT(widget), "background-color-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(widget));
	}
}

ClutterActor* tangle_widget_get_background_actor(TangleWidget* widget) {

	return widget->priv->background_actor;
}

void tangle_widget_set_background_actor(TangleWidget* widget, ClutterActor* actor) {	
	ClutterActorBox actor_box;

	if (widget->priv->background_actor != actor) {
		if (widget->priv->background_actor) {
			clutter_actor_unparent(widget->priv->background_actor);
			g_object_unref(widget->priv->background_actor);
		}
		widget->priv->background_actor = actor;
		g_object_ref_sink(widget->priv->background_actor);
		
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(widget), &actor_box);
		clutter_actor_allocate(actor, &actor_box, CLUTTER_ABSOLUTE_ORIGIN_CHANGED);
		clutter_actor_set_parent(actor, CLUTTER_ACTOR(widget));	
		g_object_notify(G_OBJECT(widget), "background-actor");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

gboolean tangle_widget_get_prefer_background_size(TangleWidget* widget) {

	return widget->priv->prefer_background_size;
}

void tangle_widget_set_prefer_background_size(TangleWidget* widget, gboolean prefer_background_size) {
	if (!widget->priv->prefer_background_size_set ||
	    widget->priv->prefer_background_size != prefer_background_size) {
		widget->priv->prefer_background_size = prefer_background_size;
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		tangle_widget_set_prefer_background_size_set(widget, TRUE);
		g_object_notify(G_OBJECT(widget), "prefer-background-size");
	}
}

gboolean tangle_widget_get_prefer_background_size_set(TangleWidget* widget) {

	return widget->priv->prefer_background_size_set;
}

void tangle_widget_set_prefer_background_size_set(TangleWidget* widget, gboolean is_set) {
	if (widget->priv->prefer_background_size_set != is_set) {
		widget->priv->prefer_background_size_set = is_set;
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
		g_object_notify(G_OBJECT(widget), "prefer-background-size-set");
	}
}

static void paint_actor_if_mapped(ClutterActor* actor) {
	if (CLUTTER_ACTOR_IS_MAPPED(actor)) {
		clutter_actor_paint(actor);
	}
}

static void tangle_widget_paint(ClutterActor* actor) {
	TangleWidget* widget;
	gdouble scale_x;
	gdouble scale_y;
	guint8 alpha;
	ClutterActorBox box;
	gboolean transition_scale_children;
	
	widget = TANGLE_WIDGET(actor);
	transition_scale_children = tangle_actor_should_transition_scale_children(TANGLE_ACTOR(widget));

	cogl_push_matrix();

	g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
	cogl_scale(scale_x, scale_y, 1.0);

	if (widget->priv->background_color_set) {
		alpha = clutter_actor_get_paint_opacity(actor) * widget->priv->background_color.alpha / 255;
		cogl_set_source_color4ub(widget->priv->background_color.red,
			 		 widget->priv->background_color.green,
			 		 widget->priv->background_color.blue,
					 alpha);
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
		cogl_rectangle(box.x1, box.y1, box.x2, box.y2);
	}
	if (widget->priv->background_actor) {
		clutter_actor_paint(widget->priv->background_actor);
	}

	if (!transition_scale_children) {
		cogl_pop_matrix();
	}

	g_list_foreach(TANGLE_WIDGET(actor)->priv->children, (GFunc)paint_actor_if_mapped, NULL);

	if (transition_scale_children) {
		cogl_pop_matrix();
	}
}

static void paint_actor_if_mapped_and_not_hiding(ClutterActor* actor) {
	if (CLUTTER_ACTOR_IS_MAPPED(actor) &&
	    (!TANGLE_IS_ACTOR(actor) || !TANGLE_ACTOR_IS_HIDING(TANGLE_ACTOR(actor)))) {
		clutter_actor_paint(actor);
	}
}

static void tangle_widget_pick(ClutterActor* actor, const ClutterColor *color) {
	GList* child_in_list;

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->pick(actor, color);

	g_list_foreach(TANGLE_WIDGET(actor)->priv->children,(GFunc)paint_actor_if_mapped_and_not_hiding, NULL);
}

static void tangle_widget_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	TangleWidget* widget;
	TangleSpacing spacing;
	gfloat min_width;
	gfloat natural_width;
	GList* child;
	gfloat child_x, child_min_width, child_natural_width;

	widget = TANGLE_WIDGET(actor);

	if (widget->priv->background_actor &&
	    ((widget->priv->prefer_background_size_set && widget->priv->prefer_background_size) ||
	     (!widget->priv->prefer_background_size_set && !widget->priv->children))) {
		if (TANGLE_IS_ACTOR(widget->priv->background_actor)) {
			tangle_actor_get_preferred_width(TANGLE_ACTOR(widget->priv->background_actor), for_height, interacting, min_width_p, natural_width_p, max_width_p);
		} else {
			clutter_actor_get_preferred_width(widget->priv->background_actor, for_height, min_width_p, natural_width_p);
			if (max_width_p) {
				*max_width_p = 0.0;
			}
		}
	} else {
		if (widget->priv->layout) {
			tangle_layout_get_preferred_width(widget->priv->layout, for_height, min_width_p, natural_width_p);
		} else {
			min_width = natural_width = 0;
			for (child = widget->priv->children; child; child = child->next) {
				child_x = clutter_actor_get_x(CLUTTER_ACTOR(child->data));
				if (TANGLE_IS_ACTOR(child->data)) {
					tangle_actor_get_preferred_size(TANGLE_ACTOR(child->data), interacting, &child_min_width, NULL, &child_natural_width, NULL, NULL, NULL);
				} else {
					clutter_actor_get_preferred_size(CLUTTER_ACTOR(child->data), &child_min_width, NULL, &child_natural_width, NULL);
				}
				if (child_x + child_min_width > min_width) {
					min_width = child_x + child_min_width;
				}
				if (child_x + child_natural_width > natural_width) {
					natural_width = child_x + child_natural_width;
				}
			}
			
			if (min_width_p) {
				*min_width_p = min_width;
			}
			if (natural_width_p) {
				*natural_width_p = natural_width;
			}
			if (max_width_p) {
				*max_width_p = 0.0;
			}
		}


		if (widget->priv->padding_set) {
			if (*min_width_p) {
				*min_width_p += widget->priv->padding.left + widget->priv->padding.right;
			}
			if (*natural_width_p) {
				*natural_width_p += widget->priv->padding.left + widget->priv->padding.right;
			}
		}
	}
}

static void tangle_widget_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	TangleWidget* widget;
	TangleSpacing spacing;
	gfloat min_height;
	gfloat natural_height;
	GList* child;
	gfloat child_x, child_min_height, child_natural_height;
	
	widget = TANGLE_WIDGET(actor);

	if (widget->priv->background_actor &&
	    ((widget->priv->prefer_background_size_set && widget->priv->prefer_background_size) ||
	     (!widget->priv->prefer_background_size_set && !widget->priv->children))) {
		if (TANGLE_IS_ACTOR(widget->priv->background_actor)) {
			tangle_actor_get_preferred_height(TANGLE_ACTOR(widget->priv->background_actor), for_width, interacting, min_height_p, natural_height_p, max_height_p);
		} else {
			clutter_actor_get_preferred_height(widget->priv->background_actor, for_width, min_height_p, natural_height_p);
			if (max_height_p) {
				*max_height_p = 0.0;
			}
		}
	} else {
		if (widget->priv->layout) {
			tangle_layout_get_preferred_height(widget->priv->layout, for_width, min_height_p, natural_height_p);
			if (max_height_p) {
				*max_height_p = 0;
			}
		} else {
			min_height = natural_height = 0;
			for (child = widget->priv->children; child; child = child->next) {
				child_x = clutter_actor_get_x(CLUTTER_ACTOR(child->data));
				if (TANGLE_IS_ACTOR(child->data)) {
					tangle_actor_get_preferred_size(TANGLE_ACTOR(child->data), interacting, NULL, &child_min_height, NULL, &child_natural_height, NULL, NULL);
				} else {
					clutter_actor_get_preferred_size(CLUTTER_ACTOR(child->data), NULL, &child_min_height, NULL, &child_natural_height);
				}
				if (child_x + child_min_height > min_height) {
					min_height = child_x + child_min_height;
				}
				if (child_x + child_natural_height > natural_height) {
					natural_height = child_x + child_natural_height;
				}
			}
			
			if (min_height_p) {
				*min_height_p = min_height;
			}
			if (natural_height_p) {
				*natural_height_p = natural_height;
			}
			if (max_height_p) {
				*max_height_p = 0.0;
			}
		}
			
		if (widget->priv->padding_set) {
			if (*min_height_p) {
				*min_height_p += widget->priv->padding.top + widget->priv->padding.bottom;
			}
			if (*natural_height_p) {
				*natural_height_p += widget->priv->padding.top + widget->priv->padding.bottom;
			}
		}
	}
}

static void tangle_widget_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleWidget* widget;
	ClutterActorBox actor_box;
	ClutterActorBox old_children_allocation;
	GList* child;

	widget = TANGLE_WIDGET(actor);

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->allocate(actor, box, flags);

	if (widget->priv->background_actor) {
		tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &actor_box);
		clutter_actor_allocate(widget->priv->background_actor, &actor_box, flags);
	}

	old_children_allocation = widget->priv->children_allocation;
	tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &widget->priv->children_allocation);
	if (widget->priv->padding_set) {
		widget->priv->children_allocation.x1 += widget->priv->padding.left;
		widget->priv->children_allocation.y1 += widget->priv->padding.top;
		widget->priv->children_allocation.x2 -= widget->priv->padding.right;
		widget->priv->children_allocation.y2 -= widget->priv->padding.bottom;
	}
	if (!clutter_actor_box_equal(&old_children_allocation, &widget->priv->children_allocation)) {
		g_object_notify(G_OBJECT(actor), "children-allocation");
	}

	if (widget->priv->layout) {
		tangle_layout_allocate(widget->priv->layout, &widget->priv->children_allocation, flags);
	} else {
		for (child = widget->priv->children; child; child = child->next) {
			clutter_actor_allocate_preferred_size(CLUTTER_ACTOR(child->data), flags);
		}
	}
}

static void tangle_widget_map(ClutterActor* actor) {
	TangleWidget* widget;
	TangleStylesheet* stylesheet;

	widget = TANGLE_WIDGET(actor);

	if (!widget->priv->stylesheet &&
	    (stylesheet = tangle_stylesheet_get_default())) {
		tangle_stylesheet_apply(stylesheet, G_OBJECT(actor));
	}

	CLUTTER_ACTOR_CLASS(tangle_widget_parent_class)->map(actor);

	if (widget->priv->background_actor) {
		clutter_actor_map(widget->priv->background_actor);
	}
}

static void tangle_widget_handle_ancestor_event(TangleActor* actor, TangleEvent* event) {
	tangle_widget_foreach_type(TANGLE_WIDGET(actor), TANGLE_TYPE_ACTOR, CLUTTER_CALLBACK(tangle_actor_handle_ancestor_event), event);

	TANGLE_ACTOR_CLASS(tangle_widget_parent_class)->handle_ancestor_event(actor, event);
}

static void tangle_widget_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleWidget* widget;
	
	widget = TANGLE_WIDGET(object);

	switch (prop_id) {
		case PROP_LAYOUT:
			tangle_widget_set_layout(widget, g_value_get_object(value));
			break;
		case PROP_BACKGROUND_COLOR:
			tangle_widget_set_background_color(widget, clutter_value_get_color(value));
			break;
		case PROP_BACKGROUND_ACTOR:
			tangle_widget_set_background_actor(widget, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_PREFER_BACKGROUND_SIZE:
			tangle_widget_set_prefer_background_size(widget, g_value_get_boolean(value));
			break;
		case PROP_PREFER_BACKGROUND_SIZE_SET:
			tangle_widget_set_prefer_background_size_set(widget, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_widget_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleWidget* widget;

	widget = TANGLE_WIDGET(object);

        switch (prop_id) {
		PROP_LAYOUT:
			g_value_set_object(value, tangle_widget_get_layout(widget));
			break;
		case PROP_CHILDREN_ALLOCATION:
			g_value_set_boxed(value, &widget->priv->children_allocation);
			break;
		case PROP_BACKGROUND_COLOR:
			clutter_value_set_color(value, &widget->priv->background_color);
			break;
		case PROP_BACKGROUND_ACTOR:
			g_value_set_object(value, tangle_widget_get_background_actor(widget));
			break;
		case PROP_PREFER_BACKGROUND_SIZE:
			g_value_set_boolean(value, widget->priv->prefer_background_size);
			break;
		case PROP_PREFER_BACKGROUND_SIZE_SET:
			g_value_set_boolean(value, widget->priv->prefer_background_size_set);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_widget_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_widget_parent_class)->finalize(object);
}

static void tangle_widget_dispose(GObject* object) {
	TangleWidget* widget;
	
	widget = TANGLE_WIDGET(object);
	
	/* TODO: layout */
	TANGLE_UNREF_AND_NULLIFY_OBJECT_POINTER(widget->priv->stylesheet);

	G_OBJECT_CLASS(tangle_widget_parent_class)->dispose(object);
}

static void tangle_widget_class_init(TangleWidgetClass* widget_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(widget_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(widget_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(widget_class);
	
	gobject_class->finalize = tangle_widget_finalize;
	gobject_class->dispose = tangle_widget_dispose;
	gobject_class->set_property = tangle_widget_set_property;
	gobject_class->get_property = tangle_widget_get_property;

	clutter_actor_class->paint = tangle_widget_paint;
	clutter_actor_class->pick = tangle_widget_pick;
	clutter_actor_class->allocate = tangle_widget_allocate;
	clutter_actor_class->map = tangle_widget_map;

	actor_class->get_preferred_width = tangle_widget_get_preferred_width;
	actor_class->get_preferred_height = tangle_widget_get_preferred_height;
	actor_class->handle_ancestor_event = tangle_widget_handle_ancestor_event;

	/**
	 * TangleWidget:layout:
	 */
	g_object_class_install_property(gobject_class, PROP_LAYOUT,
	                                g_param_spec_object("layout",
	                                "Layout",
	                                "The layout that is used to position children",
	                                TANGLE_TYPE_LAYOUT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:children-allocation:
	 */
	g_object_class_install_property(gobject_class, PROP_CHILDREN_ALLOCATION,
	                                g_param_spec_boxed("children-allocation",
	                                                   "Children allocation",
	                                                   "The allocated area for the children of the widget",
	                                                   CLUTTER_TYPE_ACTOR_BOX,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-color:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_COLOR,
	                                clutter_param_spec_color("background-color",
	                                                	 "Background color",
	                                                	 "The background color of the widget",
	                                                	 NULL,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-color_set:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_COLOR_SET,
	                                g_param_spec_boolean("background-color-set",
	                                "Background color set",
	                                "Whether the background color is set or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_ACTOR,
	                                g_param_spec_object("background-actor",
	                                "Background actor",
	                                "The actor that is used as a background of the widget",
	                                CLUTTER_TYPE_ACTOR,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:prefer-background-size:
	 *
	 * Overrides the usual size request that is based on the layout and the children of the widget.
	 * When TRUE (and #TangleWidget:prefer-background-size-set is TRUE), the size of the background actor
	 * is returned as a size request of the widget.
	 * 
	 * Writing this property sets the #TangleWidget:prefer-background-size-set property as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_PREFER_BACKGROUND_SIZE,
	                                g_param_spec_boolean("prefer-background-size",
	                                "Prefer background size",
	                                "Whether return the background size or the preferred size of children as a size request",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:prefer-background-size-set:
	 *
	 * Forces the widget to obey the value of the #TangleWidget:prefer-background-size property
	 * when returning a size request. When TRUE, the size request policy depends solely on the
	 * #TangleWidget:prefer-background-size property. When FALSE, the size request is
	 * normally based on the layout and the size requests of the children (like #TangleWidget:prefer-background-size
	 * was FALSE), but if the widget does not have any child, the size request of the background actor is used
	 * instead (like #TangleWidget:prefer-background-size was TRUE).
	 */
	g_object_class_install_property(gobject_class, PROP_PREFER_BACKGROUND_SIZE_SET,
	                                g_param_spec_boolean("prefer-background-size-set",
	                                "Prefer background size",
	                                "Whether to use the prefer-background-size property",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleWidgetPrivate));
}

static void tangle_widget_init(TangleWidget* widget) {
	widget->priv = G_TYPE_INSTANCE_GET_PRIVATE(widget, TANGLE_TYPE_WIDGET, TangleWidgetPrivate);
}

static void tangle_widget_add_impl(TangleWidget* widget, ClutterActor* actor, ClutterActor* existing, gboolean before) {
	GList* existing_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(clutter_actor_get_parent(actor) == NULL);
	g_return_if_fail(existing == NULL || CLUTTER_IS_ACTOR(existing));
	g_return_if_fail(existing == NULL || clutter_actor_get_parent(existing) == CLUTTER_ACTOR(widget));
	
	if (existing) {
		existing_in_list = g_list_find(widget->priv->children, existing);
		g_return_if_fail(existing_in_list != NULL);
	
		if (before) {
			widget->priv->children = g_list_insert_before(widget->priv->children, existing_in_list, actor);
		} else {
			if ((existing_in_list = g_list_next(existing_in_list))) {
				widget->priv->children = g_list_insert_before(widget->priv->children, existing_in_list, actor);
			} else {
				widget->priv->children = g_list_append(widget->priv->children, actor);
			}
		}
	} else {
		if (before) {
			widget->priv->children = g_list_prepend(widget->priv->children, actor);
		} else {
			widget->priv->children = g_list_append(widget->priv->children, actor);
		}		
	}

	clutter_actor_set_parent(actor, CLUTTER_ACTOR(widget));
	if (widget->priv->stylesheet) {
		tangle_stylesheet_apply(widget->priv->stylesheet, G_OBJECT(actor));
	}
	g_signal_emit_by_name(widget, "actor-added", actor);
	clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
}

static void tangle_widget_add(ClutterContainer* widget, ClutterActor* actor) {	
	tangle_widget_add_impl(TANGLE_WIDGET(widget), actor, NULL, FALSE);
}

static void tangle_widget_remove(ClutterContainer* widget, ClutterActor* actor) {
	gboolean was_visible;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(clutter_actor_get_parent(actor) == CLUTTER_ACTOR(widget));

	was_visible = CLUTTER_ACTOR_IS_VISIBLE(actor);

	TANGLE_WIDGET(widget)->priv->children = g_list_remove(TANGLE_WIDGET(widget)->priv->children, actor);
	if (TANGLE_WIDGET(widget)->priv->stylesheet) {
		tangle_stylesheet_unapply(TANGLE_WIDGET(widget)->priv->stylesheet, G_OBJECT(actor));
	}
	clutter_actor_unparent(actor);
	g_signal_emit_by_name(widget, "actor-removed", actor);
	if (was_visible) {
		clutter_actor_queue_relayout(CLUTTER_ACTOR(widget));
	}
}

static void tangle_widget_foreach(ClutterContainer* widget, ClutterCallback callback, gpointer user_data) {
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WIDGET(widget));

	for (child_in_list = TANGLE_WIDGET(widget)->priv->children; child_in_list; child_in_list = child_in_list->next) {
		callback(CLUTTER_ACTOR(child_in_list->data), user_data);
	}
}

static void clutter_container_iface_init(ClutterContainerIface* iface) {
	iface->add = tangle_widget_add;
	iface->remove = tangle_widget_remove;
	iface->foreach = tangle_widget_foreach;
}

static gboolean tangle_widget_apply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	TangleWidget* widget;
	GList* link;
	
	widget = TANGLE_WIDGET(stylable);
	if (!widget->priv->stylesheet) {
		tangle_widget_foreach_swapped(widget, TANGLE_SWAPPED_CALLBACK(tangle_stylesheet_apply), stylesheet);
		widget->priv->stylesheet = stylesheet;
		g_object_ref(stylesheet);
	}

	return FALSE;
}

static gboolean tangle_widget_unapply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	TangleWidget* widget;
	GList* link;
	
	widget = TANGLE_WIDGET(stylable);
	if (widget->priv->stylesheet == stylesheet) {
		tangle_widget_foreach_swapped(widget, TANGLE_SWAPPED_CALLBACK(tangle_stylesheet_unapply), stylesheet);
		widget->priv->stylesheet = NULL;
		g_object_unref(widget->priv->stylesheet);
	}

	return TRUE;
}

static void tangle_stylable_iface_init(TangleStylableIface* iface) {
	iface->apply_stylesheet = tangle_widget_apply_stylesheet;
	iface->unapply_stylesheet = tangle_widget_unapply_stylesheet;
}
