/*
 * tangle-warehouse.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-warehouse.h"

/**
 * SECTION:tangle-warehouse
 * @Short_description: A non-actor container for #ClutterActor objects
 * @Title: TangleWarehouse
 *
 * #TangleWarehouse implements #ClutterContainer interface. Thus, it can
 * hold any #ClutterActor objects. It can be utilized in #ClutterScript,
 * and children can be added with the special the special "children" property.
 *
 * However, #TangleWarehouse does not inherit #ClutterActor. Thus,
 * it cannot appear in a stage, and does not draw its children. In addition,
 * #TangleWarehouse does not take parentship of children.
 *
 * A #ClutterActor can be in one #TangleWarehouse at the time.
 */

static void clutter_container_iface_init(ClutterContainerIface* iface);

G_DEFINE_TYPE_WITH_CODE(TangleWarehouse, tangle_warehouse, G_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_CONTAINER, clutter_container_iface_init););

enum {
	PROP_0,
};

struct _TangleWarehousePrivate {
	GList* children;
};

static GQuark quark_warehouse_parent = 0;

static void unref_child(GObject* object);

TangleWarehouse* tangle_warehouse_new(void) {

	return TANGLE_WAREHOUSE(g_object_new(TANGLE_TYPE_WAREHOUSE, NULL));
}

TangleWarehouse* tangle_actor_get_warehouse_parent(ClutterActor* actor) {

	return TANGLE_WAREHOUSE(g_object_get_qdata(G_OBJECT(actor), quark_warehouse_parent));
}

static void tangle_warehouse_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleWarehouse* warehouse;
	
	warehouse = TANGLE_WAREHOUSE(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_warehouse_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleWarehouse* warehouse;

	warehouse = TANGLE_WAREHOUSE(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_warehouse_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_warehouse_parent_class)->finalize(object);
}

static void tangle_warehouse_dispose(GObject* object) {
	TangleWarehouse* warehouse;
	
	warehouse = TANGLE_WAREHOUSE(object);
	
	g_list_foreach(warehouse->priv->children, (GFunc)unref_child, warehouse);
	g_list_free(warehouse->priv->children);
	warehouse->priv->children = NULL;

	G_OBJECT_CLASS(tangle_warehouse_parent_class)->dispose(object);
}

static void tangle_warehouse_class_init(TangleWarehouseClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = tangle_warehouse_finalize;
	gobject_class->dispose = tangle_warehouse_dispose;
	gobject_class->set_property = tangle_warehouse_set_property;
	gobject_class->get_property = tangle_warehouse_get_property;

	quark_warehouse_parent = g_quark_from_static_string("tangle-warehouse-parent");

	g_type_class_add_private (gobject_class, sizeof (TangleWarehousePrivate));
}

static void tangle_warehouse_init(TangleWarehouse* warehouse) {
	warehouse->priv = G_TYPE_INSTANCE_GET_PRIVATE(warehouse, TANGLE_TYPE_WAREHOUSE, TangleWarehousePrivate);
}

static void tangle_warehouse_add(ClutterContainer* container, ClutterActor* actor) {	
	TangleWarehouse* warehouse;
	
	g_return_if_fail(TANGLE_IS_WAREHOUSE(container));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(tangle_actor_get_warehouse_parent(actor) == NULL);

	warehouse = TANGLE_WAREHOUSE(container);

	if (g_list_find(warehouse->priv->children, actor)) {
		g_warning("Cannot add same actor more than once.");
	} else {
		g_object_ref(G_OBJECT(actor));
		g_object_set_qdata(G_OBJECT(actor), quark_warehouse_parent, warehouse);
		warehouse->priv->children = g_list_prepend(warehouse->priv->children, actor);
	}
}

static void tangle_warehouse_remove(ClutterContainer* container, ClutterActor* actor) {
	TangleWarehouse* warehouse;
	GList* child_in_list;
	
	g_return_if_fail(TANGLE_IS_WAREHOUSE(container));
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));

	warehouse = TANGLE_WAREHOUSE(container);

	if (!(child_in_list = g_list_find(warehouse->priv->children, actor))) {
		g_warning("Cannot remove actor that is not child.");
	} else {
		unref_child(G_OBJECT(child_in_list->data));
		warehouse->priv->children = g_list_delete_link(warehouse->priv->children, child_in_list);
	}
}

static void tangle_warehouse_foreach(ClutterContainer* container, ClutterCallback callback, gpointer user_data) {
	TangleWarehouse* warehouse;
	GList* child_in_list;

	g_return_if_fail(TANGLE_IS_WAREHOUSE(container));

	warehouse = TANGLE_WAREHOUSE(container);

	for (child_in_list = warehouse->priv->children; child_in_list; child_in_list = child_in_list->next) {
		callback(CLUTTER_ACTOR(child_in_list->data), user_data);
	}
}

static void clutter_container_iface_init(ClutterContainerIface* iface) {
	iface->add = tangle_warehouse_add;
	iface->remove = tangle_warehouse_remove;
	iface->foreach = tangle_warehouse_foreach;
}

static void unref_child(GObject* object) {
	g_object_set_qdata(object, quark_warehouse_parent, NULL);
	g_object_unref(object);
}
