#include "MaemoApplicationHandler.h"


#include <QFile>
#include <QFSFileEngine>
#include <QDesktopServices>
#include <QIODevice>
#include <QDebug>


// file installer
#ifdef Q_WS_MAEMO_5

    #include <hildon-mime.h>
    #include <iostream>
    #include <stdlib.h>
#endif
// file installer end

const int MaemoApplicationHandler::LoggedIn = 1;
const int MaemoApplicationHandler::LoggedInConnectionError = 2;
const int MaemoApplicationHandler::LoggedInCredentialsError = 3;

/***  
  MaemoApplicationHandler offers the communication interface for the maemo.org
  REST API. At the moment it is done in a way, that it requests every time the data
  from the server. As this is asynchronous the interface should handle the loading time
  some how. So the request is splitted up in the function and a signal.

  @todo: - handle caching on device
         - error handling
  ***/

MaemoApplicationHandler::MaemoApplicationHandler (QString proxy,int port, int timeout) {
    connector = new HttpConnector(timeout);
    requestQueue = new QList<QueueElement*>;
    imageInUse = new QueueElement();
    queueConnected = false;
    OCS_URL="http://maemo.org/ocs/v1/content/";
    PRODUCT_URL="http://maemo.org/downloads/api/product/get/";
    //COMMENT_URL="https://maemo.org/downloads/product/Maemo5/";
    mode = "testing";
    if (!proxy.isEmpty() && port!=0) {
        connector->setProxy(proxy,port);
    }
}


/***
  requests a list of all available applications:
  @todo: parameters: os,category,page,pagesize,sortedBy,keyword,filter

  - parses the response by using (private) parseAppListResponse
  - signal when request was successful: ready
  ***/
void MaemoApplicationHandler::requestApplications(   QString OS,
                                                     QString cat,
                                                     int page,
                                                     int pageSize,
                                                     QString keyword,
                                                     QString sortmode) {
    qDebug() << "request Application: " + OS + "Category: " + cat;
    QString query = queryBuilder (OS,cat,page,pageSize,keyword,sortmode);
    QString url = OCS_URL +"data/" + query;
    qDebug()<<"URL requested: " + url;
    /***
      check if connector can handle request, otherwise add to queue
      high priority, set at begin of queue
      **/
    if (connector->isBusy()) {
        qDebug () << "connector busy - append to queue as top element";
        if (!queueConnected) {
            queueConnected = true;
            connect (connector, SIGNAL(isFree()),this,SLOT (proceedQueue()));
        }
        requestQueue->insert(requestQueue->begin(),new QueueElement (url,"","",QueueElement::LIST));
    } else {
        qDebug () << "connector free - send request";
        connector->getRequest (url);
        connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseAppListResponse(bool,QString*)));
    }
}


void MaemoApplicationHandler::requestApplications(RestRequest *req) {
    this->requestApplications(req->os(),req->category(),req->page(),req->pageSize(),req->keword(),req->sorting());
}


/***
  requests the application data of a single application

  - parses the response by using (private) parseSingleAppResponse
  - signal for a successfull request: detailsReady
  ***/
void MaemoApplicationHandler::requestApplicationData(QString uid) {
    QString url = PRODUCT_URL+uid+"/";
    qDebug ()<<"applicationDetails: uid:"<< uid.data() << "url: "<<url.data();
    /***
      request send if connector is free, otherwise send to queue (priority)
      **/
    if (connector->isBusy()) {
        qDebug () << "connector busy - add request to queue";
        if (!queueConnected) {
            queueConnected = true;
            connect (connector, SIGNAL(isFree()),this,SLOT (proceedQueue()));
        }
        requestQueue->insert (requestQueue->begin(), new QueueElement (url,"","",QueueElement::DETAILS));
    } else {
        connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseSingleAppResponse(bool,QString*)));
        connector->getRequest(url);
    }

}

/***
  request categories
  ***/
void MaemoApplicationHandler::requestCategories(QString OS) {
   QString url = OCS_URL + "categories?parent=" + OS;
   /***
     check if connector can handle request, otherwise add to queue
     high priority, set at begin of queue
     **/
   qDebug () << "Categories requested: " << url;
   if (connector->isBusy()) {
       qDebug () << "connector busy - append to queue as top element";
       if (!queueConnected) {
           queueConnected = true;
           connect (connector, SIGNAL (isFree()),this,SLOT(proceedQueue()));
       }
       requestQueue->insert(requestQueue->begin(),new QueueElement (url,"","",QueueElement::LIST));
   }
   else {
       qDebug () << "free, send request";
       connector->getRequest (url);
       connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseCategoryResponse(bool,QString*)));
   }
}

/***
  request an image

  @param: QString url -> url to the image

  - signal for a successful request: imageReady
  ***/
void MaemoApplicationHandler::requestImage (QString url, QString id, QString info, bool addToQueue) {
    if (!connector->isBusy()) {
        imageInUse->setURL(url);
        imageInUse->setID(id);
        imageInUse->setInfo(info);
        connect (connector, SIGNAL(getRawData(bool,QByteArray)),this,SLOT(proceedImageResponse(bool,QByteArray)));
        connector->getRequest(url);
    } else {
        if (!queueConnected) {
            queueConnected = true;
            connect (connector, SIGNAL(isFree()),this,SLOT (proceedQueue()));
        }
        if (addToQueue) {
            /***
              set requested images from the details-view to higher priority
              ***/
            if (info=="DETAILS"){
                requestQueue->insert(requestQueue->begin(),new QueueElement (url,id,info,QueueElement::IMAGE));
            } else {
                requestQueue->append(new QueueElement (url,id,info,QueueElement::IMAGE));
            }
        }
    }
}

/**
  Sends a request to login the user onto maemo.org
    @param: QString uname: username
    @param: QString pw: password

    emits signal loggedin (QString, bool)
**/

void MaemoApplicationHandler::requestLogin(QString uname, QString pw) {
    qDebug() << "request login: uname: " << uname;
    username = uname;
    QByteArray postParams;
    postParams.append("midcom_services_auth_frontend_form_submit=Login&");
    postParams.append("username="+uname+"&");
    postParams.append("password="+pw);

    if (!connector->isBusy()) {
        connector->postRequest(QUrl ("https://maemo.org/"),postParams);
        connect (connector,SIGNAL(getHttpResponseCode(int)),this,SLOT(loginResponse(int)));
    }
    else {
        emit loggedIn (LoggedInConnectionError);
    }
}

/**
  Returns true if a coockie is saved, false if not
**/
bool MaemoApplicationHandler::isUserLoggedIn() {
    return connector->hasCookieForUrl(QUrl ("https://maemo.org"));
}

/**
  Return username if the user is logged in
  **/
QString MaemoApplicationHandler::userName () {
    if (connector->hasCookieForUrl(QUrl ("https://maemo.org"))&&!username.isEmpty())
        return username;
    else
        return QString();

}


/**
  deletes the cookies of the connection to log the user out
 **/
void MaemoApplicationHandler::logOut() {
    connector->resetCookies();
}


void MaemoApplicationHandler::emptyQueue () {
    qDebug ("request to empty the queue");
    requestQueue->clear();
    queueConnected = false;
    disconnect (connector, SIGNAL(isFree()),this,SLOT (proceedQueue()));
    qDebug ("done");
}

/***
  slot, which handles the queue, called, when connector has handled a request
  **/
void MaemoApplicationHandler::proceedQueue () {
    //qDebug ()<<"proceed with next queueElement; Rest in queue: " << QString::number(requestQueue->count());
     if (requestQueue->count()>0) {
        QueueElement *q = requestQueue->takeFirst();
        if (q->getType()==QueueElement::IMAGE) {
            requestImage (q->getURL(),q->getID(),q->getInfo(),false);
        } else if (q->getType()==QueueElement::LIST) {
            connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseAppListResponse(bool,QString*)));
            connector->getRequest(q->getURL());
        } else if (q->getType()==QueueElement::DETAILS) {
            connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseSingleAppResponse(bool,QString*)));
            connector->getRequest(q->getURL());
        }
        delete q;
     } else {
        queueConnected = false;
        disconnect (connector, SIGNAL(isFree()),this,SLOT (proceedQueue()));
     }

 }

/***
  request the install file of an application

  @param: QString url -> url to the install file

  @todo: function at the moment just a placeholder. First http handler has to be fixed.
  ***/

void MaemoApplicationHandler::requestInstallFile (QString url) {
    connector->getRequest (url);
    QTextStream out (stdout);
    out <<"*********"+url << endl;
    connect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(proceedInstallFileResponse(bool,QString*)));
}


/***
  response of the image-request is proceeded and QByteArray is transformed as an image
  @todo: check if image is in a valid format
  ***/
void MaemoApplicationHandler::proceedImageResponse (bool err,QByteArray img)  {
    disconnect (connector, SIGNAL(getRawData(bool,QByteArray)),this,SLOT(proceedImageResponse(bool,QByteArray)));
    if (err) {
        emit raiseError (new QString("Connection error"));
    } else {
        QPixmap *pic = new QPixmap();
        pic->loadFromData (img);
        emit imageReady(pic,imageInUse->getID(),imageInUse->getInfo());
        delete pic;
    }
}

/***
  proceed the install file and initiate the installation
  @todo: everything, just a bit more than a placeholder up to now
***/
void MaemoApplicationHandler::proceedInstallFileResponse (bool error, QString *content) {
    disconnect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(proceedInstallFileResponse(bool,QString*)));
    #ifdef Q_WS_MAEMO_5
        QFile file ("/opt/appdownloader/meta/application.install");
    #else
        QFile file ("~/.maemoclient/test.install");
    #endif
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
            qDebug() << "fileerror";
        }
        QTextStream file_writer (&file);
        file_writer << *content << endl;
        file.close();
    #ifdef Q_WS_MAEMO_5
        const int suc = 1;
        int result = 0;
        DBusError err;
        dbus_error_init(&err);
        DBusConnection *connection = dbus_bus_get(DBUS_BUS_SESSION, &err);
        if (dbus_error_is_set(&err)) {
                dbus_error_free(&err);
                qDebug() << "error";
        }
        if (!connection) {
                qDebug() << "error2";
        }
        qDebug() << file.fileName();
        result = 1;
        #ifdef Q_WS_MAEMO_5
          result=hildon_mime_open_file(connection,"/opt/appdownloader/meta/application.install");
        #endif
        qDebug() << result;
        if (result != suc) {http://talk.maemo.org/showthread.php?t=50753
                qDebug() << "did not work";
        }
    #endif
}

/**
  receive the http-code from the login request
  @param int code: http code

  **/

void MaemoApplicationHandler::loginResponse(int code) {
    disconnect (connector,SIGNAL(getHttpResponseCode(int)),this,SLOT(loginResponse(int)));
    if (code==302) {
        emit loggedIn (LoggedIn);
    }
    else if (code==403){
        emit loggedIn (LoggedInCredentialsError);
    }
    else {
        emit loggedIn (LoggedInConnectionError);
    }
}

/**
  comment/rate an application:
    - first check if logged in - error otherwise
    - then build the url
    - post the request
 **/
void MaemoApplicationHandler::commentApplication(int rating, QString comment, QString applicationname) {
    if (isUserLoggedIn()) {
        QByteArray postData;
        postData.append("_qf__net_nehmer_comments=&");
        postData.append("content="+comment+"&");
        postData.append("rating="+ QString::number(rating) +"&");
        postData.append("midcom_helper_datamanager2_save=Send");


        // has to be added to the queue as well
        if (!connector->isBusy()) {
            connector->postRequest(QUrl ("https://maemo.org/downloads/product/Maemo5/"+applicationname),postData);
            connect (connector,SIGNAL(getHttpResponseCode(int)),this,SLOT(commentResponse(int)));
        }

    } else {
        emit commented (NotAuthorized);
    }

}


/**
  forward the response
  **/
void MaemoApplicationHandler::commentResponse(int resp) {
    disconnect (connector,SIGNAL(getHttpResponseCode(int)),this,SLOT(commentResponse(int)));
    if (resp==302)
        emit commented(CommentSuccess);
    else
        emit commented(CommentConnectionError);
}

/***
  xml of application list is parsed and a list of applications given back
  by the ready signal
 ***/
void MaemoApplicationHandler::parseAppListResponse(bool error,QString *content) {
    QTextStream out (stdout);
    disconnect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseAppListResponse(bool,QString*)));
    if (error) {
        out << error << endl;
        if (mode=="testing") {
            QList<MaemoApplication *> *appList = new QList<MaemoApplication *>();
            appList->append(new MaemoApplication (QString("test1"),QString("e84d47088d0a11dc9a1f9b5a44045e485e48"),QString("test1"),QString("3"),QString("6000"),QString("3")));
            appList->append(new MaemoApplication (QString("test2"),QString("64950be6721711deb1ed2953a550f700f700"),QString("test2"),QString("2"),QString("5000"),QString("1")));
            appList->append(new MaemoApplication (QString("test3"),QString("e84d47088d0a11dc9a1f9b5a44045e485e48"),QString("test3"),QString("4"),QString("4000"),QString("5")));
            appList->append(new MaemoApplication (QString("test4"),QString("64950be6721711deb1ed2953a550f700f700"),QString("test4"),QString("5"),QString("3000"),QString("0")));
            appList->append(new MaemoApplication (QString("test5"),QString("e84d47088d0a11dc9a1f9b5a44045e485e48"),QString("test5"),QString("1"),QString("1000"),QString("2")));
            emit applicationListReady (appList);
        }
        else {
            emit raiseError (content);
        }
    } else {
        xmlReader.clear();
        xmlReader.addData (*content);

        int i=0;
        QString uid;
        QString name;
        QString version;
        QString comments;
        QString downloads;
        QString title;
        QString rating;
        QString pre;
        QString bigPre;

        QList<MaemoApplication *> *appList = new QList<MaemoApplication *>();
        while (!xmlReader.atEnd()) {
           xmlReader.readNext();
            if (xmlReader.isStartElement()) {
                if (xmlReader.name() == "id") {
                    xmlReader.readNext();
                    out << "uid: " + xmlReader.text().toString() << endl;
                    uid = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "version") {
                    xmlReader.readNext();
                    out << "version: " + xmlReader.text().toString() << endl;
                    version = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "comments") {
                    xmlReader.readNext();
                    out << "comments: " + xmlReader.text().toString() << endl;
                    comments = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "downloads") {
                    xmlReader.readNext();
                    out << "downloads: " + xmlReader.text().toString() << endl;
                    downloads = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "score") {
                    xmlReader.readNext();
                    out << "rating: " + xmlReader.text().toString() << endl;
                    rating = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "name") {
                    xmlReader.readNext();
                    out << "title: " + xmlReader.text().toString() << endl;
                    title = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "smallpreviewpic1") {
                    xmlReader.readNext();
                    out << "preview-url: " + xmlReader.text().toString() << endl;
                    pre = xmlReader.text().toString();
                }
                else if (xmlReader.name() == "previewpic1") {
                    xmlReader.readNext();
                    bigPre = xmlReader.text().toString();
                }
            } else if (xmlReader.isEndElement()) {
                if (xmlReader.name() == "content") {
                    out << "end: " + xmlReader.name().toString()<< endl;
                    out << "************************************"<< endl;
                    out << QString::number(i)<< endl;
                    appList->append(new MaemoApplication (name,uid,title,rating,downloads,comments,version,pre,bigPre));
                    out << QString::number(appList->length())<< endl;
                    out << "************************************"<< endl;
                    uid="";
                    name="";
                    version="";
                    comments="";
                    downloads="";
                    title="";
                    rating="";
                    pre="";
                    bigPre = "";
                    i++;
                }
            }
        }
        emit applicationListReady (appList);
    }
    delete content;

}

/***
  xml of single application is parsed and a MaemoApplication is returned
  ***/
void MaemoApplicationHandler::parseSingleAppResponse(bool err, QString *content) {
    //@todo
    disconnect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseSingleAppResponse(bool,QString*)));
    QTextStream out (stdout);
    out << *content <<endl;
    if (err) {
        out << "error" << endl;
        if (mode=="testing") {
            MaemoApplicationDetails *appDetails = new MaemoApplicationDetails();
            appDetails->setName ("TESTAPP");
            appDetails->setRating ("4");
            appDetails->setDesc ("THIS IS A TEST!! CONNECTION ERROR FORCES TO SHOW FAKED DATA! :)");
            appDetails->setStatus ("stable");
            appDetails->setDownloads ("1111");
            appDetails->setInstallFile("a");
            emit applicationDetailsReady(appDetails);

        } else {
            emit raiseError(content);
        }
        out << "error2" << endl;
    } else {
        xmlReader.clear();
        xmlReader.addData (*content);
        bool screenshots = false;
        MaemoApplicationDetails *appDetails = new MaemoApplicationDetails();
        MaemoComment *userComment;
        while (!xmlReader.atEnd()) {
            xmlReader.readNext();
            // get the values for the interesting tags
            if (xmlReader.isStartElement()) {
                if (!screenshots) {
                    if (xmlReader.name()=="guid") {
                        xmlReader.readNext();
                        appDetails->setUid(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="unixname") {
                        xmlReader.readNext();
                        appDetails->setName(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="title") {
                        xmlReader.readNext();
                        appDetails->setTitle(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="description") {
                        xmlReader.readNext();
                        appDetails->setDesc(formalizeQString (xmlReader.text().toString()));
                    }
                    else if (xmlReader.name()=="licence") {
                        xmlReader.readNext();
                        appDetails->setLicence(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="version") {
                        xmlReader.readNext();
                       appDetails->setVersion(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="status") {
                        xmlReader.readNext();
                        appDetails->setStatus(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="installfile") {
                        xmlReader.readNext();
                        appDetails->setInstallFile(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="editable_install") {
                        xmlReader.readNext();
                        appDetails->setInstallFileContent(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="click_counter") {
                        xmlReader.readNext();
                        appDetails->setDownloads(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="revisor") {
                        xmlReader.readNext();
                        appDetails->setRevisor(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="rating") {
                        xmlReader.readNext();
                        appDetails->setRating(xmlReader.text().toString());
                    }
                    else if (xmlReader.name()=="screenshots") {
                        screenshots=true;
                        out << "start screenshots" << endl;
                    }
                    else if (xmlReader.name ()=="comments") {
                        xmlReader.readNext();
                        while (!(xmlReader.isEndElement()&&xmlReader.name()=="comments")) {
                            userComment = new MaemoComment();
                            if (xmlReader.name ()=="value"&&xmlReader.isStartElement()) {
                                while (!(xmlReader.isEndElement()&&xmlReader.name()=="value")) {
                                    //add option
                                    //if (xmlReader.isStartElement()) {
                                    if (false) {
                                        if (xmlReader.name () == "guid") {
                                            xmlReader.readNext();
                                            out << "comment-uid " + xmlReader.text().toString ()<< endl;
                                            userComment -> setUid (xmlReader.text().toString ());
                                        }
                                        else if (xmlReader.name () == "author") {
                                            xmlReader.readNext();
                                            if (!xmlReader.text().toString ().isEmpty()) {
                                                out << "comment-author " + xmlReader.text().toString ()<< endl;
                                                userComment -> setAuthor (xmlReader.text().toString ());
                                            }
                                        }
                                        else if (xmlReader.name() == "content") {
                                            xmlReader.readNext();
                                            if (!xmlReader.text().toString ().isEmpty()) {
                                                out << "comment-content " + xmlReader.text().toString ()<< endl;
                                                userComment -> setContent (xmlReader.text().toString());
                                            }
                                        }
                                        else if (xmlReader.name() == "published") {
                                            xmlReader.readNext();
                                            if (!xmlReader.text().toString ().isEmpty()) {
                                                out << "comment-published " + xmlReader.text().toString ()<< endl;
                                                userComment -> setPublished (xmlReader.text().toString ());
                                            }
                                        }
                                        else if (xmlReader.name() == "rating") {
                                            xmlReader.readNext();
                                            if (!xmlReader.text().toString ().isEmpty()) {
                                                out << "comment-rating " + xmlReader.text().toString ()<< endl;
                                                userComment -> setRating (xmlReader.text().toString());
                                            }
                                        }
                                    }
                                    xmlReader.readNext();
                                }
                            }
                            if (xmlReader.isEndElement()&&xmlReader.name()=="value") {
                                if (!userComment->getAuthor().isEmpty() && !userComment->getContent().isEmpty()) {
                                    out << "comment added" << endl;
                                    appDetails->addUserComment (userComment);
                                }
                            }
                            xmlReader.readNext();
                        }
                    }
                }
                else if (screenshots) {
                    if (xmlReader.name()=="value") {
                        xmlReader.readNext();
                        appDetails->addScreenshot(xmlReader.text().toString());
                    }
                }

            } else if (xmlReader.isEndElement()) {
                if (screenshots && xmlReader.name()=="screenshots") {
                    screenshots=false;
                }
            }
        }
        out << appDetails->toString() << endl;
        appDetails->setUnparsedComments (content);
        emit applicationDetailsReady(appDetails);
    }

}

void MaemoApplicationHandler::parseCommentsOnly (QString *com) {
     QTextStream out (stdout);
     xmlReader.clear();
     xmlReader.addData (*com);
     MaemoComment *userComment;
     QList <MaemoComment*>* comments =  new QList <MaemoComment *> ();
     while (!xmlReader.atEnd()) {
        xmlReader.readNext();
        // get the values for the interesting tags
        if (xmlReader.isStartElement()&&xmlReader.name()=="comments") {
             xmlReader.readNext();
             while (!(xmlReader.isEndElement()&&xmlReader.name()=="comments")) {
                out << "1st while start" + xmlReader.name().toString()<< endl;
                out << "1st while" + xmlReader.name().toString()<< endl;
                userComment = new MaemoComment();
                if (xmlReader.name ()=="value"&&xmlReader.isStartElement()) {
                    while (!(xmlReader.isEndElement()&&xmlReader.name()=="value")) {
                        if (xmlReader.isStartElement()) {
                            if (xmlReader.name () == "guid") {
                                xmlReader.readNext();
                                out << "comment-uid " + xmlReader.text().toString ()<< endl;
                                userComment -> setUid (xmlReader.text().toString ());
                            }
                            else if (xmlReader.name () == "author") {
                                xmlReader.readNext();
                                if (!xmlReader.text().toString ().isEmpty()) {
                                    out << "comment-author " + xmlReader.text().toString ()<< endl;
                                    userComment -> setAuthor (xmlReader.text().toString ());
                                }
                            }
                            else if (xmlReader.name() == "content") {
                                xmlReader.readNext();
                                if (!xmlReader.text().toString ().isEmpty()) {
                                    out << "comment-content " + xmlReader.text().toString ()<< endl;
                                    userComment -> setContent (xmlReader.text().toString());
                                }
                            }
                            else if (xmlReader.name() == "published") {
                                xmlReader.readNext();
                                if (!xmlReader.text().toString ().isEmpty()) {
                                    out << "comment-published " + xmlReader.text().toString ()<< endl;
                                    userComment -> setPublished (xmlReader.text().toString ());
                                }
                            }
                            else if (xmlReader.name() == "rating") {
                                xmlReader.readNext();
                                if (!xmlReader.text().toString ().isEmpty()) {
                                    out << "comment-rating " + xmlReader.text().toString ()<< endl;
                                    userComment -> setRating (xmlReader.text().toString());
                                }
                            }
                        }
                        xmlReader.readNext();
                    }
                }
                if (xmlReader.isEndElement()&&xmlReader.name()=="value") {
                    if (!userComment->getAuthor().isEmpty() && !userComment->getContent().isEmpty()) {
                        out << "comment added" << endl;
                        comments->append (userComment);
                    }
                }
                xmlReader.readNext();
            }

         }
    }
    emit commentsParsed (comments);
 }


void MaemoApplicationHandler::parseCategoryResponse(bool err, QString *content) {
    disconnect (connector, SIGNAL(getResponseContent(bool,QString*)),this,SLOT(parseCategoryResponse(bool,QString*)));
    if (err) {
        emit raiseError (content);
    }
    else {
        qDebug () << "start parsing";
        xmlReader.clear();
        xmlReader.addData(*content);
        QList <MaemoCategory*>* catList = new QList <MaemoCategory*>();
        QString id;
        QString name;
        while (!xmlReader.atEnd()) {
            xmlReader.readNext();
            if (xmlReader.isStartElement()) {
                if (xmlReader.name()=="category") {
                    while (!(xmlReader.isEndElement()&&xmlReader.name()=="category")) {
                        xmlReader.readNext();
                        if (xmlReader.isStartElement()&&xmlReader.name()=="id") {
                            xmlReader.readNext();
                            id = xmlReader.text().toString();
                        }
                        else if (xmlReader.isStartElement()&&xmlReader.name()=="name") {
                            xmlReader.readNext();
                            name = xmlReader.text().toString();
                        }
                    }
                    catList->append(new MaemoCategory(id,name));
                }

            }
        }
        emit categoriesReady(catList);
    }
    delete content;
}


QString MaemoApplicationHandler::queryBuilder ( QString OS,
                                                QString cat,
                                                int page,
                                                int pageSize,
                                                QString keyword,
                                                QString sortmode) {

    QString query="";
    // add parameter for the OS
    if (!OS.isEmpty()) {
        query+="?parent="+OS;
    }
    // add parameter for the category
    if (!cat.isEmpty()) {
        if (query.isEmpty()) {
            query+="?categories="+cat;
        } else {
            query+="&categories="+cat;
        }
    }
    // add parameter for the page size
    if (pageSize>0) {
        if (page <= 0) {
            page = 0;
        }
        if (query.isEmpty ()) {
            query+="?pagesize="+QString::number (pageSize)+"&page="+QString::number(page);
        } else {
            query+="&pagesize="+QString::number (pageSize)+"&page="+QString::number(page);
        }
    }
    // add parameter for the keyword query
    if (!keyword.isEmpty ()) {
        if (query.isEmpty()) {
            query += "?search="+keyword;
        } else {
            query += "&search="+keyword;
        }
    }
    //add sorting parameter to the query
    if (!sortmode.isEmpty()) {
        if (query.isEmpty ()) {
            query += "?sortmode="+sortmode;
        } else {
            query += "&sortmode="+sortmode;
        }
    }
    return query;
}

QString MaemoApplicationHandler::formalizeQString (QString s) {
    QTextStream out (stdout);
    s = s.replace ("\n"," ");
    return s;
}

void MaemoApplicationHandler::setHandlerMode(const QString& handlerMode)
{
    mode = handlerMode;
}


