/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003-2011 LUT. .
 *
 * @name Daemon.cc
 * @memo Implementation of the PeerHood Daemon class.
 *
 * @version 0.32
 * date     03.04.2003
 * change   19.04.2011
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include <fstream>
#include <csignal>
#include <cassert>
#include <fcntl.h>
#include <dlfcn.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <net/if.h> 
#include <pdu.h>
#include <stdlib.h>
#include "AbstractPlugin.h"
#include "DeviceStorage.h"
#include "Daemon.h"
#include "ListenerFactory.h"
#include "IFSearch.h"
#include "PluginStorage.h"

const char* PID_FILE_NAME     = "/var/run/phd.pid";
const char* LOCAL_SOCKET_NAME = "/tmp/phd.local";

#ifdef __PH_OLD_PLUGIN_STYLE__
std::map<const char*, MAbstractPlugin*> pluginMap;
#endif
CDaemon* CDaemon::iInstance = NULL;

using namespace std;

/**
TODO: iServiceList should be protected ! ! !
*/
/**
 * @memo Default constructor.
 * @doc Default constructor, opens the system log and creates the device &
 * service database.
 *
 * @return none
 */
CDaemon::CDaemon()
{
  DBG("CDaemon::CDaemon: creating new deamon");

  iInstance = this;
  openlog("phd", LOG_PID, LOG_DAEMON);
  iNumberOfPlugins = 0;
  CDeviceStorage::GetInstance();

  iFetchRequestType = 0;
  std::string fetchParameter;
  int fetchValue=0;

  iDeviceName = std::string("DeviceName");
  if(!LoadParameter(iDeviceName))
    {
      ERR("CDaemon::CDaemon : Can't find device name entry, using defaults");
      iDeviceName = std::string("Generic PeerHood device");
    }

	// TODO change to use UUID - which must be saved
	// Caclulate checksum
  iChecksum = CalculateChecksum();
  
  // Could not calculate
  if(iChecksum == 0) iChecksum = (long)getpid() ^ gethostid();
  DBG("Checksum = %u",iChecksum);
  iChecksum = htonl(iChecksum);

  iDeviceNameLength = htonl(iDeviceName.length()+1);

  //Implement here the fetch options
  fetchParameter = std::string("DeviceInfo");
  if(!LoadParameter(fetchParameter)) {
    ERR("CDaemon::CDaemon : Can't find device info entry, using defaults");
    fetchValue = D_GET_DEVICEINFO;
  }
  else {
    if(atoi(fetchParameter.c_str()) == 1) {
      fetchValue = D_GET_DEVICEINFO;
    }
    else {
      fetchValue = D_GET_DEVICEINFO;
    }
  }
  iFetchRequestType = iFetchRequestType ^ fetchValue;

  fetchParameter = std::string("ServiceInfo");
  if(!LoadParameter(fetchParameter)) {
    ERR("CDaemon::CDaemon : Can't find device info entry, using defaults");
    fetchValue = D_GET_NOINFO;
  }
  else {
    if(atoi(fetchParameter.c_str()) == 1) {
      fetchValue = D_GET_SERVICEINFO;
    }
    else {
      fetchValue = D_GET_NOINFO;
    }
  }
  iFetchRequestType = iFetchRequestType ^ fetchValue;

  fetchParameter = std::string("ProtoInfo");
  if(!LoadParameter(fetchParameter)) {
    ERR("CDaemon::CDaemon : Can't find device info entry, using defaults");
    fetchValue = D_GET_NOINFO;
  }
  else {
    if(atoi(fetchParameter.c_str()) == 1) {
      fetchValue = D_GET_PROTOINFO;
    }
    else {
      fetchValue = D_GET_NOINFO;
    }
  }
  iFetchRequestType = iFetchRequestType ^ fetchValue;

  fetchParameter = std::string("NeighInfo");
  if(!LoadParameter(fetchParameter)) {
    DBG("CDaemon::CDaemon : Can't find device info entry, using defaults");
    fetchValue = D_GET_NOINFO;
  }
  else {
    if(atoi(fetchParameter.c_str()) == 1) {
      fetchValue = D_GET_NEIGHINFO;
    }
    else {
      fetchValue = D_GET_NOINFO;
    }
  }
  iFetchRequestType = iFetchRequestType ^ fetchValue;
  DBG("iFetchRequestType %d", iFetchRequestType);
  
  // Set State variables
  iDaemonActive = true;
  iDaemonShutdown = false;
  iPluginsRunning = false;
  iRevertToPassive = false;
  iDaemonMode = true;
  
  // initialize real time informarion updater
  iRTI = PHRTI::GetInstance();
}

/**
 * @memo Returns the currently running CDaemon instance.
 * @doc Returns the currently running instance of the CDaemon class. If no
 * instances are created then an assertion will take place.
 *
 * @return pointer to the currently running CDaemon instance
 */
CDaemon* CDaemon::GetInstance()
{
  assert(iInstance != NULL);

  return iInstance;
}

void CDaemon::DaemonRevertMode(bool aMode)
{
	if(!aMode)
	{
		DBG("CDaemon::DaemonRevertMode: active");
		iRevertToPassive = false;
		iDaemonMode = true;
	}
	else
	{
		DBG("CDaemon::DaemonRevertMode: passive");
		iRevertToPassive = true;
		iDaemonMode = false;
	}
}

/**
 * @memo Creates a daemon process.
 * @doc Changes the currently running program into a daemon process. This is
 * done by forking twice and closing all open file descriptors. The pid of the
 * daemon process is written to the /var/run/phd.pid file and checked upon
 * every instantiation so that multiple instances of the program are not
 * started. All standard file descriptors (STDIN, STDOUT and STDERR) are
 * redirected to /dev/null.
 *
 * @return boolean telling whether the operation was successfull or not
 */
bool CDaemon::MakeDaemon()
{
  int fd;
  std::ofstream out;

  if (!IsUnique()) {
    ERR("Instance already running");
    return false;
  }

  // fork so that the parent can exit
  switch (fork()) {
    case -1:
      perror("fork");
      exit(EXIT_FAILURE);
    case 0: // child
      break;
    default: // parent - does nothind
      _exit(EXIT_SUCCESS);
  }

  // set the child to be session leader
  if (setsid() == -1) {
    perror("setsid");
    exit(EXIT_FAILURE);
  }

  // fork again, this will cause the session group leader to exit -> we'll never
  // ever regain a controlling terminal
  switch (fork()) {
    case -1:
      perror("fork");
      exit(EXIT_FAILURE);
    case 0:
      break;
    default:
      _exit(EXIT_SUCCESS);
  }

  // ensure that no directories get blocked...
  chdir("/");
  umask(0);

  // close inherited standard fds & reopen them to null
  close(STDIN_FILENO);
  close(STDOUT_FILENO);
  close(STDERR_FILENO);

  fd = open("/dev/null", O_RDWR);
  dup(fd);
  dup(fd);

  // write pid
  out.open(PID_FILE_NAME);
  if (!out.is_open()) {
    ERR("failed to write pidfile");
    return false;
  }
  out << (long)getpid() << std::endl;
  out.close();


  return Init();
}

bool CDaemon::Init()
{
	int listenerCount;
	
	// create the local communications socket
	if (!CreateLocalSocket()) {
		ERR("failed to create the local socket");
		return false;
	}

	if ((iNumberOfPlugins = LoadPlugins()) == -1) {
		return false;
	}
	
	// Set up plugins (load their listeners)
	SetUpPlugins();

	// Create listeners
	if((listenerCount = CreateListeners("daemon")) == 0) DBG("CDaemon::Init: No listeners found.");
	else DBG("CDaemon::Init: %d listeners found.",listenerCount);
	
	// Connect listeners 
	if(ConnectListeners() < listenerCount) DBG("CDaemon::Init: some listeners failed to start.");
	else DBG("CDaemon::Init: All listeners started");
	
	// If the daemon isn't active, do not start plugins - yet
	if(iDaemonActive == true && iDaemonMode == true) StartPlugins();

	signal(SIGTERM, CDaemon::SignalHandler);
	signal(SIGINT, CDaemon::SignalHandler);
	return true;
}


/**
 * @memo Checks if another daemon is already running.
 * @doc Checks if another daemon is already running. This is done by sending
 * the signal 0 to the process whose pid is read from /var/run/phd.pid. If the
 * result of kill is either success or EPERM then the other process exists.
 *
 * @return boolean telling if another daemon is already running
 */
bool CDaemon::IsUnique()
{
  long pid;
  int killResult;
  std::ifstream in(PID_FILE_NAME);

  if (!in.is_open()) {
    DBG("pidfile not found, assuming unique");
    return true;
  }

  in >> pid;
  in.close();

  if ((killResult = kill(pid, 0)) == 0) return false;
  if ((killResult == -1) && (errno == EPERM)) return false;

  return true;
}


/**
 * @memo Handler for the signals TERM and INT.
 * @doc Handles the signals TERM and INT so that the daemon is shut down in a
 * controlled manner. These actions include stopping all running plugins and
 * closing the local / remote sockets.
 *
 * @param aSignum The number of the signal to be handled.
 *
 * @return none
 */
void CDaemon::SignalHandler(int aSignum)
{
  DBG("Ouch! Hit by a fatal signal");

  // shut everything down here...
  //unlink(LOCAL_SOCKET_NAME);
  //unlink(PID_FILE_NAME);
  
  //DBG("stopping plugins");

  CDaemon::GetInstance()->TriggerShutdown();
  
  //std::map<const char*, MAbstractPlugin*>::iterator i;
  //for (i = pluginMap.begin();i != pluginMap.end();++i) {
  //  (*i).second->Stop();
  //  (*i).second->Unadvert();
  //}

  //DBG("fading beyond");

  //exit(EXIT_SUCCESS);
}


/**
 * @memo Creates a local socket that is used between the daemon and PeerHood
 * objects.
 * @doc Creates a local (PF_UNIX) streaming socket (SOCK_STREAM) that is used
 * as a communication channel between the daemon and PeerHood objects. The
 * created socket is set to the listening state and possibly existing socket
 * is simply removed without any further checking.
 *
 * @return boolean indicating the success of the socket creation
 */
bool CDaemon::CreateLocalSocket()
{
  struct sockaddr_un address;

  unlink(LOCAL_SOCKET_NAME);

  if ((iServerSocket = socket(PF_UNIX, SOCK_STREAM, 0)) == -1) {
    ERR("socket");
    return false;
  }

	memset(&address,0,sizeof(struct sockaddr_un));

  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, LOCAL_SOCKET_NAME);

  if (bind(iServerSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("bind");
    return false;
  }

  if (listen(iServerSocket, 32) == -1) {
    ERR("listen");
    return false;
  }
  
  /* Give access to the local socket file for others too. By default only read and execute are given,
     write access is required in order to allow applications to connect to peerhood without root
     privileges! */
  if(chmod(LOCAL_SOCKET_NAME,S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH) != 0)
  {
  	ERR("Couldn't change access rights for local socket.");
  }

  return true;
}


/**
 * @memo Main loop of the PeerHood Daemon. Listens for incoming requests and
 * handles those.
 * @doc Main loop of the PeerHood Daemon. This function listens the local
 * communications socket and handles the incoming requests. The only way to
 * exit this function is via TERM, INT and KILL signals (or because of a
 * bug of course...).
 *
 * @return none
 */
void CDaemon::Run()
{
  int newFd;
  int biggestFd;
  TCommand pdu;
  socklen_t addressLength;
  fd_set fdSet;
  struct timeval timeVal;
  struct sockaddr_un address;

  while (!iDaemonShutdown) {
    timeVal.tv_sec = 1;
    timeVal.tv_usec = 0;

    // Check states of the listeners
    std::list<MAbstractListener*>::iterator iter;
		for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
		{
			(*iter)->CheckState();
		}
		
		// iterator for plugin map
		std::map<const char*, MAbstractPlugin*>::iterator piter;

		// update states of all plugins
#ifdef __PH_OLD_PLUGIN_STYLE__
		for(piter = pluginMap.begin(); piter != pluginMap.end(); ++piter)
		{
			(*piter).second->UpdateState();
		}
#else		
		for(piter = PluginStorage::GetInstance()->Begin(); piter != PluginStorage::GetInstance()->End(); ++piter)
		{
			(*piter).second->UpdateState();
		}
		PluginStorage::GetInstance()->Stop();
#endif
		
		// If set to shut down (got message indicating shutdown), escape from the main loop
		if(iDaemonShutdown == true) break;
		
		/** @TODO Use some kind of state machine */
		// Daemon changed state to inactive and plugins were running
		if(iDaemonActive == false && iPluginsRunning == true) StopPlugins();
		
		if(iPluginsRunning == true && iDaemonMode == false) StopPlugins();
		
		// Daemon changed state to active and plugins weren't running
		if(iDaemonActive == true && 
			iPluginsRunning == false &&
			iDaemonMode == true) StartPlugins();
		
    FD_ZERO(&fdSet);
    FD_SET(iServerSocket, &fdSet);
    biggestFd = iServerSocket;

    for (std::list<ClientData*>::iterator i = iClients.begin();i != iClients.end();++i) {
      FD_SET((*i)->iFd, &fdSet);
      if ((*i)->iFd > biggestFd) biggestFd = (*i)->iFd;
    }

    switch (select(biggestFd + 1, &fdSet, NULL, NULL, &timeVal)) {
    // error
    case -1:
    	// If Daemon is shutting down go to the end of the loop
	    if(iDaemonShutdown) break;
	    else
	    {
	    	ERR("select");
	    	SignalHandler(0);
	    }

    // timeout
    case 0:
    	continue;

    	// some data...
    default:
    	if (FD_ISSET(iServerSocket, &fdSet)) {
    		DBG("got a new connection");
    		addressLength = sizeof(address);
    		newFd = accept(iServerSocket, (struct sockaddr *)&address, &addressLength);
    		if (newFd != -1) {
    			if (read(newFd, &pdu, sizeof(pdu)) != sizeof(pdu)) {
    				ERR("failed to read the PH_INIT command");
    				continue;
    			}

    			ClientData* temp = new ClientData;
    			temp->iFd = newFd;
    			temp->iPid = pdu.iPid;
    			iClients.push_back(temp);
    			
    			// If list was previously empty and the daemon is in revert mode
    			if(iClients.size() == 1 && iRevertToPassive)
    			{
    				iDaemonMode = true;
    				DBG("CDaemon::Run: First client and daemon was started in revert mode, now active");
    			}
    		}
    		continue;
    	}
    	else {
    		for (std::list<ClientData*>::iterator i = iClients.begin();i != iClients.end();++i) {
    			if (FD_ISSET((*i)->iFd, &fdSet)) {
    				//DBG("got data from the fd %d", (*i)->iFd);

    				switch (read((*i)->iFd, &pdu, sizeof(pdu))) {
    				// fallthrough to the next case is desired
    				case -1:
    				ERR("error while reading");

    				case 0:
    					DBG("connection closed, erasing the client");
    					close((*i)->iFd);

    					// delete services that belong to the closed client
    					for (std::list<CService*>::iterator l = iServiceList.begin();l != iServiceList.end();) {
    						if ((*l)->GetPid() == (*i)->iPid) {
    							std::list<CService*>::iterator sentenced = l++;
    							delete *sentenced;
    							iServiceList.erase(sentenced);
    						}
    						else {
    							l++;
    						}
    					}

    					delete *i;
    					iClients.erase(i);
    					if(iClients.size() == 0 && iRevertToPassive)
							{
								iDaemonMode = false;
								DBG("CDaemon::Run: Last client left and daemon was started in revert mode, now passive");
							}
    					break;

    				default:
    					//DBG("got some real PDU, figuring it out...%d", pdu.iCommand);

    					// Make sure you send exactly the right amount of data
    					// Approach: 1. send PDU and 2. send DATA
    					// Data is read by the Handler functions

    					switch (pdu.iCommand) {

    					case PH_GET_DEVICELIST:
    						//DBG("PH_GET_DEVICELIST");
    						HandleGetDeviceList((*i)->iFd);
    						break;

    					case PH_INSERT_SERVICE:
    						//DBG("PH_INSERT_SERVICE");
    						HandleInsertService((*i)->iFd, pdu.iPid);
    						break;

    					case PH_REMOVE_SERVICE:
    						//DBG("PH_REMOVE_SERVICE");
    						HandleRemoveService((*i)->iFd);
    						break;

    					case PH_GET_LOCAL_SERVICELIST:
    						//DBG("PH_GET_LOCAL_SERVICES");
    						HandleGetLocalServiceList((*i)->iFd);
    						break;

    					case PH_GET_FREE_PORT:
    						//DBG("PH_GET_FREE_PORT");
    						HandleGetFreePort((*i)->iFd);
    						break;
    					
    					case PH_GET_CHECKSUM:
    						//DBG("PH_GET_CHECKSUM");
    						if(!Write((*i)->iFd,&iChecksum,sizeof(iChecksum)))
    						{
    							ERR("CDaemon::Run : cannot write checksum to requesting client");
    						}
    						break;

    					default:
    						ERR("unknown pdu");
    						break;
    					}

    					break;
    				}
    				break;
    			}
    		}
    	}
    }
  }
  DBG("CDaemon::Run: Main run loop stopped. Running shutdown function.");
  Shutdown();
}


/**
 * @memo Loads all plugins.
 * @doc Loads all plugins from the directory pointed by the environment
 * variable PH_PLUGIN_DIR. If the environment variable is not set then the
 * operation will fail and no plugins are loaded. Note that this function
 * doesn't start the plugins, it just loads them. The maximum length of the
 * plugin's path is 1024 chars.
 *
 * @return the number of loaded plugins or -1 in the case of an error
 */
int CDaemon::LoadPlugins()
{
  int numOfPlugins = 0;
  void* plugin = NULL;
  std::string pluginName = "";
  std::string pluginDir = "";
  char in[1024];
  FILE* ls = NULL;
  char* pluginEnv = NULL;

  DBG("loading plugins");

  memset(in, 0, sizeof(in));
  if (!(pluginEnv = getenv("PH_PLUGIN_DIR"))) {
  
  	struct stat str_stat;
  	if(stat("/usr/lib/peerhood",&str_stat) == 0)
  	{
  		DBG("PH_PLUGIN_DIR not set, using /usr/lib/peerhood as plugin directory.");
  		pluginEnv = "/usr/lib/peerhood";
  	}
  	else
  	{
		  ERR("PH_PLUGIN_DIR not set, no plugins will be loaded, /usr/lib/peerhood doesn't exist.");
		  return -1;
    }
  }

  pluginDir = std::string(pluginEnv);

  pluginName = std::string("ls ");
  pluginName.append(pluginDir);
  pluginName.append("/libpeerhood*plugin.so"); // */ (fool the stupid doc++)

  if (!(ls = popen(pluginName.c_str(), "r"))) {
    ERR("unable to search for plugins");
    return -1;
  }

  while (fgets(in, sizeof(in), ls)) {
    pluginName = std::string(in);
    pluginName.erase(pluginName.size() -1);
    DBG("  loading plugin: %s", pluginName.c_str());

    if (!(plugin = dlopen(pluginName.c_str(), RTLD_NOW))) {
      ERR("  plugin %s failed to load: %s", pluginName.c_str(), dlerror());
    }
    else {
      //      DBG("  plugin %s loaded succefully", pluginName.c_str());
      numOfPlugins++;
    }
  }

  pclose(ls);

  DBG("plugins loaded");

  return numOfPlugins;
}

void CDaemon::SetUpPlugins()
{
	std::map<const char*, MAbstractPlugin*>::iterator i;

#ifdef __PH_OLD_PLUGIN_STYLE__
	for (i = pluginMap.begin();i != pluginMap.end();++i)
	{
		(*i).second->LoadListeners();
	}
#else
	for (i = PluginStorage::GetInstance()->Begin();i != PluginStorage::GetInstance()->End();++i)
	{
		(*i).second->LoadListeners();
	}
	PluginStorage::GetInstance()->Stop();
#endif
}

/**
 * @memo Starts all loaded plugins.
 * @doc Starts all successfully loaded plugins. Currently it is not possible to
 * start only some certains plugins.
 *
 * @return none
 */
void CDaemon::StartPlugins()
{
	DBG("starting plugins");

	iPluginsRunning = true;

	std::map<const char*, MAbstractPlugin*>::iterator i;

#ifdef __PH_OLD_PLUGIN_STYLE__
	for (i = pluginMap.begin(); i != pluginMap.end(); ++i)
	{
		if ((*i).second->Start()) {

			DBG("  plugin \"%s\" started", (*i).first);

			char *buffer;
			buffer = new char[sizeof((*i).first)];
			sprintf(buffer,"%s",(*i).first);
			iSupportedPlugins.push_back(buffer);

			if (!((*i).second->Advert())) {
				ERR("  adverting failed");
			}
		}
		else {
			DBG("  plugin \"%s\" failed to start", (*i).first);
			//iPluginsRunning = false;
		}
	}
#else
	for (i = PluginStorage::GetInstance()->Begin();i != PluginStorage::GetInstance()->End();++i)
	{
		if ((*i).second->Start()) {

			DBG("  plugin \"%s\" (%s) started", (*i).first,(*i).second->GetAdapter().c_str());

			char *buffer;
			buffer = new char[sizeof((*i).first)];
			sprintf(buffer,"%s",(*i).first);
			iSupportedPlugins.push_back(buffer);

			if (!((*i).second->Advert())) {
				ERR("  adverting failed");
			}
		}
		else {
			DBG("  plugin \"%s\" failed to start", (*i).first);
			//iPluginsRunning = false;
		}
	}
	PluginStorage::GetInstance()->Stop();
#endif

  // This part was for checking if pluginnames are correctly saved into plugin list
  //  std::list<char *>::iterator c;
  //  for(c=iSupportedPlugins.begin();c != iSupportedPlugins.end();++c) {
  //    DBG("Supported plugin %s", (*c));
  //  }
}

void CDaemon::StopPlugins()
{
	DBG("stopping plugins");
		
	std::map<const char*, MAbstractPlugin*>::iterator i;

	if(iPluginsRunning == false)
	{
		DBG("plugins aren't running, doing nothing.");
		return;
	}

#ifdef __PH_OLD_PLUGIN_STYLE__
	for (i = pluginMap.begin();i != pluginMap.end();++i)
	{
		(*i).second->Stop();
 		if(!(*i).second->Unadvert()) DBG("Plugin \"%s\" failed to stop advert",(*i).first);
	}
#else
	for (i = PluginStorage::GetInstance()->Begin();i != PluginStorage::GetInstance()->End();++i)
	{
		DBG("  plugin \"%s\" (%s) stopping", (*i).first,(*i).second->GetAdapter().c_str());
		(*i).second->Stop();
 		if(!(*i).second->Unadvert()) DBG("Plugin \"%s\" failed to stop advert",(*i).first);
	}
	PluginStorage::GetInstance()->Stop();
#endif
	  
	DBG("plugins stopped");
	iPluginsRunning = false;
}

/**
 * @memo
 * @doc
 * 
 */
void CDaemon::Shutdown()
{
	if(iRTI)
	{
		delete iRTI;
		iRTI = NULL;
	}

	unlink(LOCAL_SOCKET_NAME);
	unlink(PID_FILE_NAME);
	
	/** @TODO Notify clients! */
	
	DBG("shutting down plugins");
	StopPlugins();
	
	DBG("stopping listeners");
	RemoveListeners();

	DBG("fading beyond");
	
	exit(EXIT_SUCCESS);
}

/**
 * @memo Handles the PH_GET_DEVICELIST request.
 * @doc Handles the PH_GET_DEVICELIST request. First, the list of all devices
 * is fecthed from the DeviceStorage database. Then it is inserted into pdus
 * and finally the whole list is sent to the given file descriptor.
 *
 * @param aFd The file descriptor where the response should be sent to.
 *
 * @return none
 */
void CDaemon::HandleGetDeviceList(const int aFd)
{
  int length;
  char *data = NULL;
  TDEVICE_LIST list = CDeviceStorage::GetInstance()->GetDeviceList();

  // Just print the devicelist
  //  for (TDEVICE_ITERATOR iter = list.begin();iter != list.end();++iter) {
  //    DBG("Address %s has the following parameters", (*iter)->GetAddress().c_str());
  //    if((*iter)->HasPeerHood()) {
  //      DBG("   Name: %s", (*iter)->GetName().c_str());
  //      DBG("   Check: %d", (*iter)->GetChecksum());
  //      DBG("   Prototypes %d", (*iter)->GetProtoListSize());
  //      DBG("   Services %d", (*iter)->GetServiceListSize());
  //      DBG("   Neighbors %d", (*iter)->GetNeighborListSize());
  //    }
  //    else {
  //      DBG("   Device does not support peerhood");
  //    }
  //  }

  length = list.size();

  DBG("  device list size is: %d", length);

  if (!Write(aFd, &length, sizeof(length))) {
    ERR("  sending the number of devices failed");
    CDeviceStorage::GetInstance()->ReleaseDeviceList();
    return;
  }

  for (TDEVICE_ITERATOR iter = list.begin();iter != list.end();++iter) {
    DBG("GET_DEVICE_LIST: starting the upload...");

    // Marshalling packs the data
    data = (*iter)->MarshallL(&length);

    if (!Write(aFd, &length, sizeof(length))) {
      ERR("  sending device information failed");
      delete[] data;
      CDeviceStorage::GetInstance()->ReleaseDeviceList();
      return;
    }

    if (!Write(aFd, data, length)) {
      ERR("  sending device information failed");
      delete[] data;
      CDeviceStorage::GetInstance()->ReleaseDeviceList();
      return;
    }

    delete[] data;
  }

  DBG("GET_DEVICE_LIST: all device data sent");

  CDeviceStorage::GetInstance()->ReleaseDeviceList();
}


/**
 * @memo Handles the PH_GET_LOCAL_SERVICELIST request.
 * @doc Handles the PH_GET_LOCAL_SERVICELIST request. Each locally registered
 * service is marshalled into octet data and sent to the given unix scoket.
 *
 * @param aFd The file descriptor where the response should be sent to.
 *
 * @return none
 */
void CDaemon::HandleGetLocalServiceList(const int aFd)
{
  int length;
  char* data = NULL;

  length = iServiceList.size();

  if (!Write(aFd, &length, sizeof(length))) {
    ERR("sending the number of services failed");
    return;
  }

  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    data = (*i)->MarshallL(&length);

    if (!Write(aFd, &length, sizeof(length))) {
      ERR("sending service information failed");
      delete[] data;
      return;
    }

    if (!Write(aFd, data, length)) {
      ERR("sending service information failed");
      delete[] data;
      return;
    }

    delete[] data;
  }
}


/**
 * @memo Handles the PH_INSERT_SERVICE request.
 * @doc Handles the PH_INSERT_SERVICE request. The corresponding pdu is read
 * and a new CService object created using the read marshalled information.
 * Unmarshalling is done by the CService's constructor.
 *
 * @param aFd The file descriptor where the marshalled data should be read.
 *
 * @return none
 */
void CDaemon::HandleInsertService(const int aFd, int aPid)
{
  unsigned char response = PH_OK;
  char buffer[514];
  CService* temp;

  if (read(aFd, &buffer, sizeof(buffer)) == -1) {
    ERR("reading failed");
    return;
  }

  temp = new CService((const char *)&buffer, aPid);

  // Not actually needed any more as all services are registered as the name limit was removed
  // Could be used in future to check other issues of the service
  if (Write(aFd, &response, sizeof(response)) != sizeof(response)) {
    delete temp;
    return;
  }

  iServiceList.push_back(temp);
}


/**
 * @memo Handles the PH_REMOVE_SERVICE request.
 * @doc Handles the PH_REMOVE_SERVICE request. The service corresponding to the
 * read name is removed from the list of available local services. Note that an
 * application can remove only the services it has inserted! (removing other
 * services is prevented on the library side).
 *
 * @param aFd The file descriptor where the name should be read from.
 *
 * @return none
 */
void CDaemon::HandleRemoveService(const int aFd)
{
  unsigned short port;

  //  DBG("CDaemon::HandleRemoveService : starting to remove service");

  if (!read(aFd, &port, sizeof(port))) {
    ERR("CDaemon::HandleRemoveService read not successfull");
    return;
  }

  //  DBG("CDaemon::HandleRemoveService : removing service in port %d", (int) port);

  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    if ((*i)->GetPort() == port) {
      //      DBG("removing the service %s", (*i)->GetName());
      delete *i;
      iServiceList.erase(i);
      return;
    }
  }
}


/**
 * @memo Handles the PH_GET_FREE_PORT request.
 * @doc Handles the PH_GET_FREE_PORT request.
 * This function looks if the given port is free. If not or if the port is not
 * given, this function looks for the next available port.
 *
 * @param aFd The file descriptor where the response should be sent to.
 *
 * @return none
 */
void CDaemon::HandleGetFreePort(const int aFd)
{

  unsigned short port=0;
  bool free = true;

  if (!read(aFd, &port, sizeof(port))) {
    ERR("CDaemon::HandleGetFreePort read not successfull");
    return;
  }

  // First we check if the given port is free, if even given
  if (port) {
    // check if the given port if available
    for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
      if ((*i)->GetPort() == port) {
	DBG("service called \"%s\" has port already in use", (*i)->GetName().c_str());
	port=0;
	break;
      }
    }
  }

  // if port was not given or was already in use, find a free one
  // Start with the existing ports and always add 1
  // There is a hard coded starting port is the list is empty
  if (!port) {
     if (iServiceList.size()==0) {
	port = 32323;
    }

    for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
      port = (*i)->GetPort() + 1;
      free = true;

      for (std::list<CService*>::iterator j = iServiceList.begin();j != iServiceList.end();++j) {
	if (port == (*j)->GetPort()) {
	  DBG("Port already reserved, next one please");
	  free = false;
	  break;
	}
      }

      if (free) {
	break;
      }
    }
  }

  if (!Write(aFd, &port, sizeof(port))) {
    ERR("sending port information failed");
    return;
  }
}


/**
 * @memo Writes data to the given file descriptor.
 * @doc Writes data to the given file descriptor. The file descriptor is
 * assumed to be open. The fd will not be closed even in the case of an error.
 *
 * @param aFd The file desriptor where the data should be written.
 * @param aBuffer The data to be written.
 * @param aLength The length of the data buffer.
 *
 * @return true if the data was written succesfully
 */
bool CDaemon::Write(const int aFd, void *aBuffer, int aLength)
{
  if (write(aFd, aBuffer, aLength) != aLength) {
    ERR("write failed or size mishmatch");
    return false;
  }

  return true;
}


/**
 * @memo Sends device information
 * @doc
 *
 * @param aConnection The connection that should be used when sending the
 * service list. This parameter cannot be NULL.
 *
 * @return true if the services were sent succesfully
 */
bool CDaemon::SendDeviceInfo(MAbstractConnection* aConnection)
{
  assert(aConnection != NULL);

  //send device name length
  //  DBG("CDaemon::SendDeviceInfo : sending device name length %d", iDeviceName.length()+1);
  if (aConnection->Write(&iDeviceNameLength, sizeof(iDeviceNameLength)) == -1) {
    ERR("CDaemon::SendServiceList : sending the device name length failed");
    return false;
  }

  //send device name
  //  DBG("   SendDeviceInfo : sending device name %s",iDeviceName.c_str());
  if (aConnection->Write(iDeviceName.c_str(), iDeviceName.length()+1) == -1) {
    ERR("CDaemon::SendDeviceInfo : sending the device name failed");
    return false;
  }

  //send device name checksum
  //  DBG("   SendDeviceInfo : sending device name checksum %d", iChecksum);

  if (aConnection->Write(&iChecksum, sizeof(iChecksum)) == -1) {
    ERR("CDaemon::SendServiceList : sending the device name failed");
    return false;
  }

  DBG("CDaemon::SendDeviceInfo : OK");
  return true;
}

/**
 * @memo Sends all local services to the given connection.
 * @doc Sends all local services to the given connection. This method should
 * be called when a plugin gets the service list request from another PeerHood
 * device. The connection must be opened before calling this function. Note
 * that the connection object will be destroyed before this function returns.
 *
 * @param aConnection The connection that should be used when sending the
 * service list. This parameter cannot be NULL.
 *
 * @return true if the services were sent succesfully
 */
bool CDaemon::SendServiceList(MAbstractConnection* aConnection)
{
  char* buffer = NULL;
  unsigned short numberOfServices = 0;

  assert(aConnection != NULL);

  //  DBG("CDaemon::SendServiceList :  sending the list of services");

  // send the number of services first...
  //numberOfServices = iServiceList.size();

  //  DBG("SendServiceList : sending number of services is %d", numberOfServices);
  //  DBG("CDaemon::SendServiceList : size is %d", sizeof(numberOfServices));

  numberOfServices = htons((unsigned short)iServiceList.size());
  
  if (aConnection->Write(&numberOfServices, sizeof(numberOfServices)) == -1) {
    ERR("CDaemon::SendServiceList : sending the number of services failed");
    return false;
  }

  // ...then the services
  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    //    DBG("SendServiceList :  name : %s ", (*i)->GetName().c_str());
    //    DBG("SendServiceList :  attr : %s", (*i)->GetAttributeString().c_str());
    //    DBG("SendServiceList :  port : %d", (*i)->GetPort());

    unsigned int length = 0;
    buffer = (*i)->MarshallL((int *)&length);
    length = htonl(length);

    if (aConnection->Write(&length, sizeof(length)) == -1) {
      ERR("CDaemon::SendServiceList : sending service datas size failed");
      delete[] buffer;
      //      ERR("Okay now returning...");
      return false;
    }

    length = ntohl(length);

    if (aConnection->Write(buffer, length) == -1) {
      ERR("CDaemon::SendServiceList : sending service list failed");
      delete[] buffer;
      return false;
    }

    delete[] buffer;

  }

  DBG("CDaemon::SendServiceList : OK");
  return true;
}

/**
 * @memo Sends all local services to the given connection.
 * @doc Sends all local services to the given connection. This method should
 * be called when a plugin gets the service list request from another PeerHood
 * device. The connection must be opened before calling this function. Note
 * that the connection object will be destroyed before this function returns.
 *
 * @param aConnection The connection that should be used when sending the
 * service list. This parameter cannot be NULL.
 *
 * @return true if the services were sent succesfully
 */
bool CDaemon::SendPrototypes(MAbstractConnection* aConnection)
{
  char* buffer = NULL;
  unsigned short numberOfPlugins = 0;
  unsigned short length = 0;

  assert(aConnection != NULL);

  //  DBG("CDaemon::SendPrototypes :  sending the list of plugins");

  // send the number of plugins first...
  //numberOfPlugins = iSupportedPlugins.size();

  //  DBG("SendPrototypes : sending number of plugins is %d", numberOfPlugins);
  //  DBG("CDaemon::SendPrototypes : size is %d", sizeof(numberOfPlugins));

  numberOfPlugins = htons((unsigned short)iSupportedPlugins.size());
  if (aConnection->Write(&numberOfPlugins, sizeof(numberOfPlugins)) == -1) {
    ERR("CDaemon::SendPrototypes : sending the number of services failed");
    return false;
  }

  // ...then the plugins
  for (std::list<char*>::iterator i = iSupportedPlugins.begin();i != iSupportedPlugins.end();++i) {

    //    DBG("SendPrototypes :  sending the plugin %s", (*i));

    // calculate length here
    length = strlen(*i)+1;
    length = htons(length);

    if (aConnection->Write(&length, sizeof(length)) == -1) {
      ERR("CDaemon::SendPrototypes : sending service datas size failed");
      delete[] buffer;
      return false;
    }

    length = ntohs(length);
    if (aConnection->Write((*i), length) == -1) {
      ERR("CDaemon::SendPrototypes : sending service list failed");
      delete[] buffer;
      return false;
    }

    delete[] buffer;

  }

  DBG("CDaemon::SendPrototypes : OK");
  return true;
}

/**
 * @memo Sends all devices in local database to the given connection.
 * @doc Sends all devices in local database to the given connection. This method should
 * be called when a plugin gets the neighborhood device list request from another PeerHood
 * device. The connection must be opened before calling this function. Note
 * that the connection object will be destroyed before this function returns.
 *
 * @param aConnection The connection that should be used when sending the
 * service list. This parameter cannot be NULL.
 *
 * @return true if the neighborhood devices were sent succesfully
 */
bool CDaemon::SendNeighborList(MAbstractConnection* aConnection)
{
  unsigned int length;
  char* buffer = NULL;

  unsigned short sizeOfNeighborList;
  TDEVICE_LIST list = CDeviceStorage::GetInstance()->GetDeviceList();

  assert(aConnection != NULL);

  sizeOfNeighborList = list.size();

  //  DBG("SendNeighbors: Device list size: %d", sizeOfNeighborList);

  sizeOfNeighborList = htons((unsigned short) list.size());
  if ((aConnection->Write(&sizeOfNeighborList, sizeof(sizeOfNeighborList))) == -1) {
    ERR("CDaemon::SendNeighborhoodDeviceList : sending the number of devices failed");
    CDeviceStorage::GetInstance()->ReleaseDeviceList();
    return false;
  }

  for (TDEVICE_ITERATOR iter = list.begin();iter != list.end();++iter) {
    //    DBG("CDaemon::SendNeighborhoodDeviceList : starting the upload...");

    //    DBG("SendNeighbors: Address %s", (*iter)->GetAddress().c_str());
    //    DBG("SendNeighbors: HasPeerHood %d", (*iter)->HasPeerHood());

    buffer = (*iter)->MarshallL((int*)&length);
    length = htonl(length);

    if ((aConnection->Write(&length, sizeof(length))) == -1) {
      ERR("CDaemon::SendNeighborhoodDeviceList : sending device information failed");
      delete[] buffer;
      CDeviceStorage::GetInstance()->ReleaseDeviceList();
      return false;
    }

    length = ntohl(length);

    if ((aConnection->Write(buffer, length)) == -1) {
      ERR("CDaemon::SendNeighborhoodDeviceList : sending device information failed");
      delete[] buffer;
      CDeviceStorage::GetInstance()->ReleaseDeviceList();
      return false;
    }

    delete[] buffer;
  }

  DBG("CDaemon::SendNeighborhoodDeviceList : OK");

  CDeviceStorage::GetInstance()->ReleaseDeviceList();

  return true;
}

/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CDaemon::LoadParameter(string& aParameter)
{
  char* home;
  string line;
  string path;

  if ((home = getenv("HOME")) == NULL) {
  	path = string("/etc/default/phconfig");
    //return false;
  }
  else
  {
  	path = string(home);
  	path.append("/.phconfig");
  }
  
  struct stat st;
  
	if(stat(path.c_str(),&st) == -1)
  {
  	path.clear();
  	path = std::string("/etc/default/phconfig");
  }
  
  ifstream in(path.c_str());

  if (!in.is_open()) {
    ERR("Failed to open the configuration file, using /etc/default/phconfig");
    in.close();
   	return false;
  }

  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    string::size_type position = line.find_first_of(':');

    if (position != string::npos) {
      string parameter = line.substr(0, position);
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}

int CDaemon::GetFetchRequestType()
{
  return iFetchRequestType;
}

int CDaemon::CreateListeners(const std::string& aType)
{
	int listeners = 0;
	listeners = ListenerFactory::GetInstance()->CreateListeners(aType,this);
	
	return listeners;
}

int CDaemon::ConnectListeners()
{
	int count = 0;
	std::list<MAbstractListener*>::iterator iter;
	
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		if((*iter)->Connect())
		{
			(*iter)->CheckInitialState();
			count++;
		}
	}
	
	return count;
}
/**
 * @memo Removes all listeners (disconnects and deletes) 
 * @doc
 */
void CDaemon::RemoveListeners()
{
	std::list<MAbstractListener*>::iterator iter;
	// Go through all listeners and disconnect, delete object and erase from list
	for(iter = iListenerList.begin(); iter != iListenerList.end();)
	{
		(*iter)->Disconnect();
		delete *iter;
		iter = iListenerList.erase(iter);
	}
}


/**
 * @memo Register a listener with daemon
 * @doc Add (register) new system state listener for daemon. New listeners
 * are added to the end of iListenerList.
 * 
 * @param aListener New listener to be registered
 */
void CDaemon::RegisterListener(MAbstractListener* aListener)
{
	// Add listener if given.
	if(aListener != NULL)
	{
		iListenerList.push_back(aListener);
		DBG("CDaemon::RegisterListener: new system listener: %s", aListener->GetName().c_str());
	}
}

/**
 * @memo Change daemon activity state
 * @doc Change the activity state of the daemon from passive to active and vice versa. Changing
 * daemon state from active to passive makes the daemon to stop all plugins and when back in
 * active state daemon restarts all plugins.
 * 
 * @param aActive New activity state (true = active, false = passive)
 * 
 */
void CDaemon::SetState(bool aActive)
{
	if(iDaemonActive != aActive)
	{
		DBG("CDaemon::SetActive: Activity state change to %s", aActive ? "active" : "passive");
		iDaemonActive = aActive;
	}
	else DBG("CDaemon::SetActive: No change in activity state");
}

/**
 * @memo Set daemon to shut down state
 * @doc Set daemon to be shut down nicely. Multiple calls of this function do not mess up
 * the shutdown process, when already shutting down new shutdown calls are just ignored
 * - no need to change state e.g. when sending kill signal.
 */
void CDaemon::TriggerShutdown()
{
	if(!iDaemonShutdown)
	{
		iDaemonShutdown = true;
		DBG("CDaemon::TriggerShutdown: daemon shutting down");
	}
	else DBG("CDaemon::TriggerShutdown: already shutting down");
}

void CDaemon::SetAdapter(const std::string& aInterface, int aId)
{
	return;
}

const std::string& CDaemon::GetAdapter()
{
	return iDeviceName;
}



u_int32_t CDaemon::CalculateChecksum()
{
	CIFSearch ifSearch;
	const string* ifname = NULL;
	unsigned char* if_hwaddr = NULL;
	bool if_physical = true, if_up = false; // Do not select if that is up
	u_int32_t checksum = 0;
	u_int32_t manufacturer = 0;
	u_int32_t device = 0;
	u_int8_t byteorder = 1; // 0 = big endian, 1 = little endian
	
#ifdef __BYTE_ORDER
#if __BYTE_ORDER == __BIG_ENDIAN
	byteorder = 0;
#elif __BYTE_ORDER == __LITTLE_ENDIAN
	byteorder = 1;
#endif // if __BYTE_ORDER
#else
	short one = 1;
	char *charp = (char*)&one;
	if(*charp == 0) byteorder = 0;
	if(*charp == 1) byteorder = 1;
#endif // ifdef __BYTE_ORDER
	
	// No suitable interface (no need to be active)
	if(!(ifname = ifSearch.SelectInterface(if_up))) return 0;
	else DBG("CDaemon::CalculateChecksum: using %s",ifname->c_str());
	
	// No address?
	if(!(if_hwaddr = ifSearch.RetrieveHWaddr(*ifname,if_physical))) return 0;
	else
	{
		if(!if_physical) DBG("CDaemon::CalculateChecksum: Warning, %s is not a physical interface",ifname->c_str());
		for(int i = 0; i < IFHWADDRLEN; i++)
		{
			/* Manufacturer id - 3 first bytes - 8 first bits for manufacturer middle id
			 * in our checksum, 8 bits of manufacturer last id xor'd with first 8 bits of
			 * device id and followed by rest (16 bits) of device id
			 */
			if(byteorder == 1) // little endian
			{
				if(i<3) manufacturer |= (unsigned char)if_hwaddr[i] << (16-i*8);
				else device |= (unsigned char)if_hwaddr[i] << (16-(i-3)*8);
			}
			else // big endian
			{
				if(i<3) manufacturer |= (unsigned char)if_hwaddr[i] << (8+i*8);
				else device |= (unsigned char)if_hwaddr[i] << (8+(i-3)*8);
			}
		}
		
		if(byteorder == 1) // little endian
		{
			// Does not provide really unique checksum...
			checksum |= manufacturer << 16; // Lose 8 bits of information
			checksum ^= device;
		}
		else
		{
			checksum |= manufacturer >> 16; // Lose 8 bits of information
			checksum ^= device;
		}
		
		delete ifname;
		free(if_hwaddr);
		return checksum;
	}
}

