/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003-2011 LUT. .
 *
 * @name IFSearch.cc
 * @memo Implementation of the PeerHood IFSearch class.
 *
 * @version 0.2
 * date     22.07.2003
 * change   15.04.2011
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/select.h>
#include <errno.h>
#include <sys/uio.h>
#include <sys/stat.h>
// defined in linux/wireless.h
//#include <net/if.h> 
#include <linux/sockios.h>
#include <linux/wireless.h>
#include "IFSearch.h"

#include <iostream>
#include <fstream>
#include <list>
#include <string>

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

using namespace std;

/**
 * @memo Default constructor.
 * @doc Default constructor, Does nothing
 *
 * @return none
 */
CIFSearch::CIFSearch() {
	iSockIOCTL = -1;
}

CIFSearch::~CIFSearch() {
	if(iSockIOCTL > 0) close(iSockIOCTL);
}


/*
 * TODO: NULL should checked on every GetIFInfo, causes segfault.
 */
/**
 * @memo Returns all the found network interfaces
 * @doc Returns all the found network interfaces.
 *
 * @return Pointer to the CIFInfo linked list which
 * contains information about found interfaces. Return NULL if no interfaces
 * are found.
 */

struct IFInfo *CIFSearch::GetIFInfo(const std::string& aIface)
{
 int sockfd, len, lastlen, flags, myflags;
  char *ptr, *buf, lastname[IFNAMSIZ+48], *cptr;
  struct ifconf	ifc;
  struct ifreq *ifr, ifrcopy;
  struct sockaddr_in *sinptr;
  int family = AF_INET;
  int doaliases = 1;
  char str[16];
  struct IFInfo *ifaces = new struct IFInfo;
  bool found = false;

  sockfd = socket(AF_INET, SOCK_DGRAM, 0);

  lastlen = 0;
  len = 100 * sizeof(struct ifreq);// initial buffer size guess
  for ( ; ; ) {
    buf = new char[len];
    ifc.ifc_len = len;
    ifc.ifc_buf = buf;

    if (ioctl(sockfd, SIOCGIFCONF, &ifc) < 0) {
      if (errno != EINVAL || lastlen != 0)
	{
	  ERR("CIFSearch::GetIFInfo : ioctl error");
	  delete ifaces;
	  close(sockfd);
	  delete[] buf;
	  return NULL;
	}
    } else {
    	/* A fix for searching the interfaces - much simpler way to traverse through interfaces
    	 * On 64 bit OS caused problems -> only loopback was found because of the STETSON HARRISON
    	 * values used... */
    	
    	unsigned int slot = 0;
    	struct ifreq *ifr_element = NULL;
    	for(slot = 0; slot < ifc.ifc_len/sizeof(struct ifreq); slot++) // Go through ifconf struct interfaces
    	{
    		ifr_element = &ifc.ifc_req[slot]; // Get a single element

    		// Interface is up and it is the same as requested
    		if (((ifr_element->ifr_flags & IFF_UP) == 0) && (strcmp(aIface.c_str(),ifr_element->ifr_name) == 0))
    		{
					//unicast address 
					sinptr = (struct sockaddr_in *) &ifr_element->ifr_addr;
					memcpy(&ifaces->iUcast, sinptr, sizeof(struct sockaddr_in));

					//broadcast address
#ifdef	SIOCGIFBRDADDR
					if (ifr_element->ifr_flags & IFF_BROADCAST)
					{
						ifrcopy = *ifr_element;
						ioctl(sockfd, SIOCGIFBRDADDR, &ifrcopy);
						sinptr = (struct sockaddr_in *) &ifrcopy.ifr_broadaddr;
						memcpy(&ifaces->iBcast, sinptr, sizeof(struct sockaddr_in));
					}
#endif

    			found = true;
	    		
	    		close(sockfd);
	    		delete[] buf;
	    		return ifaces;
	    	}
    	}
      if (ifc.ifc_len == lastlen) break;		// success, len has not changed
      lastlen = ifc.ifc_len;
    }
    len += 10 * sizeof(struct ifreq);	// increment
    delete[] buf;
  }

  lastname[0] = 0;
  
  

  for (ptr = buf; ptr < buf + ifc.ifc_len; ) {
    ifr = (struct ifreq *) ptr;

#ifdef	HAVE_SOCKADDR_SA_LEN
    len = max(sizeof(struct sockaddr), ifr->ifr_addr.sa_len);
#else
    switch (ifr->ifr_addr.sa_family) {
#ifdef	IPV6
    case AF_INET6:
      len = sizeof(struct sockaddr_in6);
      break;
#endif
    case AF_INET:
    default:
      len = sizeof(struct sockaddr);
      break;
    }
#endif	// HAVE_SOCKADDR_SA_LEN
    ptr += sizeof(ifr->ifr_name) + len;	// for next one in buffer

    if (ifr->ifr_addr.sa_family != family)
      continue;	// ignore if not desired address family

    myflags = 0;
    if ( (cptr = strchr(ifr->ifr_name, ':')) != NULL)
      //*cptr = 0;		// replace colon will null
    if (strncmp(lastname, ifr->ifr_name, IFNAMSIZ+48) == 0) {
      if (doaliases == 0)
	continue;	// already processed this interface
      myflags = IFI_ALIAS;
    }
    memcpy(lastname, ifr->ifr_name, IFNAMSIZ+48);

    ifrcopy = *ifr;
    ioctl(sockfd, SIOCGIFFLAGS, &ifrcopy);

    flags = ifrcopy.ifr_flags;
    if ((flags & IFF_UP) == 0)
      continue;	// ignore if interface not up

    memcpy(str, ifr->ifr_name, IFI_NAME);

    str[IFI_NAME-1] = '\0';

		//printf("IFSearch : given: %s , value: %s\n", aIface.c_str(), str); 
    if(strcmp(aIface.c_str(), str) == 0)
    {
			//unicast
			sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
			memcpy(&ifaces->iUcast, sinptr, sizeof(struct sockaddr_in));

			//bcast
#ifdef	SIOCGIFBRDADDR
			if (flags & IFF_BROADCAST) {
	  		ioctl(sockfd, SIOCGIFBRDADDR, &ifrcopy);
	  		sinptr = (struct sockaddr_in *) &ifrcopy.ifr_broadaddr;
	  		memcpy(&ifaces->iBcast, sinptr, sizeof(struct sockaddr_in));
			}
#endif
			found = true;
    }
  }

  close(sockfd);
  delete[] buf;

  if(!found)
  {
    delete ifaces;
    return NULL;
	}
  return ifaces;
}

bool CIFSearch::RetrieveInterfaces(list <std::string> &ifaces)
{
	unsigned int i = 0, type = 0;
	string line;
	string devpath = string("/proc/net/dev");
	
	// load /proc/net/dev
	ifstream dev_in(devpath.c_str()); 
	
	if(!dev_in.is_open())
	{
		dev_in.close();
		return false;
	}

	// get adapter names
	for(i = 0; !dev_in.eof(); i++)
	{
		getline(dev_in, line);
		if(i<2) continue; // do not handle first two lines (headers)

		// Get occurence of ':'
		string::size_type position = line.find(':');

		if (position != string::npos)
		{
			string ifname = line.substr(0, position);
			
			// Trim white space 
			string::size_type sposition = ifname.find_last_of(' ');
			if(sposition != string::npos) sposition++;
			else sposition = 0; // no white space
			
			type = CheckInterfaceType(string(ifname.substr(sposition)));
			
			switch(type)
			{
				case D_IF_ETHERNET:
				case D_IF_WIRELESS:
					ifaces.push_back(ifname.substr(sposition));
					//DBG("Interface: %s (type %u)",string(ifname.substr(sposition)).c_str(),type);
					break;
				default:
					break;
			}
		}
	}

  dev_in.close();
  
  // Only header found, no interfaces
  if(i<2) return false;
  else return true;
}

unsigned int CIFSearch::CheckInterfaceType(const std::string &ifname)
{
	if(iSockIOCTL < 0) iSockIOCTL = socket(PF_INET,SOCK_DGRAM,0);
	
	struct ifreq ifr;
	char ifname_buf[2 * IFNAMSIZ + 1];
	unsigned char* hwaddr = NULL;
	bool is_physical = true;
	
	memset(&ifr,0,sizeof(ifr));
	memset(&ifname_buf,0,sizeof(ifname_buf));
	
	strncpy(ifr.ifr_name, ifname.c_str(), ifname.length());
	
	// Is loopback?
	if(ioctl(iSockIOCTL,SIOCGIFFLAGS,&ifr) == 0)
	{
		if(ifr.ifr_flags & IFF_LOOPBACK) return D_IF_OTHER;
	}
	else return D_IF_NOT_FOUND; // Cannot be found?
	
	// Too long interface name, treat as other
	if(ifname.length() > (2 * IFNAMSIZ + 1)) return D_IF_OTHER;
	
	// Is wireless?
	strncpy(ifname_buf, ifname.c_str(), ifname.length());
	if(ioctl(iSockIOCTL,SIOCGIWNAME,&ifname_buf) == 0) return D_IF_WIRELESS;
	
	// Check if the adapter is a logical or virtual
	if((hwaddr = RetrieveHWaddr(ifname,is_physical)) == NULL) return D_IF_NOT_FOUND;
	else free(hwaddr);
	
	// Is a physical interface
	if(is_physical) return D_IF_ETHERNET;
	else return D_IF_OTHER;
}

unsigned char* CIFSearch::RetrieveHWaddr(const std::string &ifname, bool &isphysical)
{
	struct ifreq ifr;
	memset(&ifr,0,sizeof(ifr));
	
	if(ifname.length() == 0) return NULL;
	
	strncpy(ifr.ifr_name,ifname.c_str(),ifname.length());
	
	if(iSockIOCTL < 0) iSockIOCTL = socket(PF_INET,SOCK_DGRAM,0);
	
	// Get mac address
	if(ioctl(iSockIOCTL, SIOCGIFHWADDR, &ifr) == 0)
	{
		unsigned char* mac_addr = (unsigned char*)malloc(sizeof(unsigned char) * IFHWADDRLEN);
		memcpy(mac_addr,&(ifr.ifr_hwaddr.sa_data),IFHWADDRLEN);
		
		// From lshw http://lshw.org GPL v2
		if(ifr.ifr_hwaddr.sa_family >= 256) isphysical = false;
		
		if(IsVirtualInterface(mac_addr)) isphysical = false;
		
		return mac_addr;
	}
	else return NULL;
}

/* Used similar approach as in lshw http://lshw.org GPL v2 */
bool CIFSearch::IsVirtualInterface(unsigned char* mac_addr)
{
	char manufacturer[7];
	memset(&manufacturer,0,sizeof(manufacturer));
	
	if(!mac_addr) return false;
	
	snprintf(manufacturer,7,"%.02X%.02X%.02X",mac_addr[0],mac_addr[1],mac_addr[2]);
	
	if(strncasecmp(manufacturer,"000569",6) == 0) return true; // VMWare
	if(strncasecmp(manufacturer,"000C29",6) == 0) return true; // VMWare
	if(strncasecmp(manufacturer,"005056",6) == 0) return true; // VMWare
	if(strncasecmp(manufacturer,"001C42",6) == 0) return true; // Parallels
	
	return false;
}

bool CIFSearch::IsInterfaceUp(const std::string &ifname)
{
	if(iSockIOCTL < 0) iSockIOCTL = socket(PF_INET,SOCK_DGRAM,0);
	
	struct ifreq ifr;
	memset(&ifr,0,sizeof(ifr));
	strncpy(ifr.ifr_name, ifname.c_str(), ifname.length());
	
	// Is loopback?
	if(ioctl(iSockIOCTL,SIOCGIFFLAGS,&ifr) == 0)
	{
		if(ifr.ifr_flags & IFF_UP) return true;
		else return false; // Down
	}
	else return false; // Cannot be found?
}

const std::string* CIFSearch::SelectInterface(bool &select_up_if)
{
	bool configured_if = false;
	list <string> interfaces;
	string selected_if = string("");
	
	// Get all interfaces
	if(RetrieveInterfaces(interfaces))
	{
		string search_param = string("WLANInterface");
		list<string>::iterator iter;
		
		// Found interface from config
		if(LoadParameter(search_param)) configured_if = true;
		
		// Go through interfaces
		for(iter = interfaces.begin(); iter != interfaces.end(); ++iter)
		{
			// Configured interface exists on the system
			if(configured_if)
			{
				// When configured is found, use it instead of already selected
				if(search_param.compare(*iter) == 0)
				{
					selected_if = string(*iter); // Use the selected, no check for up state!
					break;
				}
			}
			
			// If the adapter has wireless extensions, use it
			if(CheckInterfaceType(*iter) == D_IF_WIRELESS)
			{
				if(!select_up_if) selected_if = string(*iter);
				// Need to select interface that is up
				else if(IsInterfaceUp(*iter)) selected_if = string(*iter);
			}
		}
		
		// No suitable interface found, use the first ethernet interface in the list
		if(selected_if.length() == 0) selected_if = string(interfaces.front());
		
		return new string(selected_if);
	}
	// No interfaces found
	else return NULL;
}

bool CIFSearch::LoadParameter(string& aParameter)
{
  char* home;
  string line;
  string path;

  if ((home = getenv("HOME")) == NULL) {
  	path = string("/etc/default/phconfig");
    //return false;
  }
  else
  {
  	path = string(home);
  	path.append("/.phconfig");
  }
  
  struct stat st;
  
	if(stat(path.c_str(),&st) == -1)
  {
  	path.clear();
  	path = std::string("/etc/default/phconfig");
  }
  
  ifstream in(path.c_str());

  if (!in.is_open()) {
    ERR("Failed to open the configuration file, using /etc/default/phconfig");
    in.close();
   	return false;
  }

  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    string::size_type position = line.find_first_of(':');

    if (position != string::npos) {
      string parameter = line.substr(0, position);
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}


