/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name BluezBTListener.cc
 * @memo Bluez-specific implementation of MAbstractListener
 *
 * @version 0.22
 * date     14.11.2008
 * change   11.03.2009
 */

#include "BluezBTListener.h"
#include <string.h>
#include <string>
#include <locale>

#define BLUEZ_SERVICE "org.bluez"

#define BLUEZ_REQUEST_PATH_NEW "/"
#define BLUEZ_REQUEST_PATH_OLD "/org/bluez"

#define BLUEZ_ADAPTER_REQUEST_INTERFACE "org.bluez.Adapter"
#define BLUEZ_MANAGER_REQUEST_INTERFACE "org.bluez.Manager"

#define BLUEZ_MODE_REQ "GetMode"
#define BLUEZ_PROPERTY_REQ "GetProperties"
#define BLUEZ_DEFAULT_ADAPTER_REQ "DefaultAdapter"

#define BLUEZ_MODE_OFF "off"
#define BLUEZ_MODE_DISCOVERABLE "discoverable"
#define BLUEZ_MODE_CONNECTABLE "connectable"
#define BLUEZ_MODE_LIMITED "limited"

#define BLUEZ_SIGNAL_MODE_CHANGED "ModeChanged"
#define BLUEZ_SIGNAL_ADDED "AdapterAdded"
#define BLUEZ_SIGNAL_REMOVED "AdapterRemoved"
#define BLUEZ_SIGNAL_DEFAULT_CHANGED "DefaultAdapterChanged"
#define BLUEZ_SIGNAL_PROPERTY_CHANGED "PropertyChanged"

#define BLUEZ_ERROR_NO_SUCH_ADAPTER "org.bluez.Error.NoSuchAdapter"
#define BLUEZ_ERROR_INVALID_ARGUMENTS "org.bluez.Error.InvalidArguments"

/**
 * @memo Constructor, as private - not used.
 * @doc Default constructor isn't used (cannot be used)
 */
CBluezBTListener::CBluezBTListener()
{
	iConverter = NULL;
	iConnection = NULL;
	iName = std::string("BluezBTListener");
	
	iDictParsing = false;
	iDictKey = std::string("");
}

/**
 * @memo Constructor
 * @doc Use this constructor to create instances of this class. A reference to the 
 * component that will own the instance will have to be given as a parameter. Created
 * component will be registered into that component. 
 * 
 * @param Reference to component that owns the instance. CANNOT BE NULL!
 */
CBluezBTListener::CBluezBTListener(MAbstractStateConverter* aConverter)
{
	iConverter = aConverter;
	iConnection = NULL;
	iName = std::string("BluezBTListener");

	iAdapterChange = std::string("");
	iChanges = false;
	iPowered = true;
	
	if(iConverter != NULL) iConverter->RegisterListener(this);
	
	iDictParsing = false;
	iDictKey = std::string("");
	
	iOldBluez = true;
#ifdef __PH_BLUEZ
	if(__PH_BLUEZ >= 4.00) iOldBluez = false;
	else iOldBluez = true;
#endif
}

/**
 * @memo Destructor, closes connection to D-Bus.
 * @doc Closes the connection if it wasn't closed.
 */
CBluezBTListener::~CBluezBTListener()
{
	if(iConnection != NULL)
	{
		// Returns void - cannot be checked
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

/**
 * @memo Connects the listener to source
 * @doc Connects this listener to D-Bus. Uses private connection.
 * Also registers this listener to listen for signals from appropriate
 * interface. Uses SetupConnection() and RegisterSignals().
 * 
 * @return bool true if listener could connect
 */
bool CBluezBTListener::Connect()
{
	if(!SetupConnection()) return false;
	
	if(!RegisterSignals()) return false;
	
	return true;
}

/**
 * @memo Disconnects this listener from D-Bus
 * @doc Disconnects listener if it wasn't already disconnected (error)
 */
void CBluezBTListener::Disconnect()
{
	if(iConnection != NULL)
		{
			dbus_connection_close(iConnection);
			iConnection = NULL;
		}
}

/**
 * @memo Checks the initial state of the bluetooth device via Bluez
 * @doc Used by the plugin that owns this listener, this function is called only
 * after successful connection to D-Bus. Sends a pending call via D-Bus to
 * BlueZ request interface. Uses a blocking call to get the reply and reacts to
 * the reply by changing the state of the component that owns this listener.
 * Uses hci0 device at the moment. 
 * 
 */
void CBluezBTListener::CheckInitialState()
{
	DBusMessage* msg = NULL;
	DBusPendingCall* statecall = NULL;
	DBusMessageIter iter;
	DBusError error;
	std::string request_path = "";
	const char* mdata;
	int msgtype;
	
	// Initialize error variable
	dbus_error_init(&error);
	
	// Check if bluez is running
	if(!dbus_bus_name_has_owner(iConnection,BLUEZ_SERVICE,&error))
	{
		DBG("CBluezBTListener::CheckinitialState: no BLUEZ running.");
		iConverter->SetState(false);
		return;
	}
	
	// A error is set	
	if(dbus_error_is_set(&error))
	{
		DBG("CBluezBTListener::CheckinitialState: Check BLUEZ, error: %s",error.message);
		dbus_error_free(&error);
		
		iConverter->SetState(false);
		return;
	}
	
	// Create method call to bluez manager interface requesting default adapter
	
	if(iOldBluez)
	{
		msg = dbus_message_new_method_call(BLUEZ_SERVICE,
			BLUEZ_REQUEST_PATH_OLD,
			BLUEZ_MANAGER_REQUEST_INTERFACE,
			BLUEZ_DEFAULT_ADAPTER_REQ);
	}
	else
	{
		msg = dbus_message_new_method_call(BLUEZ_SERVICE,
			BLUEZ_REQUEST_PATH_NEW,
			BLUEZ_MANAGER_REQUEST_INTERFACE,
			BLUEZ_DEFAULT_ADAPTER_REQ);
	}
	
	if(msg == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot create DefaultAdapter call.");
		return;
	}

	// Send message
	if(!dbus_connection_send_with_reply(iConnection,msg,&statecall,-1))
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot send method call (DefaultAdapter) with reply.");
		return;
	}

	// Cannot do a pending call
	if(statecall == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot execute a pending call (DefaultAdapter).");
		return;
	}

	dbus_connection_flush(iConnection);
	dbus_message_unref(msg);
	msg = NULL;

	// Block until reply received
	dbus_pending_call_block(statecall);

	// Get the replymessage
	if((msg = dbus_pending_call_steal_reply(statecall)) == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: No reply received for DefaultAdapter call.");
		return;
	}

	// If there is no adapter, NoSuchAdapter error is sent
	if(dbus_message_is_error(msg,BLUEZ_ERROR_NO_SUCH_ADAPTER))
	{
		DBG("CBluezBTListener::CheckInitialState: No adapter present (%s)", dbus_message_get_error_name(msg));
		iConverter->SetState(false);
		dbus_pending_call_unref(statecall);
		statecall = NULL;

		dbus_message_unref(msg);
		msg = NULL;
		return;
	}
	
	// Arguments are invalid - should not happen.
	if(dbus_message_is_error(msg,BLUEZ_ERROR_INVALID_ARGUMENTS))
	{
		DBG("CBluezBTListener::CheckInitialState: Invalid arguments for default adapter request - should not happen.");
		iConverter->SetState(false);
		dbus_pending_call_unref(statecall);
		statecall = NULL;

		dbus_message_unref(msg);
		msg = NULL;
		return;
	}
	
	// Otherwise check the value
	dbus_message_iter_init(msg,&iter);
	
	if((msgtype = dbus_message_iter_get_arg_type(&iter)) == DBUS_TYPE_STRING)
	{
		dbus_message_iter_get_basic(&iter,&mdata);
		DBG("CBluezBTListener::CheckInitialState: Default adapter: %s",mdata);
		request_path = mdata; // Use the received path
	}
	else if(msgtype == DBUS_TYPE_OBJECT_PATH)
	{
		dbus_message_iter_get_basic(&iter,&mdata);
		DBG("CBluezBTListener::CheckInitialState: Got object path for default adapter: %s",mdata);
		request_path = mdata;
	}
	else
	{
		DBG("CBluezBTListener::CheckInitialState: no default adapter? (type: %d)",msgtype);
	}
	
	dbus_pending_call_unref(statecall);
	statecall = NULL;
	
	dbus_message_unref(msg);
	msg = NULL;
	mdata = NULL;
	
	// Do not perform further checks if no request path received;
	if(request_path.size() == 0) return;
	
	// Otherwise parse and set the adapter
	else ParseAndSetAdapter(request_path);
	
	// Create new method call to request the adapter mode
	if(iOldBluez)
	{
		msg = dbus_message_new_method_call(BLUEZ_SERVICE,
			request_path.c_str(),
			BLUEZ_ADAPTER_REQUEST_INTERFACE,
			BLUEZ_MODE_REQ);
	}
	else
	{
		msg = dbus_message_new_method_call(BLUEZ_SERVICE,
			request_path.c_str(),
			BLUEZ_ADAPTER_REQUEST_INTERFACE,
			BLUEZ_PROPERTY_REQ);
	}
	
	if(msg == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot create new method call.");
		return;
	}

	// Send
	if(!dbus_connection_send_with_reply(iConnection,msg,&statecall,-1))
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot send method call with reply.");
		return;
	}
	
	// Cannot do a pending call
	if(statecall == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: Cannot execute a pending call.");
		return;
	}

	dbus_connection_flush(iConnection);
	dbus_message_unref(msg);

	msg = NULL;

	// Block until reply received
	dbus_pending_call_block(statecall);

	// Get the replymessage
	if((msg = dbus_pending_call_steal_reply(statecall)) == NULL)
	{
		DBG("CBluezBTListener::CheckInitialState: No reply received.");
		return;
	}
	
	// Check the mode
	dbus_message_iter_init(msg,&iter);
	
	if(!iOldBluez) IterateDBusMessage(&iter);
	else
	{
		if((msgtype = dbus_message_iter_get_arg_type(&iter)) == DBUS_TYPE_STRING)
		{
			dbus_message_iter_get_basic(&iter,&mdata);

			// Set off
			if(strcmp(mdata,BLUEZ_MODE_OFF) == 0)
			{
				DBG("CBluezBTListener::CheckInitialState: Bluetooth adapter off (passive)");
				iConverter->SetState(false);
			}
		
			// BT adapter on, can connect
			else if(strcmp(mdata,BLUEZ_MODE_CONNECTABLE) == 0)
			{
				DBG("CBluezBTListener::CheckInitialState: Bluetooth adapter connectable (active)");
				iConverter->SetState(true);
			}
		
			// BT adapter set as visible, can be seen as other devices
			else if(strcmp(mdata,BLUEZ_MODE_DISCOVERABLE) == 0)
			{
				DBG("CBluezBTListener::CheckInitialState: Bluetooth adapter discoverable (active)");
				iConverter->SetState(true);
			}
		
			// BT adapter set as visible, can be seen as other devices
			else if(strcmp(mdata,BLUEZ_MODE_LIMITED) == 0)
			{
				DBG("CBluezBTListener::CheckInitialState: Bluetooth adapter in limited visibility (active)");
				iConverter->SetState(true);
			}
			// Error
			else
			{
				DBG("CBluezBTListener::CheckInitialState: message not recognized: %s (passive)",mdata);
				iConverter->SetState(false);
			}
		}
	}

	// Free pending call
	dbus_pending_call_unref(statecall);
	statecall = NULL;
	
	dbus_message_unref(msg);
	msg = NULL;
	
	mdata = NULL;
}

/**
 * @memo Check the current state - not implemented
 * @doc Not ipmlemented - does nothing.
 */
void CBluezBTListener::CheckState()
{
	DBusMessage* message = NULL;

	// Check connection
	if(iConnection == NULL) return;

	// Allows messages to be read and marshalled from the wire on non-blocking connection
	// Second parameter is blocking time in milliseconds
	dbus_connection_read_write(iConnection,0);

	// Pop the first message
	if((message = dbus_connection_pop_message(iConnection)) != NULL)
	{
		HandleMessage(message);
		dbus_message_unref(message);
		message = NULL;
	}
	else return;
}

/**
 * @memo Returns the name of this listener
 * @doc Returns the name that was set in the constructor for this
 * listener.
 * 
 * @return The name of this listener
 */
const std::string& CBluezBTListener::GetName()
{
	return iName;
}

bool CBluezBTListener::RegisterSignals()
{
	DBusError error;
	std::string signal_and_if;

	// Check connection
	if(iConnection == NULL) return false;

	// Create signal that is listened
	// form: type='signal',interface='org.bluez.Manager'
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += BLUEZ_MANAGER_REQUEST_INTERFACE;
	signal_and_if += "\'";

	// Initialize error variable
	dbus_error_init(&error);

	// Connect to com.nokia.btcond.signal interface listening for 'signal'
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	// Make sure it is sent
	dbus_connection_flush(iConnection);

	// If cannot listen 
	if (dbus_error_is_set(&error))
	{
		ERR("CBluezBTListener::RegisterSignals: Cannot add listening to signal: %s, reason: %s", signal_and_if.c_str(), error.message);
		dbus_error_free(&error);
		return false;
	}
	
	signal_and_if.clear();
	
	// Create signal that is listened
	// form: type='signal',interface='org.bluez.Manager'
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += BLUEZ_ADAPTER_REQUEST_INTERFACE;
	signal_and_if += "\'";

	// Initialize error variable
	dbus_error_init(&error);

	// Connect to interface listening for 'signal'
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	// Make sure it is sent
	dbus_connection_flush(iConnection);

	// If cannot listen 
	if (dbus_error_is_set(&error))
	{
		ERR("CBluezBTListener::RegisterSignals: Cannot add listening to signal: %s, reason: %s", signal_and_if.c_str(), error.message);
		dbus_error_free(&error);
		return false;
	}

	signal_and_if.clear();
	
	return true;
}

bool CBluezBTListener::SetupConnection()
{
	DBusError error;

	// Initialize error
	dbus_error_init(&error);

	// Get system bus, private connection
	iConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM, &error);

	// Error
	if (iConnection == NULL)
	{
		if (dbus_error_is_set(&error))
		{
			ERR("CBluezBTListener::SetupConnection : Connection error: %s", error.message);

			// Free error
			dbus_error_free(&error);
		}
		return false;
	}
	else return true;
}

void CBluezBTListener::HandleMessage(DBusMessage* aMessage)
{
	const char *mdata;
	std::string path = "";
	DBusMessageIter iter;
	
	dbus_message_iter_init(aMessage,&iter);
	
	// Returns object path
	if(dbus_message_is_signal(aMessage,BLUEZ_MANAGER_REQUEST_INTERFACE,BLUEZ_SIGNAL_ADDED))
	{
		dbus_message_iter_get_basic(&iter,&mdata);
		/** TODO Get the path of added adapter! */
		DBG("CBluezBTListener::HandleMessage: new adapter: %s", mdata);
		SetAdapterChange(path);
		CommitChanges();
		mdata = NULL;
	}
	// Returns object path
	else if(dbus_message_is_signal(aMessage,BLUEZ_MANAGER_REQUEST_INTERFACE,BLUEZ_SIGNAL_DEFAULT_CHANGED))
	{
		dbus_message_iter_get_basic(&iter,&mdata);
		/** TODO De-register previous adapter interface listening and replace it with new one! */
		DBG("CBluezBTListener::HandleMessage: default adapter changed (%s)",mdata);
		path = mdata;
		ParseAndSetAdapter(path);
		mdata = NULL;
	}
	
	// Returns object path
	else if(dbus_message_is_signal(aMessage,BLUEZ_MANAGER_REQUEST_INTERFACE,BLUEZ_SIGNAL_REMOVED))
	{
		dbus_message_iter_get_basic(&iter,&mdata);
		/** TODO Check the adapter! */
		DBG("CBluezBTListener::HandleMessage: adapter removed (%s)",mdata);
		path = mdata;
		SetAdapterChange(path);
		CommitChanges();
		mdata = NULL;
	}
	else if(dbus_message_is_signal(aMessage,BLUEZ_ADAPTER_REQUEST_INTERFACE,BLUEZ_SIGNAL_PROPERTY_CHANGED))
	{
		iDictParsing = true;
		iDictKey = "";
		std::string rval = IterateDBusMessage(&iter);
		std::string prop = "none";
		if(dbus_message_iter_next(&iter)) prop = IterateDBusMessage(&iter);
		else DBG("CBluezBTListener::HandleMessage: cannot iterate");
		
		//DBG("CBluezBTListener::HandleMessage: property changed (key: %s, value: %s)",rval.c_str(), prop.c_str());
		
		iDictParsing = false;
	}
	/** TODO After adding a signal for org.bluez.Adapter check the mode changes that are received */
	
	path.clear();
	return;
}

const std::string CBluezBTListener::IterateDBusMessage(DBusMessageIter *iter)
{
	DBusMessageIter sub_iter;
	const char *mdata;
	int msgtype;
	std::string returnmsg = "";
	
	if((msgtype = dbus_message_iter_get_arg_type(iter)) == DBUS_TYPE_STRING)
	{
		dbus_message_iter_get_basic(iter,&mdata);
		
		if(iDictParsing)
		{
			if((iDictKey.compare("") == 0) &&
				((strcmp(mdata,"Address") == 0)||
				(strcmp(mdata,"Name") == 0)||
				(strcmp(mdata,"Powered") == 0)||
				(strcmp(mdata,"Discoverable") == 0)||
				(strcmp(mdata,"Pairable") == 0)||
				(strcmp(mdata,"DiscoverableTimeout") == 0)||
				(strcmp(mdata,"PairableTimeout") == 0)||
				(strcmp(mdata,"Discovering") == 0)||
				(strcmp(mdata,"Devices") == 0)))
			{
				iDictKey = std::string(mdata);
				
				returnmsg = "Dictionary key: ";
				returnmsg += iDictKey;
			}
			else
			{
				//DBG("Dict: key: %s, value: %s",iDictKey.c_str(),mdata);
				returnmsg = ": ";
				returnmsg += mdata;
				
				// Address has value string
				if(iDictKey.compare("Address") == 0) ;
				// Name has value string
				else if(iDictKey.compare("Name") == 0) ;
			}
		}

		/*************** ADAPTER MODES AS STRING - OLD BLUEZ *************************/
		// Set off
		else if(strcmp(mdata,BLUEZ_MODE_OFF) == 0)
		{
			iConverter->SetState(false);
			returnmsg = "Bluetooth adapter off (passive)";
		}
		
		// BT adapter on, can connect
		else if(strcmp(mdata,BLUEZ_MODE_CONNECTABLE) == 0)
		{
			iConverter->SetState(true);
			returnmsg = "Bluetooth adapter connectable (active)";
		}
		
		// BT adapter set as visible, can be seen as other devices
		else if(strcmp(mdata,BLUEZ_MODE_DISCOVERABLE) == 0)
		{
			iConverter->SetState(true);
			returnmsg = "Bluetooth adapter discoverable (active)";
		}
		
		// BT adapter set as visible, can be seen as other devices
		else if(strcmp(mdata,BLUEZ_MODE_LIMITED) == 0)
		{
			iConverter->SetState(true);
			returnmsg = "Bluetooth adapter in limited visibility (active)";
		}
		/******************* END OF OLD BLUEZ ADAPTER TYPES ***************************/
		
		// 
		else
		{
			returnmsg = "message not recognized: ";
			returnmsg += mdata;
			
			//dbus_message_iter_recurse(iter,&sub_iter);
			//if(dbus_message_iter_get_arg_type(&sub_iter) != DBUS_TYPE_INVALID)
			//	returnmsg += CBluezBTListener::IterateDBusMessage(&sub_iter);
			//iConverter->SetState(false);
		}

	}
	else if(msgtype == DBUS_TYPE_BOOLEAN)
	{
		dbus_bool_t bdata = false;
		dbus_message_iter_get_basic(iter,&bdata);
		
		if((iDictParsing)&&(iDictKey.compare("") != 0))
		{
			//DBG("Dict: key: %s, value: %s",iDictKey.c_str(), bdata ? "true" : "false");
			returnmsg = ": ";
			returnmsg += (bdata ? "true" : "false");
			
			if(iDictKey.compare("Powered") == 0)
			{
				//if(bdata) iConverter->SetState(true);
				//else iConverter->SetState(false);
				if(!iChanges)
				{
					iPowered = bdata;
					iChanges = true;
					CommitChanges();
				}
			}
			else if(iDictKey.compare("Discoverable") == 0)
			{
				iDiscoverable = bdata; // TODO force to discoverable?
				//iChanges = true;
			}
			else if(iDictKey.compare("Pairable") == 0)
			{
				iPairable = bdata;
				//iChanges = true;
			}
			else if(iDictKey.compare("Discovering") == 0)
			{
				iDiscovering = bdata;
				//iChanges = true;
			}
		}
		else
		{
			DBG("BOOLEAN %s", bdata ? "true" : "false");
			returnmsg = (bdata ? "true" : "false");
		}
		
	}
	else if(msgtype == DBUS_TYPE_UINT32)
	{
		dbus_uint32_t data;
		dbus_message_iter_get_basic(iter,&data);
	
		if((iDictParsing)&&(iDictKey.compare("") != 0))
		{
			//DBG("Dict: key: %s, value: %d",iDictKey.c_str(), data);
			returnmsg = ": ";
			returnmsg += (int)data;
			
			if(iDictKey.compare("DiscoverableTimeout") == 0) ;
			else if (iDictKey.compare("PairableTimeout") == 0) ;
		}
		else
		{
			DBG("UINT32 %d", data);
		}
	}
	else if(msgtype == DBUS_TYPE_DICT_ENTRY)
	{
		dbus_message_iter_recurse(iter,&sub_iter);
		
		iDictParsing = true;
		returnmsg = "";
		
		while(dbus_message_iter_get_arg_type(&sub_iter) != DBUS_TYPE_INVALID)
		{
			returnmsg += CBluezBTListener::IterateDBusMessage(&sub_iter);
			dbus_message_iter_next(&sub_iter);
		}
		
		iDictParsing = false;
		iDictKey = "";
	}
	else if(msgtype == DBUS_TYPE_VARIANT)
	{
		dbus_message_iter_recurse(iter,&sub_iter);
		returnmsg = "Variant: ";
		if(dbus_message_iter_get_arg_type(&sub_iter) != DBUS_TYPE_INVALID)
		{
			returnmsg += CBluezBTListener::IterateDBusMessage(&sub_iter);
		}
	}
	else if(msgtype == DBUS_TYPE_ARRAY)
	{	
		dbus_message_iter_recurse(iter,&sub_iter);
		returnmsg = "Array: ";
		while(dbus_message_iter_get_arg_type(&sub_iter) != DBUS_TYPE_INVALID)
		{
			std::string dbg = CBluezBTListener::IterateDBusMessage(&sub_iter);
			//DBG("CBluezBTListener::IterateDBusMessage: %s",dbg.c_str());
			dbus_message_iter_next(&sub_iter);
			returnmsg += dbg;
			returnmsg += ", ";
		}
	}
	else
	{
		returnmsg = "unknown (";
		returnmsg += msgtype;
		returnmsg += ")";
	}
	
	return returnmsg;
}

void CBluezBTListener::ParseAndSetAdapter(const std::string& aPath)
{
	int pos = -1, id = -1;
	
	if(aPath.size() == 0) return;
	
	// Get the last of "/" if found
	if((pos = aPath.find_last_of("/")) != -1)
	{	
		// Get the adapter at the end
		const std::string adapter = aPath.substr(pos+1);
	
		// Get the id number of adapter if it is a digit
		if(isdigit(adapter.at(adapter.size()-1)))
		{
			// ASCII character to digit "conversion"
			id = adapter.at(adapter.size()-1) - 48;
			DBG("CBluezBTListener::ParseAndSetAdapter: Set adapter %s with id %d.", adapter.c_str(), id);
			iConverter->SetAdapter(adapter,id);
			iAdapterChange = adapter;
			CommitChanges();
		}
		else
		{
			DBG("CBluezBTListener::ParseAndSetAdapter: Cannot determine adapter id for adapter %s. Nothing done.", adapter.c_str());
		}
	}
	// Position was not found
	else DBG("CBluezBTListener::ParseAndSetAdapter: no \"/\" in given path %s", aPath.c_str());
}

void CBluezBTListener::SetAdapterChange(const std::string& aPath)
{
	int pos = -1;
	
	if(aPath.size() == 0) return;
	
	// Get the last of "/" if found
	if((pos = aPath.find_last_of("/")) != -1)
	{	
		// Get the adapter at the end
		iAdapterChange = aPath.substr(pos+1);
		DBG("CBluezBTListener::SetAdapterChange: adapter which has changes: %s", iAdapterChange.c_str());
	}
}

void CBluezBTListener::CommitChanges()
{
	// If changes made and adapter set
	if(iChanges && iAdapterChange.size() != 0)
	{
		DBG("CBluezBTListener::CommitChanges: Changes for adapter %s.",iAdapterChange.c_str());
		
		if(iConverter->GetAdapter().compare(iAdapterChange) == 0)
		{
			DBG("CBluezBTListener::CommitChanges: Changes were made for default adapter.");
			// TODO utilize other values too?
			iConverter->SetState(iPowered);
			iChanges = false;
			iAdapterChange = "";
		}
		else DBG("CBluezBTListener::CommitChanges: Adapters do not match.");
	}
	else if(iAdapterChange.size() == 0) DBG("CBluezBTListener::CommitChanges: No adapter set.");
	else DBG("CBluezBTListener::CommitChanges: No changes for adapter %s.", iAdapterChange.c_str());
}
