/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003-2010 LUT. .
 *
 * @name AbstractConnection.h
 * @memo Definition of the PeerHood Connection interface.
 *
 * @version 0.31
 * date     11.06.2003
 * change   28.04.2010
 */

#ifndef __ABSTRACTCONNECTION_H__
#define __ABSTRACTCONNECTION_H__

#ifdef __cplusplus

#include <string>

/**
 * @memo Defines the PeerHood Connection interface.
 * @doc Defines the PeerHood Connection interface. The interface is pure &
 * virtual so each networking technology needs its own implementation of the
 * interface. The PeerHood system uses this interface when communicating with
 * other PeerHood devices. The goal of this interface is to completely hide the
 * underlying networking tehchnology from the PeerHood system. This means that
 * the very same logic can be used when working with different networking
 * technologies, so no modifications to the PeerHood core are required if a new
 * networking technology is to be supported.
 *
 * @see CBTConnection
 */
class MAbstractConnection
{
 public:
  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * virtual base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MAbstractConnection() {};


  /**
   * @memo Accepts an incoming connection.
   * @doc Accepts an incoming connection. If the connection is accepted
   * succesfully then it's possible to send and receive data using the returned
   * connection object.
   *
   * @return new connection object or NULL in the case of an error
   */
  virtual MAbstractConnection* AcceptL() = 0;


  /**
   * @memo Tells if the connection is in the listening state.
   * @doc Tells if the connection is in the listening state. The listening
   * state doesn't necessarily mean the "listen" state of a traditional socket.
   * Instead, it can mean any state such that the actual networking technology
   * is able to accept incoming connection requests.
   *
   * @return true if the connection is in the listening state
   */
  virtual bool IsListening() = 0;


  /**
   * @memo Tells if a connection is established to a remote device.
   * @doc Tells if connection to a remote device is established.
   *
   * @return true if connection to a remote device is established
   */
  virtual bool IsConnected() = 0;


  /**
   * @memo Moves the connection object to the listening state.
   * @doc Moves the connection object to the listening state. After this
   * fuction call the connection object should be able to receive
   * incoming connection requests.
   *
   * @return true is the connection is moved to the listening state, otherwise
   * false
   */
  virtual bool Listen() = 0;

  /**
   * @memo Moves the connection object to the listening state.
   * @doc Moves the connection object to the listening state. After this
   * fuction call the connection object should be able to receive
   * incoming connection requests.
   *
   * @return true is the connection is moved to the listening state, otherwise
   * false
   */
  virtual bool HasData() = 0;

  /**
   * @memo Moves the connection object to the listening state.
   * @doc Moves the connection object to the listening state. After this
   * fuction call the connection object should be able to receive
   * incoming connection requests. This version of the <code>Listen</code>
   * call should be used if some defined port (or equivalent) should be used
   * instead of the networking technology's default port. Note that using other
   * than the default port can have hazardous effects to the system!
   *
   * @param aPort The port that should be listened.
   *
   * @return true if the connection is moved to the listening state, otherwise
   * false
   */
  virtual bool Listen(int aPort) = 0;


  /**
   * @memo Closes the connection in uncontrolled way.
   * @doc Closes an open connection without any further notice. No error codes 
   * are returned even if this function called for a closed connection. In that 
   * case the program execution should continue normally. <b>This function 
   * should not be called except in extreme situations</b> - the normal way to 
   * close a connection is the <code>Disconnect</code> function!.
   *
   * @return none
   */
  virtual void Close() = 0;


  /**
  * @memo Tries to establish a new connection.
  * @doc Tries to establish new connection to a remote device. Note that users
  * should not call this function directly. Instead, they are supposed to use
  * the <code>Connect</code> function in the PeerHood interface. Remote
  * device's address etc. are obtained as a part of PeerHood neighborhood
  * information.
  *
  * @param aAddress Remote device's address.
  * @param aPort Remote device's port.
  *
  * @return true if the connection establishment succeeded
  */
  virtual bool Connect(const std::string& aAddress, int aPort) = 0;


  /**
   * @memo Closes the connection in a controlled way.
   * @doc Closes the connection in a controlled way. This is the normal way to 
   * close the connection. The similar function <code>Close</code> should not
   * be used unless necessary.
   *
   * @return true if the connection was closed normally
   */
  virtual bool Disconnect() = 0;


  /**
   * @memo Sends data to the remote device.
   * @doc Sends data to the remote device. A connection must be established
   * before any data can be transmitted. There's no limit for the data lenght
   * so if the underlying networking techonlogy requires segmentation it must
   * be hidden from the user.
   *
   * @param aOutBuf Buffer containing the data to be sent.
   * @param aLength The length of the data to be sent.
   *
   * @return the number of bytes sent
   */
  virtual int Write(const void* aOutBuf, int aLength) = 0;


  /**
   * @memo Reads data send by a remote device.
   * @doc Reads data send by a remote device. The reserved buffer must be large
   * enough to hold the requested amount of data. If the underlying networking
   * technology requires segmentation then it must be implemented transparently
   * to the user.
   *
   * @param aInBuf The destination buffer.
   * @param aLength The number of bytes to be read.
   * 
   * @return the number of bytes read
   */
  virtual int Read(void* aInBuf, int aLength) = 0;


  /**
   * @memo Returns the file descriptor attached to the connection.
   * @doc Return the file descriptor attached to the current active connection.
   * 
   * @return the active file descriptor or -1 if there's no connection
   */
  virtual int GetFd() = 0;


  /**
   * @memo Returns the address of the remote device.
   * @doc Returns the address of the remote device. If there's no connection
   * then the returned address is empty.
   *
   * @return the address of the remote device
   */
  virtual const std::string& GetRemoteAddress() = 0;
  
  /**
	 * @memo Returns the checksum of the peer device
	 * @doc Returns the checksum of the peer device.
	 *
	 * @param _aAbstractConnection pointer to connection object to use
	 *
	 * @return checksum in host byte order
	 */
  virtual unsigned int GetDeviceChecksum() = 0;
  
  /**
   * @memo Set device checksum for connection
   * @doc Set device checksum for connection
   *
   * @param checksum in host byte order
   */
  virtual void SetDeviceChecksum(unsigned int checksum) = 0;
};

#else
	typedef
		struct MAbstractConnection
			MAbstractConnection;
#endif // __cplusplus

#ifdef __cplusplus
extern "C" {
#endif

void ph_c_connection_delete(MAbstractConnection* _aAbstractConnection);

int ph_c_connection_is_listening(MAbstractConnection* _aAbstractConnection);

int ph_c_connection_is_connected(MAbstractConnection* _aAbstractConnection);

int ph_c_connection_has_data(MAbstractConnection* _aAbstractConnection);

int ph_c_connection_disconnect(MAbstractConnection* _aAbstractConnection);

int ph_c_connection_write(MAbstractConnection* _aAbstractConnection, const void* _aOutBuf, int _aLength);

int ph_c_connection_read(MAbstractConnection* _aAbstractConnection, void* _aInBuf, int _aLength);

int ph_c_connection_get_fd(MAbstractConnection* _aAbstractConnection);

const char* ph_c_connection_get_remote_address(MAbstractConnection* _aAbstractConnection);

unsigned int ph_c_connection_get_device_checksum(MAbstractConnection* _aAbstractConnection);

#ifdef __cplusplus
}
#endif



#endif
