/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003-2010 LUT. .
 *
 * @name PluginStorage.cc
 * @memo Implementation of plugin storage.
 *
 * @version 0.11
 * date     23.02.2010
 * change   17.02.2011
 */

#include "PluginStorage.h"
#include <iostream>

PluginStorage* PluginStorage::iInstance = NULL;

PluginStorage::PluginStorage()
{
	iInstance = this;
	std::map<const char*, MAbstractPlugin*> iPlugins;
	pthread_mutex_init(&iPluginStorageLock, NULL);
}

PluginStorage::~PluginStorage()
{
	delete iInstance;
	iInstance = NULL;
	iPlugins.clear();
	pthread_mutex_destroy(&iPluginStorageLock);
}

PluginStorage* PluginStorage::GetInstance()
{
	if(iInstance == NULL) new PluginStorage;
	
	return iInstance;
}

void PluginStorage::AddPlugin(const char* aName, MAbstractPlugin* aPlugin)
{
	Lock();
	if(iPlugins.find(aName) == iPlugins.end())
	{
		DBG("Pluginstorage::AddPlugin: %s.", aName);
		iPlugins[aName] = aPlugin;
	}
	else DBG("Pluginstorage::AddPlugin: %s already in db.", aName);
	UnLock();
}

std::map<const char*, MAbstractPlugin*>::iterator PluginStorage::Begin()
{
	Lock();
	return iPlugins.begin();
}

std::map<const char*, MAbstractPlugin*>::iterator PluginStorage::End()
{
	return iPlugins.end();
}

void PluginStorage::Stop()
{
	UnLock();
}

void PluginStorage::Lock()
{
	pthread_mutex_lock(&iPluginStorageLock);
}

void PluginStorage::UnLock()
{
	pthread_mutex_unlock(&iPluginStorageLock);
}
