/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name DaemonDevice.cc
 * @memo Implementation of the MAbstractDevice. This version is used by the
 * PeerHood daemon and the plugins.
 *
 * @version 0.1
 * date     08.04.2003
 * change   08.04.2003
 */
#include <iostream>
#include <cstring>
#include <DaemonDevice.h>

//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

/**
 * @memo Copy constructor.
 * @doc Copy constructor. This makes the new object to be an exact copy of the
 * original object.
 *
 * @param CDaemonDevice The object to be copied.
 *
 * @return none
 */
CDaemonDevice::CDaemonDevice(const CDaemonDevice& aOriginal)
{
  CDaemonDevice *newDaemonDevice;
  char *newProto;

  iTimestamp = aOriginal.iTimestamp;
  iHasPeerHood = aOriginal.iHasPeerHood;
  iName = aOriginal.iName;
  iAddress = aOriginal.iAddress;
  iPrototype = aOriginal.iPrototype;
  iChecksum = aOriginal.iChecksum;
  iReferrerName = aOriginal.iReferrerName;

  //  DBG("Device %s created", iAddress.c_str());
  //  DBG("Daemondevice: servicelist size %d", aOriginal.iServiceList.size());

  // add services into this new daemon
  for (std::list<CService*>::const_iterator i = aOriginal.iServiceList.begin();i != aOriginal.iServiceList.end();++i) {
    //    DBG("   New ddev: name %s", (*i)->GetName().c_str());
    //    DBG("   New ddev: attr %s", (*i)->GetAttributeString().c_str());
    //    DBG("   New ddev: pid %d", (*i)->GetPid());
    //    DBG("   New ddev: port %d", (*i)->GetPort());

    // Add service to this new daemon device, use deamondevice internal function
    AddService(new CService((**i)));
  }

  //  DBG("Daemondevice: protolist size %d", aOriginal.iSupportedPrototypes.size());
  // add prototypes into this new daemon
  for (std::list<char*>::const_iterator j = aOriginal.iSupportedPrototypes.begin();j != aOriginal.iSupportedPrototypes.end();++j) {

    newProto = new char[strlen((*j))+1];
    strncpy(newProto,(*j), strlen((*j)));
    newProto[strlen((*j))] = '\0';

    // Add prototypes into this new daemon device
    AddPrototype(newProto);
  }


  //  DBG("Daemondevice: neigh list size %d", aOriginal.iNeighboringDevices.size());
  // add neighbors into this new daemon
  for (std::list<CDaemonDevice*>::const_iterator k = aOriginal.iNeighboringDevices.begin();k != aOriginal.iNeighboringDevices.end();++k) {

    //    DBG("Add neighbor %s into %s", (*k)->GetAddress().c_str(), iName.c_str());

    newDaemonDevice = new CDaemonDevice((*k)->GetAddress());
    newDaemonDevice->SetPeerHood((*k)->HasPeerHood());
    newDaemonDevice->SetDeviceName((*k)->GetName());
    newDaemonDevice->SetPrototype((*k)->GetPrototype().c_str());
    //    newDaemonDevice->SetChecksum((*k)->GetChecksum());
    //    newDaemonDevice->SetTimestamp((*k)->GetTimestamp());

    AddNeighboringDevice(newDaemonDevice);
  }
}

/**
 * @memo Constructor, that creates a device from the marshalled data.
 * @doc Constructor, that creates a device from the marshalled data. The
 + values will be copied from the data
 *
 * @param aData Marshalled data of the device information
 *
 * @return none
 */
CDaemonDevice::CDaemonDevice(const char* aData)
{
  int position;
  CService* temp;

  //  DBG("New daemon device created");

  iAddress = std::string(&aData[10]);
  iPrototype = std::string(&aData[10 + aData[0] + 1]);
  iName = std::string(&aData[10 + aData[0] + aData[1] + 2]);
  iReferrerName = std::string(&aData[10 + aData[0] + aData[1] + aData[3] + 3]);
  iHasPeerHood = (bool)aData[2];
  memcpy(&iChecksum, &aData[5], sizeof(iChecksum));

  //  if (iHasPeerHood) {
  //    DBG("   Device %s with Peerhood", iAddress.c_str());
  //    DBG("      Address %s", iAddress.c_str());
  //    DBG("      Name %s", iName.c_str());
  //    DBG("      Checksum %d", iChecksum);
  //    DBG("      Proto %s", iPrototype.c_str());
  //    DBG("      Referrer %s", iReferrerName.c_str());
  //  }
  //  else {
  //    DBG("   Device %s does not support Peerhood", iAddress.c_str());
  //  }

  // could also use aData fields (0,1,3,4)
  position = 10 + iAddress.length() + iPrototype.length() + iName.length() + iReferrerName.length() + 4;

  // unmarshal services
  for (int i = 0;i < aData[9];i++) {
    temp = new CService(&aData[position]);
    position += temp->MarshalledLength();
    AddService(temp);
  }

}

/**
 * @memo Constructor, sets internal variables to their default values.
 * @doc Constructor, sets internal variables to their default values. These
 * values include timestamp, name, the number of services, PeerHood awaresness
 * etc.
 *
 * @param aDeviceAddress the address of the device
 *
 * @return none
 */
CDaemonDevice::CDaemonDevice(const std::string& aDeviceAddress)
{
  //  DBG("CDeamonDevice(addr) constructor for %s", aDeviceAddress.c_str());

  iAddress = std::string(aDeviceAddress);
  iHasPeerHood = false;
  iName = std::string("empty");
  iPrototype = std::string("empty");
  iReferrerName = std::string("empty");
  iChecksum = 0;
  iTimestamp = 0;
}


/**
 * @memo Constructor with referrer, sets internal variables to their default values.
 * @doc Constructor, sets internal variables to their default values. These
 * values include timestamp, name, the number of services, PeerHood awaresness,
 * referrer device for neighbourhood information etc.
 *
 * @param aDeviceAddress the address of the device,
 +        aReferrerName the name of the referring neighbourhood
 *
 * @return none
 */
CDaemonDevice::CDaemonDevice(const std::string& aDeviceAddress, const std::string& aReferrerName)
{
  iAddress = std::string(aDeviceAddress);
  iHasPeerHood = false;
  iName = std::string("empty");
  iPrototype = std::string("empty");
  iReferrerName = aReferrerName;
  iChecksum = 0;
  iTimestamp = 0;
}

/**
 * @memo Sets device's timestamp value. This value can be used by the plugins.
 * @doc Sets device's timestamp value. This value can be used by the plugins
 * e.g. to determine if the device should be removed from the device & service
 * database.
 *
 * @param aValue new timestamp value
 *
 * @return none
 */
void CDaemonDevice::SetTimestamp(int aValue)
{
  iTimestamp = aValue;
}


/**
 * @memo Increases device's timestamp by one.
 * @doc Increases device's timestamp by one.
 *
 * @return none
 */
void CDaemonDevice::IncreaseTimestamp()
{
  iTimestamp++;
}


/**
 * @memo Returns device's current timestamp value.
 * @doc Returns device's current timestamp value.
 *
 * @return the current timestamp of the device
 */
int CDaemonDevice::GetTimestamp()
{
  return iTimestamp;
}


/**
 * @memo Sets device's prototype id.
 * @doc Sets device's prototype id. This id is unique for each network, so
 * in practice each plugin should use a different id.
 *
 * @return none
 */
void CDaemonDevice::SetPrototype(const char* aProto)
{
  iPrototype = std::string(aProto);
}


/**
 * @memo Converts class' information to an octet stream.
 * @doc Converts class' information to an octet stream. Note that memory
 * allocated for the return value must be freed by the caller. The format of
 * the returned stream is (in octets):
 * <ll>
 * <li>0 length of the name field</li>
 * <li>1 length of the prototype field</li>
 * <li>2..5 device's port number</li>
 * <li>6 flag indicating the presence of PeerHood (0 = no, 1 = yes)
 * <li>7.. name of the device</li>
 * <li>n.. prototype id of the device</li>
 * </ll>
 *
 * @param aLength Integer that on return will hold the length of the created
 * octet stream.
 *
 * @return octet presentation of the class' information
 */
char* CDaemonDevice::MarshallL(int* aLength)
{
  int length;
  int position;
  char* retval;
  char* temp;

  *aLength = 10; // reserved for fields 1, 2, 3, 4, 5, 6
  *aLength += iPrototype.size() + 1;
  *aLength += iAddress.size() + 1;
  *aLength += iName.size() + 1;
  *aLength += iReferrerName.size() + 1;
  position = *aLength;

  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    *aLength += (*i)->MarshalledLength();
  }

  retval = new char[*aLength];
  bzero(retval, *aLength);
  retval[0] = (unsigned char)iAddress.length();
  retval[1] = (unsigned char)iPrototype.length();
  retval[2] = (unsigned char)iHasPeerHood;
  retval[3] = (unsigned char)iName.length();
  retval[4] = (unsigned char)iReferrerName.length();
  memcpy(&retval[5], &iChecksum, sizeof(iChecksum));
  retval[9] = (unsigned char)iServiceList.size();

  strcpy(&retval[10], iAddress.c_str());
  strcpy(&retval[10 + iAddress.length() + 1], iPrototype.c_str());
  strcpy(&retval[10 + iAddress.length() + iPrototype.length() + 2], iName.c_str());
  strcpy(&retval[10 + iAddress.length() + iPrototype.length() + iName.length() + 3], iReferrerName.c_str());

  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    temp = (*i)->MarshallL(&length);
    memcpy(&retval[position], temp, length);

    //    DBG("Setting service to position %d, length %d", position, length);

    position += (*i)->MarshalledLength();
    delete[] temp;
  }

  return retval;
}


/**
 * @memo Sets device's PeerHood flag.
 * @doc Sets device's PeerHood flag. If the flag is set then the device has
 * PeerHood capability, otherwise there's no sign of it.
 *
 * @param aHasPeerHood New value of the PeerHood flag.
 *
 * @return none
 */
void CDaemonDevice::SetPeerHood(bool aHasPeerHood)
{
  iHasPeerHood = aHasPeerHood;
}

/**
 * @memo Sets device's name.
 * @doc Sets devices's name.
 *
 * @param aDeviceName User-defined device name
 *
 * @return none
 */
void CDaemonDevice::SetDeviceName(const std::string& aDeviceName)
{
  iName = std::string(aDeviceName);
}

/**
 * @memo Sets device's checksum number.
 * @doc Sets device's checksum number
 *
 * @param aChecksum New checksum number for device
 *
 * @return none
 */
void CDaemonDevice::SetChecksum(unsigned int aChecksum)
{
  iChecksum = aChecksum;
}

/**
 * @memo Sets referrer's name.
 * @doc Sets referrer's name. If set, this device is not connected directly but through
 * a neighbour. Used when neighbourhood information is used.
 *
 * @param aReferrerName User-defined device name
 *
 * @return none
 */
void CDaemonDevice::SetReferrerName(const std::string& aReferrerName)
{
  iReferrerName = std::string(aReferrerName);
}

const std::string& CDaemonDevice::GetReferrerName()
{
  return iReferrerName;
}


/**
 * @memo Adds a new neigboring device to the device.
 * @doc Adds a new neigboring device to the device. No checking to the validity of the
 * given service entry is performed, except the NULL check. If the given
 * service is NULL then an assertion will take place.
 *
 * @return none
 */
void CDaemonDevice::AddNeighboringDevice(CDaemonDevice* aDaemonDevice)
{
  assert(aDaemonDevice != NULL);
  iNeighboringDevices.push_back(aDaemonDevice);
}

/**
 * @memo Adds a new neigboring device to the device.
 * @doc Adds a new neigboring device to the device. No checking to the validity of the
 * given service entry is performed, except the NULL check. If the given
 * service is NULL then an assertion will take place.
 *
 * @return none
 */
void CDaemonDevice::AddPrototype(char* aPrototype)
{
  iSupportedPrototypes.push_back(aPrototype);
}

int CDaemonDevice::GetNeighborListSize()
{
  return iNeighboringDevices.size();
}


void CDaemonDevice::DebugDevice()
{

  DBG("DDevice debug: Addr %s", iAddress.c_str());
  DBG("DDevice debug: Name %s", iName.c_str());
  DBG("DDevice debug: CSum %d", iChecksum);
  DBG("DDevice debug: Ref  %s", iReferrerName.c_str());
  DBG("DDevice debug: PH   %d", iHasPeerHood);
  DBG("DDevice debug: Prot %s", iPrototype.c_str());

  DBG("DDevice debug: supported prototypes %zu", iSupportedPrototypes.size());
  for (std::list<char*>::const_iterator j = iSupportedPrototypes.begin();j != iSupportedPrototypes.end();++j) {
    DBG("   Prototype: %s",(*j));
  }

  DBG("DDevice debug: services %zu", iServiceList.size());
  for (std::list<CService*>::const_iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    DBG("   Service: name %s", (*i)->GetName().c_str());
    DBG("   Service: attr %s", (*i)->GetAttributeString().c_str());
    DBG("   Service: pid %d", (*i)->GetPid());
    DBG("   Service: port %d", (*i)->GetPort());
  }

  DBG("DDevice debug: neighboring devices %zu", iNeighboringDevices.size());
  for (std::list<CDaemonDevice*>::const_iterator k = iNeighboringDevices.begin();k != iNeighboringDevices.end();++k) {
    DBG("   Neighbor: Address %s", (*k)->GetAddress().c_str());
  }
}

bool CDaemonDevice::CheckNeighbors(const std::string& aAddress)
{
  bool foundNeighbor = false;

  for (std::list<CDaemonDevice*>::const_iterator k = iNeighboringDevices.begin();k != iNeighboringDevices.end();++k) {
    //    DBG("Comparing %s and %s", aAddress.c_str(), (*k)->GetAddress().c_str());
    if(aAddress.compare((*k)->GetAddress())==0) {
      foundNeighbor = true;
      break;
    }
  }

  return foundNeighbor;

}
