/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2010 LUT. .
 *
 * @name AbstractConnectionCWrapper.cc
 * @memo Implementations of C API functions for MAbstractConnection usage.
 *
 * @version 0.21
 * date     04.01.2010
 * change   28.04.2010
 */
 
#include "AbstractConnection.h"

/**
 * @memo Deletes the MAbstractConnection object.
 * @doc Deletes the connection object. Tries to disconnect before
 * deletion of the object.
 *
 * @param _aAbstractConnection pointer to connection object to use
 */
void ph_c_connection_delete(MAbstractConnection* _aAbstractConnection)
{
	_aAbstractConnection->Disconnect();
	delete _aAbstractConnection;
}

/**
 * @memo Tells if the connection is in the listening state.
 * @doc Tells if the connection is in the listening state. The listening
 * state doesn't necessarily mean the "listen" state of a traditional socket.
 * Instead, it can mean any state such that the actual networking technology
 * is able to accept incoming connection requests.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return 1 (TRUE) if the connection is in the listening state
 */
int ph_c_connection_is_listening(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->IsListening() ? 1 : 0;
}

/**
 * @memo Tells if a connection is established to a remote device.
 * @doc Tells if connection to a remote device is established.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return 1 (TRUE) if connection to a remote device is established
 */
int ph_c_connection_is_connected(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->IsConnected() ? 1 : 0;
}

/**
 * @memo Moves the connection object to the listening state.
 * @doc Moves the connection object to the listening state. After this
 * fuction call the connection object should be able to receive
 * incoming connection requests.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return 1 (TRUE) is the connection is moved to the listening state, otherwise
 * 0 (FALSE)
 */
int ph_c_connection_has_data(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->HasData() ? 1 : 0;
}

/**
 * @memo Closes the connection in a controlled way.
 * @doc Closes the connection in a controlled way. This is the normal way to 
 * close the connection. The similar function <code>Close</code> should not
 * be used unless necessary.
 *
 * @param _aAbstractConnection pointer to connection object to close
 *
 * @return 1 (TRUE) if the connection was closed normally, 0 (FALSE) otherwise
 */
int ph_c_connection_disconnect(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->Disconnect() ? 1 : 0;
}

/**
 * @memo Sends data to the remote device.
 * @doc Sends data to the remote device. A connection must be established
 * before any data can be transmitted. There's no limit for the data lenght
 * so if the underlying networking techonlogy requires segmentation it must
 * be hidden from the user.
 *
 * @param _aAbstractConnection pointer to connection object to use
 * @param _aOutBuf Buffer containing the data to be sent.
 * @param _aLength The length of the data to be sent.
 *
 * @return the number of bytes sent
 */
int ph_c_connection_write(MAbstractConnection* _aAbstractConnection, const void* _aOutBuf, int _aLength)
{
	return _aAbstractConnection->Write(_aOutBuf, _aLength);
}

/**
 * @memo Reads data send by a remote device.
 * @doc Reads data send by a remote device. The reserved buffer must be large
 * enough to hold the requested amount of data. If the underlying networking
 * technology requires segmentation then it must be implemented transparently
 * to the user.
 *
 * @param _aAbstractConnection pointer to connection object to use
 * @param _aInBuf The destination buffer.
 * @param _aLength The number of bytes to be read.
 * 
 * @return the number of bytes read
 */
int ph_c_connection_read(MAbstractConnection* _aAbstractConnection, void* _aInBuf, int _aLength)
{
	return _aAbstractConnection->Read(_aInBuf, _aLength);
}

/**
 * @memo Returns the file descriptor attached to the connection.
 * @doc Return the file descriptor attached to the current active connection.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return the active file descriptor or -1 if there's no connection
 */
int ph_c_connection_get_fd(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->GetFd();
}

/**
 * @memo Returns the address of the remote device.
 * @doc Returns the address of the remote device. If there's no connection
 * then the returned address is empty.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return the address of the remote device
 */
const char* ph_c_connection_get_remote_address(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->GetRemoteAddress().c_str();
}

/**
 * @memo Returns the checksum of the peer device
 * @doc Returns the checksum of the peer device.
 *
 * @param _aAbstractConnection pointer to connection object to use
 *
 * @return checksum in host byte order
 */
unsigned int ph_c_connection_get_device_checksum(MAbstractConnection* _aAbstractConnection)
{
	return _aAbstractConnection->GetDeviceChecksum();
}

