/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name WLANPinger.cc
 * @memo WLAN implementation of the MAbstractPinger interface.
 *
 * @version 0.1
 * date     30.06.2003
 * change   30.06.2003
 */

#include <stdio.h>
#include <errno.h>
#include <iostream>
#include <string>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include "WLANPinger.h"

#define MAX_PACKET 10*1024     // Maximum packet size to be sent or received

/**
 * @memo Constructor.
 * @doc Constructor, initializes the pinger object so that it can be used
 * immediately after the construction has finished.
 *
 * @param aAddress Remote device's address.
 *
 * @return none
 */
CWLANPinger::CWLANPinger(const std::string& aAddress) : MAbstractPinger(aAddress)
{
  iAddress = std::string(aAddress);
  iInRange = true;
}


/**
 * @memo Checks if a remote device is in range.
 * @doc Checks if a remote device is in range. The check is done by sending
 * the ECHO request. If a response isn't received during one second then the
 * remote device is assumed to be out of range.
 *
 * @return true if the remote device is in range, otherwise false
 */
bool CWLANPinger::Ping()
{
  struct sockaddr_in from;         // Socket address struct for receiving
  struct sockaddr_in to;           // Socket address struct for sending
  struct pollfd pf[1];             // Poll filedescriptor
  socklen_t addrlen;               // Storage for address length
  char    buf[MAX_PACKET];         // Send buffer
  char    rbuf[MAX_PACKET];        // Receive buffer
  int     len, rlen;               // Size variables for messages
  struct  icmp *icmpHdr;          // ICMP message header
  struct  icmp *recvIcmp;         // ICMP header for received messages
  int     s;                       // Descriptor for socket
  struct hostent *rmtHost;        // Hostent struct for address resolving
  int pollvalue;                   // Storage for poll return value
  int pingcount = 0;               // Counter for number of pings
  u_int16_t icd_seq = 0;           // Sequence number of ICMP echo message



  // Get host name for given address
  if ((rmtHost = gethostbyname(iAddress.c_str())) == (struct hostent *)NULL) {
      std::cout << "unresolvable hostname:" << iAddress << std::endl;
    return false;
  }

  // Create socket
  if ((s = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0) {
    perror("socket");
    return false;
  }

  // Set socket to non-blocking state
  if (fcntl (s, F_SETFL, O_NONBLOCK) == -1) {
      perror("fcntl");
      close(s);
      return false;
  }
  
  // Zero the buffer
  memset( buf, 0, sizeof(buf));
  
  // Build ICMP header for ICMP echo message
  icmpHdr = (struct icmp *)buf;
  icmpHdr->icmp_type = ICMP_ECHO;
  icmpHdr->icmp_cksum = 0;
  memset(buf + sizeof(struct icmp),'X', MAX_PACKET - sizeof(struct icmp));
  len = 64;
  
  // Fill to sockaddr struct
  memset(&to,0,sizeof(to));
  memcpy((void *)&to.sin_addr, (void *)rmtHost->h_addr_list[0],
	 sizeof(to.sin_addr));
  to.sin_family = AF_INET;

  icmpHdr->icmp_hun.ih_idseq.icd_seq=++icd_seq;
  icmpHdr->icmp_cksum = in_cksum((u_short *)icmpHdr,64);

  // Send message
  len = sendto( s, buf, len, 0, (struct sockaddr *)&to, sizeof(to) );
  if( len == -1 ){
      perror("sendto");
      close(s);
      return false;
  }

  while (1) {
    pf[0].fd = s; pf[0].events = POLLIN;
    pollvalue = poll(pf, 1, 1000);
    if(pollvalue < 0 ) {
      perror("Poll");
      close(s);
      return false;
    }
    if (pollvalue == 0) {
      if (++pingcount > 3) {
	close(s);
	return false;
      }
    }
    
    if (pf[0].revents &POLLIN) {
      addrlen=(socklen_t)sizeof(from);
      rlen = 1;
      while (rlen > 0) {
	rlen = recvfrom( s, rbuf,
			 sizeof(rbuf), 0, (struct sockaddr *) &from,
			 &addrlen );
	if( rlen == -1 ){
	  if (errno == EAGAIN) {
	    rlen = 0;
	    continue;
	  }
	  perror("recvfrom");
	  close(s);
	  return false;
	}
	
	// Check if packet contained IP header
	if (rbuf[0] == 0x45) {
	  recvIcmp = (struct icmp *)((char *)rbuf + sizeof(iphdr));
	}
	else {
	  recvIcmp = (struct icmp *)rbuf;
	}
	
	// Check if received packet was ICMP reply	
	if (recvIcmp->icmp_type != ICMP_ECHOREPLY) {
	  continue;
	}
	
	// Check that sequence number is correct	
	if (recvIcmp->icmp_hun.ih_idseq.icd_seq != icd_seq) {
	  continue;
	}

	close(s);
	return true;
      }
    } else {
      break;
    }
  }  

  close(s);
  return false;
}


/**
 * @memo Returns remote device's WLAN address.
 * @doc Return the WLAN address of the device under monitoring.
 *
 * @return remote device's address
 */
const std::string& CWLANPinger::GetAddress()
{
  return iAddress;
}


/**
 * @memo Tells whether a device is in range or not.
 * @doc Tells whether a device is in range or not.
 *
 * @return true if the device is in range
 */
bool CWLANPinger::InRange()
{
  return iInRange;
}

/**
 * @memo Calculates the checksum for ICMP header.
 * @doc Calculates the checksum for ICMP header.
 *
 * @return checksum
 */
u_short CWLANPinger::in_cksum(u_short *addr, int len)
 {
  register int nleft = len;
  register u_short *w = addr;
  register int sum = 0;
  u_short answer = 0;
  while( nleft > 1 )  {
    sum += *w++; 
    nleft -= 2;
  }
  if( nleft == 1 ) {
    *(u_char *)(&answer) = *(u_char *)w ;
    sum += answer;
  }
  sum = (sum >> 16) + (sum & 0xffff);     // add hi 16 to low 16
  sum += (sum >> 16);                     // add carry
  answer = ~sum;                          // truncate to 16 bits
  return (answer);
}
