/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name BluezBTListener.h
 * @memo Bluez-specific implementation of MAbstractListener
 *
 * @version 0.18
 * date     14.11.2008
 * change   11.03.2009
 */

#ifndef __BLUEZBTLISTENER_H_
#define __BLUEZBTLISTENER_H_

#include <dbus/dbus.h>
#include <syslog.h>
#include <stdio.h>
#include "AbstractListener.h"
#include "AbstractStateConverter.h"

#define ERR(format, msg...) printf(format "\n",## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif

/**
 * @memo Implementation of MAbstracListner for Bluetooth listerner using Bluez D-Bus
 * interface. 
 * @doc A listener for Bluetooth adapter, uses the default adapter reported by hcid.
 * This listener plugin can be used in normal Linux that has BlueZ libraries (including
 * development) installed. Paths,interfaces etc. of BlueZ D-Bus API are defined by hand,
 * doesn't use any BlueZ D-Bus-header (there isn't one?). Uses a private connection to
 * D-Bus. Currently only changes the state of the Bluetooth plugin if the actual state of
 * the device (removed or added) is changed.
 * 
 * @see MAbstractListener
 * @see MAbstractStateConverter
 */
class CBluezBTListener : public MAbstractListener
{
public:
	CBluezBTListener(MAbstractStateConverter *aConverter);
	virtual ~CBluezBTListener();
	
	// Listener interface
	bool Connect();
	void Disconnect();
	void CheckInitialState();
	void CheckState();
	const std::string& GetName();
	
private:
	CBluezBTListener();
	bool RegisterSignals();
	bool SetupConnection();
	void HandleMessage(DBusMessage* aMessage);
	const std::string IterateDBusMessage(DBusMessageIter *iter);
	void ParseAndSetAdapter(const std::string& aPath);
	void SetAdapterChange(const std::string& aPath);
	void CommitChanges();
	
	// Connection object
	DBusConnection* iConnection;

	// Name of the listener
	std::string iName;
	
	// Name of the default adapter
	std::string iAdapter;

	// Connection to "owner"
	MAbstractStateConverter* iConverter;
	
	// Whether using old bluez ( < 4.00)
	bool iOldBluez;
	
	//Message parsing
	std::string iDictKey;
	bool iDictParsing;
	
	// Things to react
	bool iChanges;
	bool iPowered;
	bool iDiscoverable;
	bool iPairable;
	bool iDiscovering;
	std::string iAdapterChange;
};

#endif /*__BLUEZBTLISTENER_H_*/
