/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name VirtualConnection.cc
 * @memo Virtual implementation of the MAbstractConnection interface.
 *
 * @version 0.2
 * date     07.01.2004
 * change   28.04.2010
 */

#include <cassert>
#include <sstream>
#include <stdio.h>
#include <iostream>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <assert.h>

#include "pdu.h"
#include "Factory.h"
#include "PeerHoodImpl.h"
#include "LibDevice.h"
#include "VirtualConnection.h"

//const char* LOCAL_SOCKET_NAME = "/tmp/phd.local";

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

using namespace std;

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * This method creates new technology specific connection object and points the
 * virtualconnection to it.
 *
 * @param aPrototype Prototype for new networking technology
 * @return none
 */
CVirtualConnection::CVirtualConnection(const std::string& aPrototype)
{
  iVirtualConnection = Factory::CreateConnectionL(aPrototype);
  iThreadInfo = NULL;
}

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * This method creates new technology specific connection object and points the
 * virtualconnection to it. Method also inserts connection information to iThreadList
 * 
 * @param aPrototype Prototype for new networking technology
 * @param aThreadInfo Pointer to ThreadInfo container
 * @return none
 */
CVirtualConnection::CVirtualConnection(const std::string& aPrototype, CThreadInfo* aThreadInfo)
{
  iVirtualConnection = Factory::CreateConnectionL(aPrototype);
  iThreadInfo = aThreadInfo;

  //list is locked, wait
  if(iThreadInfo->iInstance->iCopying)
    {
      sleep(1);
    }
  aThreadInfo->iInstance->iThreadList.push_back(aThreadInfo);
}

/**
 * @memo Default Destructor.
 * @doc Default destructor, removes the current connection from the iThreadList
 *
 * @return none
 */
CVirtualConnection::~CVirtualConnection()
{
  
  if(iThreadInfo)
	{
		//list is locked, wait
		if(iThreadInfo->iInstance->iCopying)
		{
			sleep(1);
		}
      
		for (list<CThreadInfo*>::iterator i = iThreadInfo->iInstance->iThreadList.begin();i != iThreadInfo->iInstance->iThreadList.end();++i){
			bool found = false;
			if((iThreadInfo->iDeviceName.compare((*i)->iDeviceName) == 0)
				&& (iThreadInfo->iServiceName.compare((*i)->iServiceName) == 0)
				&& (iThreadInfo->iPrototype.compare((*i)->iPrototype) == 0)
				&& (iThreadInfo->iAddress.compare((*i)->iAddress) == 0)
				&& (iThreadInfo->iPort == (*i)->iPort)
				&& (iThreadInfo->iConnectionId == (*i)->iConnectionId)
				&& (iThreadInfo->iChecksum == (*i)->iChecksum))
			{
				//might be smoother way to erase item?
				delete (*i);
				(*i)->iInstance->iThreadList.erase(i);
				found = true;
				break;
			}
		}
	}
  
  if(iVirtualConnection) delete iVirtualConnection;
}

/**
 * @memo Creates a network specific connection
 * @doc Creates a network specific connection. VirtualConnection
 * points to specific networking specific connection class
 * and calls the connect method.
 *
 * @param aAddress The destination address.
 * @param aPort Target port.
 *
 * @return true if the connection could be established
 */
bool CVirtualConnection::Connect(const std::string& aAddress, int aPort)
{
	if(!iVirtualConnection) return false;
	
  if(!iVirtualConnection->Connect(aAddress, aPort))
    return false;
  
  return true;
}

/**
 * @memo Kills the existing connection.
 * @doc Kills the existing connection. VirtualConnection 
 * points to specific networking specific connection class
 * and calls the disconnect method.
 *
 * @return true if the connection was dropped in a controlled manner
 */
bool CVirtualConnection::Disconnect()
{
	if(!iVirtualConnection) return true;
	
  if (!iVirtualConnection->IsConnected()) {
    return false;
  }
  iVirtualConnection->Disconnect();
  //iVirtualConnection->Close();
  return true;
}


/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CVirtualConnection::Close()
{
  if(iVirtualConnection) iVirtualConnection->Close();
}

/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. Note that this
 * function is not endianess-safe.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CVirtualConnection::Write(const void* aOutBuf, int aLength)
{
  int written = 0;
  
  if(!iVirtualConnection) return -1;
  
  written = iVirtualConnection->Write((void *)aOutBuf, aLength);
  return written;
}

/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection. Just like 
 * <code>Read</code> this function is not endianess-safe so it should be
 * treated more like a standard Posix sequential socket.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CVirtualConnection::Read(void* aInBuf, int aLength)
{
  int retval = 0;

	if(!iVirtualConnection) return -1;
	
  retval = iVirtualConnection->Read((void *)aInBuf, aLength);
  return retval;
   
}

/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const std::string& CVirtualConnection::GetRemoteAddress()
{  
	if(!iVirtualConnection) return NULL;
  return iVirtualConnection->GetRemoteAddress();
}

/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CVirtualConnection::GetFd()
{
	if(!iVirtualConnection) return -1;
  return iVirtualConnection->GetFd();
}

/**
 * @memo Sets the connection to the listening state. Not used currently.
 * @doc Sets the connection to the listening state. Not used currently.
 *
 * @return true always, except false when there is no connection.
 */
bool CVirtualConnection::Listen()
{
	if(!iVirtualConnection) return false;
  return true;
}

/**
 * @memo Sets the connection to the listening state. Not used currently. 
 * @doc Sets the connection to the listening state. Not used currently.
 *
 * @param aPort Not used currently. 
 *
 * @return true always, except false when there is no connection.
 */
bool CVirtualConnection::Listen(int aPort)
{
	if(!iVirtualConnection) return false;
  return true;
}

/**
 * @memo Accepts the incoming connection and returns a new connection object. Not used currently. 
 * @doc Accepts the incoming connection and returns a new connection object. Not used currently. 
 *
 * @return NULL always
 */
MAbstractConnection* CVirtualConnection::AcceptL()
{
  return NULL;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CVirtualConnection::IsListening()
{
	if(!iVirtualConnection) return false;
  return iVirtualConnection->IsListening();
}


/**
 * @memo Tells if the connection is established.
 * @doc Tells if the connection is established or not.
 *
 * @return true if the connection is established
 */
bool CVirtualConnection::IsConnected()
{
	if(!iVirtualConnection) return false;
  return iVirtualConnection->IsConnected();
}


/**
 * @memo Tells if the connection has data waiting
 * @doc Checks if the connection has data to read.
 *
 * @return true if the data is ready
 */
bool CVirtualConnection::HasData()
{
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
  fd_set set;
  FD_ZERO(&set);
  FD_SET(iVirtualConnection->GetFd(), &set);

	if(!iVirtualConnection) return false;

  switch (select(iVirtualConnection->GetFd() + 1, &set, NULL, NULL, &timeout)) {
  case -1:
    ERR("CVirtualConnection::HasData(): Select error");
    break;
    
  case 0:
    break;

  default:
    if (FD_ISSET(iVirtualConnection->GetFd(), &set)) {
      return true;
    }
    else {
      return false;
    }
  }
  return false;
}


/**
 * @memo Sets the virtual connection to point to a specific technology connection object.
 * @doc Sets the virtual connection to point to a specific technology connection object. This method
 * updates the connection information.
 *
 * @param aConnection Connection information which updates the old information
 * @param aPrototype New prototype which updates the old prototype
 * @param aAddress New address which updates the old address
 * @param aPort New port which updates the old port
 * @return none
 */
void CVirtualConnection::SetConnectionType(MAbstractConnection* aConnection, std::string& aPrototype, std::string& aAddress, int aPort)
{
  iVirtualConnection = aConnection;
  iThreadInfo->iLowCount = 0;
  bool found = false;

  //search our connection from the list and update prototype, port and address
  for (list<CThreadInfo*>::iterator i = iThreadInfo->iInstance->iThreadList.begin();i != iThreadInfo->iInstance->iThreadList.end();++i){
    if((iThreadInfo->iDeviceName.compare((*i)->iDeviceName) == 0)
      && (iThreadInfo->iServiceName.compare((*i)->iServiceName) == 0)
	&& (iThreadInfo->iPrototype.compare((*i)->iPrototype) == 0)
	&& (iThreadInfo->iAddress.compare((*i)->iAddress) == 0)
	&& (iThreadInfo->iPort == (*i)->iPort)
	&& (iThreadInfo->iConnectionId == (*i)->iConnectionId)
	&& (iThreadInfo->iChecksum == (*i)->iChecksum))
      
    //    if(memcmp((*i), iThreadInfo, sizeof(iThreadInfo)))
      {
	found = true;
	std::cerr << "CVirtualConnection::SetConnectionType : updated connection info" << std::endl;
	(*i)->iPrototype = aPrototype;
	(*i)->iPort = aPort;
	(*i)->iAddress = aAddress;
	break;
      }
  }
  if(!found)
    std::cerr << "CVirtualConnection::SetConnectionType : can't find connection info" << std::endl;
}

/**
 * @memo Sets the virtual connection to point to a specific technology connection object.
 * @doc Sets the virtual connection to point to a specific technology connection object. 
 * This method updates only the connection object which the virtulconnection points to.
 *
 * @return none
 */
void CVirtualConnection::SetConnectionType(MAbstractConnection* aConnection)
{
  iVirtualConnection = aConnection;
}

unsigned int CVirtualConnection::GetDeviceChecksum()
{
	if(iThreadInfo == NULL) return iVirtualConnection->GetDeviceChecksum();
	return iThreadInfo->iChecksum;
}

void CVirtualConnection::SetDeviceChecksum(unsigned int aChecksum)
{
	if(iVirtualConnection != NULL) iVirtualConnection->SetDeviceChecksum(aChecksum);
}

