/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name Service.cc
 * @memo Implementation of the PeerHood system's service class.
 *
 * @version 0.1
 * date     08.04.2003
 * change   22.04.2003
 */

#include <sstream>
#include <cassert>
#include <netinet/in.h>
#include "Service.h"
#include <pdu.h>
#include <string.h>

//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

#include <iostream>
using namespace std;

/**
 * @memo Copy constructor.
 * @doc Copy constructor. Creates an exact copy of the original object.
 *
 * @param aOriginal The original object to be copied.
 *
 * @return none
 */
CService::CService(const CService& aOriginal)
{
  iName = string(aOriginal.iName);
  iAttributes = string(aOriginal.iAttributes);
  iPort = aOriginal.iPort;
  iPid = aOriginal.iPid;
}

/**
 * @memo Constructor that creates the object from the given marshalled data.
 * @doc Constructor that creates the object from the given marshalled data.
 *
 * @param aData Marshalled data to be encoded. This parameter cannot be NULL.
 *
 * @return none
 */
CService::CService(const char* aData)
{
  assert(aData != NULL);

  TServiceData* data = (TServiceData *)aData;
  iAttributes = string(&aData[ntohs(data->iAttributePosition)]);
  iName = string(&aData[ntohs(data->iNamePosition)]);
  iPort = ntohs(data->iPort);
  iPid = ntohl(data->iPid);
}

/**
 * @memo Constructor that creates the object from the given marshalled data.
 * @doc Constructor that creates the object from the given marshalled data.
 *
 * @param aData Marshalled data to be encoded. This parameter cannot be NULL.
 *
 * @return none
 */
CService::CService(const char* aData, int aPid)
{
  assert(aData != NULL);

  TServiceData* data = (TServiceData *)aData;
  iAttributes = string(&aData[ntohs(data->iAttributePosition)]);
  iName = string(&aData[ntohs(data->iNamePosition)]);
  iPort = ntohs(data->iPort);
  iPid = ntohl(data->iPid);
}

/**
 * @memo Constructor that creates the object from the given marshalled data.
 * @doc Constructor that creates the object from the given marshalled data.
 * In addition, the service's connection prototype is set.
 *
 * @param aData Marshalled data to be encoded. This parameter cannot be NULL.
 * @param aPrototype Service's connection prototype.
 *
 * @return none
 */

CService::CService(const char* aData, const string& aPrototype)
{
  assert(aData != NULL);

  TServiceData* data = (TServiceData *)aData;
  iAttributes = string(&aData[ntohs(data->iAttributePosition)]);
  iName = string(&aData[ntohs(data->iNamePosition)]);
  iPort = ntohs(data->iPort);
  iPid = ntohl(data->iPid);
}

/**
 * @memo Constructor, sets the internal variables.
 * @doc Constructor, sets the internal variables.
 *
 * @param aName Service's name.
 * @param aAttributes Service's attributes. This is a string of arbitrary length.
 *
 * @return none
 */
/*
CService::CService(const string& aName, const string& aAttributes, int aPid)
{
  iAttributes = string(aAttributes);
  iName = string(aName);
  iPid = aPid; 
  iPort = aPid;
}
*/
/**
 * @memo Constructor, sets the internal variables.
 * @doc Constructor, sets the internal variables.
 *
 * @param aName Service's name.
 * @param aAttributes Service's attributes. This is a string of arbitrary length.
 * @param aPid Service's process id.
 * @param aPort Service's port.
 *
 * @return none
 */

CService::CService(const string& aName, const string& aAttributes, int aPid, unsigned short aPort)
{
  iAttributes = string(aAttributes);
  iName = string(aName);
  iPid = aPid; 
  iPort = aPort;
}

/**
 * @memo Returns service's attributes.
 * @doc Returns service's attributes. The format is "key value", with space 
 * between each key-value pair.
 *
 * @return service's attribute string
 */
const string& CService::GetAttributeString()
{
  return iAttributes;
}

/**
 * @memo Returns the value of one attribute.
 * @doc Returns the value of one attribute. If the given attribute is found
 * then its value is inserted to the first parameter. Otherwise the parameter
 * is left intact.
 *
 * @param aAttribute The asked attribute. If the asked attribute is found then
 * it is inserted into this parameter upon return.
 * 
 * @return true if the given attribute was found
 */
bool CService::GetAttribute(string& aAttribute)
{
  bool found = false;

  string::size_type position = iAttributes.find_first_of(" ");
  string::size_type lastPosition = iAttributes.find_first_not_of(" ", 0);

  while (position != string::npos || lastPosition != string::npos) {
    string temp = iAttributes.substr(lastPosition, position - lastPosition);
    if (found) {
      aAttribute = temp;
      if (!temp.empty()) return true;
      else return false;
    }
    if (temp.compare(aAttribute) == 0) {
      found = true;
    }
    lastPosition = iAttributes.find_first_not_of(" ", position);
    position = iAttributes.find_first_of(" ", lastPosition);
  }
  
  return false;
}


/**
 * @memo Returns service's name.
 * @doc Returns service's name. Each service should have a unique name. If two 
 * services with the same name exist then the behavior is undefined.
 *
 * @return service's name
 */
const string& CService::GetName()
{
  return iName;
}


/**
 * @memo Converts service's information to an octet stream.
 * @doc Converts service's information to an octet stream. Note that memory
 * allocated for the return value must be freed by the caller. The format of
 * the returned stream is defined in the TServiceData structure.
 *
 * @param aLength Integer that on return will hold the length of the created
 * octet stream.
 *
 * @return octet presentation of the service's information
 */
char* CService::MarshallL(int* aLength)
{
  *aLength = MarshalledLength();
  char* retval = new char[*aLength];
  TServiceData* data = (TServiceData *)retval;

  // fill the header
  data->iAttributePosition = sizeof(TServiceData);
  data->iNamePosition = data->iAttributePosition + iAttributes.length() + 1;
  data->iPort = iPort;
  data->iPid = iPid;

  // copy the payload
  strcpy(&retval[data->iAttributePosition], iAttributes.c_str());
  strcpy(&retval[data->iNamePosition], iName.c_str());

  //  DBG("Marshalled service: name %s", iName.c_str());
  //  DBG("Marshalled service: attr %s", iAttributes.c_str());
  //  DBG("Marshalled service: pid %d", data->iPid);
  //  DBG("Marshalled service: port %d", data->iPort);

  // convert to network byte order
  data->iAttributePosition = htons(data->iAttributePosition);
  data->iNamePosition = htons(data->iNamePosition);
  data->iPort = htons(data->iPort);
  data->iPid = htonl(data->iPid);

  return retval;  
}


/**
 * @memo Returns the number of octets needed to marshall the service.
 * @doc Returns the number of octets needed to marshall the service.
 *
 * @return the number of octets required to marshall the service
 */
int CService::MarshalledLength()
{
  int retval = sizeof(TServiceData);
  retval += iAttributes.length() + 1;
  retval += iName.length() + 1;
  return retval;
}


/**
 * @memo Returns the process id of the parent library
 * @doc Returns the process id of the parent library. On the remote side this
 * function always returns -1.
 *
 * @return the process id of the parent application
 */
int CService::GetPid()
{
  return iPid;
}

/**
 * @memo Returns the port of service's listening connection.
 * @doc Returns the port of service's listening connection.
 *
 * @return the port of service's listening connection
 */
unsigned short CService::GetPort()
{
  return iPort;
}


