/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/* 
 * @name PeerHoodCWrapper.cc
 * @memo 
 *
 * @version 0.31
 * date     12.11.2009
 * change   28.02.2010
 */

#include "PeerHood.h"

using namespace std;

// ----------- PH GENERIC ---------------------------------------------------//


/**
 * @memo Delete the PeerHood instance
 * @doc Deletes the given peerhood instance.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 *
 */
void ph_c_delete_peerhood(MPeerHood* _aPeerHood)
{
	delete _aPeerHood;
}

/**
 * @memo Method used to create a new instance of the PeerHood interface.
 * @doc Method used to create a new instance of the PeerHood interface. The
 * only way to create the instance is via this method. When this method is
 * called for the very first time a new instance of the interface is created.
 * Subsequent calls will return a pointer to the existing instance.
 *
 * @param _aCallback Pointer to the callback that will receive notifications
 * from the PeerHood library.
 *
 * @return pointer to a PeerHood instance
 */
MPeerHood* ph_c_get_instance(C_Callback* _aCallback)
{
	return MPeerHood::GetInstance(_aCallback);
}

/**
 * @memo Initializes the PeerHood instance.
 * @doc Initializes the PeerHood instance. These routines include connecting
 * to the PeerHood daemon and setting up the debug output. In addition, all
 * internal variables are initialized. This method should be called only
 * once.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aArgc The number of parameters.
 * @param _aArgv Array containing the parameters.
 *
 * @return 1 (TRUE) if the object was initialized succesfully
 */
int ph_c_init(MPeerHood* _aPeerHood,int _argc, char** _argv)
{
	return _aPeerHood->Init(_argc,_argv);
}


// ------------ LISTS -------------------------------------------------------//

/**
 * @memo Gets a list of all nearby devices and their services.
 * @doc Gets a list containing all nearby devices and their services and 
 * resources. Note that this function reserves the memory required by the
 * list and it's caller's responsibility to free it. Also note that the 
 * returned list contains <i>all</i> devices in range - inluding those
 * without PeerHood capability. If no devices are found then the returned
 * list is empty.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 *
 * @return a pointer to the list of found devices or NULL if an error occurred
 */
TDeviceList* ph_c_get_devicelist(MPeerHood* _aPeerHood)
{
	return _aPeerHood->GetDeviceListL();
}

/**
 * @memo Returns a list of devices that offer the asked service.
 * @doc This function builds and returns a list that contains all devices
 * that offer the requested service. Note that the caller must free the 
 * memory allocated for the returned list. If no devices are found then the
 * returned list will be empty.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aServiceName The service that should be looked for.
 *
 * @return a pointer to the list of devices that offer the requested service or
 * NULL if an error occurred
 */
TDeviceList* ph_c_get_devicelist_with_services(MPeerHood* _aPeerHood, const char* _aServiceName)
{
	string aServiceName = string(_aServiceName);
	return _aPeerHood->GetDeviceListL(&aServiceName);
}

/**
 * @memo Delete the list of devices
 * @doc Delete the given list of devices.
 *
 * @param _aDeviceList pointer to list to use
 */
void ph_c_delete_devicelist(TDeviceList* _aDeviceList)
{
	delete _aDeviceList;
}

/**
 * @memo Returns all locally registered services.
 * @doc Returns all locally registered services on a list. The memory
 * allocated for the returned list is not freed automatically so the caller
 * must take care of it. If no services are registered then the returned list
 * will be empty.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 *
 * @return a list of locally registered services or NULL in the case of an
 * error
 */
TServiceList* ph_c_get_localservicelist(MPeerHood* _aPeerHood)
{
	return _aPeerHood->GetLocalServiceListL();
}

/**
 * @memo Delete the list of services
 * @doc Delete the given list of services.
 *
 * @param _aServiceList pointer to list to use
 */
void ph_c_delete_servicelist(TServiceList* _aServiceList)
{
	delete _aServiceList;
}


//---------------- CONNECT TO -----------------------------------------------//

/**
 * @memo Creates a connection to a local service.
 * @doc Creates a connetion to a local service. 
 * Destination address and technology prototype are taken from the parameters.
 * If a connection object is returned then it's caller's responsibility to
 * delete it in a controlled way. For supporting C-API of PeerHood.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aService The service to connect to. CService class objects can be used
 * directly.
 *
 * @return a new connection object or NULL if an error happened
 */
MAbstractConnection* ph_c_connect_localservice(MPeerHood* _aPeerHood, CService* _aService)
{
	return _aPeerHood->Connect(*_aService);
}

/**
 * @memo Creates a connection to a service on another PeerHood capable device.
 * @doc Creates a connetion to a service on another PeerHood capable device. 
 * Destination address and technology prototype are taken from the parameters.
 * If a connection object is returned then it's caller's responsibility to
 * delete it in a controlled way.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aDevice The remote device.
 * @param _aServiceName Remote service's name.
 *
 * @return a new connection object or NULL if an error happened
 */
MAbstractConnection* ph_c_connect_remoteservice(MPeerHood* _aPeerHood, MAbstractDevice* _aDevice, const char* _aServiceName)
{
	const string aServiceName = _aServiceName;
	return _aPeerHood->Connect(*_aDevice,aServiceName);
}


//-------------- REGISTER SERVICE -------------------------------------------//

/**
 * @memo Registers a service so that other PeerHood devices can find it. 
 * @doc Registers a service so that other PeerHood devices can find and use 
 * it. This method contacts the PeerHood daemon that in turns starts to 
 * advert the service through its currently running plugins.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aName The name of the service.
 * @param _aAttributes Service's attributes in one string.
 *
 * @return port number if the service could be registered, otherwise 0
 */
unsigned short ph_c_register_service(MPeerHood* _aPeerHood,const char* _aName, const char* _aAttributes)
{
	return _aPeerHood->RegisterService(string(_aName), string(_aAttributes));
}

/**
 * @memo Registers a service so that other PeerHood devices can find it. 
 * @doc Registers a service so that other PeerHood devices can find and use 
 * it. This method contacts the PeerHood daemon that in turns starts to 
 * advert the service through its currently running plugins.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aName The name of the service.
 * @param _aAttributes Service's attributes in one string.
 * @param _aPort Service's port.
 *
 * @return port number if the service could be registered, otherwise 0
 */
unsigned short ph_c_register_service_with_port(MPeerHood* _aPeerHood,const char* _aName, const char* _aAttributes, const char* _aPort)
{
	return _aPeerHood->RegisterService(string(_aName), string(_aAttributes), string(_aPort));
}

/**
 * @memo  Unregisters a previously registered service.
 * @doc Unregisters a previously registered service. After unregistration
 * other devices are unable to find and call the unregistered service. Note
 * that the unregistration procedure doesn't delete the service object so
 * this should be done by the actual server application.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aName The name of the service to be unregistered.
 *
 * @return 1 (TRUE) if the service could be unregistered
 */
int ph_c_unregister_service(MPeerHood* _aPeerHood, const char* _aName)
{
	return _aPeerHood->UnregisterService(string(_aName)) ? 1 : 0 ;
}

/**
 * @memo  Unregisters a previously registered service.
 * @doc Unregisters a previously registered service. After unregistration
 * other devices are unable to find and call the unregistered service. Note
 * that the unregistration procedure doesn't delete the service object so
 * this should be done by the actual server application.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aName The name of the service to be unregistered.
 * @param _aPort Service's port.
 *
 * @return 1 (TRUE) if the service could be unregistered
 */
int ph_c_unregister_service_with_port(MPeerHood* _aPeerHood, const char* _aName, const char* _aPort)
{
	return _aPeerHood->UnregisterService(string(_aName), string(_aPort)) ? 1 : 0 ;
}


// ----------- MONITORING --------------------------------------------------//

/**
 * @memo Sets a device under constant monitoring.
 * @doc Sets a device under constant monitoring. If a change (out of range, 
 * back in range) takes place then the registered callback interface is 
 * notified. An application must derive from the <code>CBasicCallback</code>
 * class and implement the defined methods in order to receive callback
 * events.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aDevice The device that should be monitored.
 *
 * @return 1 (TRUE) if the monitoring could be started
 */
int ph_c_monitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice)
{
	return _aPeerHood->MonitorDevice(*_aDevice) ? 1 : 0 ;
}

/**
 * @memo Stops the monitoring of a device.
 * @doc Stops the monitoring of a device. After this function is called the
 * given device is no longer monitored.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aDevice The target device.
 *
 * @return 1 (TRUE) if the monitoring could be canceled succesfully
 */
int ph_c_unmonitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice)
{
	return _aPeerHood->UnmonitorDevice(*_aDevice) ? 1 : 0 ;
}

/**
 * @memo Sets a device under constant monitoring using signal-level monitoring.
 * @doc Sets a device under constant monitoring using signal-level monitoring.
 * If a change (out of range, 
 * back in range) takes place then the registered callback interface is 
 * notified (Not currently used). An application must derive from the <code>CBasicCallback</code>
 * class and implement the defined methods in order to receive callback
 * events.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aDevice The device that should be monitored.
 *
 * @return 1 (TRUE) if the monitoring could be started
 */
int ph_c_signal_monitor_device(MPeerHood* _aPeerHood, TDeviceIterator* _aDevice)
{
	return _aPeerHood->SignalMonitorDevice(*_aDevice) ? 1 : 0 ;
}

/**
 * @memo Stops the signal-level monitoring of a device.
 * @doc Stops the signal-level monitoring of a device. After this function is called the
 * given device (what device?) is no longer monitored.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 *
 * @return 1 (TRUE) if the monitoring could be canceled succesfully
 */
int ph_c_signal_unmonitor_device(MPeerHood* _aPeerHood)
{
	return _aPeerHood->SignalUnmonitorDevice() ? 1 : 0 ;
}

/**
 * @memo Sets the plugin prefered by the current application (Not used currently).
 * @doc Sets the plugin prefered by the current application. This means that 
 * PeerHood will try to use the given plugin whenever possible. This method 
 * will override the value read from the configuration file. The prefered
 * plugin can be changed during runtime. However, it affects only the actions
 * performed after the call i.e. the running services are not affected.
 *
 * @param _aPeerHood pointer to MPeerHood instance to use.
 * @param _aPluginName The name of the prefered plugin.
 */
void ph_c_set_prefered_plugin(MPeerHood* _aPeerHood, const char* _aPluginName)
{
	_aPeerHood->SetPreferedPlugin(_aPluginName);
}


// --------- DEVICELIST ----------------------------------------------------------------------//

/**
 * @memo Check if the device list is empty
 * @doc Calls the Empty() -function of the list to check if the list is empty.
 *
 * @param _aDeviceList pointer to devicelist to use.
 *
 * @return 1 (TRUE) if list is empty
 */
int ph_c_devicelist_is_empty(TDeviceList* _aDeviceList)
{
	return _aDeviceList->Empty() ? 1 : 0 ;
}

/**
 * @memo Get the size of the list
 * @doc Get the amount of devices in the list
 *
 * @param _aDeviceList pointer to list to use.
 *
 * @return the size of the list
 */
int ph_c_devicelist_size(TDeviceList* _aDeviceList)
{
	return _aDeviceList->Size();
}

/**
 * @memo Check if the service list is empty
 * @doc Calls the Empty() -function of the list to check if the list is empty.
 *
 * @param _aServiceList pointer to servicelist to use.
 *
 * @return 1 (TRUE) if list is empty
 */
int ph_c_servicelist_is_empty(TServiceList* _aServiceList)
{
	return _aServiceList->Empty() ? 1 : 0 ;
}

/**
 * @memo Get the size of the list
 * @doc Get the amount of services in the list
 *
 * @param _aServiceList pointer to list to use.
 *
 * @return the size of the list
 */
int ph_c_servicelist_size(TServiceList* _aServiceList)
{
	return _aServiceList->Size();
}
